SET QUOTED_IDENTIFIER ON
GO

SET NOCOUNT ON
GO

SET NOEXEC OFF
SET NOCOUNT ON

IF NOT EXISTS
  (SELECT TOP 1 1
     FROM SYS.objects [SO]
    WHERE [SO].[name] = 'kyNoExec')
  BEGIN
    CREATE TABLE [dbo].[kyNoExec] ([NoExec] BIT)
    INSERT INTO [dbo].[kyNoExec] ([NoExec]) VALUES(0)
  END
GO

UPDATE [dbo].[kyNoExec] SET [NoExec] = 0
GO

SET NOCOUNT OFF
GO

SET NOCOUNT ON
GO

DECLARE @dbname NVARCHAR(128)

SET @dbname = DB_NAME()

IF @dbname IN ('master', 'tempdb', 'model', 'msdb')
  BEGIN
    SELECT 'do not run this against a system database'
    SET NOCOUNT ON
    UPDATE [dbo].[kyNoExec] SET [NoExec] = 1
    SET NOCOUNT OFF
    SET NOEXEC ON
  END

IF @dbname LIKE ('ReportServer$%')
  BEGIN
    SELECT 'do not run this against a reporting server database'
    SET NOCOUNT ON
    UPDATE [dbo].[kyNoExec] SET [NoExec] = 1
    SET NOCOUNT OFF
    SET NOEXEC ON
  END

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[databases] [DB]
    WHERE [DB].[name] = @dbname
      AND [DB].[is_distributor] = 1)
  BEGIN
    SELECT 'do not run this against a distributor database'
    SET NOCOUNT ON
    UPDATE [dbo].[kyNoExec] SET [NoExec] = 1
    SET NOCOUNT OFF
    SET NOEXEC ON
  END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETFNSanitiseProc'
      AND SO.[type] = 'FN')
  BEGIN
    DROP FUNCTION [dbo].[ky_NETFNSanitiseProc]
  END
GO

CREATE FUNCTION
       [dbo].[ky_NETFNSanitiseProc]
      (@ORG NVARCHAR(MAX))
RETURNS NVARCHAR(MAX)
AS
/***************************************************************************************
 *
 * [dbo].[ky_NETFNSanitiseProc]
 *
 * Replaces tabs with four spaces, and removes trailing blanks from lines in the code
 * for SQL procedures, functions etc
 *
 * Modification History
 * 2015-09-01    Pino Carafa             Created
 * 2015-09-02    Pino Carafa             Remove leading and trailing newlines
 *
 ***************************************************************************************/
BEGIN
  SET @ORG = REPLACE(@ORG, NCHAR(9), N'    ')

  SET @ORG = REPLACE(@ORG, NCHAR(13) + NCHAR(10), NCHAR(13))

  SET @ORG = REPLACE(@ORG, NCHAR(10), NCHAR(13))

  WHILE PATINDEX(N'% ' + NCHAR(13) + '%', @ORG) > 0
    BEGIN
      SET @ORG = REPLACE(@ORG, N' ' + NCHAR(13), NCHAR(13))
    END

  SET @ORG = REPLACE(@ORG, NCHAR(13), NCHAR(13) + NCHAR(10))

  WHILE @ORG LIKE NCHAR(13) + NCHAR(10) + '%'
    SET @ORG = SUBSTRING(@ORG, 3, LEN(@ORG) - 2)

  WHILE @ORG LIKE '%' + NCHAR(13) + NCHAR(10)
    SET @ORG = SUBSTRING(@ORG, 1, LEN(@ORG) - 2)

  RETURN @ORG
END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETFNGetScript'
      AND SO.[type] = 'FN')
  BEGIN
    DROP FUNCTION [dbo].[ky_NETFNGetScript]
  END
GO

CREATE FUNCTION
       [dbo].[ky_NETFNGetScript]
      (@rewriteproc      SYSNAME,
       @deleteifexist    BIT,
       @checkifnotexist  BIT)
RETURNS NVARCHAR(MAX)
AS
/***************************************************************************************
 *
 * [dbo].[ky_NETFNGetScript]
 *
 * Gets the code to rewrite a SQL Procedure, Function or View. Also replaces tabs
 * with four blanks and removes trailing blanks from lines.
 *
 * Modification History
 * 2015-09-01    Pino Carafa             Created
 * 2015-09-02    Pino Carafa             Modified ky_NETFNSanitiseProc to remove
 *                                       leading and trailing newlines. Added final
 *                                       newline after final GO.
 * 2019-09-12    Pino Carafa             Added TRIGGER to IF EXIST statement
 *
 ***************************************************************************************/
BEGIN

DECLARE @NCOMMAND NVARCHAR(MAX)

SET @NCOMMAND = N''

IF @deleteifexist = 1
  BEGIN
    SELECT @NCOMMAND = @NCOMMAND + N'IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N''' + SO.[name] + '''
      AND SO.[type] = ''' + RTRIM(CONVERT(NVARCHAR(10), SO.[type])) + N''')
  BEGIN
    DROP ' + CASE SO.[type]
                  WHEN 'P' THEN N'PROCEDURE'
                  WHEN 'FN' THEN N'FUNCTION'
                  WHEN 'TF' THEN N'FUNCTION'
                  WHEN 'IF' THEN N'FUNCTION'
                  WHEN 'TR' THEN N'TRIGGER'
                  WHEN 'V' THEN N'VIEW'
                  ELSE N'UNKNOWN' END + N' [dbo].[' + SO.[name] + ']
  END
GO
'
      from SYS.[objects] SO
           LEFT OUTER JOIN SYS.[sql_modules] SM
                        ON SM.[object_id] = SO.[object_id]
     where so.name = @rewriteproc
  END

IF @checkifnotexist = 1
  BEGIN
    SELECT @NCOMMAND = @NCOMMAND + N'
IF NOT EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N''' + SO.[name] + '''
      AND SO.[type] = ''' + RTRIM(CONVERT(NVARCHAR(10), SO.[type])) + N''')
  BEGIN
EXEC (''' + REPLACE([dbo].[ky_NETFNSanitiseProc](SM.[definition]), N'''', N'''''') + N'
'')
  END
'
      from SYS.[objects] SO
           LEFT OUTER JOIN SYS.[sql_modules] SM
                        ON SM.[object_id] = SO.[object_id]
     where so.name = @rewriteproc
  END
ELSE
  BEGIN
    SELECT @NCOMMAND = @NCOMMAND + N'
' + [dbo].[ky_NETFNSanitiseProc](SM.[definition]) + N'
'
      from SYS.[objects] SO
           LEFT OUTER JOIN SYS.[sql_modules] SM
                        ON SM.[object_id] = SO.[object_id]
     where so.name = @rewriteproc
  END

SELECT @NCOMMAND = @NCOMMAND + '
GO
'
  from SYS.[objects] SO
       LEFT OUTER JOIN SYS.[sql_modules] SM
                    ON SM.[object_id] = SO.[object_id]
 where so.name = @rewriteproc

RETURN @NCOMMAND

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPGetScript'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPGetScript]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETSPGetScript]
      (@rewriteproc SYSNAME)
AS
/***************************************************************************************
 *
 * [dbo].[ky_NETSPGetScript]
 *
 * Gets the code to rewrite a SQL Procedure, Function or View. Also replaces tabs
 * with four blanks and removes trailing blanks from lines.
 *
 * Modification History
 * 2015-09-01    Pino Carafa             Created
 * 2015-09-02    Pino Carafa             Modified ky_NETFNSanitiseProc to remove
 *                                       leading and trailing newlines. Added final
 *                                       newline after final GO.
 * 2019-09-12    Pino Carafa             Support for Triggers and Large Procedures
 *
 ***************************************************************************************/
BEGIN

SET NOCOUNT ON

DECLARE @NCOMMAND NVARCHAR(MAX)
DECLARE @ATTEMPTS INT

SET @ATTEMPTS = 200

SELECT @NCOMMAND = [dbo].[ky_NETFNGetScript](@rewriteproc, 1, 0)

DECLARE @START INT
DECLARE @POS   INT
DECLARE @LEN   INT
DECLARE @CRLF  NVARCHAR(4)

DECLARE @LINES TABLE
       ([id]         INT           IDENTITY(1, 1)            PRIMARY KEY,
        [LINE]       NVARCHAR(MAX)                           NOT NULL)

SET @LEN = LEN(@NCOMMAND)

SET @CRLF = NCHAR(13)

IF LEN(@NCOMMAND) > 0
  BEGIN
    SET @START = 1
    SET @POS = CHARINDEX(@CRLF, @NCOMMAND, @START)
  END
ELSE
  BEGIN
    SET @START = 0
    SET @POS   = 0
  END

IF @POS > 0
  BEGIN
    INSERT
      INTO @LINES
          ([LINE])
    VALUES(SUBSTRING(@NCOMMAND, @START, @POS - @START))
    --SELECT @START, @POS, @LEN, SUBSTRING(@NCOMMAND, @START, @POS - @START)
  END

WHILE @POS > 0
  BEGIN
    SET @START = @POS + 2
    SET @POS = CHARINDEX(@CRLF, @NCOMMAND, @START)

    IF @POS > 0
      BEGIN
        INSERT
          INTO @LINES
              ([LINE])
        VALUES(SUBSTRING(@NCOMMAND, @START, @POS - @START))
        --SELECT @START, @POS, @LEN, SUBSTRING(@NCOMMAND, @START, @POS - @START)
      END
  END

DECLARE @LINE NVARCHAR(MAX)
SELECT TOP 1 @LINE = [LINE] FROM @LINES ORDER BY [id] DESC

IF @LINE <> N''
  BEGIN
    INSERT
      INTO @LINES
          ([LINE])
    VALUES(N'')
  END

  SELECT [LINE]
    FROM @LINES
ORDER BY [id]

SET NOCOUNT OFF
END

GO

IF NOT EXISTS
  (SELECT TOP 1 1
     FROM sys.types [ST]
    WHERE [ST].[name] = 'MatterSel')
  BEGIN
CREATE 
  TYPE MatterSel
    AS TABLE
      ([CASECODE]                      VARCHAR(20)                         PRIMARY KEY      NOT NULL)
  END
GO

IF NOT EXISTS
  (SELECT TOP 1 1
     FROM sys.types [ST]
    WHERE [ST].[name] = 'TrackRefsND')
  BEGIN
CREATE 
  TYPE TrackRefsND
    AS TABLE
      ([TrackReference]                INT                                 PRIMARY KEY      NOT NULL)
  END
GO

IF NOT EXISTS
  (SELECT TOP 1 1
     FROM sys.types [ST]
    WHERE [ST].[name] = 'TrackRefsTable')
  BEGIN
CREATE 
  TYPE TrackRefsTable
    AS TABLE
      ([id]                            INT              IDENTITY(1, 1)     PRIMARY KEY      NOT NULL,
       [TrackReference]                INT                                                  NOT NULL,
       [RESID]                         INT                                                  NOT NULL,
       [SORTER]                        VARCHAR(50)                                          NOT NULL,
       UNIQUE NONCLUSTERED ([SORTER], [id]))
  END
GO

IF NOT EXISTS
  (SELECT TOP 1 1
     FROM sys.types [ST]
    WHERE [ST].[name] = 'DocRes2Table')
  BEGIN
CREATE 
  TYPE DocRes2Table
    AS TABLE
      ([id]                       INT               IDENTITY(1, 1)        PRIMARY KEY,
       [CASECODE]                 VARCHAR(20),
       [MatterUniqueID]           INT,
       [ActionId]                 INT,
       [TrackReference]           INT,
       [IMDocID]                  VARCHAR(500),
       [IMClass]                  VARCHAR(20),
       [DocClass]                 VARCHAR(10),
       [Document]                 VARCHAR(2000),
       [DiaryDate]                DATETIME,
       [DocumentDate]             DATETIME,
       [TYPE]                     VARCHAR(20),
       [Attachments]              CHAR(1),
       [EMAIL]                    VARCHAR(1),
       [EMAILFROM]                VARCHAR(200),
       [EMAILSENT]                DATETIME,
       [EMAILTO]                  VARCHAR(1000),
       [CurrentVersion]           INT,
       [IsInDocFolder]            BIT,
       [IsShared]                 BIT)
  END
GO

--This type should no longer be used. Superseded by DocRes2Table
IF NOT EXISTS
  (SELECT TOP 1 1
     FROM sys.types [ST]
    WHERE [ST].[name] = 'DocResTable')
  BEGIN
CREATE 
  TYPE DocResTable
    AS TABLE
      ([id]                       INT               IDENTITY(1, 1)        PRIMARY KEY,
       [ActionId]                 INT,
       [TrackReference]           INT,
       [IMDocID]                  VARCHAR(500),
       [IMClass]                  VARCHAR(20),
       [DocClass]                 VARCHAR(10),
       [Document]                 VARCHAR(2000),
       [DiaryDate]                DATETIME,
       [DocumentDate]             DATETIME,
       [TYPE]                     VARCHAR(20),
       [Attachments]              CHAR(1),
       [EMAIL]                    VARCHAR(1),
       [EMAILFROM]                VARCHAR(200),
       [EMAILSENT]                DATETIME,
       [EMAILTO]                  VARCHAR(1000),
       [CurrentVersion]           INT,
       [IsInDocFolder]            BIT,
       [IsShared]                 BIT)
  END
GO

IF NOT EXISTS
  (SELECT TOP 1 1
     FROM sys.types [ST]
    WHERE [ST].[name] = 'IMDocTable')
  BEGIN
CREATE 
  TYPE IMDocTable
    AS TABLE
      ([IMDocID]           VARCHAR(500)        PRIMARY KEY,
       [Description]       VARCHAR(2000),
       [IMClass]           VARCHAR(20),
       [IMType]            VARCHAR(20),
       [EmailFrom]         VARCHAR(200),
       [EmailTo]           VARCHAR(1000),
       [EmailSent]         DATETIME,
       [DocumentDate]      DATETIME)
  END
GO

IF NOT EXISTS
  (SELECT TOP 1 1
     FROM sys.types [ST]
    WHERE [ST].[name] = 'StringList')
  BEGIN
CREATE 
  TYPE StringList
    AS TABLE
      ([String]                VARCHAR(50)                                 PRIMARY KEY      NOT NULL)
  END
GO

IF NOT EXISTS
  (SELECT TOP 1 1
     FROM sys.types [ST]
    WHERE [ST].[name] = 'IntegerList')
  BEGIN
CREATE 
  TYPE IntegerList
    AS TABLE
      ([Integer]               INT                                         PRIMARY KEY      NOT NULL)
  END
GO

IF NOT EXISTS
  (SELECT TOP 1 1
     FROM sys.types [ST]
    WHERE [ST].[name] = 'NTEID')
  BEGIN
CREATE 
  TYPE [NTEID]
    AS TABLE
      ([id]                            INT              IDENTITY(1, 1)     PRIMARY KEY      NOT NULL,
       [NEWID]                         INT                                                      NULL,
       [RECORDID]                      INT                                                      NULL)
  END
GO

IF NOT EXISTS
  (SELECT TOP 1 1
     FROM sys.triggers [ST]
    WHERE [ST].[name] = 'LOG_DROP_TABLE'
      AND [ST].[type] = 'TR'
      AND [ST].[parent_class] = 0)
BEGIN

EXEC ('Create TRIGGER [LOG_DROP_TABLE]
ON Database
for DROP_Table
AS
BEGIN
  IF NOT EXISTS
    (SELECT TOP 1 1
       FROM SYS.objects [SO]
      WHERE [SO].[name] = ''DROPTABLEINVOKED''
        AND [SO].[type] = ''U'')
    BEGIN
CREATE
 TABLE [dbo].[DROPTABLEINVOKED]
      ([TABLE DROPPED]     NVARCHAR(260),
       [DROPPED ON]        DATETIME,
       [hostname]          NVARCHAR(128),
       [program_name]      NVARCHAR(128),
       [cmd]               NVARCHAR(128),
       [nt_domain]         NVARCHAR(128),
       [nt_username]       NVARCHAR(128),
       [loginame]          NVARCHAR(128))
    END

  DECLARE @SCHEMADROPPED NVARCHAR(128)
  DECLARE @TABLEDROPPED NVARCHAR(128)

  SET @SCHEMADROPPED = quotename(eventdata().value(''(/EVENT_INSTANCE/SchemaName)[1]'', ''sysname''))
  SET @TABLEDROPPED = quotename(eventdata().value(''(/EVENT_INSTANCE/ObjectName)[1]'', ''sysname''))

  DECLARE @COMMAND NVARCHAR(MAX)

  SET @COMMAND = N''
INSERT
  INTO [dbo].[DROPTABLEINVOKED]
SELECT '''''' + @SCHEMADROPPED + ''.'' + @TABLEDROPPED + '''''' AS [TABLE DROPPED],
       GETDATE()                                                AS [DROPPED ON],
       CONVERT(NVARCHAR(128), [SP].[hostname])                  AS [hostname],
       CONVERT(NVARCHAR(128), [SP].[program_name])              AS [program_name],
       CONVERT(NVARCHAR(128), [SP].[cmd])                       AS [cmd],
       CONVERT(NVARCHAR(128), [SP].[nt_domain])                 AS [nt_domain],
       CONVERT(NVARCHAR(128), [SP].[nt_username])               AS [nt_username],
       CONVERT(NVARCHAR(128), [SP].[loginame])                  AS [loginame]
  FROM sys.sysprocesses [SP]
 WHERE [SP].[spid] = @@SPID''

 EXEC (@COMMAND)
End')

EXEC ('ENABLE TRIGGER [LOG_DROP_TABLE] ON DATABASE')

END

GO

--Please note that this function cannot be deleted as it will be used in a Schema Bound View
IF NOT EXISTS
  (SELECT TOP 1 1
     FROM SYS.objects [SO]
    WHERE [SO].[name] = 'ky_RemoveSpuriousWhitespaceSB'
      AND [SO].[type] = 'FN')
  BEGIN
    EXEC ('/**************************************************************************************
 *
 * ky_RemoveSpuriousWhitespaceSB
 *
 * Removes newlines, line feeds, non-breaking spaces and multiple spaces from the input
 *   string in order to produce an output format suitable for display in lists.
 *
 * Schema Bound version that can be used in Indexed Schema Bound Views
 *
 **************************************************************************************/
create function
       [dbo].[ky_RemoveSpuriousWhitespaceSB]
      (@original varchar(MAX),
       @maxlen   int)
returns varchar(5000)
WITH SCHEMABINDING
AS
BEGIN
  DECLARE @len int

  SET @original = ISNULL(@original, '''')
  IF LEN(@original) > (2 * @maxlen)
    BEGIN
      SET @original = SUBSTRING(@original, 1, (2 * @maxlen))
    END

  SET @original = replace(@original, char(10), '' '')
  SET @original = replace(@original, char(13), '' '')
  SET @original = replace(@original, char(160), '' '')

  SET @original = RTRIM(@original)

  SET @len = 0
    WHILE (@len <> len(@original))
      BEGIN
        SET @len = len(@original)
        SET @original = replace(@original, ''  '', '' '')
      END

  RETURN substring(@original, 1, @maxlen)

END')
  END
GO

--Please note that this function cannot be deleted as it will be used in a Schema Bound View
IF NOT EXISTS
  (SELECT TOP 1 1
     FROM SYS.objects [SO]
    WHERE [SO].[name] = 'ky_NETFNGetCardinalityDate112'
      AND [SO].[type] = 'FN')
  BEGIN
    EXEC ('CREATE FUNCTION
       [dbo].[ky_NETFNGetCardinalityDate112]
      (@date DATETIME)
RETURNS VARCHAR(8)
WITH SCHEMABINDING
AS
BEGIN

DECLARE @YEAR VARCHAR(4)
DECLARE @MONTH VARCHAR(2)
DECLARE @DAY VARCHAR(2)

IF @date IS NULL
  BEGIN
    RETURN CONVERT(VARCHAR(8), '''')
  END

SET @YEAR = SUBSTRING(CONVERT(VARCHAR(5), 10000 + YEAR(@date)), 2, 4)
SET @MONTH = SUBSTRING(CONVERT(VARCHAR(3), 100 + MONTH(@date)), 2, 2)
SET @DAY = SUBSTRING(CONVERT(VARCHAR(3), 100 + DAY(@date)), 2, 2)

RETURN CONVERT(VARCHAR(8), @YEAR + @MONTH + @DAY)

END')
  END

GO

--Please note that the following function cannot be deleted as it's used in a Schema Bound View
IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETFNConvertClarionTimeToHHMM'
      AND SO.[type] = 'FN')
  BEGIN
    SET NOEXEC ON
  END
GO

create function
       [dbo].[ky_NETFNConvertClarionTimeToHHMM]
      (@ClarionTime INT)
returns VARCHAR(5)
AS
/**************************************************************************************
 *
 * ky_NETFNConvertClarionTimeToHHMM
 *
 * converts an integer value between 1 and 8640000, representing a Clarion Time Value
 * in 1/100 of a Second since Midnight to a Time representation in HH:MM format
 *
 * This function prevents Cardinality Estimate Warnings when used in SELECT statements
 *
 **************************************************************************************/
BEGIN

DECLARE @HHMM   VARCHAR(5)
DECLARE @HOUR   INT
DECLARE @MINUTE INT

IF @ClarionTime IS NULL
  BEGIN
    SET @HHMM = ''
  END
ELSE IF @ClarionTime < 1
  BEGIN
    SET @HHMM = ''
  END
ELSE IF @ClarionTime > 8640000
  BEGIN
    SET @HHMM = ''
  END
ELSE
  BEGIN
    SET @ClarionTime = @ClarionTime - 1

    SET @HOUR = FLOOR(@ClarionTime / 360000)

    SET @ClarionTime = @ClarionTime - (@HOUR * 360000)

    SET @MINUTE = FLOOR(@ClarionTime / 6000)

    SET @HHMM = SUBSTRING(CONVERT(VARCHAR(3), 100 + @HOUR), 2, 2) + ':' + SUBSTRING(CONVERT(VARCHAR(3), 100 + @MINUTE), 2, 2)
  END

RETURN @HHMM

END

GO

SET NOEXEC OFF
IF EXISTS
  (SELECT TOP 1 1
     FROM [dbo].[kyNoExec]
    WHERE [NoExec] = 1)
  BEGIN
    --Noexec set script wide
    SET NOEXEC ON
  END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETFNConvertDateToUTC'
      AND SO.[type] = 'FN')
  BEGIN
    DROP FUNCTION [dbo].[ky_NETFNConvertDateToUTC]
  END
GO

DECLARE @Compatible BIT

--AT TIME ZONE is only available in SQL Azure or in SQL 2016 or later.
--We'll create the proper Function for those editions.
--For anything else, we'll create a version that assumes GMT Standard Time
--and works on the basis of "Last Sunday in March to Last Sunday in October"
--and good luck to anyone who is still on any such version when the EU
--abandons Daylight Saving in 2021 or at least we can defer any further
--modifications to this until then.
SET @Compatible = CASE WHEN ServerProperty('EngineEdition') = 5
                            THEN 1
                       WHEN @@VERSION LIKE '%2008%'
                            THEN 0
                       WHEN @@VERSION LIKE '%2012%'
                            THEN 0
                       WHEN @@VERSION LIKE '%2014%'
                            THEN 0
                       ELSE 1 END

IF @Compatible = 1
  BEGIN
    EXEC ('CREATE FUNCTION
       [dbo].[ky_NETFNConvertDateToUTC]
      (@Date            DATETIME,
       @TimeZone        VARCHAR(50))
RETURNS DATETIME
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETFNConvertDateToUTC]                                                                          *
 * Converts a DATETIME originally entered On Premise back to its correct UTC equivalent                      *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * Original:                                -                                                        *   *
 *   * First compatible version:                5.7.2.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification history                                                                                      *
 *                                                                                                           *
 * Pino Carafa       2019-07-03 Created                                                                      *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

SET @TimeZone = CASE WHEN RTRIM(ISNULL(@TimeZone, '''')) = ''''
                          THEN ''GMT Standard Time''
                     ELSE @TimeZone END

DECLARE @Original  DATETIMEOFFSET
DECLARE @Converted DATETIMEOFFSET

SET @Original = @Date AT TIME ZONE @TimeZone

SET @Converted = CONVERT(DATETIMEOFFSET, @Original) AT TIME ZONE ''UTC''

RETURN CONVERT(DATETIME, @Converted)

END')
  END
ELSE
  BEGIN
    EXEC ('CREATE FUNCTION
       [dbo].[ky_NETFNConvertDateToUTC]
      (@Date            DATETIME,
       @TimeZone        VARCHAR(50))
RETURNS DATETIME
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETFNConvertDateToUTC]                                                                          *
 * Converts a DATETIME originally entered On Premise back to its correct UTC equivalent                      *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * Original:                                -                                                        *   *
 *   * First compatible version:                5.7.2.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification history                                                                                      *
 *                                                                                                           *
 * Pino Carafa       2019-07-03 Created                                                                      *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

DECLARE @YEAR INT
DECLARE @BSTFROM DATETIME
DECLARE @BSTTO   DATETIME

SET @YEAR = YEAR(@Date)

DECLARE @DATENR INT

SET @DATENR = (10000 * @YEAR) + 331

--Last Sunday in March
WHILE DATENAME(WEEKDAY, CONVERT(VARCHAR(8), @DATENR)) <> ''Sunday''
  BEGIN
    SET @DATENR = @DATENR - 1
  END

SET @BSTFROM = CONVERT(DATETIME, CONVERT(VARCHAR(8), @DATENR))

SET @DATENR = (10000 * @YEAR) + 1031

--Last Sunday in October
WHILE DATENAME(WEEKDAY, CONVERT(VARCHAR(8), @DATENR)) <> ''Sunday''
  BEGIN
    SET @DATENR = @DATENR - 1
  END

SET @BSTTO = DATEADD(DAY, 1, CONVERT(DATETIME, CONVERT(VARCHAR(8), @DATENR)))

IF @Date < @BSTFROM
  BEGIN
    RETURN @Date
  END

IF @Date >= @BSTTO
  BEGIN
    RETURN @Date
  END

RETURN DATEADD(HOUR, -1, @Date)

END')
  END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM sys.objects [SO]
    WHERE [SO].[name] = 'ky_NETFNConvertDateToLocal'
      AND [SO].[type] = 'FN')
  BEGIN
    EXEC ('DROP FUNCTION [dbo].[ky_NETFNConvertDateToLocal]')
  END
GO

DECLARE @Compatible BIT

--This function is only required for SQL Azure databases
--We'll create a "placeholder" function that will simply return the
--unconverted date for On Premise databases
SET @Compatible = CASE WHEN ServerProperty('EngineEdition') = 5
                            THEN 1
                       ELSE 0 END

IF @Compatible = 1
  BEGIN
    EXEC ('CREATE FUNCTION
       [dbo].[ky_NETFNConvertDateToLocal]
      (@Date            DATETIME,
       @TimeZone        VARCHAR(50))
RETURNS DATETIME
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETFNConvertDateToLocal]                                                                        *
 * Converts a DATETIME in UTC to its equivalent at a certain Time Zone                                       *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * Original:                                -                                                        *   *
 *   * First compatible version:                5.7.2.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification history                                                                                      *
 *                                                                                                           *
 * Pino Carafa       2019-07-04 Created                                                                      *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

SET @TimeZone = CASE WHEN RTRIM(ISNULL(@TimeZone, '''')) = ''''
                          THEN ''GMT Standard Time''
                     ELSE @TimeZone END

DECLARE @Converted DATETIMEOFFSET

SET @Converted = CONVERT(DATETIMEOFFSET, @Date) AT TIME ZONE @TimeZone

RETURN CONVERT(DATETIME, @Converted)

END')
  END
ELSE
  BEGIN
    EXEC ('CREATE FUNCTION
       [dbo].[ky_NETFNConvertDateToLocal]
      (@Date            DATETIME,
       @TimeZone        VARCHAR(50))
RETURNS DATETIME
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETFNConvertDateToLocal]                                                                        *
 * Placeholder function for incompatible database. Return original date and ignore "Time Zone"               *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * Original:                                -                                                        *   *
 *   * First compatible version:                5.7.2.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification history                                                                                      *
 *                                                                                                           *
 * Pino Carafa       2019-07-04 Created                                                                      *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

RETURN @Date

END')
  END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETFNExtensionToFileType'
      AND SO.[type] = 'FN')
  BEGIN
    DROP FUNCTION [dbo].[ky_NETFNExtensionToFileType]
  END
GO

CREATE FUNCTION
       [dbo].[ky_NETFNExtensionToFileType]
      (@Extension VARCHAR(20))
RETURNS VARCHAR(20)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETFNExtensionToFileType]                                                                       *
 *                                                                                                           *
 * Gets the Type of File associated with a known extension                                                   *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.7.2.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2019-06-26   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

RETURN CASE WHEN @Extension IN ('BMP', 'GIF', 'G3N', 'JPG', 'JPEG', 'JPE', 'PNG', 'TIF', 'TIFF')
                 THEN 'Image'
            WHEN @Extension IN ('MOV', 'MP4', 'MPG', 'MPEG', 'AVI', 'WMV', 'SWF', 'FLV', 'MKV')
                 THEN 'Multimedia'
            WHEN @Extension IN ('MP3', '3GP', 'WAV', 'WMA', 'OGG', 'FLAC')
                 Then 'Audio'
            WHEN @Extension IN ('HTML', 'HTM', 'XML')
                 THEN 'Web Page'
            WHEN @Extension IN ('XLS', 'XLSX', 'CSV', 'ODS')
                 THEN 'Spreadsheet'
            WHEN @Extension IN ('MSG', 'EML', 'OFT')
                 THEN 'E-Mail'
            WHEN @Extension IN ('DOC', 'DOCX', 'RTF', 'INI', 'TXT', 'ODT')
                 THEN 'Document'
            WHEN @Extension = 'PDF'
                 THEN 'PDF'
            WHEN @Extension IN ('PPT', 'PPTX', 'ODP')
                 THEN 'Presentation'
            WHEN @Extension = 'XPS'
                 THEN 'XPS'
            ELSE 'Unknown' END

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_FNXMLFixLinefeed'
      AND SO.[type] = 'FN')
  BEGIN
    DROP FUNCTION [dbo].[ky_FNXMLFixLinefeed]
  END
GO

CREATE FUNCTION
       [dbo].[ky_FNXMLFixLinefeed]
      (@ORIGINAL NVARCHAR(MAX))
RETURNS NVARCHAR(MAX)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_FNXMLFixLinefeed]                                                                               *
 *                                                                                                           *
 * Ensure that New Lines are correctly returned as CRLFs                                                     *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.7.1.9                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2019-04-29   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

SET @ORIGINAL = REPLACE(@ORIGINAL, NCHAR(13) + NCHAR(10), N'**LF**')
SET @ORIGINAL = REPLACE(@ORIGINAL, NCHAR(13), N'**LF**')
SET @ORIGINAL = REPLACE(@ORIGINAL, NCHAR(10), N'**LF**')
SET @ORIGINAL = REPLACE(@ORIGINAL, '**LF**', NCHAR(13) + NCHAR(10))

RETURN @ORIGINAL

END

GO

--KEYD-6242
INSERT
  INTO [dbo].[DiaryDelActionType]
      ([ActionType],
       [ActionTypeDescription])
SELECT [NEW].[ActionType],
       [NEW].[ActionTypeDescription]
  FROM (SELECT 'A' AS [ActionType],
               'Action' AS [ActionTypeDescription]
        UNION
        SELECT 'C' AS [ActionType],
               'Critical Date' AS [ActionTypeDescription]
        UNION
        SELECT 'D' AS [ActionType],
               'Dictation' AS [ActionTypeDescription]
        UNION
        SELECT 'E' AS [ActionType],
               'Email' AS [ActionTypeDescription]
        UNION
        SELECT 'F' AS [ActionType],
               'Fixed Dated Action' AS [ActionTypeDescription]
        UNION
        SELECT 'I' AS [ActionType],
               'Non Photo ID' AS [ActionTypeDescription]
        UNION
        SELECT 'K' AS [ActionType],
               'Photo ID' AS [ActionTypeDescription]
        UNION
        SELECT 'M' AS [ActionType],
               'Scanned Post/Mail' AS [ActionTypeDescription]
        UNION
        SELECT 'N' AS [ActionType],
               'Note' AS [ActionTypeDescription]
        UNION
        SELECT 'O' AS [ActionType],
               'Court Date' AS [ActionTypeDescription]
        UNION
        SELECT 'P' AS [ActionType],
               'Appointment' AS [ActionTypeDescription]
        UNION
        SELECT 'Q' AS [ActionType],
               'Cheque Request' AS [ActionTypeDescription]
        UNION
        SELECT 'R' AS [ActionType],
               'Reminder' AS [ActionTypeDescription]
        UNION
        SELECT 'S' AS [ActionType],
               'Statute Date' AS [ActionTypeDescription]
        UNION
        SELECT 'T' AS [ActionType],
               'Phone Messages' AS [ActionTypeDescription]
        UNION
        SELECT 'U' AS [ActionType],
               'Undertaking' AS [ActionTypeDescription]
        UNION
        SELECT 'V' AS [ActionType],
               'Corporate Entities Action' AS [ActionTypeDescription]
        UNION
        SELECT 'X' AS [ActionType],
               'SMS Message' AS [ActionTypeDescription]) [NEW]
       LEFT OUTER JOIN [dbo].[DiaryDelActionType] [DDA]
                    ON [DDA].[ActionType] = [NEW].[ActionType]
 WHERE [DDA].[ActionType] IS NULL
GO

IF EXISTS
  (SELECT [DIA].[ACTIONID]
     FROM [dbo].[diary] [DIA]
          LEFT OUTER JOIN [dbo].[DiaryDelActionType] [AT]
                       ON CONVERT(VARCHAR(1), [AT].[ActionType]) = CONVERT(VARCHAR(1), [DIA].[ACTIONTYPE])
    WHERE ISNULL(CONVERT(VARCHAR(1), [DIA].[ACTIONTYPE]), '') <> ''
      AND [AT].[ActionType] IS NULL)
  BEGIN
    UPDATE [DIA]
       SET [DIA].[ACTIONTYPE] = 'A'
      FROM (SELECT [DIA].[ACTIONID]
              FROM [dbo].[diary] [DIA]
                   LEFT OUTER JOIN [dbo].[DiaryDelActionType] [AT]
                                ON CONVERT(VARCHAR(1), [AT].[ActionType]) = CONVERT(VARCHAR(1), [DIA].[ACTIONTYPE])
             WHERE ISNULL(CONVERT(VARCHAR(1), [DIA].[ACTIONTYPE]), '') <> ''
               AND [AT].[ActionType] IS NULL) [WUT]
           INNER JOIN [dbo].[diary] [DIA]
                   ON [DIA].[ACTIONID] = [WUT].[ACTIONID]
  END
GO

IF NOT EXISTS
  (SELECT TOP 1 1
     FROM sys.objects [SO]
    WHERE [SO].[name] = 'DiaryAttachmentOriginalMatterLinks'
      AND [SO].[type] = 'U')
  BEGIN
    EXEC ('CREATE TABLE
       [dbo].[DiaryAttachmentOriginalMatterLinks]
      ([TrackReference]      INT            NOT NULL,
       [OriginalUniqueID]    INT            NOT NULL,
       CONSTRAINT [PK_DAOML]
                  PRIMARY KEY CLUSTERED
                 ([TrackReference]))
    ON [PRIMARY]')
  END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM sys.objects [SO]
          LEFT OUTER JOIN sys.indexes [SI]
                       ON [SI].object_id = [SO].object_id
                      AND [SI].[name] = 'IDX_UniqueID'
    WHERE [SO].[name] = 'DiaryAttachmentOriginalMatterLinks'
      AND [SO].[type] = 'U'
      AND [SI].[index_id] IS NULL)
  BEGIN
    EXEC ('    CREATE NONCLUSTERED
     INDEX [IDX_UniqueID]
        ON [dbo].[DiaryAttachmentOriginalMatterLinks]
          ([OriginalUniqueID])
           INCLUDE ([TrackReference])')
  END
GO

DELETE [DAT]
  FROM [dbo].[DiaryAttachments] [DAT]
       LEFT OUTER JOIN [dbo].[Diary] [DIA]
                    ON [DIA].[ActionID] = [DAT].[DiaryID]
 WHERE [DIA].[ActionID] IS NULL
GO

INSERT
  INTO [dbo].[DiaryAttachmentOriginalMatterLinks]
      ([TrackReference],
       [OriginalUniqueID])
SELECT [DAT].[TrackReference],
       [MAT].[UniqueID]
  FROM [dbo].[DiaryAttachments] [DAT]
       INNER JOIN [dbo].[matters] [MAT]
               ON [MAT].[Code] = [DAT].[CASECODE]
       LEFT OUTER JOIN [dbo].[DiaryAttachmentOriginalMatterLinks] [DAOML]
                    ON [DAOML].[TrackReference] = [DAT].[TrackReference]
 WHERE [DAOML].[TrackReference] IS NULL
GO

/****************************************************************************
 *
 * ky_NETFNFallbackHandler
 *
 * PLEASE NOTE THAT THIS FUNCTION IS USED IN A DEFAULT CONSTRAINT, SO
 * IF YOU NEED TO UPDATE THIS YOU NEED TO UPDATE ALL OF THE CONTENT BELOW
 * AND MAKE SURE IT ALL GETS INCLUDED PROPERLY IN SP_SCRIPTS
 *
 * START
 *
 ****************************************************************************/
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
          INNER JOIN SYS.[default_constraints] DF
                  ON DF.[parent_object_id] = SO.[object_id]
                 AND DF.[name] = 'DF_HandlerLogFor'
    WHERE SO.[name] = 'TimeEntryLogNew'
      AND SO.[type] = 'U')
  BEGIN
    ALTER TABLE [dbo].[TimeEntryLogNew] DROP CONSTRAINT [DF_HandlerLogFor]
  END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = 'ky_NETFNFallbackHandler'
      AND SO.[type] = 'FN')
  BEGIN
    DROP FUNCTION [dbo].[ky_NETFNFallbackHandler]
  END
GO

CREATE FUNCTION
       [dbo].[ky_NETFNFallbackHandler]
      ()
RETURNS VARCHAR(10)
AS
/**********************************************************************************
 *
 * [dbo].[ky_NETFNFallbackHandler]
 * Tries to establish the default handler to use
 *
 * 1) When invoked from Framework, the current process will have its [context_info]
 *    set to hold the current handler code
 * 2) Otherwise, use 'ADM' if such a handler exists
 * 3) If all else fails, use blank ('')
 *
 * Modification history:
 * ????-??-??   ???????        Created
 * 2019-10-09   John Ginnane   KEYD-6197 - Changed for to use function
 *
 **********************************************************************************/
BEGIN
  DECLARE @HAN VARCHAR(10)

  SET @HAN = [dbo].[ky_NETFNGetLoggedOnHandler]()

  RETURN @HAN
END
GO

ALTER TABLE [dbo].[TimeEntryLogNew] ADD  CONSTRAINT [DF_HandlerLogFor]  DEFAULT ([dbo].[ky_NETFNFallbackHandler]()) FOR [HandlerLogFor]
GO

/****************************************************************************
 *
 * ky_NETFNFallbackHandler
 *
 * PLEASE NOTE THAT THIS FUNCTION IS USED IN A DEFAULT CONSTRAINT, SO
 * IF YOU NEED TO UPDATE THIS YOU NEED TO UPDATE ALL OF THE CONTENT ABOVE
 * AND MAKE SURE IT ALL GETS INCLUDED PROPERLY IN SP_SCRIPTS
 *
 * END
 *
 ****************************************************************************/
GO


/****************************************************************************
 *
 * ky_NETFNLogin
 *
 * PLEASE NOTE THAT THIS FUNCTION IS USED IN A DEFAULT CONSTRAINT, SO
 * IF YOU NEED TO UPDATE THIS YOU NEED TO UPDATE ALL OF THE CONTENT BELOW
 * AND MAKE SURE IT ALL GETS INCLUDED PROPERLY IN SP_SCRIPTS
 *
 * START
 *
 ****************************************************************************/
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
          INNER JOIN SYS.[default_constraints] DF
                  ON DF.[parent_object_id] = SO.[object_id]
                 AND DF.[name] = 'DF_LoginLogFor'
    WHERE SO.[name] = 'TimeEntryLogNew'
      AND SO.[type] = 'U')
  BEGIN
    ALTER TABLE [dbo].[TimeEntryLogNew] DROP CONSTRAINT [DF_LoginLogFor]
  END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = 'ky_NETFNLogin'
      AND SO.[type] = 'FN')
  BEGIN
    DROP FUNCTION [dbo].[ky_NETFNLogin]
  END
GO

CREATE FUNCTION
        [dbo].[ky_NETFNLogin]
      ()
RETURNS NVARCHAR(128)
AS
/************************************************************
 *
 * Returns the currently logged in WINDOWS user
 * or SQL User
 *
 * Modification history:
 * ????-??-??   ???????        Created
 * 2019-10-09   John Ginnane   KEYD-6197 - Prepended [sysprocesses] with [sys].
 *
 ************************************************************/
BEGIN
  DECLARE @LOGIN NVARCHAR(128)

  SELECT @LOGIN = [P].[loginame]
    FROM [sys].[sysprocesses] [P]
    WHERE [P].[spid] = @@SPID

  SET @LOGIN = ISNULL(@LOGIN, '')

  RETURN @LOGIN
END
GO

ALTER TABLE [dbo].[TimeEntryLogNew] ADD  CONSTRAINT [DF_LoginLogFor]  DEFAULT ([dbo].[ky_NETFNLogin]()) FOR [LoginLogFor]
GO

/****************************************************************************
 *
 * ky_NETFNLogin
 *
 * PLEASE NOTE THAT THIS FUNCTION IS USED IN A DEFAULT CONSTRAINT, SO
 * IF YOU NEED TO UPDATE THIS YOU NEED TO UPDATE ALL OF THE CONTENT ABOVE
 * AND MAKE SURE IT ALL GETS INCLUDED PROPERLY IN SP_SCRIPTS
 *
 * END
 *
 ****************************************************************************/
GO

---- Example use of the next two functions [dbo].[ky_NETFNGetParameters] and [dbo].[ky_NETFNGetFunctionDescription]
--  SELECT REPLACE([SO].[name], 'ky_SUF', 'f.') + [dbo].[ky_NETFNGetParameters]([SO].object_id, 1) + ' ' + [dbo].[ky_NETFNGetParameters]([SO].object_id, 0) AS [Function],
--         [dbo].[ky_NETFNGetFunctionDescription]([SO].object_id) AS [Description]
--    FROM sys.objects [SO]
--   WHERE [SO].[name] LIKE 'ky_SUF%'
--ORDER BY [SO].[name]
IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETFNGetParameters'
      AND SO.[type] = 'FN')
  BEGIN
    DROP FUNCTION [dbo].[ky_NETFNGetParameters]
  END
GO

CREATE FUNCTION
       [dbo].[ky_NETFNGetParameters]
      (@OBJECT_ID    INT,
       @INPUT        BIT)
RETURNS VARCHAR(MAX)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETFNGetParameters]                                                                             *
 *                                                                                                           *
 * Returns the parameter definition for a function.                                                          *
 * @INPUT = 0 - The Return value                                                                             *
 * @INPUT = 1 - The Input parameters                                                                         *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.6.4.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2019-01-15   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

DECLARE @PARMS VARCHAR(MAX)

SET @PARMS = ''

  SELECT @PARMS = @PARMS + CASE WHEN (@PARMS = '') AND (@INPUT = 1) 
                                     THEN '('
                                WHEN (@PARMS = '') AND (@INPUT = 0) 
                                     THEN ''
                                ELSE ', ' END
                + CASE WHEN @INPUT = 1
                            THEN [P].[name]
                       ELSE 'RETURNS' END
                + ' '
                + [ST].[name]
                + CASE WHEN [ST].[name] IN ('char', 'nchar', 'varchar', 'nvarchar')
                            THEN '(' + CASE WHEN [P].[max_length] = -1
                                                 THEN 'max'
                                            ELSE convert(VARCHAR, [P].[max_length]) END + ')'
                       WHEN [ST].[name] IN ('numeric', 'decimal')
                            THEN '(' + convert(VARCHAR, [P].[precision]) + ','
                                     + convert(VARCHAR, [P].[scale]) + ')'
                       ELSE '' END
                + CASE WHEN @INPUT = 1
                            THEN CASE WHEN [P].[is_output] = 1
                                           THEN ' OUT'
                                      ELSE '' END
                       ELSE '' END
    FROM sys.parameters [P]
         INNER JOIN sys.types [ST]
                 ON [ST].[system_type_id] = [P].[system_type_id]
   WHERE [P].object_id = @OBJECT_ID
     AND (   (    @INPUT = 1
              AND [P].parameter_id <> 0)
          OR (    @INPUT = 0
              AND [P].[parameter_id] = 0))
ORDER BY [P].[parameter_id]

SET @PARMS = @PARMS + CASE WHEN @PARMS = '' THEN '' WHEN @INPUT = 0 THEN '' ELSE ')' END

RETURN @PARMS

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETFNGetFunctionDescription'
      AND SO.[type] = 'FN')
  BEGIN
    DROP FUNCTION [dbo].[ky_NETFNGetFunctionDescription]
  END
GO

CREATE FUNCTION
       [dbo].[ky_NETFNGetFunctionDescription]
      (@OBJECT_ID    INT)
RETURNS VARCHAR(MAX)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETFNGetFunctionDescription]                                                                    *
 *                                                                                                           *
 * Returns the description for a function                                                                    *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.6.4.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2019-01-15   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

DECLARE @DEF VARCHAR(MAX)
DECLARE @POS INT

SELECT @DEF = [SM].[definition]
  FROM sys.sql_modules [SM]
 WHERE [SM].object_id = @OBJECT_ID

SET @POS = PATINDEX('%/*%', @DEF)

IF @POS = 0
  BEGIN
    RETURN 'Function description does not conform to coding standards'
  END

SET @DEF = SUBSTRING(@DEF, @POS, LEN(@DEF) - @POS + 1)

SET @POS = PATINDEX('%*/%', @DEF)

SET @DEF = SUBSTRING(@DEF, 2, @POS - 1)

IF @POS = 0
  BEGIN
    RETURN 'Function description does not conform to coding standards'
  END

SET @POS = PATINDEX('%[[]dbo]%', @DEF)

IF @POS > 0
  BEGIN
    SET @DEF = SUBSTRING(@DEF, @POS + 5, LEN(@DEF) - @POS - 4)
  END

SET @POS = CHARINDEX(']', @DEF)
IF @POS > 0
  BEGIN
    SET @DEF = SUBSTRING(@DEF, @POS + 1, LEN(@DEF) - @POS)
  END

SET @POS = PATINDEX('%compatibility information%', @DEF)

IF @POS > 0
  BEGIN
    SET @DEF = SUBSTRING(@DEF, 1, @POS - 1)
  END

SET @POS = PATINDEX('%modification history%', @DEF)

IF @POS > 0
  BEGIN
    SET @DEF = SUBSTRING(@DEF, 1, @POS - 1)
  END

SET @DEF = RTRIM(LTRIM(REPLACE(@DEF, '*', '')))

SET @DEF = [dbo].[ky_RemoveSpuriousWhitespace2](@DEF, 500)

RETURN @DEF

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = 'ky_NETFNNumberInWordsCleanup'
      AND SO.[type] = 'FN')
  BEGIN
    DROP FUNCTION [dbo].[ky_NETFNNumberInWordsCleanup]
  END
GO

CREATE FUNCTION
       [dbo].[ky_NETFNNumberInWordsCleanup]
      (@NumberInWords VARCHAR(MAX))
RETURNS VARCHAR(MAX)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETFNNumberInWords]                                                                             *
 *                                                                                                           *
 * Removes superfluous text ("and"s) from a number in words                                                  *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.6.4.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2019-01-31   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN
  IF PATINDEX('%billion and%', @NumberInWords) > 0
    IF (PATINDEX('%million and%', @NumberInWords) > 0)
    OR (PATINDEX('%thousand and%', @NumberInWords) > 0)
    OR (PATINDEX('%hundred and%', @NumberInWords) > 0)
      BEGIN
        SET @NumberInWords = REPLACE(@NumberInWords, 'billion and ', 'billion ')
      END

  IF PATINDEX('%million and%', @NumberInWords) > 0
    IF (PATINDEX('%thousand and%', @NumberInWords) > 0)
    OR (PATINDEX('%hundred and%', @NumberInWords) > 0)
      BEGIN
        SET @NumberInWords = REPLACE(@NumberInWords, 'million and ', 'million ')
      END

  IF PATINDEX('%thousand and%', @NumberInWords) > 0
    IF (PATINDEX('%hundred and%', @NumberInWords) > 0)
      BEGIN
        SET @NumberInWords = REPLACE(@NumberInWords, 'thousand and ', 'thousand ')
      END

  RETURN @NumberInWords
END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = 'ky_NETFNNumberInWords'
      AND SO.[type] = 'FN')
  BEGIN
    DROP FUNCTION [dbo].[ky_NETFNNumberInWords]
  END
GO

CREATE FUNCTION
       [dbo].[ky_NETFNNumberInWords]
      (@Number   DECIMAL(20, 2),
       @Currency BIT)
RETURNS VARCHAR(MAX)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETFNNumberInWords]                                                                             *
 *                                                                                                           *
 * Converts a number into words. The @Currency parameter will currently only result in output in euro/cent   *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.6.4.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2019-01-31   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

DECLARE @WORK VARCHAR(MAX)
DECLARE @WORK2 VARCHAR(MAX)
DECLARE @INT INTEGER

IF @Number < 100
  BEGIN
    SET @INT = CONVERT(INT, @Number)

    SET @WORK = CASE WHEN @INT = 0
                          THEN ''
                     WHEN @INT = 1
                          THEN 'one'
                     WHEN @INT = 2
                          THEN 'two'
                     WHEN @INT = 3
                          THEN 'three'
                     WHEN @INT = 4
                          THEN 'four'
                     WHEN @INT = 5
                          THEN 'five'
                     WHEN @INT = 6
                          THEN 'six'
                     WHEN @INT = 7
                          THEN 'seven'
                     WHEN @INT = 8
                          THEN 'eight'
                     WHEN @INT = 9
                          THEN 'nine'
                     WHEN @INT = 10
                          THEN 'ten'
                     WHEN @INT = 11
                          THEN 'eleven'
                     WHEN @INT = 12
                          THEN 'twelve'
                     WHEN @INT = 13
                          THEN 'thirteen'
                     WHEN @INT = 14
                          THEN 'fourteen'
                     WHEN @INT = 15
                          THEN 'fifteen'
                     WHEN @INT = 16
                          THEN 'sixteen'
                     WHEN @INT = 17
                          THEN 'seventeen'
                     WHEN @INT = 18
                          THEN 'eighteen'
                     WHEN @INT = 19
                          THEN 'nineteen'
                     WHEN @INT > 90
                          THEN 'ninety-' + [dbo].[ky_NETFNNumberInWords](CONVERT(decimal(20, 2), @INT-90), 0)
                     WHEN @INT = 90
                          THEN 'ninety'
                     WHEN @INT > 80
                          THEN 'eighty-' + [dbo].[ky_NETFNNumberInWords](CONVERT(decimal(20, 2), @INT-80), 0)
                     WHEN @INT = 80
                          THEN 'eighty'
                     WHEN @INT > 70
                          THEN 'seventy-' + [dbo].[ky_NETFNNumberInWords](CONVERT(decimal(20, 2), @INT-70), 0)
                     WHEN @INT = 70
                          THEN 'seventy'
                     WHEN @INT > 60
                          THEN 'sixty-' + [dbo].[ky_NETFNNumberInWords](CONVERT(decimal(20, 2), @INT-60), 0)
                     WHEN @INT = 60
                          THEN 'sixty'
                     WHEN @INT > 50
                          THEN 'fifty-' + [dbo].[ky_NETFNNumberInWords](CONVERT(decimal(20, 2), @INT-50), 0)
                     WHEN @INT = 50
                          THEN 'fifty'
                     WHEN @INT > 40
                          THEN 'fourty-' + [dbo].[ky_NETFNNumberInWords](CONVERT(decimal(20, 2), @INT-40), 0)
                     WHEN @INT = 40
                          THEN 'fourty'
                     WHEN @INT > 30
                          THEN 'thirty-' + [dbo].[ky_NETFNNumberInWords](CONVERT(decimal(20, 2), @INT-30), 0)
                     WHEN @INT = 30
                          THEN 'thirty'
                     WHEN @INT > 20
                          THEN 'twenty-' + [dbo].[ky_NETFNNumberInWords](CONVERT(decimal(20, 2), @INT-20), 0)
                     WHEN @INT = 20
                          THEN 'twenty'
                     ELSE '' END

    SET @Number = CONVERT(DECIMAL(20, 2), 100) * (@Number - Convert(DECIMAL(20, 2), @INT))

    IF @Number = 0
      BEGIN
        SET @WORK2 = ''
      END
    ELSE
      BEGIN
        SET @WORK2 = [dbo].[ky_NETFNNumberInWords](@Number, 0)
      END

    IF @WORK2 <> ''
      BEGIN
        IF @Currency = 1
          BEGIN
            SET @WORK2 = @WORK2 + ' cent'
          END
        ELSE
          BEGIN
            SET @WORK2 = 'zero point ' + @WORK2
          END
      END

    IF @WORK = ''
      BEGIN
        RETURN @WORK2
      END
    ELSE
      BEGIN
        IF @WORK2 = ''
          BEGIN
            IF @Currency = 1
              BEGIN
                RETURN @WORK + ' euro'
              END
            ELSE
              BEGIN
                RETURN @WORK
              END
          END
        IF @Currency = 1
          BEGIN
            RETURN @WORK + ' euro and ' + @WORK2
          END
        ELSE
          BEGIN
            RETURN @WORK + ' ' + REPLACE(@WORK2, 'zero point ', 'point ')
          END
      END

    RETURN @WORK
  END

IF @Number < 1000
  BEGIN
    SET @INT = CONVERT(INT, @Number / CONVERT(DECIMAL(20, 2), 100))
    SET @WORK = [dbo].[ky_NETFNNumberInWords](@INT, 0) + ' hundred'
    SET @Number = @Number - (CONVERT(DECIMAL(20, 2), @INT) * CONVERT(DECIMAL(20, 2), 100))
    SET @WORK2 = [dbo].[ky_NETFNNumberInWords](@Number, @Currency)
    IF @WORK2 = ''
      BEGIN
        IF @Currency = 1
          BEGIN
            SET @WORK = @WORK + ' euro'
          END
        RETURN @WORK
      END

    IF @WORK = ''
      BEGIN
        RETURN @WORK2
      END

    IF @Currency = 1
      BEGIN
        IF PATINDEX('%euro%', @WORK) = 0 AND PATINDEX('%euro%', @WORK2) = 0
          BEGIN
            SET @WORK = @WORK + ' euro'
          END
      END

    RETURN @WORK + ' and ' + REPLACE(@WORK2, 'zero point ', 'point ')
  END

IF @Number < 1000000
  BEGIN
    SET @INT = CONVERT(INT, @Number / CONVERT(DECIMAL(20, 2), 1000))
    SET @WORK = [dbo].[ky_NETFNNumberInWords](@INT, 0) + ' thousand'
    SET @Number = @Number - (CONVERT(DECIMAL(20, 2), @INT) * CONVERT(DECIMAL(20, 2), 1000))
    SET @WORK2 = [dbo].[ky_NETFNNumberInWords](@Number, @Currency)
    IF @WORK2 = ''
      BEGIN
        IF @Currency = 1
          BEGIN
            SET @WORK = @WORK + ' euro'
          END
        RETURN @WORK
      END

    IF @WORK = ''
      BEGIN
        RETURN @WORK2
      END

    IF @Currency = 1
      BEGIN
        IF PATINDEX('%euro%', @WORK) = 0 AND PATINDEX('%euro%', @WORK2) = 0
          BEGIN
            SET @WORK = @WORK + ' euro'
          END
      END

    RETURN @WORK + ' and ' + REPLACE(@WORK2, 'zero point ', 'point ')
  END

IF @Number < 1000000000
  BEGIN
    SET @INT = CONVERT(INT, @Number / CONVERT(DECIMAL(20, 2), 1000000))
    SET @WORK = [dbo].[ky_NETFNNumberInWords](@INT, 0) + ' million'
    SET @Number = @Number - (CONVERT(DECIMAL(20, 2), @INT) * CONVERT(DECIMAL(20, 2), 1000000))
    SET @WORK2 = [dbo].[ky_NETFNNumberInWords](@Number, @Currency)
    IF @WORK2 = ''
      BEGIN
        IF @Currency = 1
          BEGIN
            SET @WORK = @WORK + ' euro'
          END
        RETURN @WORK
      END

    IF @WORK = ''
      BEGIN
        RETURN @WORK2
      END

    IF @Currency = 1
      BEGIN
        IF PATINDEX('%euro%', @WORK) = 0 AND PATINDEX('%euro%', @WORK2) = 0
          BEGIN
            SET @WORK = @WORK + ' euro'
          END
      END

    RETURN @WORK + ' and ' + REPLACE(@WORK2, 'zero point ', 'point ')
  END

IF @Number < 1000000000000
  BEGIN
    SET @INT = CONVERT(INT, @Number / CONVERT(DECIMAL(20, 2), 1000000000))
    SET @WORK = [dbo].[ky_NETFNNumberInWords](@INT, 0) + ' billion'
    SET @Number = @Number - (CONVERT(DECIMAL(20, 2), @INT) * CONVERT(DECIMAL(20, 2), 1000000000))
    SET @WORK2 = [dbo].[ky_NETFNNumberInWords](@Number, @Currency)
    IF @WORK2 = ''
      BEGIN
        IF @Currency = 1
          BEGIN
            SET @WORK = @WORK + ' euro'
          END
        RETURN @WORK
      END

    IF @WORK = ''
      BEGIN
        RETURN @WORK2
      END

    IF @Currency = 1
      BEGIN
        IF PATINDEX('%euro%', @WORK) = 0 AND PATINDEX('%euro%', @WORK2) = 0
          BEGIN
            SET @WORK = @WORK + ' euro'
          END
      END

    RETURN @WORK + ' and ' + REPLACE(@WORK2, 'zero point ', 'point ')
  END

RETURN ''

END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETFNCURNumberInWords'
      AND SO.[type] = 'FN')
  BEGIN
    DROP FUNCTION [dbo].[ky_NETFNCURNumberInWords]
  END
GO

CREATE FUNCTION
       [dbo].[ky_NETFNCURNumberInWords]
      (@Number   DECIMAL(20, 2),
       @Currency BIT,
       @CurMain  VARCHAR(10),
       @CurCent  VARCHAR(10))
RETURNS VARCHAR(MAX)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETFNCurNumberInWords]                                                                          *
 *                                                                                                           *
 * Converts a number into words. The @Currency parameter will currently only result in output in             *
 * @CurMain/@CurCent (default euro/cent)                                                                     *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.7.1.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2019-05-22   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

DECLARE @WORK VARCHAR(MAX)
DECLARE @WORK2 VARCHAR(MAX)
DECLARE @INT INTEGER

IF RTRIM(ISNULL(@CurMain, '')) = ''
  BEGIN
    SET @CurMain = 'euro'
  END
SET @CurMain = RTRIM(@CurMain)

IF RTRIM(ISNULL(@CurCent, '')) = ''
  BEGIN
    SET @CurCent = 'cent'
  END
SET @CurCent = RTRIM(@CurCent)

IF @Number < 100
  BEGIN
    SET @INT = CONVERT(INT, @Number)

    SET @WORK = CASE WHEN @INT = 0
                          THEN ''
                     WHEN @INT = 1
                          THEN 'one'
                     WHEN @INT = 2
                          THEN 'two'
                     WHEN @INT = 3
                          THEN 'three'
                     WHEN @INT = 4
                          THEN 'four'
                     WHEN @INT = 5
                          THEN 'five'
                     WHEN @INT = 6
                          THEN 'six'
                     WHEN @INT = 7
                          THEN 'seven'
                     WHEN @INT = 8
                          THEN 'eight'
                     WHEN @INT = 9
                          THEN 'nine'
                     WHEN @INT = 10
                          THEN 'ten'
                     WHEN @INT = 11
                          THEN 'eleven'
                     WHEN @INT = 12
                          THEN 'twelve'
                     WHEN @INT = 13
                          THEN 'thirteen'
                     WHEN @INT = 14
                          THEN 'fourteen'
                     WHEN @INT = 15
                          THEN 'fifteen'
                     WHEN @INT = 16
                          THEN 'sixteen'
                     WHEN @INT = 17
                          THEN 'seventeen'
                     WHEN @INT = 18
                          THEN 'eighteen'
                     WHEN @INT = 19
                          THEN 'nineteen'
                     WHEN @INT > 90
                          THEN 'ninety-' + [dbo].[ky_NETFNNumberInWords](CONVERT(decimal(20, 2), @INT-90), 0)
                     WHEN @INT = 90
                          THEN 'ninety'
                     WHEN @INT > 80
                          THEN 'eighty-' + [dbo].[ky_NETFNNumberInWords](CONVERT(decimal(20, 2), @INT-80), 0)
                     WHEN @INT = 80
                          THEN 'eighty'
                     WHEN @INT > 70
                          THEN 'seventy-' + [dbo].[ky_NETFNNumberInWords](CONVERT(decimal(20, 2), @INT-70), 0)
                     WHEN @INT = 70
                          THEN 'seventy'
                     WHEN @INT > 60
                          THEN 'sixty-' + [dbo].[ky_NETFNNumberInWords](CONVERT(decimal(20, 2), @INT-60), 0)
                     WHEN @INT = 60
                          THEN 'sixty'
                     WHEN @INT > 50
                          THEN 'fifty-' + [dbo].[ky_NETFNNumberInWords](CONVERT(decimal(20, 2), @INT-50), 0)
                     WHEN @INT = 50
                          THEN 'fifty'
                     WHEN @INT > 40
                          THEN 'fourty-' + [dbo].[ky_NETFNNumberInWords](CONVERT(decimal(20, 2), @INT-40), 0)
                     WHEN @INT = 40
                          THEN 'fourty'
                     WHEN @INT > 30
                          THEN 'thirty-' + [dbo].[ky_NETFNNumberInWords](CONVERT(decimal(20, 2), @INT-30), 0)
                     WHEN @INT = 30
                          THEN 'thirty'
                     WHEN @INT > 20
                          THEN 'twenty-' + [dbo].[ky_NETFNNumberInWords](CONVERT(decimal(20, 2), @INT-20), 0)
                     WHEN @INT = 20
                          THEN 'twenty'
                     ELSE '' END

    SET @Number = CONVERT(DECIMAL(20, 2), 100) * (@Number - Convert(DECIMAL(20, 2), @INT))

    IF @Number = 0
      BEGIN
        SET @WORK2 = ''
      END
    ELSE
      BEGIN
        SET @WORK2 = [dbo].[ky_NETFNNumberInWords](@Number, 0)
      END

    IF @WORK2 <> ''
      BEGIN
        IF @Currency = 1
          BEGIN
            SET @WORK2 = @WORK2 + ' ' + @CurCent
          END
        ELSE
          BEGIN
            SET @WORK2 = 'zero point ' + @WORK2
          END
      END

    IF @WORK = ''
      BEGIN
        RETURN @WORK2
      END
    ELSE
      BEGIN
        IF @WORK2 = ''
          BEGIN
            IF @Currency = 1
              BEGIN
                RETURN @WORK + ' ' + @CurMain
              END
            ELSE
              BEGIN
                RETURN @WORK
              END
          END
        IF @Currency = 1
          BEGIN
            RETURN @WORK + ' ' + @CurMain + ' and ' + @WORK2
          END
        ELSE
          BEGIN
            RETURN @WORK + ' ' + REPLACE(@WORK2, 'zero point ', 'point ')
          END
      END

    RETURN @WORK
  END

IF @Number < 1000
  BEGIN
    SET @INT = CONVERT(INT, @Number / CONVERT(DECIMAL(20, 2), 100))
    SET @WORK = [dbo].[ky_NETFNNumberInWords](@INT, 0) + ' hundred'
    SET @Number = @Number - (CONVERT(DECIMAL(20, 2), @INT) * CONVERT(DECIMAL(20, 2), 100))
    SET @WORK2 = [dbo].[ky_NETFNCURNumberInWords](@Number, @Currency, @CurMain, @CurCent)
    IF @WORK2 = ''
      BEGIN
        IF @Currency = 1
          BEGIN
            SET @WORK = @WORK + ' ' + @CurMain
          END
        RETURN @WORK
      END

    IF @WORK = ''
      BEGIN
        RETURN @WORK2
      END

    IF @Currency = 1
      BEGIN
        IF PATINDEX('%' + @CurMain + '%', @WORK) = 0 AND PATINDEX('%' + @CurMain + '%', @WORK2) = 0
          BEGIN
            SET @WORK = @WORK + ' ' + @CurMain
          END
      END

    RETURN @WORK + ' and ' + REPLACE(@WORK2, 'zero point ', 'point ')
  END

IF @Number < 1000000
  BEGIN
    SET @INT = CONVERT(INT, @Number / CONVERT(DECIMAL(20, 2), 1000))
    SET @WORK = [dbo].[ky_NETFNNumberInWords](@INT, 0) + ' thousand'
    SET @Number = @Number - (CONVERT(DECIMAL(20, 2), @INT) * CONVERT(DECIMAL(20, 2), 1000))
    SET @WORK2 = [dbo].[ky_NETFNCURNumberInWords](@Number, @Currency, @CurMain, @CurCent)
    IF @WORK2 = ''
      BEGIN
        IF @Currency = 1
          BEGIN
            SET @WORK = @WORK + ' ' + @CurMain
          END
        RETURN @WORK
      END

    IF @WORK = ''
      BEGIN
        RETURN @WORK2
      END

    IF @Currency = 1
      BEGIN
        IF PATINDEX('%' + @CurMain + '%', @WORK) = 0 AND PATINDEX('%' + @CurMain + '%', @WORK2) = 0
          BEGIN
            SET @WORK = @WORK + ' ' + @CurMain
          END
      END

    RETURN @WORK + ' and ' + REPLACE(@WORK2, 'zero point ', 'point ')
  END

IF @Number < 1000000000
  BEGIN
    SET @INT = CONVERT(INT, @Number / CONVERT(DECIMAL(20, 2), 1000000))
    SET @WORK = [dbo].[ky_NETFNNumberInWords](@INT, 0) + ' million'
    SET @Number = @Number - (CONVERT(DECIMAL(20, 2), @INT) * CONVERT(DECIMAL(20, 2), 1000000))
    SET @WORK2 = [dbo].[ky_NETFNCURNumberInWords](@Number, @Currency, @CurMain, @CurCent)
    IF @WORK2 = ''
      BEGIN
        IF @Currency = 1
          BEGIN
            SET @WORK = @WORK + ' ' + @CurMain
          END
        RETURN @WORK
      END

    IF @WORK = ''
      BEGIN
        RETURN @WORK2
      END

    IF @Currency = 1
      BEGIN
        IF PATINDEX('%' + @CurMain + '%', @WORK) = 0 AND PATINDEX('%' + @CurMain + '%', @WORK2) = 0
          BEGIN
            SET @WORK = @WORK + ' ' + @CurMain
          END
      END

    RETURN @WORK + ' and ' + REPLACE(@WORK2, 'zero point ', 'point ')
  END

IF @Number < 1000000000000
  BEGIN
    SET @INT = CONVERT(INT, @Number / CONVERT(DECIMAL(20, 2), 1000000000))
    SET @WORK = [dbo].[ky_NETFNNumberInWords](@INT, 0) + ' billion'
    SET @Number = @Number - (CONVERT(DECIMAL(20, 2), @INT) * CONVERT(DECIMAL(20, 2), 1000000000))
    SET @WORK2 = [dbo].[ky_NETFNCURNumberInWords](@Number, @Currency, @CurMain, @CurCent)
    IF @WORK2 = ''
      BEGIN
        IF @Currency = 1
          BEGIN
            SET @WORK = @WORK + ' ' + @CurMain
          END
        RETURN @WORK
      END

    IF @WORK = ''
      BEGIN
        RETURN @WORK2
      END

    IF @Currency = 1
      BEGIN
        IF PATINDEX('%' + @CurMain + '%', @WORK) = 0 AND PATINDEX('%' + @CurMain + '%', @WORK2) = 0
          BEGIN
            SET @WORK = @WORK + ' ' + @CurMain
          END
      END

    RETURN @WORK + ' and ' + REPLACE(@WORK2, 'zero point ', 'point ')
  END

RETURN ''

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = 'ky_SUFNumberInWords'
      AND SO.[type] = 'FN')
  BEGIN
    DROP FUNCTION [dbo].[ky_SUFNumberInWords]
  END
GO

CREATE FUNCTION
       [dbo].[ky_SUFNumberInWords]
      (@NumberV  VARCHAR(30),
       @Currency BIT)
RETURNS VARCHAR(MAX)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_SUFNumberInWords]                                                                               *
 *                                                                                                           *
 * Converts a number into words The @Currency parameter will currently only result in output in euro/cent    *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.6.4.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2019-01-31   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

DECLARE @CHECK VARCHAR(8)
DECLARE @THOUSEP CHAR(1)
SET @CHECK = CONVERT(VARCHAR, CONVERT(MONEY, 1000), 1)
SET @THOUSEP = SUBSTRING(@CHECK, 2, 1)

DECLARE @number DECIMAL(20, 2)
DECLARE @result VARCHAR(MAX)

SET @number = CONVERT(DECIMAL(20, 2), REPLACE(@NumberV, @THOUSEP, ''))

IF @number < 0
  BEGIN
    SET @result = [dbo].[ky_NETFNNumberInWordsCleanup]([dbo].[ky_NETFNNumberInWords](-@number, @Currency))
    IF RTRIM(ISNULL(@result, '')) <> ''
      BEGIN
        IF @Currency = 1
          BEGIN
            SET @result = @result + ' CR'
          END
        ELSE
          BEGIN
            SET @result = 'minus ' + @result
          END
      END
  END
ELSE
  BEGIN
    SET @result = [dbo].[ky_NETFNNumberInWordsCleanup]([dbo].[ky_NETFNNumberInWords](@number, @Currency))
  END

RETURN @result

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_SUFCURNumberInWords'
      AND SO.[type] = 'FN')
  BEGIN
    DROP FUNCTION [dbo].[ky_SUFCURNumberInWords]
  END
GO

CREATE FUNCTION
       [dbo].[ky_SUFCURNumberInWords]
      (@NumberV  VARCHAR(30),
       @Currency BIT,
       @CurMain  VARCHAR(10),
       @CurCent  VARCHAR(10))
RETURNS VARCHAR(MAX)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_SUFCURNumberInWords]                                                                            *
 *                                                                                                           *
 * Converts a number into words. The @Currency parameter will currently only result in output in             *
 * @CurMain/@CurCent (default euro/cent)                                                                     *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.7.1.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2019-05-22   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

DECLARE @CHECK VARCHAR(8)
DECLARE @THOUSEP CHAR(1)
SET @CHECK = CONVERT(VARCHAR, CONVERT(MONEY, 1000), 1)
SET @THOUSEP = SUBSTRING(@CHECK, 2, 1)

DECLARE @number DECIMAL(20, 2)
DECLARE @result VARCHAR(MAX)

SET @number = CONVERT(DECIMAL(20, 2), REPLACE(@NumberV, @THOUSEP, ''))

IF @number < 0
  BEGIN
    SET @result = [dbo].[ky_NETFNNumberInWordsCleanup]([dbo].[ky_NETFNCURNumberInWords](-@number, @Currency, @CurMain, @CurCent))
    IF RTRIM(ISNULL(@result, '')) <> ''
      BEGIN
        IF @Currency = 1
          BEGIN
            SET @result = @result + ' CR'
          END
        ELSE
          BEGIN
            SET @result = 'minus ' + @result
          END
      END
  END
ELSE
  BEGIN
    SET @result = [dbo].[ky_NETFNNumberInWordsCleanup]([dbo].[ky_NETFNCURNumberInWords](@number, @Currency, @CurMain, @CurCent))
  END

RETURN @result

END

GO

-- Create dummy function to stop the code below complaining
IF NOT EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_SUFGetFormattedDate'
      AND SO.[type] = 'FN')
  BEGIN
    EXEC ('CREATE FUNCTION [dbo].[ky_SUFGetFormattedDate](@DATE datetime, @FORMAT VARCHAR(200)) RETURNS VARCHAR(MAX) AS BEGIN RETURN ''dummy'' END')
  END
GO

-- Create dummy function to stop the code below complaining
IF NOT EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_SUFGetFormattedValue'
      AND SO.[type] = 'FN')
  BEGIN
    EXEC ('CREATE FUNCTION [dbo].[ky_SUFGetFormattedValue](@NUMBER FLOAT, @FORMAT VARCHAR(200), @THOUSANDSEPARATORS BIT, @NEGBRACKETS BIT, @CULTURE VARCHAR(10)) RETURNS VARCHAR(MAX) AS BEGIN RETURN ''dummy'' END')
  END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_SUFFormat'
      AND SO.[type] = 'FN')
  BEGIN
    DROP FUNCTION [dbo].[ky_SUFFormat]
  END
GO

CREATE FUNCTION
       [dbo].[ky_SUFFormat]
      (@ORIGINAL  VARCHAR(MAX),
       @FORMAT    VARCHAR(200))
RETURNS VARCHAR(MAX)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_SUFFormat]                                                                                      *
 *                                                                                                           *
 * To be used as Format function for Document Generator codes                                                *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.6.4.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2019-01-15   Pino Carafa       Created                                                                    *
 * 2019-01-16   Pino Carafa       Handle date formats that SQL doesn't like                                  *
 * 2019-01-18   Pino Carafa       Check the x/y/z pattern before using IsDate as SQL will interpret strings  *
 *                                in xx/yy/zzzz format as mm/dd/yyyy by default regardless of the culture    *
 * 2019-01-18   Pino Carafa       Ensure numerical values with thousand separators are interpreted correctly *
 * 2019-02-05   Pino Carafa       Ensure very large numbers are handled correctly, handle '-' and '+'        *
 * 2019-02-08   Pino Carafa       Handle any reasonable date formats that could be passed in, even things    *
 *                                like "Friday, the 8th day of February, 2019"                               *
 * 2019-02-13   Pino Carafa       KEYD-5744 - It was recognising certain integers as Dates                   *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

  DECLARE @ORIGINALDATE DATETIME
  DECLARE @PART1 VARCHAR(10)
  DECLARE @PART2 VARCHAR(10)
  DECLARE @PART3 VARCHAR(10)
  DECLARE @ALTERNATIVE VARCHAR(10)

  IF (@ORIGINAL LIKE '%/%/%') AND (@ORIGINAL NOT LIKE '%/%/%/%')
    BEGIN
      SET @PART1 = SUBSTRING(@ORIGINAL, 1, CHARINDEX('/', @ORIGINAL) - 1)
      SET @PART2 = SUBSTRING(@ORIGINAL, LEN(@PART1) + 2,  CHARINDEX('/', @ORIGINAL, LEN(@PART1) + 2) - LEN(@PART1) - 2)
      SET @PART3 = SUBSTRING(@ORIGINAL, LEN(@PART1) + LEN(@PART2) + 3, LEN(@ORIGINAL) - LEN(@PART1) - LEN(@PART2) - 2)
      IF LEN(@PART1) = 4 --YYYY/MM/DD
        BEGIN
          SET @ALTERNATIVE = CONVERT(VARCHAR(10), @PART1 + @PART2 + @PART3) --YYYYMMDD
          IF ISDATE(@ALTERNATIVE) = 1
            BEGIN
              SET @ORIGINALDATE = CONVERT(DATETIME, @ALTERNATIVE)
            END
        END
      ELSE IF LEN(@PART3) = 4 --DD/MM/YYYY
        BEGIN
          SET @ALTERNATIVE = @PART3 + @PART2 + @PART1 --YYYYMMDD
          IF ISDATE(@ALTERNATIVE) = 1
            BEGIN
              SET @ORIGINALDATE = CONVERT(DATETIME, @ALTERNATIVE)
            END
        END
    END
  ELSE IF (@ORIGINAL LIKE '%-%-%') AND (@ORIGINAL NOT LIKE '%-%-%-%')
    BEGIN
      SET @PART1 = SUBSTRING(@ORIGINAL, 1, CHARINDEX('-', @ORIGINAL) - 1)
      SET @PART2 = SUBSTRING(@ORIGINAL, LEN(@PART1) + 2,  CHARINDEX('-', @ORIGINAL, LEN(@PART1) + 2) - LEN(@PART1) - 2)
      SET @PART3 = SUBSTRING(@ORIGINAL, LEN(@PART1) + LEN(@PART2) + 3, LEN(@ORIGINAL) - LEN(@PART1) - LEN(@PART2) - 2)
      IF LEN(@PART1) = 4 --YYYY-MM-DD
        BEGIN
          SET @ALTERNATIVE = CONVERT(VARCHAR(10), @PART1 + @PART2 + @PART3) --YYYYMMDD
          IF ISDATE(@ALTERNATIVE) = 1
            BEGIN
              SET @ORIGINALDATE = CONVERT(DATETIME, @ALTERNATIVE)
            END
        END
      ELSE IF LEN(@PART3) = 4 --DD-MM-YYYY
        BEGIN
          SET @ALTERNATIVE = @PART3 + @PART2 + @PART1 --YYYYMMDD
          IF ISDATE(@ALTERNATIVE) = 1
            BEGIN
              SET @ORIGINALDATE = CONVERT(DATETIME, @ALTERNATIVE)
            END
        END
    END
  ELSE IF (CHARINDEX(' ', @ORIGINAL) > 0) OR (CHARINDEX('/', @ORIGINAL) > 0) OR (CHARINDEX('-', @ORIGINAL) > 0) --KEYD-5744
    BEGIN
      DECLARE @orgtest VARCHAR(MAX)
      SET @orgtest = @original
      SET @orgtest = REPLACE(@orgtest, 'sunday', '')
      SET @orgtest = REPLACE(@orgtest, 'monday', '')
      SET @orgtest = REPLACE(@orgtest, 'tuesday', '')
      SET @orgtest = REPLACE(@orgtest, 'wednesday', '')
      SET @orgtest = REPLACE(@orgtest, 'thursday', '')
      SET @orgtest = REPLACE(@orgtest, 'friday', '')
      SET @orgtest = REPLACE(@orgtest, 'saturday', '')
      SET @orgtest = REPLACE(@orgtest, 'the', '')
      SET @orgtest = REPLACE(@orgtest, 'day', '')
      SET @orgtest = REPLACE(@orgtest, 'of', '')
      SET @orgtest = REPLACE(@orgtest, ',', '')
      SET @orgtest = REPLACE(REPLACE(@orgtest, 'st', ''), 'augu', 'august') --there's always one, isn't there!
      SET @orgtest = REPLACE(@orgtest, 'nd', '')
      SET @orgtest = REPLACE(@orgtest, 'rd', '')
      SET @orgtest = REPLACE(@orgtest, 'th', '')
      SET @orgtest = REPLACE(@orgtest, '  ', ' ')
      SET @orgtest = REPLACE(@orgtest, '  ', ' ')
      SET @orgtest = REPLACE(@orgtest, '  ', ' ')

      IF ISDATE(@orgtest) = 1
        BEGIN
          SET @ORIGINAL = @orgtest
        END

      IF ISDATE(@ORIGINAL) = 1
        BEGIN
          SET @ORIGINALDATE = CONVERT(DATETIME, @ORIGINAL)
        END
    END
  ELSE IF (@Format LIKE '@d%') OR (@Format LIKE '%DD%') OR (@Format LIKE '%MM%') OR (@Format LIKE '%YY%')
    --KEYD-5744
    BEGIN
      IF ISDATE(@ORIGINAL) = 1
        BEGIN
          SET @ORIGINALDATE = CONVERT(DATETIME, @ORIGINAL)
        END
    END

  IF @ORIGINALDATE IS NOT NULL
    BEGIN
      RETURN [dbo].[ky_SUFGetFormattedDate](@ORIGINALDATE, @FORMAT)
    END

  IF @FORMAT LIKE '@d%'
    BEGIN
      RETURN ''
    END

  IF NOT RTRIM(ISNULL(@FORMAT, '')) = ''
    BEGIN
      IF ISNUMERIC(@ORIGINAL) = 1
        BEGIN
          DECLARE @CHECK VARCHAR(8)
          DECLARE @THOUSEP CHAR(1)
          SET @CHECK = CONVERT(VARCHAR, CONVERT(MONEY, 1000), 1)
          SET @THOUSEP = SUBSTRING(@CHECK, 2, 1)

          IF RTRIM(ISNULL(@ORIGINAL, '')) IN ('-', '+')
            BEGIN
              SET @ORIGINAL = '0'
            END

          RETURN [dbo].[ky_SUFGetFormattedValue](REPLACE(@ORIGINAL, @THOUSEP, ''), @FORMAT, 1, 0, NULL)
        END
    END

  RETURN CONVERT(VARCHAR(MAX), @ORIGINAL)
END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_SUFToday'
      AND SO.[type] = 'FN')
  BEGIN
    DROP FUNCTION [dbo].[ky_SUFToday]
  END
GO

CREATE FUNCTION
       [dbo].[ky_SUFToday]
      ()
RETURNS VARCHAR(MAX)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_SUFToday]                                                                                       *
 *                                                                                                           *
 * Performs a Doc Assist "Today()" function. Returns Today's Date in DD MMM YYYY format. For other formats,  *
 * use format - e.g. format(Today(), '@d18')                                                                 *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.6.4.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2019-01-18   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

RETURN [dbo].[ky_SUFFormat](GetDate(), 'DD MMM YYYY')

END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_SUFNow'
      AND SO.[type] = 'FN')
  BEGIN
    DROP FUNCTION [dbo].[ky_SUFNow]
  END
GO

CREATE FUNCTION
       [dbo].[ky_SUFNow]
      ()
RETURNS VARCHAR(MAX)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_SUFNow]                                                                                         *
 *                                                                                                           *
 * Performs a Doc Assist "Now()" function. Returns the current time in DD MMM YYYY format.                   *
 * For other formats, use format - e.g. format(Now(), '@t4')                                                 *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.6.4.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2019-01-18   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

RETURN [dbo].[ky_SUFFormat](GetDate(), 'DD MMM YYYY') + ' ' + CONVERT(VARCHAR, GetDate(), 108)

END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_SUFDoComparison'
      AND SO.[type] = 'FN')
  BEGIN
    DROP FUNCTION [dbo].[ky_SUFDoComparison]
  END
GO

CREATE FUNCTION
       [dbo].[ky_SUFDoComparison]
      (@VALUE1 VARCHAR(MAX),
       @VALUE2 VARCHAR(MAX),
       @OPERATOR VARCHAR(4))
RETURNS BIT
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_SUFDoComparison]                                                                                *
 *                                                                                                           *
 * Performs a Doc Assist "COMPARISON". This does a date comparison if both values are Dates.                 *
 * Does a numerical comparison if both values are numerical, and simply compares string values otherwise     *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.6.4.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2019-01-18   Pino Carafa       Created                                                                    *
 * 2019-01-21   Pino Carafa       Handle large integers that might contain Thousand Separators               *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

DECLARE @DATEVALUE1 VARCHAR(100)
DECLARE @DATEVALUE2 VARCHAR(100)
DECLARE @NUMBERVALUE1 DECIMAL(38, 19)
DECLARE @NUMBERVALUE2 DECIMAL(38, 19)
DECLARE @INTVALUE1 INT
DECLARE @INTVALUE2 INT

IF CHARINDEX(' ', @VALUE1) > 0
OR CHARINDEX('-', @VALUE1) > 0
OR CHARINDEX('/', @VALUE1) > 0
  BEGIN
    SET @DATEVALUE1 = RTRIM(ISNULL([dbo].[ky_SUFFormat](@VALUE1, '@D18'), ''))
  END
ELSE
  BEGIN
    SET @DATEVALUE1 = ''
  END

IF CHARINDEX(' ', @VALUE2) > 0
OR CHARINDEX('-', @VALUE2) > 0
OR CHARINDEX('/', @VALUE2) > 0
  BEGIN
    SET @DATEVALUE2 = RTRIM(ISNULL([dbo].[ky_SUFFormat](@VALUE2, '@D18'), ''))
  END
ELSE
  BEGIN
    SET @DATEVALUE2 = ''
  END

DECLARE @WORK VARCHAR(MAX)
DECLARE @THOUSEP CHAR(1)
DECLARE @DECIMAL CHAR(1)

SET @WORK = CONVERT(VARCHAR, CONVERT(MONEY, 1000), 1)
SET @THOUSEP = SUBSTRING(@WORK, 2, 1)
SET @VALUE1 = REPLACE(@VALUE1, @THOUSEP, '')
SET @VALUE2 = REPLACE(@VALUE2, @THOUSEP, '')
IF @THOUSEP = ','
  BEGIN
    SET @DECIMAL = '.'
  END
ELSE
  BEGIN
    SET @DECIMAL = ','
  END
SET @WORK = ''

IF ISNUMERIC(@VALUE1) = 1
  BEGIN
    SET @NUMBERVALUE1 = CONVERT(DECIMAL(38, 19), @VALUE1)
    IF CHARINDEX(@DECIMAL, @VALUE1) = 0
      BEGIN
        SET @INTVALUE1 = CONVERT(INT, @NUMBERVALUE1)
      END
    ELSE 
      BEGIN
        SET @WORK = SUBSTRING(@VALUE1, CHARINDEX(@DECIMAL, @VALUE1) + 1, LEN(@VALUE1) - CHARINDEX(@DECIMAL, @VALUE1))
        IF (@WORK = '0') OR (@WORK = '00')
          BEGIN
            SET @INTVALUE1 = CONVERT(INT, @NUMBERVALUE1)
          END
        SET @WORK = ''
      END
  END

IF ISNUMERIC(@VALUE2) = 1
  BEGIN
    SET @NUMBERVALUE2 = CONVERT(DECIMAL(38, 19), @VALUE2)
    IF CHARINDEX(@DECIMAL, @VALUE2) = 0
      BEGIN
        SET @INTVALUE2 = CONVERT(INT, @NUMBERVALUE2)
      END
    ELSE 
      BEGIN
        SET @WORK = SUBSTRING(@VALUE2, CHARINDEX(@DECIMAL, @VALUE2) + 1, LEN(@VALUE2) - CHARINDEX(@DECIMAL, @VALUE2))
        IF (@WORK = '0') OR (@WORK = '00')
          BEGIN
            SET @INTVALUE2 = CONVERT(INT, @NUMBERVALUE2)
          END
        SET @WORK = ''
      END
  END

IF (@DATEVALUE1 <> '') AND (@DATEVALUE2 <> '')
  BEGIN
    RETURN CASE WHEN @OPERATOR = '=' AND CONVERT(DATETIME, @DATEVALUE1) = CONVERT(DATETIME, @DATEVALUE2)
                     THEN 1
                WHEN @OPERATOR = '='
                     THEN 0
                WHEN @OPERATOR = '<>' AND CONVERT(DATETIME, @DATEVALUE1) = CONVERT(DATETIME, @DATEVALUE2)
                     THEN 0
                WHEN @OPERATOR = '<>'
                     THEN 1
                WHEN @OPERATOR = '<' AND CONVERT(DATETIME, @DATEVALUE1) < CONVERT(DATETIME, @DATEVALUE2)
                     THEN 1
                WHEN @OPERATOR = '<'
                     THEN 0
                WHEN @OPERATOR = '<=' AND CONVERT(DATETIME, @DATEVALUE1) <= CONVERT(DATETIME, @DATEVALUE2)
                     THEN 1
                WHEN @OPERATOR = '<='
                     THEN 0
                WHEN @OPERATOR = '>' AND CONVERT(DATETIME, @DATEVALUE1) > CONVERT(DATETIME, @DATEVALUE2)
                     THEN 1
                WHEN @OPERATOR = '>'
                     THEN 0
                WHEN @OPERATOR = '>=' AND CONVERT(DATETIME, @DATEVALUE1) >= CONVERT(DATETIME, @DATEVALUE2)
                     THEN 1
                WHEN @OPERATOR = '>='
                     THEN 0
                ELSE 0 END
  END

 IF (CHARINDEX(@DECIMAL, @VALUE1) = 0)
AND (CHARINDEX(@DECIMAL, @VALUE2) = 0)
AND (@INTVALUE1 IS NOT NULL)
AND (@INTVALUE2 IS NOT NULL)
  BEGIN
    RETURN CASE WHEN @OPERATOR = '=' AND @INTVALUE1 = @INTVALUE2
                     THEN 1
                WHEN @OPERATOR = '='
                     THEN 0
                WHEN @OPERATOR = '<>' AND @INTVALUE1 = @INTVALUE2
                     THEN 0
                WHEN @OPERATOR = '<>'
                     THEN 1
                WHEN @OPERATOR = '<' AND @INTVALUE1 < @INTVALUE2
                     THEN 1
                WHEN @OPERATOR = '<'
                     THEN 0
                WHEN @OPERATOR = '<=' AND @INTVALUE1 <= @INTVALUE2
                     THEN 1
                WHEN @OPERATOR = '<='
                     THEN 0
                WHEN @OPERATOR = '>' AND @INTVALUE1 > @INTVALUE2
                     THEN 1
                WHEN @OPERATOR = '>'
                     THEN 0
                WHEN @OPERATOR = '>=' AND @INTVALUE1 >= @INTVALUE2
                     THEN 1
                WHEN @OPERATOR = '>='
                     THEN 0
                ELSE 0 END
  END

 IF (@NUMBERVALUE1 IS NOT NULL)
AND (@NUMBERVALUE2 IS NOT NULL)
  BEGIN
    RETURN CASE WHEN @OPERATOR = '=' AND @NUMBERVALUE1 = @NUMBERVALUE2
                     THEN 1
                WHEN @OPERATOR = '='
                     THEN 0
                WHEN @OPERATOR = '<>' AND @NUMBERVALUE1 = @NUMBERVALUE2
                     THEN 0
                WHEN @OPERATOR = '<>'
                     THEN 1
                WHEN @OPERATOR = '<' AND @NUMBERVALUE1 < @NUMBERVALUE2
                     THEN 1
                WHEN @OPERATOR = '<'
                     THEN 0
                WHEN @OPERATOR = '<=' AND @NUMBERVALUE1 <= @NUMBERVALUE2
                     THEN 1
                WHEN @OPERATOR = '<='
                     THEN 0
                WHEN @OPERATOR = '>' AND @NUMBERVALUE1 > @NUMBERVALUE2
                     THEN 1
                WHEN @OPERATOR = '>'
                     THEN 0
                WHEN @OPERATOR = '>=' AND @NUMBERVALUE1 >= @NUMBERVALUE2
                     THEN 1
                WHEN @OPERATOR = '>='
                     THEN 0
                ELSE 0 END
  END

--Else just compare strings
RETURN CASE WHEN @OPERATOR = '=' AND @VALUE1 = @VALUE2
                 THEN 1
            WHEN @OPERATOR = '='
                 THEN 0
            WHEN @OPERATOR = '<>' AND @VALUE1 = @VALUE2
                 THEN 0
            WHEN @OPERATOR = '<>'
                 THEN 1
            WHEN @OPERATOR = '<' AND @VALUE1 < @VALUE2
                 THEN 1
            WHEN @OPERATOR = '<'
                 THEN 0
            WHEN @OPERATOR = '<=' AND @VALUE1 <= @VALUE2
                 THEN 1
            WHEN @OPERATOR = '<='
                 THEN 0
            WHEN @OPERATOR = '>' AND @VALUE1 > @VALUE2
                 THEN 1
            WHEN @OPERATOR = '>'
                 THEN 0
            WHEN @OPERATOR = '>=' AND @VALUE1 >= @VALUE2
                 THEN 1
            WHEN @OPERATOR = '>='
                 THEN 0
            ELSE 0 END

END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_SUFDateDiff'
      AND SO.[type] = 'FN')
  BEGIN
    DROP FUNCTION [dbo].[ky_SUFDateDiff]
  END
GO

CREATE FUNCTION
       [dbo].[ky_SUFDateDiff]
      (@DIFF   VARCHAR(10),
       @VALUE1 VARCHAR(MAX),
       @VALUE2 VARCHAR(MAX),
       @FORAGE BIT)
RETURNS VARCHAR(MAX)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_SUFDateDiff]                                                                                    *
 *                                                                                                           *
 * @DIFF Can be 'Day', 'Week', 'Month', 'Year'                                                               *
 *                                                                                                           *
 * Returns the difference between two dates as number of days/weeks/months/years                             *
 *                                                                                                           *
 * The function will simply return the difference between the requested value (Day/Week/Month/Year) if       *
 * @FORAGE = 0, but will calculate an "age" if it is set to 1                                                *
 *                                                                                                           *
 * Example: f.SUFDateDiff('year', '17 July 2003', '16 July 2019', 0) will return 16 as there are simply 16   *
 *          years between 2003 and 2019 but                                                                  *
 *          f.SUFDateDiff('year', '17 July 2003', '16 July 2019', 1) will return 15 because the "birthday"   *
 *          has not passed yet so this person is still "15 years old"                                        *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.6.4.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2019-02-19   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

DECLARE @DATEVALUE1 VARCHAR(100)
DECLARE @DATEVALUE2 VARCHAR(100)
DECLARE @DATE1 DATETIME
DECLARE @DATE2 DATETIME

IF CHARINDEX(' ', @VALUE1) > 0
OR CHARINDEX('-', @VALUE1) > 0
OR CHARINDEX('/', @VALUE1) > 0
  BEGIN
    SET @DATEVALUE1 = RTRIM(ISNULL([dbo].[ky_SUFFormat](@VALUE1, '@D18'), ''))
  END
ELSE
  BEGIN
    SET @DATEVALUE1 = ''
  END

IF CHARINDEX(' ', @VALUE2) > 0
OR CHARINDEX('-', @VALUE2) > 0
OR CHARINDEX('/', @VALUE2) > 0
  BEGIN
    SET @DATEVALUE2 = RTRIM(ISNULL([dbo].[ky_SUFFormat](@VALUE2, '@D18'), ''))
  END
ELSE
  BEGIN
    SET @DATEVALUE2 = ''
  END

DECLARE @MINUS BIT
DECLARE @RESULT VARCHAR(MAX)

IF (@DATEVALUE1 <> '') AND (@DATEVALUE2 <> '')
  BEGIN
    SET @DATE1 = CONVERT(DATETIME, @DATEVALUE1)
    SET @DATE2 = CONVERT(DATETIME, @DATEVALUE2)
    IF @DATE1 > @DATE2
      BEGIN
        SET @MINUS = 1
        SET @DATE1 = CONVERT(DATETIME, @DATEVALUE2)
        SET @DATE2 = CONVERT(DATETIME, @DATEVALUE1)
      END

    DECLARE @YEAR INT
    DECLARE @MONTH1 INT
    DECLARE @MONTH2 INT
    DECLARE @DAY1 INT
    DECLARE @DAY2 INT

    IF @MINUS = 1
      BEGIN
        SET @RESULT = '-'
      END
    ELSE
      BEGIN
        SET @RESULT = ''
      END

    IF @DIFF = 'day'
      BEGIN
        SET @RESULT = @RESULT + CONVERT(VARCHAR(10), DATEDIFF(DAY, @DATE1, @DATE2))
      END        
        
    IF @DIFF = 'week'
      BEGIN
        SET @RESULT = @RESULT + CONVERT(VARCHAR(10), DATEDIFF(WEEK, @DATE1, @DATE2))
      END        
        
    IF @DIFF = 'month'
      BEGIN
        SET @MONTH1 = DATEDIFF(MONTH, @DATE1, @DATE2)
        IF @FORAGE = 1
          BEGIN
            SET @DAY1   = DAY(@DATE1)
            SET @DAY2   = DAY(@DATE2)
            IF @DAY1 > @DAY2
              BEGIN
                SET @MONTH1 = @MONTH1 - 1
              END
          END

        SET @RESULT = @RESULT + CONVERT(VARCHAR(10), @MONTH1)
      END        
        
    IF @DIFF = 'year'
      BEGIN
        SET @YEAR = DATEDIFF(YEAR, @DATE1, @DATE2)
        SET @MONTH1 = MONTH(@DATE1)
        SET @MONTH2 = MONTH(@DATE2)
        SET @DAY1   = DAY(@DATE1)
        SET @DAY2   = DAY(@DATE2)
        IF @FORAGE = 1
          BEGIN
            IF @MONTH1 > @MONTH2
              BEGIN
                SET @YEAR = @YEAR - 1
              END
            ELSE IF @MONTH1 = @MONTH2
              BEGIN
                IF @DAY1 > @DAY2
                  BEGIN
                    SET @YEAR = @YEAR - 1
                  END
              END
          END

        SET @RESULT = @RESULT + CONVERT(VARCHAR(10), @YEAR)
      END        
  END

RETURN @RESULT

END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_SUFDoAdd'
      AND SO.[type] = 'FN')
  BEGIN
    DROP FUNCTION [dbo].[ky_SUFDoAdd]
  END
GO

CREATE FUNCTION
       [dbo].[ky_SUFDoAdd]
      (@VALUE1 VARCHAR(MAX),
       @VALUE2 VARCHAR(MAX))
RETURNS VARCHAR(MAX)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_SUFDoAdd]                                                                                       *
 *                                                                                                           *
 * Performs a Doc Assist "ADD". This adds days if one of the values is a date time and the other is a number.*
 * Does a numerical add if both values are numerical, and simply concatenates string values otherwise        *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.6.4.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2019-01-18   Pino Carafa       Created                                                                    *
 * 2019-01-21   Pino Carafa       Handle large integers that might contain Thousand Separators               *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

DECLARE @DATEVALUE1 VARCHAR(100)
DECLARE @DATEVALUE2 VARCHAR(100)
DECLARE @NUMBERVALUE1 DECIMAL(38, 19)
DECLARE @NUMBERVALUE2 DECIMAL(38, 19)
DECLARE @INTVALUE1 INT
DECLARE @INTVALUE2 INT

IF CHARINDEX(' ', @VALUE1) > 0
OR CHARINDEX('-', @VALUE1) > 0
OR CHARINDEX('/', @VALUE1) > 0
  BEGIN
    SET @DATEVALUE1 = RTRIM(ISNULL([dbo].[ky_SUFFormat](@VALUE1, '@D18'), ''))
  END
ELSE
  BEGIN
    SET @DATEVALUE1 = ''
  END

IF CHARINDEX(' ', @VALUE2) > 0
OR CHARINDEX('-', @VALUE2) > 0
OR CHARINDEX('/', @VALUE2) > 0
  BEGIN
    SET @DATEVALUE2 = RTRIM(ISNULL([dbo].[ky_SUFFormat](@VALUE2, '@D18'), ''))
  END
ELSE
  BEGIN
    SET @DATEVALUE2 = ''
  END

DECLARE @WORK VARCHAR(MAX)
DECLARE @THOUSEP CHAR(1)
DECLARE @DECIMAL CHAR(1)

SET @WORK = CONVERT(VARCHAR, CONVERT(MONEY, 1000), 1)
SET @THOUSEP = SUBSTRING(@WORK, 2, 1)
SET @VALUE1 = REPLACE(@VALUE1, @THOUSEP, '')
SET @VALUE2 = REPLACE(@VALUE2, @THOUSEP, '')
IF @THOUSEP = ','
  BEGIN
    SET @DECIMAL = '.'
  END
ELSE
  BEGIN
    SET @DECIMAL = ','
  END
SET @WORK = ''

IF ISNUMERIC(@VALUE1) = 1
  BEGIN
    SET @NUMBERVALUE1 = CONVERT(DECIMAL(38, 19), @VALUE1)
    IF CHARINDEX(@DECIMAL, @VALUE1) = 0
      BEGIN
        SET @INTVALUE1 = CONVERT(INT, @NUMBERVALUE1)
      END
    ELSE 
      BEGIN
        SET @WORK = SUBSTRING(@VALUE1, CHARINDEX(@DECIMAL, @VALUE1) + 1, LEN(@VALUE1) - CHARINDEX(@DECIMAL, @VALUE1))
        IF (@WORK = '0') OR (@WORK = '00')
          BEGIN
            SET @INTVALUE1 = CONVERT(INT, @NUMBERVALUE1)
          END
        SET @WORK = ''
      END
  END

IF ISNUMERIC(@VALUE2) = 1
  BEGIN
    SET @NUMBERVALUE2 = CONVERT(DECIMAL(38, 19), @VALUE2)
    IF CHARINDEX(@DECIMAL, @VALUE2) = 0
      BEGIN
        SET @INTVALUE2 = CONVERT(INT, @NUMBERVALUE2)
      END
    ELSE 
      BEGIN
        SET @WORK = SUBSTRING(@VALUE2, CHARINDEX(@DECIMAL, @VALUE2) + 1, LEN(@VALUE2) - CHARINDEX(@DECIMAL, @VALUE2))
        IF (@WORK = '0') OR (@WORK = '00')
          BEGIN
            SET @INTVALUE2 = CONVERT(INT, @NUMBERVALUE2)
          END
        SET @WORK = ''
      END
  END

IF (@DATEVALUE1 <> '') AND (@INTVALUE2 IS NOT NULL)
  BEGIN
    RETURN [dbo].[ky_SUFFormat](DATEADD(DAY, @INTVALUE2, CONVERT(DATETIME, @DATEVALUE1)), '@d18')
  END

IF (@DATEVALUE2 <> '') AND (@INTVALUE1 IS NOT NULL)
  BEGIN
    RETURN [dbo].[ky_SUFFormat](DATEADD(DAY, @INTVALUE1, CONVERT(DATETIME, @DATEVALUE2)), '@d18')
  END

 IF (CHARINDEX(@DECIMAL, @VALUE1) = 0)
AND (CHARINDEX(@DECIMAL, @VALUE2) = 0)
AND (@INTVALUE1 IS NOT NULL)
AND (@INTVALUE2 IS NOT NULL)
  BEGIN
    RETURN [dbo].[ky_SUFFormat](CONVERT(VARCHAR(30), @INTVALUE1 + @INTVALUE2), 'N0')
  END

 IF (@NUMBERVALUE1 IS NOT NULL)
AND (@NUMBERVALUE2 IS NOT NULL)
  BEGIN
    RETURN [dbo].[ky_SUFFormat](CONVERT(VARCHAR(30), @NUMBERVALUE1 + @NUMBERVALUE2), 'N2')
  END

--Else just concatenate
RETURN @VALUE1 + @VALUE2

END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_SUFConcatenate'
      AND SO.[type] = 'FN')
  BEGIN
    DROP FUNCTION [dbo].[ky_SUFConcatenate]
  END
GO

CREATE FUNCTION
       [dbo].[ky_SUFConcatenate]
      (@VALUE1 VARCHAR(MAX),
       @VALUE2 VARCHAR(MAX))
RETURNS VARCHAR(MAX)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_SUFConcatenate]                                                                                 *
 *                                                                                                           *
 * Use this when a standard Add (+, & or f.DoAdd) causes the system to produce the wrong result.             *
 * This can happen because f.DoAdd will interpret Date and Numerical values and interpret the + as an actual *
 * operator.                                                                                                 *
 * For example SYS:Today() + '10' will return a date 10 days from today.                                     *
 * While this is exactly what you'd want most of the time, if you ever need to literally concatenate the     *
 * string '10' to a date, you can now use (for example) concatenate(Today(), '10') instead                   *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.6.4.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2019-01-21   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

RETURN @VALUE1 + @VALUE2

END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_SUFDoSubtract'
      AND SO.[type] = 'FN')
  BEGIN
    DROP FUNCTION [dbo].[ky_SUFDoSubtract]
  END
GO

CREATE FUNCTION
       [dbo].[ky_SUFDoSubtract]
      (@VALUE1 VARCHAR(MAX),
       @VALUE2 VARCHAR(MAX))
RETURNS VARCHAR(MAX)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_SUFDoSubtract]                                                                                  *
 *                                                                                                           *
 * Performs a Doc Assist "SUBTRACT". This subtracts days if the first value is a date time and the second    *
 * is an integer. Does a numerical add if both values are numerical. Returns blank otherwise.                *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.6.4.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2019-01-18   Pino Carafa       Created                                                                    *
 * 2019-01-21   Pino Carafa       Handle large integers that might contain Thousand Separators               *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

DECLARE @DATEVALUE1 VARCHAR(100)
DECLARE @NUMBERVALUE1 DECIMAL(38, 19)
DECLARE @NUMBERVALUE2 DECIMAL(38, 19)
DECLARE @INTVALUE1 INT
DECLARE @INTVALUE2 INT

IF CHARINDEX(' ', @VALUE1) > 0
OR CHARINDEX('-', @VALUE1) > 0
OR CHARINDEX('/', @VALUE1) > 0
  BEGIN
    SET @DATEVALUE1 = RTRIM(ISNULL([dbo].[ky_SUFFormat](@VALUE1, '@D18'), ''))
  END
ELSE
  BEGIN
    SET @DATEVALUE1 = ''
  END

DECLARE @WORK VARCHAR(MAX)
DECLARE @THOUSEP CHAR(1)
DECLARE @DECIMAL CHAR(1)

SET @WORK = CONVERT(VARCHAR, CONVERT(MONEY, 1000), 1)
SET @THOUSEP = SUBSTRING(@WORK, 2, 1)
SET @VALUE1 = REPLACE(@VALUE1, @THOUSEP, '')
SET @VALUE2 = REPLACE(@VALUE2, @THOUSEP, '')
IF @THOUSEP = ','
  BEGIN
    SET @DECIMAL = '.'
  END
ELSE
  BEGIN
    SET @DECIMAL = ','
  END
SET @WORK = ''

IF ISNUMERIC(@VALUE1) = 1
  BEGIN
    SET @NUMBERVALUE1 = CONVERT(DECIMAL(38, 19), @VALUE1)
    IF CHARINDEX(@DECIMAL, @VALUE1) = 0
      BEGIN
        SET @INTVALUE1 = CONVERT(INT, @NUMBERVALUE1)
      END
    ELSE 
      BEGIN
        SET @WORK = SUBSTRING(@VALUE1, CHARINDEX(@DECIMAL, @VALUE1) + 1, LEN(@VALUE1) - CHARINDEX(@DECIMAL, @VALUE1))
        IF (@WORK = '0') OR (@WORK = '00')
          BEGIN
            SET @INTVALUE1 = CONVERT(INT, @NUMBERVALUE1)
          END
        SET @WORK = ''
      END
  END

IF ISNUMERIC(@VALUE2) = 1
  BEGIN
    SET @NUMBERVALUE2 = CONVERT(DECIMAL(38, 19), @VALUE2)
    IF CHARINDEX(@DECIMAL, @VALUE2) = 0
      BEGIN
        SET @INTVALUE2 = CONVERT(INT, @NUMBERVALUE2)
      END
    ELSE 
      BEGIN
        SET @WORK = SUBSTRING(@VALUE2, CHARINDEX(@DECIMAL, @VALUE2) + 1, LEN(@VALUE2) - CHARINDEX(@DECIMAL, @VALUE2))
        IF (@WORK = '0') OR (@WORK = '00')
          BEGIN
            SET @INTVALUE2 = CONVERT(INT, @NUMBERVALUE2)
          END
        SET @WORK = ''
      END
  END

IF (@DATEVALUE1 <> '') AND (@INTVALUE2 IS NOT NULL)
  BEGIN
    RETURN [dbo].[ky_SUFFormat](DATEADD(DAY, -@INTVALUE2, CONVERT(DATETIME, @DATEVALUE1)), '@d18')
  END

 IF (CHARINDEX(@DECIMAL, @VALUE1) = 0)
AND (CHARINDEX(@DECIMAL, @VALUE2) = 0)
AND (@INTVALUE1 IS NOT NULL)
AND (@INTVALUE2 IS NOT NULL)
  BEGIN
    RETURN [dbo].[ky_SUFFormat](CONVERT(VARCHAR(30), @INTVALUE1 - @INTVALUE2), 'N0')
  END

 IF (@NUMBERVALUE1 IS NOT NULL)
AND (@NUMBERVALUE2 IS NOT NULL)
  BEGIN
    RETURN [dbo].[ky_SUFFormat](CONVERT(VARCHAR(30), @NUMBERVALUE1 - @NUMBERVALUE2), 'N2')
  END

--Else return blank
RETURN ''

END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_SUFDoMultiply'
      AND SO.[type] = 'FN')
  BEGIN
    DROP FUNCTION [dbo].[ky_SUFDoMultiply]
  END
GO

CREATE FUNCTION
       [dbo].[ky_SUFDoMultiply]
      (@VALUE1 VARCHAR(MAX),
       @VALUE2 VARCHAR(MAX))
RETURNS VARCHAR(MAX)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_SUFDoMultiply]                                                                                  *
 *                                                                                                           *
 * Performs a Doc Assist "MULTIPLY". Does a numerical multiply if both values are numerical.                 *
 * Returns blank otherwise.                                                                                  *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.6.4.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2019-01-18   Pino Carafa       Created                                                                    *
 * 2019-01-21   Pino Carafa       Handle large integers that might contain Thousand Separators               *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

DECLARE @NUMBERVALUE1 DECIMAL(38, 19)
DECLARE @NUMBERVALUE2 DECIMAL(38, 19)
DECLARE @INTVALUE1 INT
DECLARE @INTVALUE2 INT

DECLARE @WORK VARCHAR(MAX)
DECLARE @THOUSEP CHAR(1)
DECLARE @DECIMAL CHAR(1)

SET @WORK = CONVERT(VARCHAR, CONVERT(MONEY, 1000), 1)
SET @THOUSEP = SUBSTRING(@WORK, 2, 1)
SET @VALUE1 = REPLACE(@VALUE1, @THOUSEP, '')
SET @VALUE2 = REPLACE(@VALUE2, @THOUSEP, '')
IF @THOUSEP = ','
  BEGIN
    SET @DECIMAL = '.'
  END
ELSE
  BEGIN
    SET @DECIMAL = ','
  END
SET @WORK = ''

IF ISNUMERIC(@VALUE1) = 1
  BEGIN
    SET @NUMBERVALUE1 = CONVERT(DECIMAL(38, 19), @VALUE1)
    IF CHARINDEX(@DECIMAL, @VALUE1) = 0
      BEGIN
        SET @INTVALUE1 = CONVERT(INT, @NUMBERVALUE1)
      END
    ELSE 
      BEGIN
        SET @WORK = SUBSTRING(@VALUE1, CHARINDEX(@DECIMAL, @VALUE1) + 1, LEN(@VALUE1) - CHARINDEX(@DECIMAL, @VALUE1))
        IF (@WORK = '0') OR (@WORK = '00')
          BEGIN
            SET @INTVALUE1 = CONVERT(INT, @NUMBERVALUE1)
          END
        SET @WORK = ''
      END
  END

IF ISNUMERIC(@VALUE2) = 1
  BEGIN
    SET @NUMBERVALUE2 = CONVERT(DECIMAL(38, 19), @VALUE2)
    IF CHARINDEX(@DECIMAL, @VALUE2) = 0
      BEGIN
        SET @INTVALUE2 = CONVERT(INT, @NUMBERVALUE2)
      END
    ELSE 
      BEGIN
        SET @WORK = SUBSTRING(@VALUE2, CHARINDEX(@DECIMAL, @VALUE2) + 1, LEN(@VALUE2) - CHARINDEX(@DECIMAL, @VALUE2))
        IF (@WORK = '0') OR (@WORK = '00')
          BEGIN
            SET @INTVALUE2 = CONVERT(INT, @NUMBERVALUE2)
          END
        SET @WORK = ''
      END
  END

 IF (CHARINDEX(@DECIMAL, @VALUE1) = 0)
AND (CHARINDEX(@DECIMAL, @VALUE2) = 0)
AND (@INTVALUE1 IS NOT NULL)
AND (@INTVALUE2 IS NOT NULL)
  BEGIN
    RETURN [dbo].[ky_SUFFormat](CONVERT(VARCHAR(30), @INTVALUE1 * @INTVALUE2), 'N0')
  END

 IF (@NUMBERVALUE1 IS NOT NULL)
AND (@NUMBERVALUE2 IS NOT NULL)
  BEGIN
    RETURN [dbo].[ky_SUFFormat](CONVERT(VARCHAR(30), @NUMBERVALUE1 * @NUMBERVALUE2), 'N2')
  END

--Else return blank
RETURN ''

END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_SUFDoDivide'
      AND SO.[type] = 'FN')
  BEGIN
    DROP FUNCTION [dbo].[ky_SUFDoDivide]
  END
GO

CREATE FUNCTION
       [dbo].[ky_SUFDoDivide]
      (@VALUE1 VARCHAR(MAX),
       @VALUE2 VARCHAR(MAX))
RETURNS VARCHAR(MAX)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_SUFDoDivide]                                                                                    *
 *                                                                                                           *
 * Performs a Doc Assist "Divide". Does a numerical divide if both values are numerical.                     *
 * Returns blank otherwise.                                                                                  *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.6.4.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2019-01-18   Pino Carafa       Created                                                                    *
 * 2019-01-21   Pino Carafa       Handle large integers that might contain Thousand Separators               *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

DECLARE @NUMBERVALUE1 DECIMAL(38, 19)
DECLARE @NUMBERVALUE2 DECIMAL(38, 19)
DECLARE @INTVALUE1 INT
DECLARE @INTVALUE2 INT

DECLARE @WORK VARCHAR(MAX)
DECLARE @THOUSEP CHAR(1)
DECLARE @DECIMAL CHAR(1)

SET @WORK = CONVERT(VARCHAR, CONVERT(MONEY, 1000), 1)
SET @THOUSEP = SUBSTRING(@WORK, 2, 1)
SET @VALUE1 = REPLACE(@VALUE1, @THOUSEP, '')
SET @VALUE2 = REPLACE(@VALUE2, @THOUSEP, '')
IF @THOUSEP = ','
  BEGIN
    SET @DECIMAL = '.'
  END
ELSE
  BEGIN
    SET @DECIMAL = ','
  END
SET @WORK = ''

IF ISNUMERIC(@VALUE1) = 1
  BEGIN
    SET @NUMBERVALUE1 = CONVERT(DECIMAL(38, 19), @VALUE1)
    IF CHARINDEX(@DECIMAL, @VALUE1) = 0
      BEGIN
        SET @INTVALUE1 = CONVERT(INT, @NUMBERVALUE1)
      END
    ELSE 
      BEGIN
        SET @WORK = SUBSTRING(@VALUE1, CHARINDEX(@DECIMAL, @VALUE1) + 1, LEN(@VALUE1) - CHARINDEX(@DECIMAL, @VALUE1))
        IF (@WORK = '0') OR (@WORK = '00')
          BEGIN
            SET @INTVALUE1 = CONVERT(INT, @NUMBERVALUE1)
          END
        SET @WORK = ''
      END
  END

IF ISNUMERIC(@VALUE2) = 1
  BEGIN
    SET @NUMBERVALUE2 = CONVERT(DECIMAL(38, 19), @VALUE2)
    IF CHARINDEX(@DECIMAL, @VALUE2) = 0
      BEGIN
        SET @INTVALUE2 = CONVERT(INT, @NUMBERVALUE2)
      END
    ELSE 
      BEGIN
        SET @WORK = SUBSTRING(@VALUE2, CHARINDEX(@DECIMAL, @VALUE2) + 1, LEN(@VALUE2) - CHARINDEX(@DECIMAL, @VALUE2))
        IF (@WORK = '0') OR (@WORK = '00')
          BEGIN
            SET @INTVALUE2 = CONVERT(INT, @NUMBERVALUE2)
          END
        SET @WORK = ''
      END
  END

 IF (CHARINDEX(@DECIMAL, @VALUE1) = 0)
AND (CHARINDEX(@DECIMAL, @VALUE2) = 0)
AND (@INTVALUE1 IS NOT NULL)
AND (@INTVALUE2 IS NOT NULL)
  BEGIN
    IF @INTVALUE2 <> 0
      BEGIN
        RETURN [dbo].[ky_SUFFormat](CONVERT(VARCHAR(30), @INTVALUE1 / @INTVALUE2), 'N0')
      END
    ELSE
      BEGIN
        RETURN ''
      END
  END

 IF (@NUMBERVALUE1 IS NOT NULL)
AND (@NUMBERVALUE2 IS NOT NULL)
  BEGIN
    IF @NUMBERVALUE2 <> 0
      BEGIN
        RETURN [dbo].[ky_SUFFormat](CONVERT(VARCHAR(30), @NUMBERVALUE1 / @NUMBERVALUE2), 'N2')
      END
    ELSE
      BEGIN
        RETURN ''
      END
  END

--Else return blank
RETURN ''

END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = 'ky_SUFDateNumber'
      AND SO.[type] = 'FN')
  BEGIN
    DROP FUNCTION [dbo].[ky_SUFDateNumber]
  END
GO

CREATE FUNCTION
       [dbo].[ky_SUFDateNumber]
      (@ORIGINAL  DATETIME,
       @VALUE     VARCHAR(10))
RETURNS INT
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_SUFDateNumber]                                                                                  *
 *                                                                                                           *
 * To be used to get DAY, MONTH, YEAR for Document Generator codes                                           *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.6.4.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2019-01-17   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

  IF @ORIGINAL IS NULL
    RETURN 0

  IF @VALUE = 'DAY'
    RETURN DAY(@ORIGINAL)

  IF @VALUE = 'MONTH'
    RETURN MONTH(@ORIGINAL)

  IF @VALUE = 'YEAR'
    RETURN YEAR(@ORIGINAL)

  RETURN 0
END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_SUFAddressLine'
      AND SO.[type] = 'FN')
  BEGIN
    DROP FUNCTION [dbo].[ky_SUFAddressLine]
  END
GO

CREATE FUNCTION
       [dbo].[ky_SUFAddressLine]
      (@Address VARCHAR(MAX),
       @LINENO INT)
RETURNS VARCHAR(MAX)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_SUFAddressLine]                                                                                 *
 *                                                                                                           *
 * Get an address line from an address.                                                                      *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.6.4.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2019-01-21   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

RETURN [dbo].[KY_NETFNAddressLine](@Address, @LINENO)

END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETFNGetUnembeddedCharacterPosFrom'
      AND SO.[type] = 'FN')
  BEGIN
    DROP FUNCTION [dbo].[ky_NETFNGetUnembeddedCharacterPosFrom]
  END
GO

CREATE FUNCTION
       [dbo].[ky_NETFNGetUnembeddedCharacterPosFrom]
      (@CHARACTER      CHAR(1),
       @ORIGINAL       VARCHAR(MAX),
       @START          INT)
RETURNS INT
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETFNGetUnembeddedCharacterPosFrom]                                                             *
 *                                                                                                           *
 * Equivalent to CHARINDEX except this one doesn't count anything that's quoted or embedded within brackets  *
 * - with START position                                                                                     *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.6.4.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2019-01-17   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

DECLARE @SIMPLEQUOTE VARCHAR(MAX)

DECLARE @POS                 INT
DECLARE @SIMPLEQUOTEINDENT   BIT
DECLARE @QUOTEINDENT         BIT
DECLARE @LEN                 INT
DECLARE @CHAR                CHAR(1)
DECLARE @ORIGINALCHAR        VARCHAR(1)
DECLARE @BRACKINDENT         INT

SET @SIMPLEQUOTE = REPLACE(REPLACE(@ORIGINAL, '', ''''), '', '''')
SET @POS = @START - 1
IF @POS < 0
  BEGIN
    SET @POS = 0
  END
SET @SIMPLEQUOTEINDENT = 0
SET @QUOTEINDENT = 0
SET @BRACKINDENT = 0
SET @LEN = LEN(@ORIGINAL)

WHILE @POS < @LEN
  BEGIN
    SET @POS = @POS + 1

    SET @CHAR = SUBSTRING(@SIMPLEQUOTE, @POS, 1)
    SET @ORIGINALCHAR = SUBSTRING(@ORIGINAL, @POS, 1)

    --First we establish whether we're in a Quoted or an Unembedded part of the string
    IF @CHAR = ''''
      BEGIN
        IF @ORIGINALCHAR = ''''
          BEGIN
            SET @SIMPLEQUOTEINDENT = 1 - @SIMPLEQUOTEINDENT
          END
        ELSE
          --it's a smart quote
          BEGIN
            IF @SIMPLEQUOTEINDENT = 0
              BEGIN
                SET @QUOTEINDENT = 1 - @QUOTEINDENT
                SET @ORIGINALCHAR = ''''
              END
          END

      END

    IF (@SIMPLEQUOTEINDENT = 0) AND (@QUOTEINDENT = 0)
      BEGIN
        IF @CHAR = '('
          BEGIN
            SET @BRACKINDENT = @BRACKINDENT + 1
          END
        ELSE IF @CHAR = ')'
          BEGIN
            SET @BRACKINDENT = @BRACKINDENT - 1
          END

        IF @BRACKINDENT = 0
          BEGIN
            IF @CHAR = @CHARACTER
              BEGIN
                RETURN @POS
              END
          END
      END
  END

RETURN 0

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETFNGetUnembeddedCharacterPos'
      AND SO.[type] = 'FN')
  BEGIN
    DROP FUNCTION [dbo].[ky_NETFNGetUnembeddedCharacterPos]
  END
GO

CREATE FUNCTION
       [dbo].[ky_NETFNGetUnembeddedCharacterPos]
      (@CHARACTER      CHAR(1),
       @ORIGINAL       VARCHAR(MAX))
RETURNS INT
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETFNGetUnembeddedCharacterPos]                                                                 *
 *                                                                                                           *
 * Equivalent to CHARINDEX except this one doesn't count anything that's quoted or embedded within brackets  *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.6.4.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2019-01-17   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

RETURN [dbo].[ky_NETFNGetUnembeddedCharacterPosFrom](@CHARACTER, @ORIGINAL, 0)

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETFNGetUnquotedCharacterPosFrom'
      AND SO.[type] = 'FN')
  BEGIN
    DROP FUNCTION [dbo].[ky_NETFNGetUnquotedCharacterPosFrom]
  END
GO

CREATE FUNCTION
       [dbo].[ky_NETFNGetUnquotedCharacterPosFrom]
      (@CHARACTER      CHAR(1),
       @ORIGINAL       VARCHAR(MAX),
       @START          INT)
RETURNS INT
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETFNGetUnquotedCharacterPosFrom]                                                               *
 *                                                                                                           *
 * Equivalent to CHARINDEX except this one doesn't count anything that's quoted - with START position        *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.6.4.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2019-01-15   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

DECLARE @SIMPLEQUOTE VARCHAR(MAX)

DECLARE @POS                 INT
DECLARE @SIMPLEQUOTEINDENT   BIT
DECLARE @QUOTEINDENT         BIT
DECLARE @LEN                 INT
DECLARE @CHAR                CHAR(1)
DECLARE @ORIGINALCHAR        VARCHAR(1)

SET @SIMPLEQUOTE = REPLACE(REPLACE(@ORIGINAL, '', ''''), '', '''')
SET @POS = @START - 1
IF @POS < 0
  BEGIN
    SET @POS = 0
  END
SET @SIMPLEQUOTEINDENT = 0
SET @QUOTEINDENT = 0
SET @LEN = LEN(@ORIGINAL)

WHILE @POS < @LEN
  BEGIN
    SET @POS = @POS + 1

    SET @CHAR = SUBSTRING(@SIMPLEQUOTE, @POS, 1)
    SET @ORIGINALCHAR = SUBSTRING(@ORIGINAL, @POS, 1)

    --First we establish whether we're in a Quoted or an Unquoted part of the string
    IF @CHAR = ''''
      BEGIN
        IF @ORIGINALCHAR = ''''
          BEGIN
            SET @SIMPLEQUOTEINDENT = 1 - @SIMPLEQUOTEINDENT
          END
        ELSE
          --it's a smart quote
          BEGIN
            IF @SIMPLEQUOTEINDENT = 0
              BEGIN
                SET @QUOTEINDENT = 1 - @QUOTEINDENT
                SET @ORIGINALCHAR = ''''
              END
          END

      END

    IF (@SIMPLEQUOTEINDENT = 0) AND (@QUOTEINDENT = 0)
      BEGIN
        IF @CHAR = @CHARACTER
          BEGIN
            RETURN @POS
          END
      END
  END

RETURN 0

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETFNGetUnquotedCharacterPos'
      AND SO.[type] = 'FN')
  BEGIN
    DROP FUNCTION [dbo].[ky_NETFNGetUnquotedCharacterPos]
  END
GO

CREATE FUNCTION
       [dbo].[ky_NETFNGetUnquotedCharacterPos]
      (@CHARACTER      CHAR(1),
       @ORIGINAL       VARCHAR(MAX))
RETURNS INT
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETFNGetUnquotedCharacterPos]                                                                   *
 *                                                                                                           *
 * Equivalent to CHARINDEX except this one doesn't count anything that's quoted                              *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.6.4.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2019-01-14   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

RETURN [dbo].[ky_NETFNGetUnquotedCharacterPosFrom](@CHARACTER, @ORIGINAL, 0)

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = 'ky_SUFSubstring'
      AND SO.[type] = 'FN')
  BEGIN
    DROP FUNCTION [dbo].[ky_SUFSubstring]
  END
GO

CREATE FUNCTION
       [dbo].[ky_SUFSubstring]
      (@ORIGINAL  VARCHAR(MAX), @START INT, @LEN INT)
RETURNS VARCHAR(MAX)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_SUFSubstring]                                                                                   *
 *                                                                                                           *
 * To be used as SUBSTRING function for Document Generator codes                                             *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.6.4.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2019-01-11   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/

BEGIN
  RETURN SUBSTRING(@ORIGINAL, @START, @LEN)
END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = 'ky_SUFClip'
      AND SO.[type] = 'FN')
  BEGIN
    DROP FUNCTION [dbo].[ky_SUFClip]
  END
GO

CREATE FUNCTION
       [dbo].[ky_SUFClip]
      (@ORIGINAL  VARCHAR(MAX))
RETURNS VARCHAR(MAX)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_SUFClip]                                                                                        *
 *                                                                                                           *
 * To be used as Clip function for Document Generator codes                                                  *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.6.4.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2019-01-11   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/

BEGIN
  RETURN RTRIM(LTRIM(ISNULL(@ORIGINAL, '')))
END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = 'ky_SUFLTrim'
      AND SO.[type] = 'FN')
  BEGIN
    DROP FUNCTION [dbo].[ky_SUFLTrim]
  END
GO

CREATE FUNCTION
       [dbo].[ky_SUFLTrim]
      (@ORIGINAL  VARCHAR(MAX))
RETURNS VARCHAR(MAX)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_SUFLTrim]                                                                                       *
 *                                                                                                           *
 * To be used as LTrim function for Document Generator codes                                                 *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.6.4.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2019-01-11   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/

BEGIN
  RETURN LTRIM(ISNULL(@ORIGINAL, ''))
END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = 'ky_SUFRTrim'
      AND SO.[type] = 'FN')
  BEGIN
    DROP FUNCTION [dbo].[ky_SUFRTrim]
  END
GO

CREATE FUNCTION
       [dbo].[ky_SUFRTrim]
      (@ORIGINAL  VARCHAR(MAX))
RETURNS VARCHAR(MAX)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_SUFRTrim]                                                                                       *
 *                                                                                                           *
 * To be used as RTrim function for Document Generator codes                                                 *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.6.4.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2019-01-11   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/

BEGIN
  RETURN RTRIM(ISNULL(@ORIGINAL, ''))
END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = 'ky_SUFLeft'
      AND SO.[type] = 'FN')
  BEGIN
    DROP FUNCTION [dbo].[ky_SUFLeft]
  END
GO

CREATE FUNCTION
       [dbo].[ky_SUFLeft]
      (@ORIGINAL  VARCHAR(MAX), @LEN INT)
RETURNS VARCHAR(MAX)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_SUFLeft]                                                                                        *
 *                                                                                                           *
 * To be used as Left function for Document Generator codes                                                  *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.6.4.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2019-01-11   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/

BEGIN
  IF LEN(ISNULL(@ORIGINAL, '')) < @LEN
    RETURN @ORIGINAL

  RETURN SUBSTRING(ISNULL(@ORIGINAL, ''), 1, @LEN)
END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = 'ky_SUFRight'
      AND SO.[type] = 'FN')
  BEGIN
    DROP FUNCTION [dbo].[ky_SUFRight]
  END
GO

CREATE FUNCTION
       [dbo].[ky_SUFRight]
      (@ORIGINAL  VARCHAR(MAX), @LEN INT)
RETURNS VARCHAR(MAX)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_SUFRight]                                                                                       *
 *                                                                                                           *
 * To be used as Right function for Document Generator codes                                                 *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.6.4.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2019-01-11   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/

BEGIN
  IF LEN(ISNULL(@ORIGINAL, '')) < @LEN
    RETURN @ORIGINAL

  RETURN SUBSTRING(ISNULL(@ORIGINAL, ''), LEN(ISNULL(@ORIGINAL, '')) - @LEN + 1, @LEN)
END
GO

/* Associate UDFs - in SAM3 we need to make sure that the column definitions are correct.
                    in SAM4 the table doesn't exist at all so we need to create it */
IF NOT EXISTS
  (SELECT TOP 1 1
     FROM sys.objects [SO]
    WHERE [SO].[name] = 'AssociateUDF'
      AND [SO].[type] = 'U')
BEGIN
  EXEC ('CREATE 
 TABLE [dbo].[AssociateUDF]
      ([Id]          [int]            IDENTITY(1,1) NOT NULL,
       [NameCode]    [varchar](6)                   NOT NULL,
       [FilePrefix]  [varchar](5)                   NOT NULL,
       [FieldName]   [varchar](30)                  NOT NULL,
       [TextValue]   [varchar](max)                 NOT NULL,
       [DateValue]   [datetime]                         NULL,
       [NumberValue] [decimal](20, 2)               NOT NULL,
       CONSTRAINT [PK_AssociateUDF]
                  PRIMARY KEY CLUSTERED 
                 ([Id] ASC)
               ON [PRIMARY])
    ON [PRIMARY]')
END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM sys.objects [SO]
          INNER JOIN sys.columns [SC]
                  ON [SC].object_id = [SO].object_id
                 AND [SC].[name] = 'NameCode'
                 AND [SC].[system_type_id] <> 167
    WHERE [SO].[name] = 'AssociateUDF'
      AND [SO].[type] = 'U')
  BEGIN
    IF EXISTS
      (SELECT TOP 1 1
         FROM sys.objects [SO]
              INNER JOIN sys.indexes [SI]
                      ON [SI].object_id = [SO].object_id
                     AND [SI].[name] = 'KeyNameCode'
        WHERE [SO].[name] = 'AssociateUDF'
          AND [SO].[type] = 'U')
      BEGIN
        EXEC ('DROP INDEX [dbo].[AssociateUDF].[KeyNameCode]')
      END

    EXEC ('ALTER TABLE [dbo].[AssociateUDF] ALTER COLUMN [NameCode] VARCHAR(6) NOT NULL')
  END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM sys.objects [SO]
          INNER JOIN sys.columns [SC]
                  ON [SC].object_id = [SO].object_id
                 AND [SC].[name] = 'FilePrefix'
                 AND [SC].[system_type_id] <> 167
    WHERE [SO].[name] = 'AssociateUDF'
      AND [SO].[type] = 'U')
  BEGIN
    IF EXISTS
      (SELECT TOP 1 1
         FROM sys.objects [SO]
              INNER JOIN sys.indexes [SI]
                      ON [SI].object_id = [SO].object_id
                     AND [SI].[name] = 'KeyNameCode'
        WHERE [SO].[name] = 'AssociateUDF'
          AND [SO].[type] = 'U')
      BEGIN
        EXEC ('DROP INDEX [dbo].[AssociateUDF].[KeyNameCode]')
      END

    EXEC ('ALTER TABLE [dbo].[AssociateUDF] ALTER COLUMN [FilePrefix] VARCHAR(5) NOT NULL')
  END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM sys.objects [SO]
          INNER JOIN sys.columns [SC]
                  ON [SC].object_id = [SO].object_id
                 AND [SC].[name] = 'FieldName'
                 AND [SC].[system_type_id] <> 167
    WHERE [SO].[name] = 'AssociateUDF'
      AND [SO].[type] = 'U')
  BEGIN
    IF EXISTS
      (SELECT TOP 1 1
         FROM sys.objects [SO]
              INNER JOIN sys.indexes [SI]
                      ON [SI].object_id = [SO].object_id
                     AND [SI].[name] = 'KeyNameCode'
        WHERE [SO].[name] = 'AssociateUDF'
          AND [SO].[type] = 'U')
      BEGIN
        EXEC ('DROP INDEX [dbo].[AssociateUDF].[KeyNameCode]')
      END

    EXEC ('ALTER TABLE [dbo].[AssociateUDF] ALTER COLUMN [FieldName] VARCHAR(30) NOT NULL')
  END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM sys.objects [SO]
          INNER JOIN sys.columns [SC]
                  ON [SC].object_id = [SO].object_id
                 AND [SC].[name] = 'TextValue'
                 AND [SC].[system_type_id] <> 167
    WHERE [SO].[name] = 'AssociateUDF'
      AND [SO].[type] = 'U')
  BEGIN
    EXEC ('ALTER TABLE [dbo].[AssociateUDF] ALTER COLUMN [TextValue] VARCHAR(MAX) NOT NULL')
  END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM sys.objects [SO]
          INNER JOIN sys.columns [SC]
                  ON [SC].object_id = [SO].object_id
                 AND [SC].[name] = 'NumberValue'
                 AND (   [SC].[system_type_id] <> 106
                      OR [SC].[precision] < 20)
    WHERE [SO].[name] = 'AssociateUDF'
      AND [SO].[type] = 'U')
  BEGIN
    EXEC ('ALTER TABLE [dbo].[AssociateUDF] ALTER COLUMN [NumberValue] decimal(20, 2) NOT NULL')
  END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM sys.objects [SO]
          LEFT OUTER JOIN sys.indexes [SI]
                       ON [SI].object_id = [SO].object_id
                      AND [SI].[name] = 'KeyNameCode'
    WHERE [SO].[name] = 'AssociateUDF'
      AND [SO].[type] = 'U'
      AND [SI].[index_id] IS NULL)
  BEGIN
    EXEC ('CREATE UNIQUE NONCLUSTERED INDEX
       [KeyNameCode]
    ON [dbo].[AssociateUDF]
      ([NameCode]   ASC,
       [FilePrefix] ASC,
       [FieldName]  ASC)
    ON [PRIMARY]')
  END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.objects [SO]
          INNER JOIN SYS.[columns] [SC]
                  ON [SC].object_id = [SO].object_id
                 AND [SC].[name] = 'FORMAT'
                 AND [SC].[max_length] < 50
    WHERE [SO].[name] = 'SystemUserDefinedFields')
  BEGIN
    EXEC ('ALTER TABLE [dbo].[SystemUserDefinedFields] ALTER COLUMN [FORMAT] VARCHAR(50) NOT NULL')
  END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETFNFieldIDParser'
      AND SO.[type] = 'FN')
  BEGIN
    DROP FUNCTION [dbo].[ky_NETFNFieldIDParser]
  END
GO

CREATE FUNCTION
       [dbo].[ky_NETFNFieldIDParser]
      (@ORIGINAL    VARCHAR(MAX),
       @FORFUNCTION BIT)
RETURNS VARCHAR(MAX)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETFNFieldIDParser]                                                                             *
 *                                                                                                           *
 * Converts old Clarion Field IDs to the .NET Generator compatible equivalent                                *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.6.4.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2019-01-11   Pino Carafa       Created                                                                    *
 * 2019-01-15   Pino Carafa       Include CAN: fields                                                        *
 * 2019-01-16   Pino Carafa       Can now handle ALL prefixes. When it cannot parse the field itself         *
 *                                it hands it back to the calling procedure to use the old parser instead.   *
 * 2019-01-21   Pino Carafa       It didn't add the [CDE].[Code] parameter to the f.UDF call                 *
 * 2019-01-21   Pino Carafa       Use the OLDPARSEQ alternative for quoted strings, so that the calling      *
 *                                procedure knows to double quotes in the returned values. KEYD-5663         *
 * 2019-01-22   Pino Carafa       Allow for a space between the File Prefix and the Field Name as that       *
 *                                situation can arise when copying and pasting codes into a Precedent        *
 * 2019-02-01   Pino Carafa       Allow for a embedded LCN aggregate codes                                   *
 * 2019-02-06   Pino Carafa       Use Doc Assist format for AUD fields                                       *
 * 2019-02-06   Pino Carafa       Make allowance for longer FORMAT strings                                   *
 * 2019-02-18   Pino Carafa       Use functions for CAN fields                                               *
 * 2019-02-19   Pino Carafa       Use functions for LCN fields                                               *
 * 2019-02-19   Pino Carafa       Support aggregate AUD fields                                               *
 * 2019-12-09   Pino Carafa       KEYD-5950 added functionality to return "?" when passing in invalid        *
 *                                field identifiers e.g. [MAT:ThisFieldDoesNotExistInTheMattersTable]        *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

DECLARE @RESULT VARCHAR(MAX)
DECLARE @ACTUALNAME VARCHAR(200)
DECLARE @SQLSTATEMENT VARCHAR(MAX)
DECLARE @SUFFMT VARCHAR(50)
DECLARE @FOUND BIT

IF CHARINDEX('''', @ORIGINAL) > 0
  BEGIN
    RETURN @ORIGINAL
    --RETURN
  END
                 
IF CHARINDEX('', @ORIGINAL) > 0
  BEGIN
    RETURN @ORIGINAL
    --RETURN
  END
                 
IF CHARINDEX('', @ORIGINAL) > 0
  BEGIN
    RETURN @ORIGINAL
    --RETURN
  END
                 
IF CHARINDEX('(', @ORIGINAL) > 0
  BEGIN
    RETURN @ORIGINAL
    --RETURN
  END
                 
IF CHARINDEX(')', @ORIGINAL) > 0
  BEGIN
    RETURN @ORIGINAL
    --RETURN
  END

IF CHARINDEX(':', @ORIGINAL) = 0
  BEGIN
    RETURN @ORIGINAL
    --RETURN
  END

DECLARE @REST VARCHAR(MAX)
DECLARE @PREFIX VARCHAR(MAX)
DECLARE @Field VARCHAR(MAX)
DECLARE @ASSOCTYPE VARCHAR(MAX)
DECLARE @CONNUM INT
DECLARE @VCCONNUM VARCHAR(3)

SET @PREFIX = SUBSTRING(@ORIGINAL, 1, CHARINDEX(':', @ORIGINAL))
SET @REST = SUBSTRING(@ORIGINAL, CHARINDEX(':', @ORIGINAL) + 1, LEN(@ORIGINAL) - CHARINDEX(':', @ORIGINAL))

IF @PREFIX IN ('AUD:', 'AUS:', 'CAN:')
  BEGIN
    IF CHARINDEX('.', @REST) = 0
      BEGIN
        RETURN @ORIGINAL
        --RETURN
      END

    SET @FIELD = LTRIM(RTRIM(SUBSTRING(@REST, 1, CHARINDEX('.', @REST) - 1)))

    SET @REST = SUBSTRING(@ORIGINAL, CHARINDEX('.', @ORIGINAL) + 1, LEN(@ORIGINAL) - CHARINDEX('.', @ORIGINAL))
  END
ELSE IF @PREFIX = 'LCN:'
  BEGIN
    IF CHARINDEX('#', @REST) = 0
      BEGIN
        SET @FIELD = @REST
        SET @REST = ''
        SET @CONNUM = 1
      END
    ELSE
      BEGIN
        SET @FIELD = SUBSTRING(@REST, 1, CHARINDEX('#', @REST) - 1)
        SET @REST = SUBSTRING(@REST, CHARINDEX('#', @REST) + 1, LEN(@REST) - CHARINDEX('#', @REST))
        IF @REST = '_AND_'
          SET @CONNUM = -1
        ELSE IF @REST = '_CND_'
          SET @CONNUM = -2
        ELSE IF @REST = '_NLN_'
          SET @CONNUM = -3
        ELSE
          SET @CONNUM = CONVERT(INT, @REST)

        IF @CONNUM = 0
          SET @CONNUM = 1
      END
  END
ELSE
  BEGIN
    SET @Field = LTRIM(RTRIM(@REST))
    SET @ASSOCTYPE = ''
    SET @CONNUM = 0
    SET @REST = ''
  END

IF @PREFIX IN ('AUD:', 'AUS:', 'CAN:')
  BEGIN
    IF CHARINDEX('#', @REST) = 0
      BEGIN
        SET @ASSOCTYPE = @REST
        SET @REST = ''
        SET @CONNUM = 1
      END
    ELSE
      BEGIN
        SET @ASSOCTYPE = SUBSTRING(@REST, 1, CHARINDEX('#', @REST) - 1)
        SET @REST = SUBSTRING(@ORIGINAL, CHARINDEX('#', @ORIGINAL) + 1, LEN(@ORIGINAL) - CHARINDEX('#', @ORIGINAL))
        IF ISNUMERIC(@REST) = 1
          BEGIN
            SET @CONNUM = CONVERT(INT, @REST)
          END
        ELSE
          BEGIN
            IF @REST = '_AND_'
              SET @CONNUM = -1
            ELSE IF @REST = '_CND_'
              SET @CONNUM = -2
            ELSE IF @REST = '_NLN_'
              SET @CONNUM = -3
            ELSE
              SET @CONNUM = 1
          END
        SET @REST = ''
      END
  END

-- for the moment we'll just process the new AUD: prefixes. This can be expanded on later.
--IF @PREFIX NOT IN ('AUD:', 'AUS:', 'MAT:', 'CAN:', 'CNT:')
--  BEGIN
--    RETURN @ORIGINAL
--    --RETURN
--  END

DECLARE @TYPE VARCHAR(11)

IF @PREFIX = 'AUD:'
  BEGIN
    SELECT @TYPE = RTRIM(ISNULL([SUF].[Type], 'Text')),
           @SUFFMT = RTRIM(ISNULL([SUF].[FORMAT], '')),
           @FOUND = 1
      FROM [dbo].[SystemUserDefinedFields] [SUF]
     WHERE [SUF].[FILEPREFIX] = @PREFIX
       AND [SUF].[FIELDNAME] = @FIELD

    IF ISNULL(@FOUND, 0) = 0
      BEGIN
        RETURN 'f.CStr(''?'')'
      END

    IF @TYPE = 'Numeric'
      IF @CONNUM < 0
        BEGIN
          SET @RESULT = 'f.AUDNumbers([CDE].[Code], ''' + REPLACE(@FIELD, '''', '''''') + ''', ''' + REPLACE(@ASSOCTYPE, '''', '''''') + ''', ''' + CONVERT(VARCHAR, @CONNUM) + ''', ''' + @SUFFMT + ''')'
        END
      ELSE
        BEGIN
          IF @FORFUNCTION = 0
            BEGIN
              SET @RESULT = 'f.Format(f.AUDNumber([CDE].[Code], ''' + REPLACE(@FIELD, '''', '''''') + ''', ''' + REPLACE(@ASSOCTYPE, '''', '''''') + ''', ' + CONVERT(VARCHAR, @CONNUM) + '), ''' + @SUFFMT + ''')'
            END
          ELSE
            BEGIN
              SET @RESULT = 'f.AUDNumber([CDE].[Code], ''' + REPLACE(@FIELD, '''', '''''') + ''', ''' + REPLACE(@ASSOCTYPE, '''', '''''') + ''', ' + CONVERT(VARCHAR, @CONNUM) + ')'
            END
        END
    ELSE IF @TYPE = 'Date'
      IF @CONNUM < 0
        BEGIN
          SET @RESULT = 'f.AUDDates([CDE].[Code], ''' + REPLACE(@FIELD, '''', '''''') + ''', ''' + REPLACE(@ASSOCTYPE, '''', '''''') + ''', ''' + CONVERT(VARCHAR, @CONNUM) + ''', ''' + @SUFFMT + ''')'
        END
      ELSE
        BEGIN
          IF @FORFUNCTION = 0
            BEGIN
              SET @RESULT = 'f.Format(f.AUDDate([CDE].[Code], ''' + REPLACE(@FIELD, '''', '''''') + ''', ''' + REPLACE(@ASSOCTYPE, '''', '''''') + ''', ' + CONVERT(VARCHAR, @CONNUM) + '), ''' + @SUFFMT + ''')'
            END
          ELSE
            BEGIN
              SET @RESULT = 'f.AUDDate([CDE].[Code], ''' + REPLACE(@FIELD, '''', '''''') + ''', ''' + REPLACE(@ASSOCTYPE, '''', '''''') + ''', ' + CONVERT(VARCHAR, @CONNUM) + ')'
            END
        END
    ELSE IF @CONNUM < 0
      BEGIN
        SET @RESULT = 'f.AUDTexts([CDE].[Code], ''' + REPLACE(@FIELD, '''', '''''') + ''', ''' + REPLACE(@ASSOCTYPE, '''', '''''') + ''', ''' + CONVERT(VARCHAR, @CONNUM) + ''')'
      END
    ELSE
      BEGIN
        SET @RESULT = 'f.AUDText([CDE].[Code], ''' + REPLACE(@FIELD, '''', '''''') + ''', ''' + REPLACE(@ASSOCTYPE, '''', '''''') + ''', ' + CONVERT(VARCHAR, @CONNUM) + ')'
      END

    RETURN @RESULT
  END

IF @PREFIX = 'AUS:' -- AUD: Values represented as VARCHAR for concatenations and so on
  BEGIN
    SELECT @TYPE = RTRIM(ISNULL([SUF].[Type], 'Text')),
           @SUFFMT = RTRIM(ISNULL([SUF].[FORMAT], '')),
           @FOUND = 1
      FROM [dbo].[SystemUserDefinedFields] [SUF]
     WHERE [SUF].[FILEPREFIX] = 'AUD:'
       AND [SUF].[FIELDNAME] = @FIELD

    IF ISNULL(@FOUND, 0) = 0
      BEGIN
        RETURN 'f.CStr(''?'')'
      END

    IF @TYPE = 'Numeric'
      BEGIN
        SET @RESULT = 'f.GetFormattedValue(f.AUDNumber([CDE].[Code], ''' + REPLACE(@FIELD, '''', '''''') + ''', ''' + REPLACE(@ASSOCTYPE, '''', '''''') + ''', ' + CONVERT(VARCHAR, @CONNUM) + '), ''N2'', 1, 0, NULL)'
      END
    ELSE IF @TYPE = 'Date'
      BEGIN
        SET @RESULT = 'f.AddDays(f.AUDDate([CDE].[Code], ''' + REPLACE(@FIELD, '''', '''''') + ''', ''' + REPLACE(@ASSOCTYPE, '''', '''''') + ''', ' + CONVERT(VARCHAR, @CONNUM) + '), 0, ''DD MMM YYYY'')'
      END
    ELSE
      BEGIN
        SET @RESULT = 'f.AUDText([CDE].[Code], ''' + REPLACE(@FIELD, '''', '''''') + ''', ''' + REPLACE(@ASSOCTYPE, '''', '''''') + ''', ' + CONVERT(VARCHAR, @CONNUM) + ')'
      END

    RETURN @RESULT
  END

IF @PREFIX = 'CAN:'
  BEGIN
    SELECT @TYPE = RTRIM(ISNULL([SUF].[Type], 'Text')),
           @SUFFMT = RTRIM(ISNULL([SUF].[FORMAT], '')),
           @ACTUALNAME = RTRIM(ISNULL([SUF].[ActualName], '')),
           @SQLSTATEMENT = RTRIM(ISNULL([SUF].[SQLStatement], ''))
      FROM [dbo].[SystemUserDefinedFields] [SUF]
     WHERE [SUF].[FILEPREFIX] = @PREFIX
       AND [SUF].[FIELDNAME] = @FIELD

    IF EXISTS
      (SELECT TOP 1 1
         FROM SYS.objects [SO]
        WHERE [SO].[name] = 'ky_SUFCAS' + @Field
          AND [SO].[type] = 'FN')
      BEGIN
        RETURN 'f.CAS' + @Field + '([CDE].[Code], ''' + REPLACE(@ASSOCTYPE, '''', '''''') + ''', ' + CONVERT(VARCHAR, @CONNUM) + ')'
      END

    IF (@ACTUALNAME <> '' OR @SQLSTATEMENT <> '') AND (@ACTUALNAME <> 'NAM:' + @FIELD)
      BEGIN
        -- Leave this one to the old parser
        IF @SUFFMT LIKE '@n%'
          BEGIN
            SET @TYPE = 'Numeric'
          END
        ELSE IF @SUFFMT LIKE '@d%'
          BEGIN
            SET @TYPE = 'Date'
          END
        ELSE
          BEGIN
            SET @TYPE = 'Text'
          END

        IF @TYPE = 'Numeric'
          BEGIN
            IF @FORFUNCTION = 0
              BEGIN
                SET @RESULT = 'f.GetFormattedValue(OLDPARSER**FIELD**CAN**' + @FIELD + '.' + @ASSOCTYPE + '#' + SUBSTRING(CONVERT(VARCHAR(3), 100 + @CONNUM), 2, 2) + '**TYPE**' + @TYPE + '**RESRAPDLO**, ''N2'', 1, 0, NULL)'
              END
            ELSE
              BEGIN
                SET @RESULT = '''OLDPARSEQ**FIELD**CAN**' + @FIELD + '.' + @ASSOCTYPE + '#' + SUBSTRING(CONVERT(VARCHAR(3), 100 + @CONNUM), 2, 2) + '**TYPE**' + @TYPE + '**QESRAPDLO**'''
              END
            RETURN @RESULT
          END
        ELSE IF @TYPE = 'Date'
          BEGIN
            IF @FORFUNCTION = 0
              BEGIN
                IF @SUFFMT = '@d6'
                  BEGIN
                    SET @RESULT = 'f.Format(OLDPARSER**FIELD**CAN**' + @FIELD + '.' + @ASSOCTYPE + '#' + SUBSTRING(CONVERT(VARCHAR(3), 100 + @CONNUM), 2, 2) + '**TYPE**' + @TYPE + '**RESRAPDLO**, ''DD/MM/YY'')'
                  END
                ELSE IF @SUFFMT = '@d8'
                  BEGIN
                    SET @RESULT = 'f.Format(OLDPARSER**FIELD**CAN**' + @FIELD + '.' + @ASSOCTYPE + '#' + SUBSTRING(CONVERT(VARCHAR(3), 100 + @CONNUM), 2, 2) + '**TYPE**' + @TYPE + '**RESRAPDLO**, ''DD MMM YYYY'')'
                  END
                ELSE
                  BEGIN
                    SET @RESULT = 'f.Format(OLDPARSER**FIELD**CAN**' + @FIELD + '.' + @ASSOCTYPE + '#' + SUBSTRING(CONVERT(VARCHAR(3), 100 + @CONNUM), 2, 2) + '**TYPE**' + @TYPE + '**RESRAPDLO**, ''D MMMM YYYY'')'
                  END
              END
            ELSE
              BEGIN
                SET @RESULT = '''OLDPARSEQ**FIELD**CAN**' + @FIELD + '.' + @ASSOCTYPE + '#' + SUBSTRING(CONVERT(VARCHAR(3), 100 + @CONNUM), 2, 2) + '**TYPE**' + @TYPE + '**QESRAPDLO**'''
              END
            RETURN @RESULT
          END

        RETURN '''OLDPARSEQ**FIELD**CAN**' + @FIELD + '.' + @ASSOCTYPE + '#' + SUBSTRING(CONVERT(VARCHAR(3), 100 + @CONNUM), 2, 2) + '**TYPE**' + @TYPE + '**QESRAPDLO**'''
      END

    SET @TYPE = NULL

    SELECT @TYPE = ISNULL([ST].[name], ''),
           @SUFFMT = RTRIM(ISNULL([SUF].[FORMAT], '')),
           @FOUND = 1
      FROM sys.objects [SO]
           INNER JOIN sys.columns [SC]
                      INNER JOIN sys.types [ST]
                              ON [ST].[system_type_id] = [SC].[system_type_id]
                      LEFT OUTER JOIN [dbo].[SystemUserDefinedFields] [SUF]
                                   ON [SUF].[FilePrefix] = @PREFIX
                                  AND [SUF].[FieldName] = @FIELD
                   ON [SC].object_id = [SO].object_id
                  AND [SC].[name] = @FIELD
     WHERE [SO].[name] = 'CaseAssociatesNames'
       AND [SO].[type] = 'U'

    IF ISNULL(@FOUND, 0) = 0
      BEGIN
        RETURN 'f.CStr(''?'')'
      END

    IF RTRIM(ISNULL(@TYPE, '')) = ''
      BEGIN
        SET @TYPE = 'System'
        RETURN '''OLDPARSEQ**FIELD**CAN**' + @FIELD + '.' + @ASSOCTYPE + '#' + SUBSTRING(CONVERT(VARCHAR(3), 100 + @CONNUM), 2, 2) + '**TYPE**' + @TYPE + '**QESRAPDLO**'''
      END

    IF (@TYPE = 'datetime') OR (@TYPE = 'datetime2') OR (@TYPE = 'date') OR (@TYPE = 'smalldatetime')
      BEGIN
        IF @FORFUNCTION = 0
          BEGIN
            IF @SUFFMT = '@d6'
              BEGIN
                SET @RESULT = 'f.format(f.CANDate([CDE].[Code], ''' + REPLACE(@FIELD, '''', '''''') + ''', ''' + REPLACE(@ASSOCTYPE, '''', '''''') + ''', ' + CONVERT(VARCHAR, @CONNUM) + '), ''DD/MM/YY'')'
              END
            ELSE IF @SUFFMT = '@d8'
              BEGIN
                SET @RESULT = 'f.format(f.CANDate([CDE].[Code], ''' + REPLACE(@FIELD, '''', '''''') + ''', ''' + REPLACE(@ASSOCTYPE, '''', '''''') + ''', ' + CONVERT(VARCHAR, @CONNUM) + '), ''DD MMM YYYY'')'
              END
            ELSE
              BEGIN
                SET @RESULT = 'f.format(f.CANDate([CDE].[Code], ''' + REPLACE(@FIELD, '''', '''''') + ''', ''' + REPLACE(@ASSOCTYPE, '''', '''''') + ''', ' + CONVERT(VARCHAR, @CONNUM) + '), ''D MMMM YYYY'')'
              END
          END
        ELSE
          BEGIN
            SET @RESULT = 'f.CANDate([CDE].[Code], ''' + REPLACE(@FIELD, '''', '''''') + ''', ''' + REPLACE(@ASSOCTYPE, '''', '''''') + ''', ' + CONVERT(VARCHAR, @CONNUM) + ')'
          END

        RETURN @RESULT
      END

    IF (@TYPE = 'money') OR (@TYPE = 'decimal') OR (@TYPE = 'float') OR (@TYPE = 'numeric')
      BEGIN
        IF @FORFUNCTION = 0
          BEGIN
            SET @RESULT = 'f.format(f.CANNumber([CDE].[Code], ''' + REPLACE(@FIELD, '''', '''''') + ''', ''' + REPLACE(@ASSOCTYPE, '''', '''''') + ''', ' + CONVERT(VARCHAR, @CONNUM) + '), ''N2'')'
          END
        ELSE
          BEGIN
            SET @RESULT = 'f.CANNumber([CDE].[Code], ''' + REPLACE(@FIELD, '''', '''''') + ''', ''' + REPLACE(@ASSOCTYPE, '''', '''''') + ''', ' + CONVERT(VARCHAR, @CONNUM) + ')'
          END
        RETURN @RESULT
      END

    IF (@TYPE = 'int') OR (@TYPE = 'bigint') OR (@TYPE = 'smallint') OR (@TYPE = 'tinyint')
      BEGIN
        IF @FORFUNCTION = 0
          BEGIN
            SET @RESULT = 'f.CStr(f.CANInt([CDE].[Code], ''' + REPLACE(@FIELD, '''', '''''') + ''', ''' + REPLACE(@ASSOCTYPE, '''', '''''') + ''', ' + CONVERT(VARCHAR, @CONNUM) + '))'
          END
        ELSE
          BEGIN
            SET @RESULT = 'f.CANInt([CDE].[Code], ''' + REPLACE(@FIELD, '''', '''''') + ''', ''' + REPLACE(@ASSOCTYPE, '''', '''''') + ''', ' + CONVERT(VARCHAR, @CONNUM) + ')'
          END

        RETURN @RESULT
      END

    IF EXISTS
      (SELECT TOP 1 1
         FROM SYS.objects [SO]
        WHERE [SO].[name] = 'ky_SUFCAS' + @Field
          AND [SO].[type] = 'FN')
      BEGIN
        RETURN 'f.CAS' + @Field + '([CDE].[Code], ''' + REPLACE(@ASSOCTYPE, '''', '''''') + ''', ' + CONVERT(VARCHAR, @CONNUM) + ')'
      END
    ELSE
      BEGIN
        RETURN 'f.CANText([CDE].[Code], ''' + REPLACE(@FIELD, '''', '''''') + ''', ''' + REPLACE(@ASSOCTYPE, '''', '''''') + ''', ' + CONVERT(VARCHAR, @CONNUM) + ')'
      END

  END

IF @PREFIX = 'LCN:'
  BEGIN
    IF @CONNUM = -1
      BEGIN
        SET @VCCONNUM = '&'
      END
    ELSE IF @CONNUM = -2
      BEGIN
        SET @VCCONNUM = ','
      END
    ELSE IF @CONNUM < 0
      BEGIN
        SET @VCCONNUM = 'r'
      END
    ELSE
      BEGIN
        SET @VCCONNUM = CONVERT(VARCHAR(3), @CONNUM)
      END

    IF EXISTS
      (SELECT TOP 1 1
         FROM SYS.objects [SO]
        WHERE [SO].[name] = 'ky_SUFLCS' + @Field
          AND [SO].[type] = 'FN')
      BEGIN
        RETURN 'f.LCS' + @Field + '([CDE].[Code], ''' + @VCCONNUM + ''')'
      END
    ELSE
      BEGIN
        RETURN 'f.GETLCNInfo([CDE].[Code], ''' + @VCCONNUM + ''', ''' + @FIELD + ''', '''')'
      END
  END

IF @PREFIX = 'UDF:'
  BEGIN
    SELECT @TYPE = RTRIM(ISNULL([SUF].[Type], 'Text')),
           @SUFFMT = RTRIM(ISNULL([SUF].[FORMAT], '')),
           @ACTUALNAME = RTRIM(ISNULL([SUF].[ActualName], '')),
           @SQLSTATEMENT = RTRIM(ISNULL([SUF].[SQLStatement], '')),
           @FOUND = 1
      FROM [dbo].[SystemUserDefinedFields] [SUF]
     WHERE [SUF].[FILEPREFIX] = 'MAT:'
       AND [SUF].[FIELDNAME] = @FIELD

    IF ISNULL(@FOUND, 0) = 0
      BEGIN
        RETURN 'f.CStr(''?'')'
      END

    IF @ACTUALNAME <> '' OR @SQLSTATEMENT <> ''
      BEGIN
        -- Leave this one to the old parser
        IF @SUFFMT LIKE '@n%'
          BEGIN
            SET @TYPE = 'Numeric'
          END
        ELSE IF @SUFFMT LIKE '@d%'
          BEGIN
            SET @TYPE = 'Date'
          END
        ELSE
          BEGIN
            SET @TYPE = 'Text'
          END

        IF @TYPE = 'Numeric'
          BEGIN
            IF @FORFUNCTION = 0
              BEGIN
                SET @RESULT = 'f.GetFormattedValue(OLDPARSER**FIELD**UDF**' + @FIELD + '**TYPE**' + @TYPE + '**RESRAPDLO**, ''N2'', 1, 0, NULL)'
              END
            ELSE
              BEGIN
                SET @RESULT = '''OLDPARSEQ**FIELD**UDF**' + @FIELD + '**TYPE**' + @TYPE + '**QESRAPDLO**'''
              END
            RETURN @RESULT
          END
        ELSE IF @TYPE = 'Date'
          BEGIN
            IF @FORFUNCTION = 0
              BEGIN
                IF @SUFFMT = '@d6'
                  BEGIN
                    SET @RESULT = 'f.Format(OLDPARSER**FIELD**UDF**' + @FIELD + '**TYPE**' + @TYPE + '**RESRAPDLO**, ''DD/MM/YY'')'
                  END
                ELSE IF @SUFFMT = '@d8'
                  BEGIN
                    SET @RESULT = 'f.Format(OLDPARSER**FIELD**UDF**' + @FIELD + '**TYPE**' + @TYPE + '**RESRAPDLO**, ''DD MMM YYYY'')'
                  END
                ELSE
                  BEGIN
                    SET @RESULT = 'f.Format(OLDPARSER**FIELD**UDF**' + @FIELD + '**TYPE**' + @TYPE + '**RESRAPDLO**, ''D MMMM YYYY'')'
                  END
              END
            ELSE
              BEGIN
                SET @RESULT = '''OLDPARSEQ**FIELD**UDF**' + @FIELD + '**TYPE**' + @TYPE + '**QESRAPDLO**'''
              END
            RETURN @RESULT
          END

        RETURN '''OLDPARSEQ**FIELD**UDF**' + @FIELD + '**TYPE**' + @TYPE + '**QESRAPDLO**'''
      END

    RETURN 'f.UDF([CDE].[Code], ''' + @FIELD + ''')'
  END

IF @PREFIX = 'MAT:'
  BEGIN
    SELECT @TYPE = RTRIM(ISNULL([SUF].[Type], 'Text')),
           @SUFFMT = RTRIM(ISNULL([SUF].[FORMAT], '')),
           @ACTUALNAME = RTRIM(ISNULL([SUF].[ActualName], '')),
           @SQLSTATEMENT = RTRIM(ISNULL([SUF].[SQLStatement], ''))
      FROM [dbo].[SystemUserDefinedFields] [SUF]
     WHERE [SUF].[FILEPREFIX] = 'MAT:'
       AND [SUF].[FIELDNAME] = @FIELD

    IF @ACTUALNAME <> '' OR @SQLSTATEMENT <> ''
      BEGIN
        -- Leave this one to the old parser
        IF @SUFFMT LIKE '@n%'
          BEGIN
            SET @TYPE = 'Numeric'
          END
        ELSE IF @SUFFMT LIKE '@d%'
          BEGIN
            SET @TYPE = 'Date'
          END
        ELSE
          BEGIN
            SET @TYPE = 'Text'
          END

        IF @TYPE = 'Numeric'
          BEGIN
            IF @FORFUNCTION = 0
              BEGIN
                SET @RESULT = 'f.GetFormattedValue(OLDPARSER**FIELD**MAT**' + @FIELD + '**TYPE**' + @TYPE + '**RESRAPDLO**, ''N2'', 1, 0, NULL)'
              END
            ELSE
              BEGIN
                SET @RESULT = '''OLDPARSEQ**FIELD**MAT**' + @FIELD + '**TYPE**' + @TYPE + '**QESRAPDLO**'''
              END
            RETURN @RESULT
          END
        ELSE IF @TYPE = 'Date'
          BEGIN
            IF @FORFUNCTION = 0
              BEGIN
                IF @SUFFMT = '@d6'
                  BEGIN
                    SET @RESULT = 'f.Format(OLDPARSER**FIELD**MAT**' + @FIELD + '**TYPE**' + @TYPE + '**RESRAPDLO**, ''DD/MM/YY'')'
                  END
                ELSE IF @SUFFMT = '@d8'
                  BEGIN
                    SET @RESULT = 'f.Format(OLDPARSER**FIELD**MAT**' + @FIELD + '**TYPE**' + @TYPE + '**RESRAPDLO**, ''DD MMM YYYY'')'
                  END
                ELSE
                  BEGIN
                    SET @RESULT = 'f.Format(OLDPARSER**FIELD**MAT**' + @FIELD + '**TYPE**' + @TYPE + '**RESRAPDLO**, ''D MMMM YYYY'')'
                  END
              END
            ELSE
              BEGIN
                SET @RESULT = '''OLDPARSEQ**FIELD**MAT**' + @FIELD + '**TYPE**' + @TYPE + '**QESRAPDLO**'''
              END
            RETURN @RESULT
          END

        RETURN '''OLDPARSEQ**FIELD**MAT**' + @FIELD + '**TYPE**' + @TYPE + '**QESRAPDLO**'''
      END

    SELECT @TYPE = ISNULL([ST].[name], ''),
           @SUFFMT = RTRIM(ISNULL([SUF].[FORMAT], '')),
           @FOUND = CASE WHEN [SUF].[FilePrefix] IS NULL
                              THEN 0
                         ELSE 1 END
      FROM sys.objects [SO]
           INNER JOIN sys.columns [SC]
                      INNER JOIN sys.types [ST]
                              ON [ST].[system_type_id] = [SC].[system_type_id]
                      LEFT OUTER JOIN [dbo].[SystemUserDefinedFields] [SUF]
                                   ON [SUF].[FilePrefix] = @PREFIX
                                  AND [SUF].[FieldName] = @FIELD
                   ON [SC].object_id = [SO].object_id
                  AND [SC].[name] = @FIELD
     WHERE [SO].[name] = 'matters'
       AND [SO].[type] = 'U'

    IF ISNULL(@FOUND, 0) = 0
      BEGIN
        RETURN 'f.CStr(''?'')'
      END

    IF (@TYPE = 'datetime') OR (@TYPE = 'datetime2') OR (@TYPE = 'date') OR (@TYPE = 'smalldatetime')
      BEGIN
        IF @FORFUNCTION = 0
          BEGIN
            IF @SUFFMT = '@d6'
              BEGIN
                SET @RESULT = 'f.format([MAT].[' + @FIELD + '], ''DD/MM/YY'')'
              END
            ELSE IF @SUFFMT = '@d8'
              BEGIN
                SET @RESULT = 'f.format([MAT].[' + @FIELD + '], ''DD MMM YYYY'')'
              END
            ELSE
              BEGIN
                SET @RESULT = 'f.format([MAT].[' + @FIELD + '], ''D MMMM YYYY'')'
              END
          END
        ELSE
          BEGIN
            SET @RESULT = '[MAT].[' + @FIELD + ']'
          END

        RETURN @RESULT
      END

    IF (@TYPE = 'money') OR (@TYPE = 'decimal') OR (@TYPE = 'float') OR (@TYPE = 'numeric')
      BEGIN
        IF @FORFUNCTION = 0
          BEGIN
            SET @RESULT = 'f.format([MAT].[' + @FIELD + '], ''N2'')'
          END
        ELSE
          BEGIN
            SET @RESULT = '[MAT].[' + @FIELD + ']'
          END

        RETURN @RESULT
      END

    IF (@TYPE = 'int') OR (@TYPE = 'bigint') OR (@TYPE = 'smallint') OR (@TYPE = 'tinyint')
      BEGIN
        IF @FORFUNCTION = 0
          BEGIN
            SET @RESULT = 'f.CStr([MAT].[' + @FIELD + '])'
          END
        ELSE
          BEGIN
            SET @RESULT = '[MAT].[' + @FIELD + ']'
          END

        RETURN @RESULT
      END

    RETURN 'f.CStr([MAT].[' + @FIELD + '])'
  END

SET @TYPE = NULL
SET @SUFFMT = NULL
SET @ACTUALNAME = NULL
SET @SQLSTATEMENT = NULL

IF NOT EXISTS
  (SELECT TOP 1 1
     FROM [dbo].[SystemUserDefinedFields] [SUF]
    WHERE [SUF].[FILEPREFIX] = @PREFIX
      AND [SUF].[FIELDNAME] = @FIELD)
  BEGIN
    RETURN @ORIGINAL
  END

SELECT @TYPE = RTRIM(ISNULL([SUF].[Type], 'Text')),
       @SUFFMT = RTRIM(ISNULL([SUF].[FORMAT], '')),
       @ACTUALNAME = RTRIM(ISNULL([SUF].[ActualName], '')),
       @SQLSTATEMENT = RTRIM(ISNULL([SUF].[SQLStatement], ''))
  FROM [dbo].[SystemUserDefinedFields] [SUF]
 WHERE [SUF].[FILEPREFIX] = @PREFIX
   AND [SUF].[FIELDNAME] = @FIELD

IF RTRIM(ISNULL(@TYPE, '')) = ''
  RETURN '''OLDPARSEQ**FIELD**' + REPLACE(@PREFIX, ':', '') + '**' + @FIELD + '**TYPE**System**QESRAPDLO**'''

IF RTRIM(ISNULL(@SUFFMT, '')) = ''
  RETURN '''OLDPARSEQ**FIELD**' + REPLACE(@PREFIX, ':', '') + '**' + @FIELD + '**TYPE**' + @TYPE + '**QESRAPDLO**'''

IF (RTRIM(ISNULL(@SUFFMT, '')) <> '') AND (RTRIM(ISNULL(@SUFFMT, '')) NOT LIKE '@s%')
  BEGIN
    RETURN 'f.Format(''OLDPARSEQ**FIELD**' + REPLACE(@PREFIX, ':', '') + '**' + @FIELD + '**TYPE**' + @TYPE + '**QESRAPDLO**'', ''' + @SUFFMT + ''')'
  END
ELSE
  BEGIN
    RETURN 'f.CStr(''OLDPARSEQ**FIELD**' + REPLACE(@PREFIX, ':', '') + '**' + @FIELD + '**TYPE**' + @TYPE + '**QESRAPDLO**'')'
  END

--SELECT @PREFIX    AS [PREFIX],
--       @FIELD     AS [Field],
--       @ASSOCTYPE AS [AssociateType],
--       @CONNUM    AS [CONNUM],
--       @TYPE      AS [TYPE]

--SELECT @RESULT

RETURN @ORIGINAL -- never gonna happen but just in case
END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETFNInterpretComparison'
      AND SO.[type] = 'FN')
  BEGIN
    DROP FUNCTION [dbo].[ky_NETFNInterpretComparison]
  END
GO

CREATE FUNCTION
       [dbo].[ky_NETFNInterpretComparison]
      (@COMPARISONSTRING VARCHAR(MAX))
RETURNS VARCHAR(MAX)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETFNInterpretComparison]                                                                       *
 *                                                                                                           *
 * Interprets a DOC ASSIST "Comparison"                                                                      *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.6.4.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2019-01-22   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

DECLARE @VALUE1 VARCHAR(MAX)
DECLARE @VALUE2 VARCHAR(MAX)

DECLARE @EQPOS INT
DECLARE @GTPOS INT
DECLARE @LTPOS INT
DECLARE @NOTPOS INT
DECLARE @OPERATOR VARCHAR(4)

SET @EQPOS = [dbo].[ky_NETFNGetUnembeddedCharacterPos]('=', @COMPARISONSTRING)
SET @GTPOS = [dbo].[ky_NETFNGetUnembeddedCharacterPos]('>', @COMPARISONSTRING)
SET @LTPOS = [dbo].[ky_NETFNGetUnembeddedCharacterPos]('<', @COMPARISONSTRING)
SET @NOTPOS = [dbo].[ky_NETFNGetUnembeddedCharacterPos]('!', @COMPARISONSTRING)

IF @NOTPOS > 0
  BEGIN
    IF @EQPOS = @NOTPOS + 1
      BEGIN
        SET @OPERATOR = '''<>'''
        SET @VALUE1 = LTRIM(RTRIM(SUBSTRING(@COMPARISONSTRING, 1, @NOTPOS - 1)))
        SET @VALUE2 = LTRIM(RTRIM(SUBSTRING(@COMPARISONSTRING, @EQPOS + 1, LEN(@COMPARISONSTRING) - @EQPOS)))
      END
  END
ELSE IF @LTPOS > 0
  BEGIN
    IF @GTPOS = @LTPOS + 1
      BEGIN
        SET @OPERATOR = '''<>'''
        SET @VALUE1 = LTRIM(RTRIM(SUBSTRING(@COMPARISONSTRING, 1, @LTPOS - 1)))
        SET @VALUE2 = LTRIM(RTRIM(SUBSTRING(@COMPARISONSTRING, @GTPOS + 1, LEN(@COMPARISONSTRING) - @GTPOS)))
      END
    ELSE IF @EQPOS = @LTPOS + 1
      BEGIN
        SET @OPERATOR = '''<='''
        SET @VALUE1 = LTRIM(RTRIM(SUBSTRING(@COMPARISONSTRING, 1, @LTPOS - 1)))
        SET @VALUE2 = LTRIM(RTRIM(SUBSTRING(@COMPARISONSTRING, @EQPOS + 1, LEN(@COMPARISONSTRING) - @EQPOS)))
      END
    ELSE
      BEGIN
        SET @OPERATOR = '''<'''
        SET @VALUE1 = LTRIM(RTRIM(SUBSTRING(@COMPARISONSTRING, 1, @LTPOS - 1)))
        SET @VALUE2 = LTRIM(RTRIM(SUBSTRING(@COMPARISONSTRING, @LTPOS + 1, LEN(@COMPARISONSTRING) - @LTPOS)))
      END
  END
ELSE IF @GTPOS > 0
  BEGIN
    IF @EQPOS = @GTPOS + 1
      BEGIN
        SET @OPERATOR = '''>='''
        SET @VALUE1 = LTRIM(RTRIM(SUBSTRING(@COMPARISONSTRING, 1, @GTPOS - 1)))
        SET @VALUE2 = LTRIM(RTRIM(SUBSTRING(@COMPARISONSTRING, @EQPOS + 1, LEN(@COMPARISONSTRING) - @EQPOS)))
      END
    ELSE
      BEGIN
        SET @OPERATOR = '''>'''
        SET @VALUE1 = LTRIM(RTRIM(SUBSTRING(@COMPARISONSTRING, 1, @GTPOS - 1)))
        SET @VALUE2 = LTRIM(RTRIM(SUBSTRING(@COMPARISONSTRING, @GTPOS + 1, LEN(@COMPARISONSTRING) - @GTPOS)))
      END
  END
ELSE IF @EQPOS > 0
  BEGIN
    SET @OPERATOR = '''='''
    SET @VALUE1 = LTRIM(RTRIM(SUBSTRING(@COMPARISONSTRING, 1, @EQPOS - 1)))
    SET @VALUE2 = LTRIM(RTRIM(SUBSTRING(@COMPARISONSTRING, @EQPOS + 1, LEN(@COMPARISONSTRING) - @EQPOS)))
  END

RETURN 'f.DoComparison(' + @VALUE1 + ', ' + @VALUE2 + ', ' + @OPERATOR + ')'

END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = 'ky_NETFNFunctionInterpreter'
      AND SO.[type] = 'FN')
  BEGIN
    DROP FUNCTION [dbo].[ky_NETFNFunctionInterpreter]
  END
GO

CREATE FUNCTION
       [dbo].[ky_NETFNFunctionInterpreter]
      (@FUNCTION   VARCHAR(100),
       @PARAMETERS VARCHAR(MAX))
RETURNS VARCHAR(MAX)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETFNFunctionInterpreter]                                                                       *
 *                                                                                                           *
 * Replaces a Clarion function with an f. Function equivalent                                                *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.6.4.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2019-01-11   Pino Carafa       Created                                                                    *
 * 2019-01-18   Pino Carafa       Added Choose, IIF, UDF                                                     *
 * 2019-01-22   Pino Carafa       Use Comparison Interpreter for Choose and IIF                              *
 * 2019-02-01   Pino Carafa       Added CANCount and LCNCount                                                *
 * 2019-02-19   Pino Carafa       Added DateDiff                                                             *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN
  DECLARE @PARMPOS   INT
  DECLARE @PARM1     VARCHAR(MAX)
  DECLARE @RESULT    VARCHAR(MAX)
  DECLARE @PARMNO    INT
  DECLARE @PARMCOUNT INT

  DECLARE @PARMTABLE TABLE
         ([id]        INT            IDENTITY(1, 1)     NOT NULL  PRIMARY KEY,
          [Parameter] VARCHAR(MAX))

  WHILE @PARAMETERS <> ''
    BEGIN
      SET @PARMPOS = PATINDEX('%****PARMSEP****%', @PARAMETERS)

      IF @PARMPOS = 0
        BEGIN
          INSERT
            INTO @PARMTABLE
                ([Parameter])
          VALUES(@PARAMETERS)

          SET @PARAMETERS = ''
        END
      ELSE
        BEGIN
          INSERT
            INTO @PARMTABLE
                ([Parameter])
          VALUES(SUBSTRING(@PARAMETERS, 1, @PARMPOS - 1))

          SET @PARAMETERS = SUBSTRING(@PARAMETERS, @PARMPOS + 15, LEN(@PARAMETERS) - 14)
        END
    END

  IF (@FUNCTION = 'sub') AND ((SELECT COUNT(1) FROM @PARMTABLE) = 3)
    BEGIN
      SET @RESULT = ''

        SELECT @RESULT = @RESULT + CASE WHEN @RESULT = '' THEN '' ELSE ', ' END + [PRM].[Parameter]
          FROM @PARMTABLE [PRM]
      ORDER BY [id]

      SET @RESULT = 'f.Substring(' + @RESULT + ')'
    END
  ELSE IF ((@FUNCTION = 'clip') OR (@FUNCTION = 'trim')) AND ((SELECT COUNT(1) FROM @PARMTABLE) = 1)
    BEGIN
      SET @RESULT = ''

        SELECT @RESULT = @RESULT + CASE WHEN @RESULT = '' THEN '' ELSE ', ' END + [PRM].[Parameter]
          FROM @PARMTABLE [PRM]
      ORDER BY [id]

      SET @RESULT = 'f.Clip(' + @RESULT + ')'
    END
  ELSE IF (@FUNCTION = 'left') AND ((SELECT COUNT(1) FROM @PARMTABLE) = 2)
    BEGIN
      SET @RESULT = ''

        SELECT @RESULT = @RESULT + CASE WHEN @RESULT = '' THEN '' ELSE ', ' END + [PRM].[Parameter]
          FROM @PARMTABLE [PRM]
      ORDER BY [id]

      SET @RESULT = 'f.Left(' + @RESULT + ')'
    END
  ELSE IF (@FUNCTION = 'right') AND ((SELECT COUNT(1) FROM @PARMTABLE) = 2)
    BEGIN
      SET @RESULT = ''

        SELECT @RESULT = @RESULT + CASE WHEN @RESULT = '' THEN '' ELSE ', ' END + [PRM].[Parameter]
          FROM @PARMTABLE [PRM]
      ORDER BY [id]

      SET @RESULT = 'f.Right(' + @RESULT + ')'
    END
  ELSE IF (@FUNCTION = 'cstr') AND ((SELECT COUNT(1) FROM @PARMTABLE) = 1)
    BEGIN
      SET @RESULT = ''

        SELECT @RESULT = @RESULT + CASE WHEN @RESULT = '' THEN '' ELSE ', ' END + [PRM].[Parameter]
          FROM @PARMTABLE [PRM]
      ORDER BY [id]

      SET @RESULT = 'f.CStr(' + @RESULT + ')'
    END
  ELSE IF (@FUNCTION = 'chr') AND ((SELECT COUNT(1) FROM @PARMTABLE) = 1)
    BEGIN
      SET @RESULT = ''

        SELECT @RESULT = @RESULT + CASE WHEN @RESULT = '' THEN '' ELSE ', ' END + [PRM].[Parameter]
          FROM @PARMTABLE [PRM]
      ORDER BY [id]

      SET @RESULT = 'f.Chr(' + @RESULT + ')'
    END
  ELSE IF (@FUNCTION = 'format') AND ((SELECT COUNT(1) FROM @PARMTABLE) = 2)
    BEGIN
      SET @RESULT = ''

        SELECT @RESULT = @RESULT + CASE WHEN @RESULT = '' THEN '' ELSE ', ' END + [PRM].[Parameter]
          FROM @PARMTABLE [PRM]
      ORDER BY [id]

      SET @RESULT = 'f.Format(' + @RESULT + ')'
    END
  ELSE IF (@FUNCTION = 'LAddress') AND ((SELECT COUNT(1) FROM @PARMTABLE) = 2)
    BEGIN
      SET @RESULT = ''

        SELECT @RESULT = @RESULT + CASE WHEN @RESULT = '' THEN '' ELSE ', ' END + [PRM].[Parameter]
          FROM @PARMTABLE [PRM]
      ORDER BY [id]

      SET @RESULT = 'f.AddressLine(' + @RESULT + ')'
    END
  ELSE IF (@FUNCTION = 'day') AND ((SELECT COUNT(1) FROM @PARMTABLE) = 1)
    BEGIN
      SET @RESULT = ''

        SELECT @RESULT = @RESULT + CASE WHEN @RESULT = '' THEN '' ELSE ', ' END + [PRM].[Parameter]
          FROM @PARMTABLE [PRM]
      ORDER BY [id]

      SET @RESULT = 'f.DateNumber(f.Format(' + @RESULT + ', ''@d8''), ''DAY'')'
    END
  ELSE IF (@FUNCTION = 'month') AND ((SELECT COUNT(1) FROM @PARMTABLE) = 1)
    BEGIN
      SET @RESULT = ''

        SELECT @RESULT = @RESULT + CASE WHEN @RESULT = '' THEN '' ELSE ', ' END + [PRM].[Parameter]
          FROM @PARMTABLE [PRM]
      ORDER BY [id]

      SET @RESULT = 'f.DateNumber(f.Format(' + @RESULT + ', ''@d8''), ''MONTH'')'
    END
  ELSE IF (@FUNCTION = 'year') AND ((SELECT COUNT(1) FROM @PARMTABLE) = 1)
    BEGIN
      SET @RESULT = ''

        SELECT @RESULT = @RESULT + CASE WHEN @RESULT = '' THEN '' ELSE ', ' END + [PRM].[Parameter]
          FROM @PARMTABLE [PRM]
      ORDER BY [id]

      SET @RESULT = 'f.DateNumber(f.Format(' + @RESULT + ', ''@d8''), ''YEAR'')'
    END
  ELSE IF (@FUNCTION = 'udf') AND ((SELECT COUNT(1) FROM @PARMTABLE) = 1)
    BEGIN
      SET @RESULT = ''

        SELECT @RESULT = @RESULT + CASE WHEN @RESULT = '' THEN '' ELSE ', ' END + [PRM].[Parameter]
          FROM @PARMTABLE [PRM]
      ORDER BY [id]

      IF LEN(@RESULT) > 0
        BEGIN
          IF (SUBSTRING(@RESULT, 1, 1) = '''') AND (SUBSTRING(@RESULT, LEN(@RESULT), 1) = '''')
            BEGIN
              SET @RESULT = SUBSTRING(@RESULT, 2, LEN(@RESULT) - 2)
            END
        END

      IF EXISTS
        (SELECT TOP 1 1
           FROM [dbo].[SystemUserDefinedFields] [SUF]
          WHERE [SUF].[FILEPREFIX] = 'UDF:'
            AND [SUF].[FIELDNAME] = @RESULT
            AND RTRIM(ISNULL([SUF].[SQLSTATEMENT], '')) NOT IN ('', '[UDF].[Value]', '[UDF].[Text1]'))
        BEGIN
          SET @RESULT = 'f.SQL' + @RESULT + '([CDE].[Code])'
        END
      ELSE
        BEGIN
          SET @RESULT = 'f.UDF([CDE].[Code], ''' + @RESULT + ''')'
        END
    END
  ELSE IF (@FUNCTION = 'lcncount') AND ((SELECT COUNT(1) FROM @PARMTABLE) = 0)
    BEGIN
      SET @RESULT = ''

        SELECT @RESULT = @RESULT + CASE WHEN @RESULT = '' THEN '' ELSE ', ' END + [PRM].[Parameter]
          FROM @PARMTABLE [PRM]
      ORDER BY [id]

      SET @RESULT = 'f.LCNCount([CDE].[Code])'
    END
  ELSE IF (@FUNCTION = 'cancount') AND ((SELECT COUNT(1) FROM @PARMTABLE) = 1)
    BEGIN
      SET @RESULT = ''

        SELECT @RESULT = @RESULT + CASE WHEN @RESULT = '' THEN '' ELSE ', ' END + [PRM].[Parameter]
          FROM @PARMTABLE [PRM]
      ORDER BY [id]

      SET @RESULT = 'f.CANCount([CDE].[Code], ' + @RESULT + ')'
    END
  ELSE IF (@FUNCTION = 'iif') AND ((SELECT COUNT(1) FROM @PARMTABLE) = 3)
    BEGIN
      SET @RESULT = 'CASE WHEN ' + (SELECT [dbo].[ky_NETFNInterpretComparison]([PRM].[Parameter])
                                      FROM @PARMTABLE [PRM]
                                     WHERE [PRM].[id] = 1) + ' = 1 THEN '
                                 + (SELECT [PRM].[Parameter]
                                      FROM @PARMTABLE [PRM]
                                     WHERE [PRM].[id] = 2) + ' ELSE '
                                 + (SELECT [PRM].[Parameter]
                                      FROM @PARMTABLE [PRM]
                                     WHERE [PRM].[id] = 3) + ' END'

      --SET @RESULT = '''' + REPLACE((SELECT [PRM].[Parameter] FROM @PARMTABLE [PRM] WHERE [PRM].[id] = 1), '''', '''''') + ''''
    END
  ELSE IF (@FUNCTION = 'datediff') AND ((SELECT COUNT(1) FROM @PARMTABLE) = 2)
    BEGIN
      --years
      SET @RESULT = 'f.DateDiff(''year'', ' + (SELECT [PRM].[PARAMETER]
                                                 FROM @PARMTABLE [PRM]
                                                WHERE [PRM].[id] = 1) + ', '
                                            + (SELECT [PRM].[Parameter]
                                                 FROM @PARMTABLE [PRM]
                                                WHERE [PRM].[id] = 2) + ', 1)'
    END
  ELSE IF (@FUNCTION = 'datediff') AND ((SELECT COUNT(1) FROM @PARMTABLE) = 3)
    BEGIN
      --date difference as an Age function only
      SET @RESULT = 'f.DateDiff(' + (SELECT [PRM].[PARAMETER]
                                       FROM @PARMTABLE [PRM]
                                      WHERE [PRM].[id] = 1) + ', '
                                  + (SELECT [PRM].[PARAMETER]
                                       FROM @PARMTABLE [PRM]
                                      WHERE [PRM].[id] = 2) + ', '
                                  + (SELECT [PRM].[Parameter]
                                       FROM @PARMTABLE [PRM]
                                      WHERE [PRM].[id] = 3) + ', 1)'
    END
  ELSE IF (@FUNCTION = 'datediff') AND ((SELECT COUNT(1) FROM @PARMTABLE) = 4)
    BEGIN
      --date difference as an Age function only
      SET @RESULT = 'f.DateDiff(' + (SELECT [PRM].[PARAMETER]
                                       FROM @PARMTABLE [PRM]
                                      WHERE [PRM].[id] = 1) + ', '
                                  + (SELECT [PRM].[PARAMETER]
                                       FROM @PARMTABLE [PRM]
                                      WHERE [PRM].[id] = 2) + ', '
                                  + (SELECT [PRM].[PARAMETER]
                                       FROM @PARMTABLE [PRM]
                                      WHERE [PRM].[id] = 3) + ', '
                                  + (SELECT [PRM].[Parameter]
                                       FROM @PARMTABLE [PRM]
                                      WHERE [PRM].[id] = 4) + ')'
    END
  ELSE IF (@FUNCTION = 'choose') AND ((SELECT COUNT(1) FROM @PARMTABLE) > 1)
    BEGIN
      SET @RESULT = ''
      IF ((SELECT COUNT(1) FROM @PARMTABLE) = 3)
        BEGIN
          SET @PARM1 = (SELECT [PRM].[Parameter]
                          FROM @PARMTABLE [PRM]
                         WHERE [PRM].[id] = 1)
          IF [dbo].[ky_NETFNGetUnembeddedCharacterPos]('=', @PARM1) > 0
          OR [dbo].[ky_NETFNGetUnembeddedCharacterPos]('!', @PARM1) > 0
          OR [dbo].[ky_NETFNGetUnembeddedCharacterPos]('<', @PARM1) > 0
          OR [dbo].[ky_NETFNGetUnembeddedCharacterPos]('>', @PARM1) > 0
            BEGIN
              --This is a "choose" used as the equivalent of an IIF
              SET @RESULT = 'CASE WHEN ' + (SELECT CONVERT(VARCHAR, [dbo].[ky_NETFNInterpretComparison]([PRM].[Parameter]))
                                              FROM @PARMTABLE [PRM]
                                             WHERE [PRM].[id] = 1) + ' = 1 THEN '
                                         + (SELECT [PRM].[Parameter]
                                              FROM @PARMTABLE [PRM]
                                             WHERE [PRM].[id] = 2) + ' ELSE '
                                         + (SELECT [PRM].[Parameter]
                                              FROM @PARMTABLE [PRM]
                                             WHERE [PRM].[id] = 3) + ' END'
            END
        END
      IF @RESULT = ''
        BEGIN
          -- Any other type of 'Choose'
          SET @RESULT = 'CASE ' + @PARM1
          SET @PARMNO = 1
          SELECT @PARMCOUNT = COUNT(1) FROM @PARMTABLE
          WHILE @PARMNO < @PARMCOUNT
            BEGIN
              SET @PARMNO = @PARMNO + 1
              SET @RESULT = @RESULT
                          + ' WHEN ' + CONVERT(VARCHAR, @PARMNO - 1)
                          + ' THEN ' + (SELECT [PRM].[Parameter]
                                          FROM @PARMTABLE [PRM]
                                         WHERE [PRM].[id] = @PARMNO)
            END
          SET @RESULT = @RESULT + ' ELSE '''' END'
        END
    END
  ELSE IF (@FUNCTION = 'today') AND ((SELECT COUNT(1) FROM @PARMTABLE) = 0)
    BEGIN
      SET @RESULT = 'f.Today()'
    END
  ELSE IF (@FUNCTION = 'now') AND ((SELECT COUNT(1) FROM @PARMTABLE) = 0)
    BEGIN
      SET @RESULT = 'f.Now()'
    END
  ELSE
    --Just pass back out whatever was passed in
    BEGIN
      SET @RESULT = ''

        SELECT @RESULT = @RESULT + CASE WHEN @RESULT = '' THEN '' ELSE ', ' END + [PRM].[Parameter]
          FROM @PARMTABLE [PRM]
      ORDER BY [id]

      SET @RESULT = @FUNCTION + '(' + @RESULT + ')'
    END

  RETURN @RESULT

END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETFNApplyOperator'
      AND SO.[type] = 'FN')
  BEGIN
    DROP FUNCTION [dbo].[ky_NETFNApplyOperator]
  END
GO

CREATE FUNCTION
       [dbo].[ky_NETFNApplyOperator]
      (@ORIGINAL       VARCHAR(MAX))
RETURNS VARCHAR(MAX)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETFNApplyOperator]                                                                             *
 *                                                                                                           *
 * Replaces standard operators with the f.Function Equivalents                                               *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.6.4.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2019-01-18   Pino Carafa       Created                                                                    *
 * 2019-01-21   Pino Carafa       Distinguish addition '+' from concatenation '&'. '+' needs type awareness  *
 *                                so that, e.g. Today() + 10 is interpreted as "10 days from today", but     *
 *                                & is a simple string concatenation                                         *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

DECLARE @OPERATORPOS INT
DECLARE @OPERATORPOS2 INT

SET @OPERATORPOS = [dbo].[ky_NETFNGetUnembeddedCharacterPos]('+', @ORIGINAL)
SET @OPERATORPOS2 = [dbo].[ky_NETFNGetUnembeddedCharacterPos]('&', @ORIGINAL)
IF (@OPERATORPOS2 > 0) AND (@OPERATORPOS2 < @OPERATORPOS)
  BEGIN
    SET @OPERATORPOS = @OPERATORPOS2
  END
ELSE IF @OPERATORPOS = 0
  BEGIN
    SET @OPERATORPOS = @OPERATORPOS2
  END
SET @OPERATORPOS2 = [dbo].[ky_NETFNGetUnembeddedCharacterPos]('-', @ORIGINAL)
IF (@OPERATORPOS2 > 0) AND (@OPERATORPOS2 < @OPERATORPOS)
  BEGIN
    SET @OPERATORPOS = @OPERATORPOS2
  END
ELSE IF @OPERATORPOS = 0
  BEGIN
    SET @OPERATORPOS = @OPERATORPOS2
  END
SET @OPERATORPOS2 = [dbo].[ky_NETFNGetUnembeddedCharacterPos]('*', @ORIGINAL)
IF (@OPERATORPOS2 > 0) AND (@OPERATORPOS2 < @OPERATORPOS)
  BEGIN
    SET @OPERATORPOS = @OPERATORPOS2
  END
ELSE IF @OPERATORPOS = 0
  BEGIN
    SET @OPERATORPOS = @OPERATORPOS2
  END
SET @OPERATORPOS2 = [dbo].[ky_NETFNGetUnembeddedCharacterPos]('/', @ORIGINAL)
IF (@OPERATORPOS2 > 0) AND (@OPERATORPOS2 < @OPERATORPOS)
  BEGIN
    SET @OPERATORPOS = @OPERATORPOS2
  END
ELSE IF @OPERATORPOS = 0
  BEGIN
    SET @OPERATORPOS = @OPERATORPOS2
  END

WHILE @OPERATORPOS > 0
  BEGIN
    DECLARE @CHAR CHAR(1)
    SET @CHAR = SUBSTRING(@ORIGINAL, @OPERATORPOS, 1)

    IF @CHAR = '+'
      BEGIN
        SET @ORIGINAL = 'f.DoAdd(' + RTRIM(SUBSTRING(@ORIGINAL, 1, @OPERATORPOS - 1)) + ', ' + [dbo].[ky_NETFNApplyOperator](LTRIM(SUBSTRING(@ORIGINAL, @OPERATORPOS + 1, LEN(@ORIGINAL) - @OPERATORPOS))) + ')'
      END
    ELSE IF @CHAR = '&'
      BEGIN
        SET @ORIGINAL = 'f.Concatenate(' + RTRIM(SUBSTRING(@ORIGINAL, 1, @OPERATORPOS - 1)) + ', ' + [dbo].[ky_NETFNApplyOperator](LTRIM(SUBSTRING(@ORIGINAL, @OPERATORPOS + 1, LEN(@ORIGINAL) - @OPERATORPOS))) + ')'
      END
    ELSE IF @CHAR = '-'
      BEGIN
        SET @ORIGINAL = 'f.DoSubtract(' + RTRIM(SUBSTRING(@ORIGINAL, 1, @OPERATORPOS - 1)) + ', ' + [dbo].[ky_NETFNApplyOperator](LTRIM(SUBSTRING(@ORIGINAL, @OPERATORPOS + 1, LEN(@ORIGINAL) - @OPERATORPOS))) + ')'
      END
    ELSE IF @CHAR = '*'
      BEGIN
        SET @ORIGINAL = 'f.DoMultiply(' + RTRIM(SUBSTRING(@ORIGINAL, 1, @OPERATORPOS - 1)) + ', ' + [dbo].[ky_NETFNApplyOperator](LTRIM(SUBSTRING(@ORIGINAL, @OPERATORPOS + 1, LEN(@ORIGINAL) - @OPERATORPOS))) + ')'
      END
    ELSE IF @CHAR = '/'
      BEGIN
        SET @ORIGINAL = 'f.DoDivide(' + RTRIM(SUBSTRING(@ORIGINAL, 1, @OPERATORPOS - 1)) + ', ' + [dbo].[ky_NETFNApplyOperator](LTRIM(SUBSTRING(@ORIGINAL, @OPERATORPOS + 1, LEN(@ORIGINAL) - @OPERATORPOS))) + ')'
      END

    SET @OPERATORPOS = [dbo].[ky_NETFNGetUnembeddedCharacterPos]('+', @ORIGINAL)
    SET @OPERATORPOS2 = [dbo].[ky_NETFNGetUnembeddedCharacterPos]('&', @ORIGINAL)
    IF (@OPERATORPOS2 > 0) AND (@OPERATORPOS2 < @OPERATORPOS)
      BEGIN
        SET @OPERATORPOS = @OPERATORPOS2
      END
    ELSE IF @OPERATORPOS = 0
      BEGIN
        SET @OPERATORPOS = @OPERATORPOS2
      END
    SET @OPERATORPOS2 = [dbo].[ky_NETFNGetUnembeddedCharacterPos]('-', @ORIGINAL)
    IF (@OPERATORPOS2 > 0) AND (@OPERATORPOS2 < @OPERATORPOS)
      BEGIN
        SET @OPERATORPOS = @OPERATORPOS2
      END
    ELSE IF @OPERATORPOS = 0
      BEGIN
        SET @OPERATORPOS = @OPERATORPOS2
      END
    SET @OPERATORPOS2 = [dbo].[ky_NETFNGetUnembeddedCharacterPos]('*', @ORIGINAL)
    IF (@OPERATORPOS2 > 0) AND (@OPERATORPOS2 < @OPERATORPOS)
      BEGIN
        SET @OPERATORPOS = @OPERATORPOS2
      END
    ELSE IF @OPERATORPOS = 0
      BEGIN
        SET @OPERATORPOS = @OPERATORPOS2
      END
    SET @OPERATORPOS2 = [dbo].[ky_NETFNGetUnembeddedCharacterPos]('/', @ORIGINAL)
    IF (@OPERATORPOS2 > 0) AND (@OPERATORPOS2 < @OPERATORPOS)
      BEGIN
        SET @OPERATORPOS = @OPERATORPOS2
      END
    ELSE IF @OPERATORPOS = 0
      BEGIN
        SET @OPERATORPOS = @OPERATORPOS2
      END
  END

RETURN @ORIGINAL

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETFNFieldIDInterpreter'
      AND SO.[type] = 'FN')
  BEGIN
    DROP FUNCTION [dbo].[ky_NETFNFieldIDInterpreter]
  END
GO

CREATE FUNCTION
       [dbo].[ky_NETFNFieldIDInterpreter]
      (@ORIGINAL       VARCHAR(MAX),
       @FORFUNCTION    BIT,
       @mattercode     VARCHAR(20),     --From original @SEARCHES, *just in case* we ever need them - START
       @actionid       INT,             
       @trackreference INT,
       @actualname     VARCHAR(500),
       @retrievalcode  VARCHAR(500))    --From original @SEARCHES, *just in case* we ever need them - END)
RETURNS VARCHAR(MAX)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETFNFieldIDInterpreter]                                                                        *
 *                                                                                                           *
 * Converts old Clarion syntax to the .NET Generator compatible equivalent                                   *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.6.4.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2019-01-10   Pino Carafa       Created                                                                    *
 * 2019-01-14   Pino Carafa       Use [dbo].[ky_NETFNGetUnquotedCharacterPos] instead of CHARINDEX           *
 * 2019-01-16   Pino Carafa       Handle PARTS when an operator like + is not preceded by a blank            *
 * 2019-01-21   Pino Carafa       Recognise the & operator as distinct from +. + will be "type sensitive"    *
 *                                whereas & is purely string-based.                                          *
 * 2019-01-23   Pino Carafa       KEYD-5668 - Distinguish negatives -Y from subtractions X-Y                 *
 * 2019-02-01   Pino Carafa       Prepare for recognising embedded Aggregate codes                           *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

SET @ORIGINAL = REPLACE(@ORIGINAL, '#@&', '#_AND_')
SET @ORIGINAL = REPLACE(@ORIGINAL, '#@,', '#_CND_')
SET @ORIGINAL = REPLACE(@ORIGINAL, '#@r', '#_NLN_')

DECLARE @SIMPLEQUOTE VARCHAR(MAX)
DECLARE @POS INT
DECLARE @LEN INT
DECLARE @PART VARCHAR(MAX)
DECLARE @WORK VARCHAR(MAX)
DECLARE @SIMPLEQUOTEINDENT BIT
DECLARE @QUOTEINDENT BIT
DECLARE @CHAR CHAR(1)
DECLARE @ORIGINALCHAR VARCHAR(1)
DECLARE @BRACKINDENT INT
DECLARE @FUNCTION VARCHAR(MAX)
DECLARE @PARAMETERS TABLE
       ([id]           INT             IDENTITY(1, 1)            NOT NULL PRIMARY KEY,
        [Parameter]    VARCHAR(MAX)                              NOT NULL)

-- Handle Negatives KEYD-5668 START 
DECLARE @NEGPOS     INT
DECLARE @NEGPOS2    INT
DECLARE @NEGSTART   VARCHAR(MAX)
DECLARE @NEGREST    VARCHAR(MAX)
DECLARE @NEGNEXTPOS INT

SET @NEGPOS = [dbo].[ky_NETFNGetUnembeddedCharacterPos]('-', @ORIGINAL)
WHILE @NEGPOS > 0
  BEGIN
    SET @NEGSTART = RTRIM(SUBSTRING(@ORIGINAL, 1, @NEGPOS - 1))
    SET @NEGREST  = LTRIM(SUBSTRING(@ORIGINAL, @NEGPOS + 1, LEN(@ORIGINAL) - @NEGPOS))
    IF @NEGSTART = ''
      BEGIN
        SET @NEGNEXTPOS = [dbo].[ky_NETFNGetUnembeddedCharacterPos]('=', @NEGREST)
        SET @NEGPOS2    = [dbo].[ky_NETFNGetUnembeddedCharacterPos]('>', @NEGREST)
        IF @NEGNEXTPOS = 0
          SET @NEGNEXTPOS = @NEGPOS2
        ELSE IF (@NEGPOS2 > 0) AND (@NEGPOS2 < @NEGNEXTPOS)
          SET @NEGNEXTPOS = @NEGPOS2
        SET @NEGPOS2    = [dbo].[ky_NETFNGetUnembeddedCharacterPos]('<', @NEGREST)
        IF @NEGNEXTPOS = 0
          SET @NEGNEXTPOS = @NEGPOS2
        ELSE IF (@NEGPOS2 > 0) AND (@NEGPOS2 < @NEGNEXTPOS)
          SET @NEGNEXTPOS = @NEGPOS2
        SET @NEGPOS2    = [dbo].[ky_NETFNGetUnembeddedCharacterPos]('+', @NEGREST)
        IF @NEGNEXTPOS = 0
          SET @NEGNEXTPOS = @NEGPOS2
        ELSE IF (@NEGPOS2 > 0) AND (@NEGPOS2 < @NEGNEXTPOS)
          SET @NEGNEXTPOS = @NEGPOS2
        SET @NEGPOS2    = [dbo].[ky_NETFNGetUnembeddedCharacterPos]('*', @NEGREST)
        IF @NEGNEXTPOS = 0
          SET @NEGNEXTPOS = @NEGPOS2
        ELSE IF (@NEGPOS2 > 0) AND (@NEGPOS2 < @NEGNEXTPOS)
          SET @NEGNEXTPOS = @NEGPOS2
        SET @NEGPOS2    = [dbo].[ky_NETFNGetUnembeddedCharacterPos]('/', @NEGREST)
        IF @NEGNEXTPOS = 0
          SET @NEGNEXTPOS = @NEGPOS2
        ELSE IF (@NEGPOS2 > 0) AND (@NEGPOS2 < @NEGNEXTPOS)
          SET @NEGNEXTPOS = @NEGPOS2
        SET @NEGPOS2    = [dbo].[ky_NETFNGetUnembeddedCharacterPos]('-', @NEGREST)
        IF @NEGNEXTPOS = 0
          SET @NEGNEXTPOS = @NEGPOS2
        ELSE IF (@NEGPOS2 > 0) AND (@NEGPOS2 < @NEGNEXTPOS)
          SET @NEGNEXTPOS = @NEGPOS2
        SET @NEGPOS2    = [dbo].[ky_NETFNGetUnembeddedCharacterPos](',', @NEGREST)
        IF @NEGNEXTPOS = 0
          SET @NEGNEXTPOS = @NEGPOS2
        ELSE IF (@NEGPOS2 > 0) AND (@NEGPOS2 < @NEGNEXTPOS)
          SET @NEGNEXTPOS = @NEGPOS2

        IF @NEGNEXTPOS = 0
          BEGIN
            SET @ORIGINAL = '(0 - ' + @NEGREST + ')'
          END
        ELSE
          BEGIN
            SET @ORIGINAL = '(0 - ' + RTRIM(SUBSTRING(@NEGREST, 1, @NEGNEXTPOS - 1)) + ') ' + LTRIM(SUBSTRING(@NEGREST, @NEGNEXTPOS, LEN(@NEGREST) - @NEGNEXTPOS + 1))
          END

        SET @NEGPOS = [dbo].[ky_NETFNGetUnembeddedCharacterPos]('-', @ORIGINAL)
      END
    ELSE IF SUBSTRING(@NEGSTART, LEN(@NEGSTART), 1) NOT IN ('=', '>', '<', '+', '*', '/', '(', ',')
      BEGIN
        SET @NEGPOS = [dbo].[ky_NETFNGetUnembeddedCharacterPosFrom]('-', @ORIGINAL, @NEGPOS + 1)
      END
    ELSE
      BEGIN
        SET @NEGNEXTPOS = [dbo].[ky_NETFNGetUnembeddedCharacterPos]('=', @NEGREST)
        SET @NEGPOS2    = [dbo].[ky_NETFNGetUnembeddedCharacterPos]('>', @NEGREST)
        IF @NEGNEXTPOS = 0
          SET @NEGNEXTPOS = @NEGPOS2
        ELSE IF (@NEGPOS2 > 0) AND (@NEGPOS2 < @NEGNEXTPOS)
          SET @NEGNEXTPOS = @NEGPOS2
        SET @NEGPOS2    = [dbo].[ky_NETFNGetUnembeddedCharacterPos]('<', @NEGREST)
        IF @NEGNEXTPOS = 0
          SET @NEGNEXTPOS = @NEGPOS2
        ELSE IF (@NEGPOS2 > 0) AND (@NEGPOS2 < @NEGNEXTPOS)
          SET @NEGNEXTPOS = @NEGPOS2
        SET @NEGPOS2    = [dbo].[ky_NETFNGetUnembeddedCharacterPos]('+', @NEGREST)
        IF @NEGNEXTPOS = 0
          SET @NEGNEXTPOS = @NEGPOS2
        ELSE IF (@NEGPOS2 > 0) AND (@NEGPOS2 < @NEGNEXTPOS)
          SET @NEGNEXTPOS = @NEGPOS2
        SET @NEGPOS2    = [dbo].[ky_NETFNGetUnembeddedCharacterPos]('*', @NEGREST)
        IF @NEGNEXTPOS = 0
          SET @NEGNEXTPOS = @NEGPOS2
        ELSE IF (@NEGPOS2 > 0) AND (@NEGPOS2 < @NEGNEXTPOS)
          SET @NEGNEXTPOS = @NEGPOS2
        SET @NEGPOS2    = [dbo].[ky_NETFNGetUnembeddedCharacterPos]('/', @NEGREST)
        IF @NEGNEXTPOS = 0
          SET @NEGNEXTPOS = @NEGPOS2
        ELSE IF (@NEGPOS2 > 0) AND (@NEGPOS2 < @NEGNEXTPOS)
          SET @NEGNEXTPOS = @NEGPOS2
        SET @NEGPOS2    = [dbo].[ky_NETFNGetUnembeddedCharacterPos]('-', @NEGREST)
        IF @NEGNEXTPOS = 0
          SET @NEGNEXTPOS = @NEGPOS2
        ELSE IF (@NEGPOS2 > 0) AND (@NEGPOS2 < @NEGNEXTPOS)
          SET @NEGNEXTPOS = @NEGPOS2
        SET @NEGPOS2    = [dbo].[ky_NETFNGetUnembeddedCharacterPos](',', @NEGREST)
        IF @NEGNEXTPOS = 0
          SET @NEGNEXTPOS = @NEGPOS2
        ELSE IF (@NEGPOS2 > 0) AND (@NEGPOS2 < @NEGNEXTPOS)
          SET @NEGNEXTPOS = @NEGPOS2

        IF @NEGNEXTPOS = 0
          BEGIN
            SET @ORIGINAL = @NEGSTART + ' (0 - ' + @NEGREST + ')'
          END
        ELSE
          BEGIN
            SET @ORIGINAL = @NEGSTART + ' (0 - ' + RTRIM(SUBSTRING(@NEGREST, 1, @NEGNEXTPOS - 1)) + ') ' + LTRIM(SUBSTRING(@NEGREST, @NEGNEXTPOS, LEN(@NEGREST) - @NEGNEXTPOS + 1))
          END

        SET @NEGPOS = [dbo].[ky_NETFNGetUnembeddedCharacterPos]('-', @ORIGINAL)
      END
  END
-- Handle Negatives KEYD-5668 END

SET @SIMPLEQUOTE = REPLACE(REPLACE(@ORIGINAL, '', ''''), '', '''')

SET @POS = 0
SET @LEN = LEN(@ORIGINAL)
SET @PART = ''
SET @WORK = ''
SET @SIMPLEQUOTEINDENT = 0
SET @QUOTEINDENT = 0
SET @BRACKINDENT = 0

--Split the string into parts
WHILE @POS < @LEN
  BEGIN
    SET @POS = @POS + 1
    SET @CHAR = SUBSTRING(@SIMPLEQUOTE, @POS, 1)
    SET @ORIGINALCHAR = SUBSTRING(@ORIGINAL, @POS, 1)

    --First we establish whether we're in a Quoted or an Unquoted part of the string
    IF @CHAR = ''''
      BEGIN
        IF @ORIGINALCHAR = ''''
          BEGIN
            SET @SIMPLEQUOTEINDENT = 1 - @SIMPLEQUOTEINDENT
          END
        ELSE
          --it's a smart quote
          BEGIN
            IF @SIMPLEQUOTEINDENT = 0
              BEGIN
                SET @QUOTEINDENT = 1 - @QUOTEINDENT
                SET @ORIGINALCHAR = ''''
              END
          END

      END

    --Get top level bracketing
    --We are not going to interpret anything within quotes.
    IF (@SIMPLEQUOTEINDENT = 0) AND (@QUOTEINDENT = 0)
      BEGIN
        --Now we'll establish whether we're inside a bracketed expression
        IF @CHAR = '('
          BEGIN
            IF @BRACKINDENT = 0
              BEGIN
                --Start of a top level bracketed expression
                SET @FUNCTION = @PART
                SET @PART = ''
                SET @ORIGINALCHAR = ''
              END
            SET @BRACKINDENT = @BRACKINDENT + 1
          END

        IF @CHAR = ')'
          BEGIN
            IF @BRACKINDENT = 1
              BEGIN
                --End of a top level bracketed expression
                DECLARE @PARMS VARCHAR(MAX)

                SET @PARMS = ''

                INSERT
                  INTO @PARAMETERS
                      ([Parameter])
                VALUES(LTRIM(RTRIM(ISNULL(@PART, 'NULL'))))

                  SELECT @PARMS = @PARMS + CASE WHEN @PARMS = '' 
                                                     THEN ''
                                                ELSE '****PARMSEP****' END
                                + [dbo].[ky_NETFNFieldIDInterpreter]([Parameter], 1, @mattercode, @actionid, @trackreference, @actualname, @retrievalcode)
                    FROM @PARAMETERS
                ORDER BY [id]

                SET @PARMS = [dbo].[ky_NETFNFunctionInterpreter](@FUNCTION, @PARMS)

                --SELECT @FUNCTION AS [Function]
                --SELECT [Parameter] FROM @PARAMETERS ORDER BY [id]
                DELETE @PARAMETERS

                SET @WORK = @WORK + @PARMS -- @FUNCTION + '(' + @PARMS + ')'
                SET @PART = ''
                SET @ORIGINALCHAR = ''
              END
            SET @BRACKINDENT = @BRACKINDENT - 1
          END
      END

    --Get function parameters
    --We are not going to interpret anything within quotes.
    IF (@SIMPLEQUOTEINDENT = 0) AND (@QUOTEINDENT = 0)
      BEGIN
        --We'll only do this if we're within a top level bracketed expression
        IF @BRACKINDENT = 1
          BEGIN
            -- Parameters are separated by commas.
            IF @CHAR = ','
              BEGIN
                INSERT
                  INTO @PARAMETERS
                      ([Parameter])
                VALUES(LTRIM(RTRIM(ISNULL(@PART, 'NULL'))))
                SET @PART = ''
                SET @ORIGINALCHAR = ''
              END
          END
      END

    --Process Parts separated by operators or blanks
    --We are not going to interpret anything within quotes.
    IF (@SIMPLEQUOTEINDENT = 0) AND (@QUOTEINDENT = 0)
      BEGIN
        --We are not going to split anything within brackets
        IF @BRACKINDENT = 0
          BEGIN
            DECLARE @LASTCHAR VARCHAR(1)

            IF @CHAR = CONVERT(CHAR(1), ' ')
              BEGIN
                IF @PART <> ''
                  BEGIN
                    SET @PART = [dbo].[ky_NETFNFieldIDParser](@PART, @FORFUNCTION)
                    IF [dbo].[ky_NETFNGetUnquotedCharacterPos](':', @PART) > 0
                      BEGIN
                        RETURN @ORIGINAL
                      END

                    SET @WORK = @WORK + @PART + @ORIGINALCHAR
                    SET @ORIGINALCHAR = ''
                    SET @PART = ''
                  END
                ELSE IF LEN(@WORK) > 0
                  BEGIN
                    IF SUBSTRING(@WORK + '<', LEN(@WORK + '<') - 1, 1) + '.' <> ' .'
                      BEGIN
                        SET @WORK = @WORK + ' '
                        SET @ORIGINALCHAR = ''
                        SET @PART = ''
                      END
                    ELSE
                      BEGIN
                        SET @ORIGINALCHAR = ''
                        SET @PART = ''
                      END
                  END
                ELSE
                  BEGIN
                    SET @ORIGINALCHAR = ''
                    SET @PART = ''
                  END
              END

            --IF @CHAR = '&'
            --  BEGIN
            --    --Clarion concatenation
            --    SET @CHAR = '+'
            --    SET @ORIGINALCHAR = '+'
            --  END

            IF @CHAR IN ('+', '-', '=', '*', '\', '/', '&')
              BEGIN
                IF SUBSTRING(@WORK + '<', LEN(@WORK + '<') - 1, 1) + '.' <> ' .'
                  BEGIN
                    IF @PART <> ''
                      BEGIN
                        SET @PART = [dbo].[ky_NETFNFieldIDParser](@PART, @FORFUNCTION)
                        IF [dbo].[ky_NETFNGetUnquotedCharacterPos](':', @PART) > 0
                          BEGIN
                            RETURN @ORIGINAL
                          END

                        SET @WORK = @WORK + @PART + ' '
                        SET @PART = ''
                      END
                    SET @WORK = @WORK + ' '
                  END
                ELSE
                  BEGIN
                    IF @PART <> ''
                      BEGIN
                        SET @PART = [dbo].[ky_NETFNFieldIDParser](@PART, @FORFUNCTION)
                        IF [dbo].[ky_NETFNGetUnquotedCharacterPos](':', @PART) > 0
                          BEGIN
                            RETURN @ORIGINAL
                          END

                        SET @WORK = @WORK + @PART + ' '
                        SET @PART = ''
                      END
                  END

                SET @WORK = @WORK + @CHAR
                SET @ORIGINALCHAR = ''

                IF @POS < @LEN
                  BEGIN
                    IF SUBSTRING(@ORIGINAL, @POS + 1, 1) + '.' <> ' .'
                      BEGIN
                        SET @WORK = @WORK + ' '
                      END
                  END
              END
          END
      END

    SET @PART = @PART + @ORIGINALCHAR

  END

-- Process whatever may be left at the end of the string
IF @PART <> ''
  BEGIN
    SET @PART = [dbo].[ky_NETFNFieldIDParser](@PART, @FORFUNCTION)
    IF [dbo].[ky_NETFNGetUnquotedCharacterPos](':', @PART) > 0
      BEGIN
        RETURN @ORIGINAL
      END
    SET @WORK = @WORK + @PART
  END

SET @WORK = [dbo].[ky_NETFNApplyOperator](@WORK)
RETURN @WORK
END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPRetrievalCodeInterpreter'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPRetrievalCodeInterpreter]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETSPRetrievalCodeInterpreter]
      (@mattercode     VARCHAR(20),     --From original @SEARCHES, *just in case* we ever need them - START
       @actionid       INT,             
       @trackreference INT,
       @actualname     VARCHAR(500),
       @retrievalcode  VARCHAR(500),    --From original @SEARCHES, *just in case* we ever need them - END
       @ORIGINAL       VARCHAR(MAX)    OUT)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETSPRetrievalCodeInterpreter]                                                                  *
 *                                                                                                           *
 * Replaces a Retrieval Code with its interpreted SYS: equivalent                                            *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.6.4.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2019-01-11   Pino Carafa       Created                                                                    *
 * 2019-01-31   Pino Carafa       Handle a few unusual scenarios e.g. where the retrieval code is merely a   *
 *                                literal string or a hard coded value                                       *
 * 2019-07-16   Pino Carafa       Functionality to handle Doc Naming on the basis of LCN, CAN and AUD        *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

DECLARE @WORK VARCHAR(MAX)
DECLARE @WORK2 VARCHAR(MAX)
DECLARE @LEN INT

SET @WORK = @ORIGINAL

SET @LEN = LEN(@WORK)
IF @LEN > 0
  BEGIN
    IF (SUBSTRING(@WORK, 1, 1) = '[') AND (SUBSTRING(@WORK, LEN(@WORK), 1) = ']')
      BEGIN
        SET @WORK = SUBSTRING(@WORK, 2, LEN(@WORK) - 2)
      END
  END

SET @LEN = LEN(@WORK)
IF @LEN > 0
  BEGIN
    IF PATINDEX('%SYS:%', @WORK) = 1
      BEGIN
        SET @WORK = SUBSTRING(@WORK, 5, LEN(@WORK) - 4)
      END
  END

SET @LEN = LEN(@WORK)
IF @LEN > 0
  BEGIN
    IF SUBSTRING(@WORK, 1, 1) = ''''
      BEGIN
        -- This handles the rare occasion where a SYS: field starts with a literal. To
        -- deal with that, we'll just concatenate an empty string before that, causing
        -- the Field ID Interpreter to work it out.
        SET @WORK = ''''' & ' + @WORK
      END
  END

DECLARE @originaldocnaming VARCHAR(MAX)
SET @originaldocnaming = ''
DECLARE @docnamingprefix VARCHAR(10)
DECLARE @docnamingpos INT

DECLARE @PREFIXES TABLE
       ([PREFIX]  VARCHAR(10))

INSERT
  INTO @PREFIXES
      ([PREFIX])
VALUES('%>lcn:%')

INSERT
  INTO @PREFIXES
      ([PREFIX])
VALUES('%>can:%')

INSERT
  INTO @PREFIXES
      ([PREFIX])
VALUES('%>aud:%')

WHILE EXISTS (SELECT TOP 1 1 FROM @PREFIXES)
  BEGIN
    SELECT TOP 1 @docnamingprefix = [PREFIX] FROM @PREFIXES

    SET @docnamingpos = PATINDEX(@docnamingprefix, @retrievalcode)
    IF @docnamingpos > 0
      BEGIN
        SET @originaldocnaming = SUBSTRING(@retrievalcode, 1, @docnamingpos - 1)
        SET @retrievalcode = SUBSTRING(@retrievalcode, @docnamingpos + 1, LEN(@retrievalcode) - @docnamingpos)
      END

    IF @originaldocnaming = ''
      BEGIN
        DELETE @PREFIXES WHERE [PREFIX] = @docnamingprefix
      END
    ELSE
      BEGIN
        DELETE @PREFIXES
      END
  END

IF @originaldocnaming <> ''
  IF LEN(@originaldocnaming) <= LEN(@WORK)
    IF SUBSTRING(@WORK, 1, LEN(@originaldocnaming)) = @originaldocnaming
      SET @WORK = SUBSTRING(@WORK, LEN(@originaldocnaming) + 2, LEN(@WORK) - LEN(@originaldocnaming) - 1)

SET @WORK2 = [dbo].[ky_NETFNFieldIDInterpreter](@WORK, 0, @mattercode, @actionid, @trackreference, @actualname, @retrievalcode)

IF @WORK <> @WORK2
  BEGIN
    IF @originaldocnaming = ''
      BEGIN
        SET @ORIGINAL = '[SYS:' + @WORK2 + ']'
      END
    ELSE
      BEGIN
        SET @ORIGINAL = '[' + @originaldocnaming + '>SYS:' + @WORK2 + ']'
      END
    RETURN
  END

IF CHARINDEX('''', @WORK) > 0
  BEGIN
    RETURN
  END

IF CHARINDEX('(', @WORK) > 0
  BEGIN
    RETURN
  END

IF CHARINDEX(':', @WORK) > 0
  BEGIN
    RETURN
  END

IF @originaldocnaming = ''
  BEGIN
    SET @ORIGINAL = '[SYS:'''' & ' + @WORK2 + ']'
  END
ELSE
  BEGIN
    SET @ORIGINAL = '[' + @originaldocnaming + '>SYS:'''' & ' + @WORK2 + ']'
  END

RETURN

END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETFNGetSQLConcatenator'
      AND SO.[type] = 'FN')
  BEGIN
    DROP FUNCTION [dbo].[ky_NETFNGetSQLConcatenator]
  END
GO

CREATE FUNCTION
       [dbo].[ky_NETFNGetSQLConcatenator]
      (@ORIGINAL VARCHAR(MAX))
RETURNS VARCHAR(MAX)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETFNGetSQLConcatenator]                                                                        *
 *                                                                                                           *
 * Replaces Clarion ampersands with SQL + signs in concatenators                                             *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.6.4.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2019-01-09   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

DECLARE @SIMPLEQUOTE VARCHAR(MAX)
DECLARE @WORK VARCHAR(MAX)

DECLARE @END INT
DECLARE @INDENT INT
DECLARE @SIMPLEQUOTEINDENT BIT
DECLARE @QUOTEINDENT BIT
DECLARE @CHAR VARCHAR(1)

SET @SIMPLEQUOTE = REPLACE(REPLACE(@ORIGINAL, '', ''''), '', '''')
SET @WORK = ''

SET @END = 0
SET @SIMPLEQUOTEINDENT = 0
SET @QUOTEINDENT = 0

WHILE @END < LEN(@SIMPLEQUOTE)
  BEGIN
    SET @END = @END + 1
    SET @CHAR = SUBSTRING(@ORIGINAL, @END, 1)

    IF SUBSTRING(@SIMPLEQUOTE, @END, 1) = ''''
      BEGIN
        IF SUBSTRING(@ORIGINAL, @END, 1) = ''''
          BEGIN
            SET @SIMPLEQUOTEINDENT = 1 - @SIMPLEQUOTEINDENT
          END
        ELSE
          --it's a smart quote
          BEGIN
            IF @SIMPLEQUOTEINDENT = 0
              BEGIN
                SET @QUOTEINDENT = 1 - @QUOTEINDENT
              END
          END

      END

    IF (@CHAR = '&') AND (@SIMPLEQUOTEINDENT = 0) AND (@QUOTEINDENT = 0)
      BEGIN
        SET @WORK = @WORK + '+'
      END
    ELSE
      BEGIN
        SET @WORK = @WORK + @CHAR
      END

  END

RETURN @WORK

END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETFNGetUDFDecider'
      AND SO.[type] = 'FN')
  BEGIN
    DROP FUNCTION [dbo].[ky_NETFNGetUDFDecider]
  END
GO

CREATE FUNCTION
       [dbo].[ky_NETFNGetUDFDecider]
      (@ORIGINAL VARCHAR(MAX))
RETURNS VARCHAR(MAX)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETFNGetUDFDecider]                                                                             *
 *                                                                                                           *
 * Converts an old Clarion iif(UDF(' syntax to the .NET Generator compatible equivalent                      *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.6.4.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2019-01-09   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

DECLARE @SIMPLEQUOTE VARCHAR(MAX)

SET @SIMPLEQUOTE = REPLACE(REPLACE(@ORIGINAL, '', ''''), '', '''')

DECLARE @STARTPART VARCHAR(MAX)
DECLARE @TRUEPART VARCHAR(MAX)
DECLARE @TRUEVALUE VARCHAR(MAX)
DECLARE @FALSEVALUE VARCHAR(MAX)
DECLARE @ENDPART VARCHAR(MAX)
DECLARE @UDF VARCHAR(30)

DECLARE @START INT
DECLARE @END INT
DECLARE @INDENT INT
DECLARE @SIMPLEQUOTEINDENT BIT
DECLARE @QUOTEINDENT BIT

SET @STARTPART = ''
SET @ENDPART = ''

SET @START = PATINDEX('%iif(UDF(''%', @SIMPLEQUOTE)
IF @START > 0
  BEGIN
    SET @STARTPART = SUBSTRING(@ORIGINAL, 1, @START - 1)    
  END

SET @START = CHARINDEX('''', @SIMPLEQUOTE, @START)
IF @START > 0
  BEGIN
    SET @END = CHARINDEX('''', @SIMPLEQUOTE, @START + 1)
  END

IF @END = 0
  BEGIN
    RETURN @ORIGINAL
    --RETURN
  END

SET @UDF = SUBSTRING(@ORIGINAL, @START + 1, @END - @START - 1)

SET @END = @END + 1
IF @END > LEN(@SIMPLEQUOTE)
  BEGIN
    RETURN @ORIGINAL
    --RETURN
  END

IF SUBSTRING(@SIMPLEQUOTE, @END, 1) <> ')'
  BEGIN
    RETURN @ORIGINAL
    --RETURN
  END

DECLARE @FOUND BIT
SET @FOUND = 0
WHILE (@END < LEN(@SIMPLEQUOTE)) AND (@FOUND = 0)
  BEGIN
    SET @END = @END + 1
    IF SUBSTRING(@SIMPLEQUOTE, @END, 1) = '='
      BEGIN
        SET @FOUND = 1
      END
  END

IF @FOUND = 0
  BEGIN
    RETURN @ORIGINAL
    --RETURN
  END

SET @START = @END + 1

SET @FOUND = 0

SET @FOUND = 0
SET @SIMPLEQUOTEINDENT = 0
SET @QUOTEINDENT = 0
SET @INDENT = 0
WHILE (@END < LEN(@SIMPLEQUOTE)) AND (@FOUND = 0)
  BEGIN
    SET @END = @END + 1
    IF SUBSTRING(@SIMPLEQUOTE, @END, 1) = ''''
      BEGIN
        IF SUBSTRING(@ORIGINAL, @END, 1) = ''''
          BEGIN
            SET @SIMPLEQUOTEINDENT = 1 - @SIMPLEQUOTEINDENT
          END
        ELSE
          --it's a smart quote
          BEGIN
            IF @SIMPLEQUOTEINDENT = 0
              BEGIN
                SET @QUOTEINDENT = 1 - @QUOTEINDENT
              END
          END

      END

    IF SUBSTRING(@SIMPLEQUOTE, @END, 1) = '('
      BEGIN
        IF (@QUOTEINDENT = 0) AND (@SIMPLEQUOTEINDENT = 0)
          BEGIN
            SET @INDENT = @INDENT + 1
          END
      END

    IF SUBSTRING(@SIMPLEQUOTE, @END, 1) = ')'
      BEGIN
        IF (@QUOTEINDENT = 0) AND (@SIMPLEQUOTEINDENT = 0)
          BEGIN
            SET @INDENT = @INDENT - 1
          END
      END

    IF (@SIMPLEQUOTEINDENT = 0) AND (@QUOTEINDENT = 0) AND (@INDENT = 0)
      BEGIN
        IF SUBSTRING(@SIMPLEQUOTE, @END, 1) = ','
          BEGIN
            SET @FOUND = 1
          END
      END
  END

IF @END > LEN(@SIMPLEQUOTE)
  BEGIN
    RETURN @ORIGINAL
    --RETURN
  END

SET @TRUEPART = LTRIM(RTRIM(SUBSTRING(@ORIGINAL, @START, @END - @START)))

SET @START = @END + 1

SET @FOUND = 0

SET @FOUND = 0
SET @SIMPLEQUOTEINDENT = 0
SET @QUOTEINDENT = 0
SET @INDENT = 0
WHILE (@END < LEN(@SIMPLEQUOTE)) AND (@FOUND = 0)
  BEGIN
    SET @END = @END + 1
    IF SUBSTRING(@SIMPLEQUOTE, @END, 1) = ''''
      BEGIN
        IF SUBSTRING(@ORIGINAL, @END, 1) = ''''
          BEGIN
            SET @SIMPLEQUOTEINDENT = 1 - @SIMPLEQUOTEINDENT
          END
        ELSE
          --it's a smart quote
          BEGIN
            IF @SIMPLEQUOTEINDENT = 0
              BEGIN
                SET @QUOTEINDENT = 1 - @QUOTEINDENT
              END
          END

      END

    IF SUBSTRING(@SIMPLEQUOTE, @END, 1) = '('
      BEGIN
        IF (@QUOTEINDENT = 0) AND (@SIMPLEQUOTEINDENT = 0)
          BEGIN
            SET @INDENT = @INDENT + 1
          END
      END

    IF SUBSTRING(@SIMPLEQUOTE, @END, 1) = ')'
      BEGIN
        IF (@QUOTEINDENT = 0) AND (@SIMPLEQUOTEINDENT = 0)
          BEGIN
            SET @INDENT = @INDENT - 1
          END
      END

    IF (@SIMPLEQUOTEINDENT = 0) AND (@QUOTEINDENT = 0) AND (@INDENT = 0)
      BEGIN
        IF SUBSTRING(@SIMPLEQUOTE, @END, 1) = ','
          BEGIN
            SET @FOUND = 1
          END
      END
  END

IF @END > LEN(@SIMPLEQUOTE)
  BEGIN
    RETURN @ORIGINAL
    --RETURN
  END

SET @TRUEVALUE = LTRIM(RTRIM(SUBSTRING(@ORIGINAL, @START, @END - @START)))

SET @START = @END + 1

SET @FOUND = 0

SET @FOUND = 0
SET @SIMPLEQUOTEINDENT = 0
SET @QUOTEINDENT = 0
SET @INDENT = 0
WHILE (@END < LEN(@SIMPLEQUOTE)) AND (@FOUND = 0)
  BEGIN
    SET @END = @END + 1
    IF SUBSTRING(@SIMPLEQUOTE, @END, 1) = ''''
      BEGIN
        IF SUBSTRING(@ORIGINAL, @END, 1) = ''''
          BEGIN
            SET @SIMPLEQUOTEINDENT = 1 - @SIMPLEQUOTEINDENT
          END
        ELSE
          --it's a smart quote
          BEGIN
            IF @SIMPLEQUOTEINDENT = 0
              BEGIN
                SET @QUOTEINDENT = 1 - @QUOTEINDENT
              END
          END

      END

    IF SUBSTRING(@SIMPLEQUOTE, @END, 1) = '('
      BEGIN
        IF (@QUOTEINDENT = 0) AND (@SIMPLEQUOTEINDENT = 0)
          BEGIN
            SET @INDENT = @INDENT + 1
          END
      END

    IF SUBSTRING(@SIMPLEQUOTE, @END, 1) = ')'
      BEGIN
        IF @INDENT > 0
          BEGIN
            SET @INDENT = @INDENT - 1
          END
        ELSE IF (@SIMPLEQUOTEINDENT = 0) AND (@QUOTEINDENT = 0) AND (@INDENT = 0)
          BEGIN
            SET @FOUND = 1
          END
      END

    IF (@SIMPLEQUOTEINDENT = 0) AND (@QUOTEINDENT = 0) AND (@INDENT = 0)
      BEGIN
        IF SUBSTRING(@SIMPLEQUOTE, @END, 1) = ','
          BEGIN
            SET @FOUND = 1
          END
      END
  END

IF @END > LEN(@SIMPLEQUOTE)
  BEGIN
    RETURN @ORIGINAL
    --RETURN
  END

SET @FALSEVALUE = LTRIM(RTRIM(SUBSTRING(@ORIGINAL, @START, @END - @START)))

SET @END = @END + 1
IF @END <= LEN(@ORIGINAL)
  BEGIN
    SET @ENDPART = SUBSTRING(@ORIGINAL, @END, LEN(@ORIGINAL) - @END + 1)
  END

SET @TRUEPART = [dbo].[ky_NETFNGetSQLConcatenator](@TRUEPART)
IF @TRUEPART LIKE '%iif(UDF(%'
  BEGIN
    SET @TRUEPART = [dbo].[ky_NETFNGetUDFDecider](@TRUEPART)
  END

SET @TRUEVALUE = [dbo].[ky_NETFNGetSQLConcatenator](@TRUEVALUE)
IF @TRUEVALUE LIKE '%iif(UDF(%'
  BEGIN
    SET @TRUEVALUE = [dbo].[ky_NETFNGetUDFDecider](@TRUEVALUE)
  END

SET @FALSEVALUE = [dbo].[ky_NETFNGetSQLConcatenator](@FALSEVALUE)
IF @FALSEVALUE LIKE '%iif(UDF(%'
  BEGIN
    SET @FALSEVALUE = [dbo].[ky_NETFNGetUDFDecider](@FALSEVALUE)
  END

SET @ORIGINAL = @STARTPART + 'f.UDFDecider([CDE].[Code], ''' + @UDF + ''', ' + @TRUEPART + ', ' + @TRUEVALUE + ', ' + @FALSEVALUE + ')' + @ENDPART

RETURN @ORIGINAL
END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETFNGetUDFFormatter'
      AND SO.[type] = 'FN')
  BEGIN
    DROP FUNCTION [dbo].[ky_NETFNGetUDFFormatter]
  END
GO

CREATE FUNCTION
       [dbo].[ky_NETFNGetUDFFormatter]
      (@ORIGINAL VARCHAR(MAX))
RETURNS VARCHAR(MAX)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETFNGetUDFFormatter]                                                                           *
 *                                                                                                           *
 * Converts an old Clarion format(UDF(' syntax to the .NET Generator compatible equivalent                   *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.6.4.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2019-01-09   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

DECLARE @SIMPLEQUOTE VARCHAR(MAX)

SET @SIMPLEQUOTE = REPLACE(REPLACE(@ORIGINAL, '', ''''), '', '''')

DECLARE @STARTPART VARCHAR(MAX)
DECLARE @FORMAT VARCHAR(MAX)
DECLARE @ENDPART VARCHAR(MAX)
DECLARE @UDF VARCHAR(30)

DECLARE @START INT
DECLARE @END INT
DECLARE @INDENT INT
DECLARE @SIMPLEQUOTEINDENT BIT
DECLARE @QUOTEINDENT BIT

SET @STARTPART = ''
SET @ENDPART = ''

SET @START = PATINDEX('%format(UDF(''%', @SIMPLEQUOTE)
IF @START > 0
  BEGIN
    SET @STARTPART = SUBSTRING(@ORIGINAL, 1, @START - 1)    
  END

SET @START = CHARINDEX('''', @SIMPLEQUOTE, @START)
IF @START > 0
  BEGIN
    SET @END = CHARINDEX('''', @SIMPLEQUOTE, @START + 1)
  END

IF @END = 0
  BEGIN
    RETURN @ORIGINAL
    --RETURN
  END

SET @UDF = SUBSTRING(@ORIGINAL, @START + 1, @END - @START - 1)

IF @UDF IS NULL
  BEGIN
    RETURN @ORIGINAL
    --RETURN
  END

SET @END = @END + 1
IF @END > LEN(@SIMPLEQUOTE)
  BEGIN
    RETURN @ORIGINAL
    --RETURN
  END

IF SUBSTRING(@SIMPLEQUOTE, @END, 1) <> ')'
  BEGIN
    RETURN @ORIGINAL
    --RETURN
  END

DECLARE @FOUND BIT
SET @END = @END + 1
SET @START = @END + 1

SET @FOUND = 0

SET @FOUND = 0
SET @SIMPLEQUOTEINDENT = 0
SET @QUOTEINDENT = 0
SET @INDENT = 0
WHILE (@END < LEN(@SIMPLEQUOTE)) AND (@FOUND = 0)
  BEGIN
    SET @END = @END + 1
    IF SUBSTRING(@SIMPLEQUOTE, @END, 1) = ''''
      BEGIN
        IF SUBSTRING(@ORIGINAL, @END, 1) = ''''
          BEGIN
            SET @SIMPLEQUOTEINDENT = 1 - @SIMPLEQUOTEINDENT
          END
        ELSE
          --it's a smart quote
          BEGIN
            IF @SIMPLEQUOTEINDENT = 0
              BEGIN
                SET @QUOTEINDENT = 1 - @QUOTEINDENT
              END
          END

      END

    IF SUBSTRING(@SIMPLEQUOTE, @END, 1) = '('
      BEGIN
        SET @INDENT = @INDENT + 1
      END

    IF SUBSTRING(@SIMPLEQUOTE, @END, 1) = ')'
      BEGIN
        IF @INDENT > 0
          BEGIN
            SET @INDENT = @INDENT - 1
          END
        ELSE IF (@SIMPLEQUOTEINDENT = 0) AND (@QUOTEINDENT = 0) AND (@INDENT = 0)
          BEGIN
            SET @FOUND = 1
          END
      END

    IF (@SIMPLEQUOTEINDENT = 0) AND (@QUOTEINDENT = 0) AND (@INDENT = 0)
      BEGIN
        IF SUBSTRING(@SIMPLEQUOTE, @END, 1) = ','
          BEGIN
            SET @FOUND = 1
          END
      END
  END

IF @END > LEN(@SIMPLEQUOTE)
  BEGIN
    RETURN @ORIGINAL
    --RETURN
  END

SET @FORMAT = LTRIM(RTRIM(SUBSTRING(@ORIGINAL, @START, @END - @START)))
IF @FORMAT IS NULL
  BEGIN
    RETURN @ORIGINAL
    --RETURN
  END

SET @END = @END + 1
IF @END <= LEN(@ORIGINAL)
  BEGIN
    SET @ENDPART = SUBSTRING(@ORIGINAL, @END, LEN(@ORIGINAL) - @END + 1)
  END

DECLARE @DATE BIT

IF @FORMAT LIKE('%d18%')
  BEGIN
    SET @DATE = 1
    SET @FORMAT = '''DD MMMM YYYY'''
  END

IF @FORMAT LIKE('%d8%')
  BEGIN
    SET @DATE = 1
    SET @FORMAT = '''DD MMM YYYY'''
  END

IF @FORMAT LIKE('%d6%')
  BEGIN
    SET @DATE = 1
    SET @FORMAT = '''DD/MM/YY'''
  END

IF @DATE = 1
  BEGIN
    SET @ORIGINAL = @STARTPART + 'f.AddDays(f.UDFDate([CDE].[Code], ''' + @UDF + '''), 0, ' + @FORMAT + ')' + @ENDPART
    RETURN @ORIGINAL
    --RETURN
  END

SET @ORIGINAL = @STARTPART + 'f.GetValue(f.UDFNumber([CDE].[Code], ''' + @UDF + '''), ' + @FORMAT + ', 0)' + @ENDPART
RETURN @ORIGINAL
--RETURN
END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETFNConvertToDateTime'
      AND SO.[type] = 'FN')
  BEGIN
    DROP FUNCTION [dbo].[ky_NETFNConvertToDateTime]
  END
GO

CREATE FUNCTION
       [dbo].[ky_NETFNConvertToDateTime]
      (@DATESTRING VARCHAR(MAX))
RETURNS DATETIME
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETFNConvertToDateTime]                                                                         *
 *                                                                                                           *
 * Convert a string produced by one of our own functions back to a datetime                                  *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.6.4.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2018-12-20   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

IF @DATESTRING LIKE 'Monday%'
  SET @DATESTRING = LTRIM(SUBSTRING(@DATESTRING, 7, LEN(@DATESTRING) - 6))

IF @DATESTRING LIKE 'Tuesday%'
  SET @DATESTRING = LTRIM(SUBSTRING(@DATESTRING, 8, LEN(@DATESTRING) - 7))

IF @DATESTRING LIKE 'Wednesday%'
  SET @DATESTRING = LTRIM(SUBSTRING(@DATESTRING, 10, LEN(@DATESTRING) - 9))

IF @DATESTRING LIKE 'Thursday%'
  SET @DATESTRING = LTRIM(SUBSTRING(@DATESTRING, 9, LEN(@DATESTRING) - 8))

IF @DATESTRING LIKE 'Friday%'
  SET @DATESTRING = LTRIM(SUBSTRING(@DATESTRING, 7, LEN(@DATESTRING) - 6))

IF @DATESTRING LIKE 'Saturday%'
  SET @DATESTRING = LTRIM(SUBSTRING(@DATESTRING, 9, LEN(@DATESTRING) - 8))

IF @DATESTRING LIKE 'Sunday%'
  SET @DATESTRING = LTRIM(SUBSTRING(@DATESTRING, 7, LEN(@DATESTRING) - 6))

-- Remove any of the commas and "the" "day" "of", "nd", "th", "rd" text
SET @DATESTRING = REPLACE(@DATESTRING, ',', '')
SET @DATESTRING = REPLACE(@DATESTRING, 'the ', '')
SET @DATESTRING = REPLACE(@DATESTRING, 'day ', '')
SET @DATESTRING = REPLACE(@DATESTRING, 'of ', '')
SET @DATESTRING = REPLACE(@DATESTRING, 'st', '')
--but restore august
SET @DATESTRING = REPLACE(@DATESTRING, 'Augu', 'August')
--the rest of these won't appear in Month names so we can simply remove them
SET @DATESTRING = REPLACE(@DATESTRING, 'nd', '')
SET @DATESTRING = REPLACE(@DATESTRING, 'rd', '')
SET @DATESTRING = REPLACE(@DATESTRING, 'th', '')

SET @DATESTRING = LTRIM(RTRIM(@DATESTRING))

IF CHARINDEX('/', @DATESTRING) > 4
  BEGIN
    --YYYY/MM/DD
    IF ISDATE(@DATESTRING) = 1
      BEGIN
        RETURN CONVERT(DATETIME, @DATESTRING)
      END
  END
ELSE IF CHARINDEX('/', @DATESTRING) > 0
  BEGIN
    --MUST be DD/MM/YYYY or (for shame) DD/MM/YY
    IF LEN(@DATESTRING) = 8
      RETURN CONVERT(DATETIME, @DATESTRING, 3)
    ELSE
      RETURN CONVERT(DATETIME, @DATESTRING, 103)
  END
ELSE
  BEGIN
    IF ISDATE(@DATESTRING) = 1
      BEGIN
        RETURN CONVERT(DATETIME, @DATESTRING)
      END
  END

RETURN CONVERT(DATETIME, NULL)

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_SUFExtractCounty'
      AND SO.[type] = 'FN')
  BEGIN
    DROP FUNCTION [dbo].[ky_SUFExtractCounty]
  END
GO

CREATE FUNCTION
       [dbo].[ky_SUFExtractCounty]
      (@Address VARCHAR(MAX))
RETURNS VARCHAR(MAX)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_SUFExtractCounty]                                                                               *
 *                                                                                                           *
 * User Defined Scalar Function to extract the County out of a free form address                             *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.6.4.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2018-12-20   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

DECLARE @COUNTYPOS INT

SET @COUNTYPOS = PATINDEX('%county%', @Address)
IF @COUNTYPOS > 0
  BEGIN
    SET @COUNTYPOS = @COUNTYPOS + 6
  END
ELSE
  BEGIN
    SET @COUNTYPOS = PATINDEX('%co.%', @ADDRESS)
    IF @COUNTYPOS > 0
      BEGIN
        SET @COUNTYPOS = @COUNTYPOS + 3
      END
  END

IF @COUNTYPOS = 0
  BEGIN
    RETURN ''
  END

IF @COUNTYPOS > LEN(@Address) - 1
  BEGIN
    RETURN ''
  END

DECLARE @COUNTYEND INT
SET @COUNTYEND = LEN(@Address) + 1
DECLARE @CHAR10 INT
DECLARE @CHAR11 INT
DECLARE @CHAR13 INT
DECLARE @CHARPERIOD INT
DECLARE @CHARCOMMA INT
DECLARE @CHARSPACE INT

SET @CHAR10 = CHARINDEX(CHAR(10), @Address, @COUNTYPOS + 1)
SET @CHAR11 = CHARINDEX(CHAR(11), @Address, @COUNTYPOS + 1)
SET @CHAR13 = CHARINDEX(CHAR(13), @Address, @COUNTYPOS + 1)
SET @CHARPERIOD = CHARINDEX('.', @Address, @COUNTYPOS + 1)
SET @CHARCOMMA = CHARINDEX(',', @Address, @COUNTYPOS + 1)
SET @CHARSPACE = CHARINDEX(' ', @Address, @COUNTYPOS + 1)

IF @CHAR10 > 0
  IF @CHAR10 < @COUNTYEND
    BEGIN
      SET @COUNTYEND = @CHAR10
    END

IF @CHAR11 > 0
  IF @CHAR11 < @COUNTYEND
    BEGIN
      SET @COUNTYEND = @CHAR11
    END

IF @CHAR13 > 0
  IF @CHAR13 < @COUNTYEND
    BEGIN
      SET @COUNTYEND = @CHAR13
    END

IF @CHARPERIOD > 0
  IF @CHARPERIOD < @COUNTYEND
    BEGIN
      SET @COUNTYEND = @CHARPERIOD
    END

IF @CHARCOMMA > 0
  IF @CHARCOMMA < @COUNTYEND
    BEGIN
      SET @COUNTYEND = @CHARCOMMA
    END

IF @CHARSPACE > 0
  IF @CHARSPACE < @COUNTYEND
    BEGIN
      SET @COUNTYEND = @CHARSPACE
    END

IF @COUNTYEND > LEN(@Address)
  BEGIN
    RETURN LTRIM(RTRIM(SUBSTRING(@Address, @COUNTYPOS, LEN(@Address) - @COUNTYPOS + 1)))
  END

RETURN LTRIM(RTRIM(SUBSTRING(@Address, @COUNTYPOS, @COUNTYEND - @COUNTYPOS)))

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_SUFChr'
      AND SO.[type] = 'FN')
  BEGIN
    DROP FUNCTION [dbo].[ky_SUFChr]
  END
GO

CREATE FUNCTION
       [dbo].[ky_SUFChr]
      (@CHAR INT)
RETURNS VARCHAR(MAX)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_SUFChr]                                                                                         *
 *                                                                                                           *
 * User Defined Scalar Function to return a character (for parsing)                                          *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.6.4.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2018-12-19   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

RETURN CHAR(@CHAR)

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETFNQuotedStatement'
      AND SO.[type] = 'FN')
  BEGIN
    DROP FUNCTION [dbo].[ky_NETFNQuotedStatement]
  END
GO

CREATE FUNCTION
       [dbo].[ky_NETFNQuotedStatement]
      (@ORIGINAL VARCHAR(MAX))
RETURNS VARCHAR(MAX)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETFNQuotedStatement]                                                                           *
 *                                                                                                           *
 * Pre-parse a quoted statement to remove Operators                                                          *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.6.4.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2018-12-19   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

DECLARE @New VARCHAR(MAX)
DECLARE @Quoted BIT
DECLARE @STARTPOS INT
DECLARE @ENDPOS INT
DECLARE @WORK VARCHAR(MAX)

SET @New = ''
SET @Quoted = 0

SET @STARTPOS = CHARINDEX('''', @Original)
WHILE @STARTPOS > 0
  BEGIN
    SET @WORK = SUBSTRING(@ORIGINAL, 1, @STARTPOS - 1)
    IF @Quoted = 1
      BEGIN
        SET @WORK = REPLACE(@WORK, '+', '____PLUS____')
        SET @WORK = REPLACE(@WORK, '-', '____MINUS____')
        SET @WORK = REPLACE(@WORK, '/', '____SLASH____')
        SET @WORK = REPLACE(@WORK, '\', '____BACKSLASH____')
        SET @WORK = REPLACE(@WORK, '*', '____STAR____')
        SET @WORK = REPLACE(@WORK, '&', '____AMP____')
      END

    SET @Quoted = 1 - @Quoted
    SET @NEW = @NEW + @WORK
    SET @NEW = @NEW + ''''

    SET @ORIGINAL = SUBSTRING(@ORIGINAL, @STARTPOS + 1, LEN(@ORIGINAL) - @STARTPOS)
    SET @STARTPOS = CHARINDEX('''', @Original)
  END

IF @ORIGINAL <> ''
  BEGIN
    SET @NEW = @NEW + @ORIGINAL
  END

RETURN @NEW
END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETFNUnQuotedStatement'
      AND SO.[type] = 'FN')
  BEGIN
    DROP FUNCTION [dbo].[ky_NETFNUnQuotedStatement]
  END
GO

CREATE FUNCTION
       [dbo].[ky_NETFNUnQuotedStatement]
      (@ORIGINAL VARCHAR(MAX))
RETURNS VARCHAR(MAX)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETFNUnQuotedStatement]                                                                         *
 *                                                                                                           *
 * Pre-parse a quoted statement to restore Operators                                                         *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.6.4.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2018-12-19   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

DECLARE @New VARCHAR(MAX)

SET @New = @ORIGINAL

SET @New = REPLACE(@New, '____PLUS____', '+')
SET @New = REPLACE(@New, '____MINUS____', '-')
SET @New = REPLACE(@New, '____SLASH____', '/')
SET @New = REPLACE(@New, '____BACKSLASH____', '\')
SET @New = REPLACE(@New, '____STAR____', '*')
SET @New = REPLACE(@New, '____AMP____', '&')

RETURN @NEW
END

GO

IF EXISTS
  (SELECT TOP 1 1
          FROM SYS.[objects] [SO]
               LEFT OUTER JOIN SYS.[indexes] [SI]
                            ON [SI].[object_id] = [SO].[object_id]
                           AND [SI].[name] = 'PK_BillHeader'
         WHERE [SO].[name] = 'BillHeader'
           AND [SI].[index_id] IS NULL)
  BEGIN
    DECLARE @PRIMARYKEY NVARCHAR(128)
    SELECT @PRIMARYKEY = SI.[name]
      FROM SYS.[objects] [SO]
           INNER JOIN SYS.[indexes] [SI]
                   ON [SI].[object_id] = [SO].[object_id]
                  AND [SI].[type] = 1
     WHERE [SO].[name] = 'BillHeader'

    IF NOT (@PRIMARYKEY IS NULL)
      BEGIN
        DECLARE @COMMAND VARCHAR(MAX)

        SET @COMMAND = 'ALTER TABLE [dbo].[BillHeader] DROP CONSTRAINT [' + CONVERT(VARCHAR(128), @PRIMARYKEY) + ']'

        EXEC (@COMMAND)

        SET @COMMAND = '
ALTER
TABLE [dbo].[BillHeader] 
  ADD CONSTRAINT [PK_BillHeader]
      PRIMARY KEY CLUSTERED 
     ([DRAFTNO] ASC)
'
        EXEC (@COMMAND)
      END
  END
GO

IF EXISTS
  (SELECT TOP 1 1
          FROM SYS.[objects] [SO]
               LEFT OUTER JOIN SYS.[indexes] [SI]                               
                            ON [SI].[object_id] = [SO].[object_id]
                           AND [SI].[name] = 'KeyUser'
               CROSS APPLY (SELECT COUNT(1) AS [TOTAL]
                              FROM SYS.[index_columns] [IC]
                             WHERE[IC].[object_id] = [SI].[object_id]
                               AND [IC].[index_id] = [SI].[index_id]) [TC]
         WHERE [SO].[name] = 'BillHeader'
           AND [SI].[index_id] IS NOT NULL
           AND [TC].[TOTAL] < 4)
  BEGIN
    DECLARE @COMMAND VARCHAR(MAX)

    SET @COMMAND = 'DROP INDEX [dbo].[BillHeader].[KeyUser]'

    EXEC (@COMMAND)

    SET @COMMAND = '
CREATE NONCLUSTERED INDEX
       [KeyUser]
    ON [dbo].[BillHeader]
      ([USER] ASC,
       [FeCode] ASC,
	     [DATE] ASC,
       [MATTER] ASC)
       INCLUDE ([DRAFTNO],
                [INVCR],
                [BILLNO],
                [OURREF],
                [YOURREF],
                [RELEASED],
                [ACTIONID],
                [BATCHNO],
                [BillSent])

'

    EXEC (@COMMAND)
  END
GO

IF EXISTS
  (SELECT TOP 1 1
          FROM SYS.[objects] [SO]
               LEFT OUTER JOIN SYS.[indexes] [SI]                               
                            ON [SI].[object_id] = [SO].[object_id]
                           AND [SI].[name] = 'KeyActionID'
               CROSS APPLY (SELECT COUNT(1) AS [TOTAL]
                              FROM SYS.[index_columns] [IC]
                             WHERE[IC].[object_id] = [SI].[object_id]
                               AND [IC].[index_id] = [SI].[index_id]) [TC]
         WHERE [SO].[name] = 'BillHeader'
           AND [SI].[index_id] IS NOT NULL
           AND [TC].[TOTAL] < 2)
  BEGIN
    DECLARE @COMMAND VARCHAR(MAX)

    SET @COMMAND = 'DROP INDEX [dbo].[BillHeader].[KeyActionID]'

    EXEC (@COMMAND)

    SET @COMMAND = '
CREATE NONCLUSTERED INDEX
       [KeyActionID]
    ON [dbo].[BillHeader]
      ([ACTIONID] ASC)
       INCLUDE ([DRAFTNO],
                [INVCR],
                [BILLNO],
                [OURREF],
                [YOURREF],
                [RELEASED],
                [BATCHNO],
                [BillSent],
                [USER],
                [FeCode],
	              [DATE],
                [MATTER])

'

    EXEC (@COMMAND)
  END
GO

IF EXISTS
  (SELECT TOP 1 1
          FROM SYS.[objects] [SO]
               LEFT OUTER JOIN SYS.[indexes] [SI]                               
                            ON [SI].[object_id] = [SO].[object_id]
                           AND [SI].[name] = 'KeyBatch'
               CROSS APPLY (SELECT COUNT(1) AS [TOTAL]
                              FROM SYS.[index_columns] [IC]
                             WHERE[IC].[object_id] = [SI].[object_id]
                               AND [IC].[index_id] = [SI].[index_id]) [TC]
         WHERE [SO].[name] = 'BillHeader'
           AND [SI].[index_id] IS NOT NULL
           AND [TC].[TOTAL] < 2)
  BEGIN
    DECLARE @COMMAND VARCHAR(MAX)

    SET @COMMAND = 'DROP INDEX [dbo].[BillHeader].[KeyBatch]'

    EXEC (@COMMAND)

    SET @COMMAND = '
CREATE NONCLUSTERED INDEX
       [KeyBatch]
    ON [dbo].[BillHeader]
      ([BATCHNO] ASC)
       INCLUDE ([DRAFTNO],
                [INVCR],
                [BILLNO],
                [OURREF],
                [YOURREF],
                [RELEASED],
                [ACTIONID],
                [BillSent],
                [USER],
                [FeCode],
	              [DATE],
                [MATTER])

'

    EXEC (@COMMAND)
  END
GO

IF EXISTS
  (SELECT TOP 1 1
          FROM SYS.[objects] [SO]
               LEFT OUTER JOIN SYS.[indexes] [SI]                               
                            ON [SI].[object_id] = [SO].[object_id]
                           AND [SI].[name] = 'KeyBillNo'
               CROSS APPLY (SELECT COUNT(1) AS [TOTAL]
                              FROM SYS.[index_columns] [IC]
                             WHERE[IC].[object_id] = [SI].[object_id]
                               AND [IC].[index_id] = [SI].[index_id]) [TC]
         WHERE [SO].[name] = 'BillHeader'
           AND [SI].[index_id] IS NOT NULL
           AND [TC].[TOTAL] < 3)
  BEGIN
    DECLARE @COMMAND VARCHAR(MAX)

    SET @COMMAND = 'DROP INDEX [dbo].[BillHeader].[KeyBillNo]'

    EXEC (@COMMAND)

    SET @COMMAND = '
CREATE NONCLUSTERED INDEX
       [KeyBillNo]
    ON [dbo].[BillHeader]
      ([MATTER] ASC,
       [BILLNO] ASC)
       INCLUDE ([BATCHNO],
                [DRAFTNO],
                [INVCR],
                [OURREF],
                [YOURREF],
                [RELEASED],
                [ACTIONID],
                [BillSent],
                [USER],
                [FeCode],
	              [DATE])

'

    EXEC (@COMMAND)
  END
GO

IF EXISTS
  (SELECT TOP 1 1
          FROM SYS.[objects] [SO]
               LEFT OUTER JOIN SYS.[indexes] [SI]                               
                            ON [SI].[object_id] = [SO].[object_id]
                           AND [SI].[name] = 'KeyBillNumber'
               CROSS APPLY (SELECT COUNT(1) AS [TOTAL]
                              FROM SYS.[index_columns] [IC]
                             WHERE[IC].[object_id] = [SI].[object_id]
                               AND [IC].[index_id] = [SI].[index_id]) [TC]
         WHERE [SO].[name] = 'BillHeader'
           AND [SI].[index_id] IS NOT NULL
           AND [TC].[TOTAL] < 2)
  BEGIN
    DECLARE @COMMAND VARCHAR(MAX)

    SET @COMMAND = 'DROP INDEX [dbo].[BillHeader].[KeyBillNumber]'

    EXEC (@COMMAND)

    SET @COMMAND = '
CREATE NONCLUSTERED INDEX
       [KeyBillNumber]
    ON [dbo].[BillHeader]
      ([BILLNO] ASC)
       INCLUDE ([MATTER],
                [BATCHNO],
                [DRAFTNO],
                [INVCR],
                [OURREF],
                [YOURREF],
                [RELEASED],
                [ACTIONID],
                [BillSent],
                [USER],
                [FeCode],
	              [DATE])

'

    EXEC (@COMMAND)
  END
GO

IF EXISTS
  (SELECT TOP 1 1
          FROM SYS.[objects] [SO]
               LEFT OUTER JOIN SYS.[indexes] [SI]                               
                            ON [SI].[object_id] = [SO].[object_id]
                           AND [SI].[name] = 'KeyCreditInv'
               CROSS APPLY (SELECT COUNT(1) AS [TOTAL]
                              FROM SYS.[index_columns] [IC]
                             WHERE[IC].[object_id] = [SI].[object_id]
                               AND [IC].[index_id] = [SI].[index_id]) [TC]
         WHERE [SO].[name] = 'BillHeader'
           AND [SI].[index_id] IS NOT NULL
           AND [TC].[TOTAL] < 4)
  BEGIN
    DECLARE @COMMAND VARCHAR(MAX)

    SET @COMMAND = 'DROP INDEX [dbo].[BillHeader].[KeyCreditInv]'

    EXEC (@COMMAND)

    SET @COMMAND = '
CREATE NONCLUSTERED INDEX
       [KeyCreditInv]
    ON [dbo].[BillHeader]
      ([CreditInvoiceNo] ASC,
       [BATCHNO] ASC,
       [DRAFTNO] ASC)
       INCLUDE ([MATTER],
                [BILLNO],
                [INVCR],
                [OURREF],
                [YOURREF],
                [RELEASED],
                [ACTIONID],
                [BillSent],
                [USER],
                [FeCode],
	              [DATE])

'

    EXEC (@COMMAND)
  END
GO

IF EXISTS
  (SELECT TOP 1 1
          FROM SYS.[objects] [SO]
               LEFT OUTER JOIN SYS.[indexes] [SI]                               
                            ON [SI].[object_id] = [SO].[object_id]
                           AND [SI].[name] = 'KeyFE'
               CROSS APPLY (SELECT COUNT(1) AS [TOTAL]
                              FROM SYS.[index_columns] [IC]
                             WHERE[IC].[object_id] = [SI].[object_id]
                               AND [IC].[index_id] = [SI].[index_id]) [TC]
         WHERE [SO].[name] = 'BillHeader'
           AND [SI].[index_id] IS NOT NULL
           AND [TC].[TOTAL] < 4)
  BEGIN
    DECLARE @COMMAND VARCHAR(MAX)

    SET @COMMAND = 'DROP INDEX [dbo].[BillHeader].[KeyFE]'

    EXEC (@COMMAND)

    SET @COMMAND = '
CREATE NONCLUSTERED INDEX
       [KeyFE]
    ON [dbo].[BillHeader]
      ([FeCode] ASC,
       [RELEASED] ASC)
       INCLUDE ([MATTER],
                [BATCHNO],
	              [BILLNO],
                [DRAFTNO],
                [INVCR],
                [OURREF],
                [YOURREF],
                [ACTIONID],
                [BillSent],
                [USER],
                [DATE])

'

    EXEC (@COMMAND)
  END
GO

IF EXISTS
  (SELECT TOP 1 1
          FROM SYS.[objects] [SO]
               LEFT OUTER JOIN SYS.[indexes] [SI]                               
                            ON [SI].[object_id] = [SO].[object_id]
                           AND [SI].[name] = 'KeyOpen'
               CROSS APPLY (SELECT COUNT(1) AS [TOTAL]
                              FROM SYS.[index_columns] [IC]
                             WHERE[IC].[object_id] = [SI].[object_id]
                               AND [IC].[index_id] = [SI].[index_id]) [TC]
         WHERE [SO].[name] = 'BillHeader'
           AND [SI].[index_id] IS NOT NULL
           AND [TC].[TOTAL] < 4)
  BEGIN
    DECLARE @COMMAND VARCHAR(MAX)

    SET @COMMAND = 'DROP INDEX [dbo].[BillHeader].[KeyOpen]'

    EXEC (@COMMAND)

    SET @COMMAND = '
CREATE NONCLUSTERED INDEX
       [KeyOpen]
    ON [dbo].[BillHeader]
      ([OPENINVOICE] ASC,
       [MATTER] ASC,
       [BILLNO] ASC)
       INCLUDE ([BATCHNO],
                [FeCode],
                [RELEASED],
	              [DRAFTNO],
                [INVCR],
                [OURREF],
                [YOURREF],
                [ACTIONID],
                [BillSent],
                [USER],
                [DATE])

'

    EXEC (@COMMAND)
  END
GO

IF EXISTS
  (SELECT TOP 1 1
          FROM SYS.[objects] [SO]
               LEFT OUTER JOIN SYS.[indexes] [SI]                               
                            ON [SI].[object_id] = [SO].[object_id]
                           AND [SI].[name] = 'KeyRelease'
               CROSS APPLY (SELECT COUNT(1) AS [TOTAL]
                              FROM SYS.[index_columns] [IC]
                             WHERE[IC].[object_id] = [SI].[object_id]
                               AND [IC].[index_id] = [SI].[index_id]) [TC]
         WHERE [SO].[name] = 'BillHeader'
           AND [SI].[index_id] IS NOT NULL
           AND [TC].[TOTAL] < 5)
  BEGIN
    DECLARE @COMMAND VARCHAR(MAX)

    SET @COMMAND = 'DROP INDEX [dbo].[BillHeader].[KeyRelease]'

    EXEC (@COMMAND)

    SET @COMMAND = '
CREATE NONCLUSTERED INDEX
       [KeyRelease]
    ON [dbo].[BillHeader]
      ([RELEASED] ASC,
       [USER] ASC,
       [DATE] ASC,
       [MATTER] ASC)
       INCLUDE ([FeCode],
                [BATCHNO],
	              [BILLNO],
                [DRAFTNO],
                [INVCR],
                [OURREF],
                [YOURREF],
                [ACTIONID],
                [BillSent])

'

    EXEC (@COMMAND)
  END
GO





IF EXISTS
  (SELECT TOP 1 1
          FROM SYS.[objects] [SO]
               LEFT OUTER JOIN SYS.[indexes] [SI]
                            ON [SI].[object_id] = [SO].[object_id]
                           AND [SI].[name] = 'PK_BillDetails'
         WHERE [SO].[name] = 'BillDetails'
           AND [SI].[index_id] IS NULL)
  BEGIN
    DECLARE @PRIMARYKEY NVARCHAR(128)
    SELECT @PRIMARYKEY = SI.[name]
      FROM SYS.[objects] [SO]
           INNER JOIN SYS.[indexes] [SI]
                   ON [SI].[object_id] = [SO].[object_id]
                  AND [SI].[type] = 1
     WHERE [SO].[name] = 'BillDetails'

    IF NOT (@PRIMARYKEY IS NULL)
      BEGIN
        DECLARE @COMMAND VARCHAR(MAX)

        SET @COMMAND = 'ALTER TABLE [dbo].[BillDetails] DROP CONSTRAINT [' + CONVERT(VARCHAR(128), @PRIMARYKEY) + ']'

        EXEC (@COMMAND)

        SET @COMMAND = '
ALTER
TABLE [dbo].[BillDetails] 
  ADD CONSTRAINT [PK_BillDetails]
      PRIMARY KEY CLUSTERED 
     ([KEYID] ASC)
'
        EXEC (@COMMAND)
      END
  END
GO

IF EXISTS
  (SELECT TOP 1 1
          FROM SYS.[objects] [SO]
               LEFT OUTER JOIN SYS.[indexes] [SI]                               
                            ON [SI].[object_id] = [SO].[object_id]
                           AND [SI].[name] = 'KeyBillCopy'
               CROSS APPLY (SELECT COUNT(1) AS [TOTAL]
                              FROM SYS.[index_columns] [IC]
                             WHERE[IC].[object_id] = [SI].[object_id]
                               AND [IC].[index_id] = [SI].[index_id]) [TC]
         WHERE [SO].[name] = 'BillDetails'
           AND [SI].[index_id] IS NOT NULL
           AND [TC].[TOTAL] < 3)
  BEGIN
    DECLARE @COMMAND VARCHAR(MAX)

    SET @COMMAND = 'DROP INDEX [dbo].[BillDetails].[KeyBillCopy]'

    EXEC (@COMMAND)

    SET @COMMAND = '
CREATE NONCLUSTERED INDEX
       [KeyBillCopy]
    ON [dbo].[BillDetails]
      ([MATTER] ASC,
	     [BILLNO] ASC)
       INCLUDE ([DRAFTNO],
                [TYPE],
                [LINENO],
                [KEYID])

'

    EXEC (@COMMAND)
  END
GO

IF EXISTS
  (SELECT TOP 1 1
          FROM SYS.[objects] [SO]
               LEFT OUTER JOIN SYS.[indexes] [SI]                               
                            ON [SI].[object_id] = [SO].[object_id]
                           AND [SI].[name] = 'KeyBillno'
               CROSS APPLY (SELECT COUNT(1) AS [TOTAL]
                              FROM SYS.[index_columns] [IC]
                             WHERE[IC].[object_id] = [SI].[object_id]
                               AND [IC].[index_id] = [SI].[index_id]) [TC]
         WHERE [SO].[name] = 'BillDetails'
           AND [SI].[index_id] IS NOT NULL
           AND [TC].[TOTAL] < 4)
  BEGIN
    DECLARE @COMMAND VARCHAR(MAX)

    SET @COMMAND = 'DROP INDEX [dbo].[BillDetails].[KeyBillno]'

    EXEC (@COMMAND)

    SET @COMMAND = '
CREATE NONCLUSTERED INDEX
       [KeyBillno]
    ON [dbo].[BillDetails]
      ([MATTER] ASC,
	     [BILLNO] ASC,
       [LINENO] ASC)
       INCLUDE ([DRAFTNO],
                [TYPE],
                [KEYID])

'

    EXEC (@COMMAND)
  END
GO

IF EXISTS
  (SELECT TOP 1 1
          FROM SYS.[objects] [SO]
               LEFT OUTER JOIN SYS.[indexes] [SI]                               
                            ON [SI].[object_id] = [SO].[object_id]
                           AND [SI].[name] = 'KeyChildSeq'
               CROSS APPLY (SELECT COUNT(1) AS [TOTAL]
                              FROM SYS.[index_columns] [IC]
                             WHERE[IC].[object_id] = [SI].[object_id]
                               AND [IC].[index_id] = [SI].[index_id]) [TC]
         WHERE [SO].[name] = 'BillDetails'
           AND [SI].[index_id] IS NOT NULL
           AND [TC].[TOTAL] < 4)
  BEGIN
    DECLARE @COMMAND VARCHAR(MAX)

    SET @COMMAND = 'DROP INDEX [dbo].[BillDetails].[KeyChildSeq]'

    EXEC (@COMMAND)

    SET @COMMAND = '
CREATE NONCLUSTERED INDEX
       [KeyChildSeq]
    ON [dbo].[BillDetails]
      ([DRAFTNO] ASC,
       [LINENO] ASC,
       [KEYID] ASC)
       INCLUDE ([TYPE],
                [MATTER],
	              [BILLNO])

'

    EXEC (@COMMAND)
  END
GO

IF EXISTS
  (SELECT TOP 1 1
          FROM SYS.[objects] [SO]
               LEFT OUTER JOIN SYS.[indexes] [SI]                               
                            ON [SI].[object_id] = [SO].[object_id]
                           AND [SI].[name] = 'KeyMatter'
               CROSS APPLY (SELECT COUNT(1) AS [TOTAL]
                              FROM SYS.[index_columns] [IC]
                             WHERE[IC].[object_id] = [SI].[object_id]
                               AND [IC].[index_id] = [SI].[index_id]) [TC]
         WHERE [SO].[name] = 'BillDetails'
           AND [SI].[index_id] IS NOT NULL
           AND [TC].[TOTAL] < 5)
  BEGIN
    DECLARE @COMMAND VARCHAR(MAX)

    SET @COMMAND = 'DROP INDEX [dbo].[BillDetails].[KeyMatter]'

    EXEC (@COMMAND)

    SET @COMMAND = '
CREATE NONCLUSTERED INDEX
       [KeyMatter]
    ON [dbo].[BillDetails]
      ([MATTER] ASC,
	     [DRAFTNO] ASC,
       [BILLNO] ASC,
       [LINENO] ASC)
       INCLUDE ([TYPE],
                [KEYID])

'

    EXEC (@COMMAND)
  END
GO

IF EXISTS
   (SELECT TOP 1 1
      FROM SYS.[objects] [SO]
     WHERE [SO].[name] = 'BillHeader')
  BEGIN
    EXEC ('
UPDATE [BH]
   SET [BH].[MATTER] = [BD].[MATTER]
  FROM [dbo].[BillHeader] [BH]
       INNER JOIN [dbo].[BillDetails] [BD]
               ON [BD].[DRAFTNO] = [BH].[DRAFTNO]
              AND RTRIM(ISNULL([BD].[MATTER], '''')) <> RTRIM(ISNULL([BH].[MATTER], '''')) 
 WHERE RTRIM(ISNULL([BH].[MATTER], '''')) = ''''')
  END
GO

IF EXISTS
   (SELECT TOP 1 1
      FROM SYS.[objects] [SO]
     WHERE [SO].[name] = 'BillHeader')
  BEGIN
    EXEC ('
UPDATE [BD]
   SET [BD].[MATTER] = [BH].[MATTER]
  FROM [dbo].[BillHeader] [BH]
       INNER JOIN [dbo].[BillDetails] [BD]
               ON [BD].[DRAFTNO] = [BH].[DRAFTNO]
              AND RTRIM(ISNULL([BD].[MATTER], '''')) <> RTRIM(ISNULL([BH].[MATTER], ''''))')
  END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] [SO]
          LEFT OUTER JOIN SYS.[indexes] [SI]
                       ON [SI].[object_id] = [SO].[object_id]
                      AND [SI].[name] = 'IDX_DATPath'
    WHERE [SO].[name] = 'DiaryAttachments'
      AND [SO].[type] = 'U'
      AND [SI].[index_id] IS NULL)
  BEGIN
    EXEC ('
CREATE UNIQUE NONCLUSTERED
 INDEX [IDX_DATPath]
    ON [dbo].[DiaryAttachments]
      ([FILEPATH] ASC,
       [TrackReference] ASC)
       INCLUDE ([DiaryID],
                [CASECODE],
                [NAME],
                [DOCUMENT],
                [TYPE],
                [DOCCLASS],
                [DATEENTERED],
                [ENTEREDBY],
                [LASTACCESSDATE],
                [LASTACCESSBY],
                [SYSTEM],
                [DICTATIONFILE],
                [SOURCE])
    ON [PRIMARY]')
  END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM sys.objects [SO]
          LEFT OUTER JOIN sys.[columns] [SC]
                       ON [SC].[object_id] = [SO].[object_id]
                      AND [SC].[name] = 'Domain'
    WHERE [SO].[name] = 'FileSharingServers'
      AND [SO].[type] = 'U'
      AND [SC].[column_id] IS NULL)
BEGIN
  EXEC ('
        DROP TABLE [dbo].[FileSharingDocFolders]
        DROP TABLE [dbo].[FileSharingDocuments]
        DROP TABLE [dbo].[FileSharingHandlers]
        DROP TABLE [dbo].[FileSharingMapping]
        DROP TABLE [dbo].[FileSharingMatterMapping]
        DROP TABLE [dbo].[FileSharingServers]
        DROP TABLE [dbo].[FileSharingServerTypes]')
END
GO

IF NOT EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] [SO]
    WHERE [SO].[name] = 'FileSharingServerTypes')
  BEGIN
    EXEC ('
    CREATE
     TABLE [dbo].[FileSharingServerTypes]
          ([id]                           INT            IDENTITY(1, 1)                                                             NOT NULL,
           [TypeName]                     VARCHAR(20)                                                                               NOT NULL,
           [ClientFolderURL]              VARCHAR(500)   CONSTRAINT [DF_FSST_ClientFolderURL] DEFAULT('''')                         NOT NULL,
           [ClientFolderPermissionsURL]   VARCHAR(500)   CONSTRAINT [DF_FSST_ClientFolderPermissionsURL] DEFAULT('''')              NOT NULL,
           [DefaultDomain]                VARCHAR(100)   CONSTRAINT [DF_FSST_DefaultDomain] DEFAULT('''')                           NOT NULL,
           CONSTRAINT [PK_FileSharingServerTypes] PRIMARY KEY CLUSTERED
                     ([id])
                   ON [PRIMARY])
        ON [PRIMARY]')
  END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] [SO]
          LEFT OUTER JOIN SYS.[indexes] [SI]
                       ON [SI].[object_id] = [SO].[object_id]
                      AND [SI].[name] = 'IDX_TypeName'
    WHERE [SO].[name] = 'FileSharingServerTypes'
      AND [SI].[index_id] IS NULL)
  BEGIN
    EXEC ('
    CREATE UNIQUE NONCLUSTERED INDEX
           [IDX_TypeName]
        ON [dbo].[FileSharingServerTypes]
          ([TypeName])
           INCLUDE ([id],
                    [ClientFolderURL],
                    [ClientFolderPermissionsURL],
                    [DefaultDomain])
        ON [PRIMARY]')
  END
GO

EXEC ('
INSERT
  INTO [dbo].[FileSharingServerTypes]
      ([TypeName],
       [ClientFolderURL],
       [ClientFolderPermissionsURL],
       [DefaultDomain])
SELECT [NEW].[TypeName],
       [NEW].[ClientFolderURL],
       [NEW].[ClientFolderPermissionsURL],
       [NEW].[DefaultDomain]
  FROM (SELECT ''Citrix ShareFile''                                                       AS [TypeName],
               ''https://%'' + ''02.%'' + ''04/app/#/home/shared/%'' + ''01''             AS [ClientFolderURL],
               ''https://%'' + ''02.%'' + ''04/app/#/home/shared/%'' + ''01/People''      AS [ClientFolderPermissionsURL],
               ''sharefile.com''                                                          AS [DefaultDomain]) [NEW]
       LEFT OUTER JOIN [dbo].[FileSharingServerTypes] [FSST]
                    ON [FSST].[TypeName] = [NEW].[TypeName]
 WHERE [FSST].[id] IS NULL')
GO

EXEC ('INSERT
  INTO [dbo].[FileSharingServerTypes]
      ([TypeName],
       [ClientFolderURL],
       [ClientFolderPermissionsURL],
       [DefaultDomain])
SELECT [NEW].[TypeName],
       [NEW].[ClientFolderURL],
       [NEW].[ClientFolderPermissionsURL],
       [NEW].[DefaultDomain]
  FROM (SELECT ''HighQ''                                                                  AS [TypeName],
                 ''https://%''
               + ''02.%04/%''
               + ''02/documentHome.action?metaData.siteID=%''
               + ''03&metaData.parentFolderID=%''
               + ''01&metaData.moduleView=columnView''
               + ''&metaData.paginationNo=0''
               + ''&metaData.loadMoreNo=0''
               + ''&sortOrder=0''
               + ''&filterShared=0''                                                      AS [ClientFolderURL],
               ''''                                                                       AS [ClientFolderPermissionsURL],
               ''highq.com''                                                              AS [DefaultDomain]) [NEW]
       LEFT OUTER JOIN [dbo].[FileSharingServerTypes] [FSST]
                    ON [FSST].[TypeName] = [NEW].[TypeName]
 WHERE [FSST].[id] IS NULL')
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPFileSharingListServerTypes'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPFileSharingListServerTypes]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETSPFileSharingListServerTypes]
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETSPFileSharingListServerTypes]                                                                *
 *                                                                                                           *
 * List File Sharing Server Types                                                                            *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.5.1.2                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2017-06-20   Pino Carafa       Created                                                                    *
 * 2017-08-24   Pino Carafa       Added Default Domain                                                       *
 *                                                                                                           *
 *************************************************************************************************************/
 BEGIN
     SELECT [FSST].[id],
            [FSST].[TypeName],
            [FSST].[ClientFolderURL],
            [FSST].[ClientFolderPermissionsURL],
            [FSST].[DefaultDomain]
       FROM [dbo].[FileSharingServerTypes] [FSST]
   ORDER BY [FSST].[TypeName]
 END

GO

IF NOT EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] [SO]
    WHERE [SO].[name] = 'FileSharingServers')
  BEGIN
    EXEC ('
    CREATE
     TABLE [dbo].[FileSharingServers]
          ([id]                           INT            IDENTITY(1, 1)                                                            NOT NULL,
           [TypeId]                       INT                                                                                      NOT NULL,
           [Name]                         VARCHAR(50)    CONSTRAINT [DF_FSS_Name] DEFAULT ('''')                                   NOT NULL,
           [Server]                       VARCHAR(50)    CONSTRAINT [DF_FSS_Server] DEFAULT ('''')                                 NOT NULL,
           [AdminUserId]                  VARCHAR(100)   CONSTRAINT [DF_FSS_AdminUserId] DEFAULT ('''')                            NOT NULL,
           [AdminPassword]                VARCHAR(100)   CONSTRAINT [DF_FSS_AdminPassword] DEFAULT ('''')                          NOT NULL,
           [Retired]                      BIT            CONSTRAINT [DF_FSS_Retired] DEFAULT (0)                                   NOT NULL,
           [RootFolder]                   VARCHAR(200)   CONSTRAINT [DF_FSS_RootFolder] DEFAULT ('''')                             NOT NULL,
           [ClientFolderURL]              VARCHAR(500)   CONSTRAINT [DF_FSS_ClientFolderURL] DEFAULT('''')                         NOT NULL,
           [ClientFolderPermissionsURL]   VARCHAR(500)   CONSTRAINT [DF_FSS_ClientFolderPermissionsURL] DEFAULT('''')              NOT NULL,
           [ClientID]                     VARCHAR(100)   CONSTRAINT [DF_FSS_ClientID] DEFAULT('''')                                NOT NULL,
           [ClientSecret]                 VARCHAR(500)   CONSTRAINT [DF_FSS_ClientSecret] DEFAULT('''')                            NOT NULL,
           [SiteID]                       VARCHAR(100)   CONSTRAINT [DF_FSS_SiteID] DEFAULT('''')                                  NOT NULL,
           [CollabDocFolderID]            INT                                                                                          NULL,
           [CollabMatter]                 VARCHAR(50)                                                                                  NULL,
           [CollabFolder]                 VARCHAR(50)                                                                                  NULL,
           [Domain]                       VARCHAR(100)   CONSTRAINT [DF_FSS_Domain] DEFAULT('''')                                  NOT NULL,
           [ProxyType]                    INT            CONSTRAINT [DF_FSS_ProxyType] DEFAULT(0)
                                                         CONSTRAINT [CS_FSS_ProxyType] CHECK([ProxyType] IN (0, 1, 2))             NOT NULL,
           [ProxyServer]                  VARCHAR(500)   CONSTRAINT [DF_FSS_ProxyServer] DEFAULT('''')                             NOT NULL,
           [ProxyPort]                    INT   CONSTRAINT [DF_FSS_ProxyPort] DEFAULT(0)                                           NOT NULL,
           CONSTRAINT [PK_FileSharingServers] PRIMARY KEY CLUSTERED
                     ([id])
                   ON [PRIMARY])
        ON [PRIMARY]')

    EXEC ('
    ALTER 
    TABLE [dbo].[FileSharingServers] 
     WITH CHECK 
      ADD CONSTRAINT 
          [FK_FileSharingServers_Types]
          FOREIGN KEY([TypeId])
          REFERENCES [dbo].[FileSharingServerTypes]
                    ([id])
          ON DELETE CASCADE')

    EXEC ('
    ALTER 
    TABLE [dbo].[FileSharingServers] 
    CHECK CONSTRAINT [FK_FileSharingServers_Types]')
  END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] [SO]
          LEFT OUTER JOIN SYS.[indexes] [SI]
                       ON [SI].[object_id] = [SO].[object_id]
                      AND [SI].[name] = 'IDX_Name'
    WHERE [SO].[name] = 'FileSharingServers'
      AND [SI].[index_id] IS NULL)
  BEGIN
    EXEC ('
    CREATE UNIQUE NONCLUSTERED INDEX
           [IDX_Name]
        ON [dbo].[FileSharingServers]
          ([Name])
           INCLUDE ([id],
                    [TypeId],
                    [Server],
                    [AdminUserId],
                    [AdminPassword],
                    [Retired],
                    [RootFolder],
                    [ClientFolderURL],
                    [ClientFolderPermissionsURL],
                    [ClientID],
                    [ClientSecret],
                    [SiteID],
                    [CollabDocFolderID],
                    [CollabMatter],
                    [CollabFolder],
                    [Domain],
                    [ProxyType],
                    [ProxyServer],
                    [ProxyPort])
        ON [PRIMARY]')
  END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] [SO]
          LEFT OUTER JOIN SYS.[indexes] [SI]
                       ON [SI].[object_id] = [SO].[object_id]
                      AND [SI].[name] = 'IDX_Server'
    WHERE [SO].[name] = 'FileSharingServers'
      AND [SI].[index_id] IS NULL)
  BEGIN
    EXEC ('
    CREATE UNIQUE NONCLUSTERED INDEX
           [IDX_Server]
        ON [dbo].[FileSharingServers]
          ([Server],
           [CollabDocFolderID])
           INCLUDE ([id],
                    [TypeId],
                    [Name],
                    [AdminUserId],
                    [AdminPassword],
                    [Retired],
                    [RootFolder],
                    [ClientID],
                    [ClientSecret],
                    [SiteID],
                    [ClientFolderURL],
                    [ClientFolderPermissionsURL],
                    [CollabMatter],
                    [CollabFolder],
                    [Domain],
                    [ProxyType],
                    [ProxyServer],
                    [ProxyPort])
        ON [PRIMARY]')
  END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] [SO]
          LEFT OUTER JOIN SYS.[indexes] [SI]
                       ON [SI].[object_id] = [SO].[object_id]
                      AND [SI].[name] = 'IDX_Collab'
    WHERE [SO].[name] = 'FileSharingServers'
      AND [SI].[index_id] IS NULL)
  BEGIN
    EXEC ('
    CREATE UNIQUE NONCLUSTERED INDEX
           [IDX_Collab]
        ON [dbo].[FileSharingServers]
          ([CollabDocFolderID],
           [id])
           INCLUDE ([Server],
                    [TypeId],
                    [Name],
                    [AdminUserId],
                    [AdminPassword],
                    [Retired],
                    [RootFolder],
                    [ClientFolderURL],
                    [ClientFolderPermissionsURL],
                    [ClientID],
                    [ClientSecret],
                    [SiteID],
                    [CollabMatter],
                    [CollabFolder],
                    [Domain],
                    [ProxyType],
                    [ProxyServer],
                    [ProxyPort])
        ON [PRIMARY]')
  END
GO

IF NOT EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] [SO]
    WHERE [SO].[name] = 'FileSharingDocFolders')
  BEGIN
    EXEC ('
    CREATE
     TABLE [dbo].[FileSharingDocFolders]
          ([id]              INT            IDENTITY(1, 1)                                      NOT NULL,
           [DocFolderID]     INT                                                                NOT NULL,
           [Shared]          BIT            CONSTRAINT [DF_FSDF_Shared] DEFAULT (0)             NOT NULL,
           CONSTRAINT [PK_FileSharingDocFolders] PRIMARY KEY CLUSTERED
                     ([id])
                   ON [PRIMARY])
        ON [PRIMARY]')

    EXEC ('
    ALTER 
    TABLE [dbo].[FileSharingDocFolders] 
     WITH CHECK 
      ADD CONSTRAINT 
          [FK_FileSharingDocFolders_DocFolders]
          FOREIGN KEY([DocFolderID])
          REFERENCES [dbo].[DocFolders]
                    ([id])
          ON DELETE CASCADE')

    EXEC ('
    ALTER 
    TABLE [dbo].[FileSharingDocFolders] 
    CHECK CONSTRAINT [FK_FileSharingDocFolders_DocFolders]')
  END
GO

IF NOT EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] [SO]
    WHERE [SO].[name] = 'FileSharingClientMapping')
  BEGIN
    EXEC ('
    CREATE
     TABLE [dbo].[FileSharingClientMapping]
          ([id]                          INT            IDENTITY(1, 1)                                      NOT NULL,
           [FileSharingClient]           VARCHAR(10)                                                        NOT NULL,
           [FileSharingServersID]        INT                                                                NOT NULL,
           [FileSharingServerFolderID]   VARCHAR(500)   CONSTRAINT [DF_FCM_ServerFolderID] DEFAULT ('''')   NOT NULL,
           CONSTRAINT [PK_FileSharingClientMapping] PRIMARY KEY CLUSTERED
                     ([id])
                   ON [PRIMARY])
        ON [PRIMARY]')

    EXEC ('
    ALTER 
    TABLE [dbo].[FileSharingClientMapping] 
     WITH CHECK 
      ADD CONSTRAINT 
          [FK_FileSharingClientMapping_CON]
          FOREIGN KEY([FileSharingClient])
          REFERENCES [dbo].[Contacts]
                    ([CODE])
          ON DELETE CASCADE')

    EXEC ('
    ALTER 
    TABLE [dbo].[FileSharingClientMapping] 
    CHECK CONSTRAINT [FK_FileSharingClientMapping_CON]')

    EXEC ('
    ALTER 
    TABLE [dbo].[FileSharingClientMapping] 
     WITH CHECK 
      ADD CONSTRAINT 
          [FK_FileSharingClientMapping_Servers]
          FOREIGN KEY([FileSharingServersID])
          REFERENCES [dbo].[FileSharingServers]
                    ([id])
          ON DELETE CASCADE')

    EXEC ('
    ALTER 
    TABLE [dbo].[FileSharingClientMapping] 
    CHECK CONSTRAINT [FK_FileSharingClientMapping_Servers]')
  END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] [SO]
          LEFT OUTER JOIN SYS.[indexes] [SI]
                       ON [SI].[object_id] = [SO].[object_id]
                      AND [SI].[name] = 'IDX_FSCM_Unique'
    WHERE [SO].[name] = 'FileSharingClientMapping'
      AND [SI].[index_id] IS NULL)
  BEGIN
    EXEC ('
    CREATE UNIQUE NONCLUSTERED INDEX
           [IDX_FSCM_Unique]
        ON [dbo].[FileSharingClientMapping]
          ([FileSharingClient],
           [FileSharingServersID])
           INCLUDE ([id],
                    [FileSharingServerFolderID])
        ON [PRIMARY]')
  END
GO

IF NOT EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] [SO]
    WHERE [SO].[name] = 'FileSharingMatterMapping')
  BEGIN
    EXEC ('
    CREATE
     TABLE [dbo].[FileSharingMatterMapping]
          ([id]                          INT            IDENTITY(1, 1)                                      NOT NULL,
           [FileSharingMatter]           VARCHAR(20)                                                        NOT NULL,
           [FileSharingServersID]        INT                                                                NOT NULL,
           [FileSharingServerFolderID]   VARCHAR(500)   CONSTRAINT [DF_FMM_ServerFolderID] DEFAULT ('''')   NOT NULL,
           CONSTRAINT [PK_FileSharingMatterMapping] PRIMARY KEY CLUSTERED
                     ([id])
                   ON [PRIMARY])
        ON [PRIMARY]')

    EXEC ('
    ALTER 
    TABLE [dbo].[FileSharingMatterMapping] 
     WITH CHECK 
      ADD CONSTRAINT 
          [FK_FileSharingMatterMapping_MAT]
          FOREIGN KEY([FileSharingMatter])
          REFERENCES [dbo].[SearchMatters]
                    ([CODE])
          ON DELETE CASCADE')

    EXEC ('
    ALTER 
    TABLE [dbo].[FileSharingMatterMapping] 
    CHECK CONSTRAINT [FK_FileSharingMatterMapping_MAT]')

    EXEC ('
    ALTER 
    TABLE [dbo].[FileSharingMatterMapping] 
     WITH CHECK 
      ADD CONSTRAINT 
          [FK_FileSharingMatterMapping_Servers]
          FOREIGN KEY([FileSharingServersID])
          REFERENCES [dbo].[FileSharingServers]
                    ([id])
          ON DELETE CASCADE')

    EXEC ('
    ALTER 
    TABLE [dbo].[FileSharingMatterMapping] 
    CHECK CONSTRAINT [FK_FileSharingMatterMapping_Servers]')
  END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] [SO]
          LEFT OUTER JOIN SYS.[indexes] [SI]
                       ON [SI].[object_id] = [SO].[object_id]
                      AND [SI].[name] = 'IDX_FSMM_Unique'
    WHERE [SO].[name] = 'FileSharingMatterMapping'
      AND [SI].[index_id] IS NULL)
  BEGIN
    EXEC ('
    CREATE UNIQUE NONCLUSTERED INDEX
           [IDX_FSMM_Unique]
        ON [dbo].[FileSharingMatterMapping]
          ([FileSharingMatter],
           [FileSharingServersID])
           INCLUDE ([id],
                    [FileSharingServerFolderID])
        ON [PRIMARY]')
  END
GO

IF NOT EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] [SO]
    WHERE [SO].[name] = 'FileSharingMapping')
  BEGIN
    EXEC ('
    CREATE
     TABLE [dbo].[FileSharingMapping]
          ([id]                          INT            IDENTITY(1, 1)                                      NOT NULL,
           [FileSharingDocFoldersID]     INT                                                                NOT NULL,
           [FileSharingServersID]        INT                                                                NOT NULL,
           [FileSharingServerFolderID]   VARCHAR(500)   CONSTRAINT [DF_FSM_ServerFolderID] DEFAULT ('''')   NOT NULL,
           CONSTRAINT [PK_FileSharingMapping] PRIMARY KEY CLUSTERED
                     ([id])
                   ON [PRIMARY])
        ON [PRIMARY]')

    EXEC ('
    ALTER 
    TABLE [dbo].[FileSharingMapping] 
     WITH CHECK 
      ADD CONSTRAINT 
          [FK_FileSharingMapping_FSDF]
          FOREIGN KEY([FileSharingDocFoldersID])
          REFERENCES [dbo].[DocFolders]
                    ([id])
          ON DELETE CASCADE')

    EXEC ('
    ALTER 
    TABLE [dbo].[FileSharingMapping] 
    CHECK CONSTRAINT [FK_FileSharingMapping_FSDF]')

    EXEC ('
    ALTER 
    TABLE [dbo].[FileSharingMapping] 
     WITH CHECK 
      ADD CONSTRAINT 
          [FK_FileSharingMapping_Servers]
          FOREIGN KEY([FileSharingServersID])
          REFERENCES [dbo].[FileSharingServers]
                    ([id])
          ON DELETE CASCADE')

    EXEC ('
    ALTER 
    TABLE [dbo].[FileSharingMapping] 
    CHECK CONSTRAINT [FK_FileSharingMapping_Servers]')
  END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] [SO]
          LEFT OUTER JOIN SYS.[indexes] [SI]
                       ON [SI].[object_id] = [SO].[object_id]
                      AND [SI].[name] = 'IDX_FSM_Unique'
    WHERE [SO].[name] = 'FileSharingMapping'
      AND [SI].[index_id] IS NULL)
  BEGIN
    EXEC ('
    CREATE UNIQUE NONCLUSTERED INDEX
           [IDX_FSM_Unique]
        ON [dbo].[FileSharingMapping]
          ([FileSharingDocFoldersID],
           [FileSharingServersID])
           INCLUDE ([id],
                    [FileSharingServerFolderID])
        ON [PRIMARY]')
  END
GO

IF NOT EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] [SO]
    WHERE [SO].[name] = 'FileSharingHandlers')
  BEGIN
    EXEC ('
    CREATE
     TABLE [dbo].[FileSharingHandlers]
          ([id]                          INT            IDENTITY(1, 1)                                      NOT NULL,
           [FileSharingServersID]        INT                                                                NOT NULL,
           [FileSharingHandlerCode]      VARCHAR(10)                                                        NOT NULL,
           [UserId]                      VARCHAR(100)   CONSTRAINT [DF_FSH_UserId] DEFAULT ('''')           NOT NULL,
           [Password]                    VARCHAR(100)   CONSTRAINT [DF_FSH_Password] DEFAULT ('''')         NOT NULL,
           CONSTRAINT [PK_FileSharingHandlers] PRIMARY KEY CLUSTERED
                     ([id])
                   ON [PRIMARY])
        ON [PRIMARY]')

    EXEC ('
    ALTER 
    TABLE [dbo].[FileSharingHandlers] 
     WITH CHECK 
      ADD CONSTRAINT 
          [FK_FileSharingHandlers_Servers]
          FOREIGN KEY([FileSharingServersID])
          REFERENCES [dbo].[FileSharingServers]
                    ([id])
          ON DELETE CASCADE')

    EXEC ('
    ALTER 
    TABLE [dbo].[FileSharingHandlers] 
    CHECK CONSTRAINT [FK_FileSharingHandlers_Servers]')

    EXEC ('
    ALTER 
    TABLE [dbo].[FileSharingHandlers] 
     WITH CHECK 
      ADD CONSTRAINT 
          [FK_FileSharingHandlers_Handlers]
          FOREIGN KEY([FileSharingHandlerCode])
          REFERENCES [dbo].[Handlers]
                    ([CODE])
          ON DELETE CASCADE')

    EXEC ('
    ALTER 
    TABLE [dbo].[FileSharingHandlers] 
    CHECK CONSTRAINT [FK_FileSharingHandlers_Handlers]')
  END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] [SO]
          LEFT OUTER JOIN SYS.[indexes] [SI]
                       ON [SI].[object_id] = [SO].[object_id]
                      AND [SI].[name] = 'IDX_FSH_Unique'
    WHERE [SO].[name] = 'FileSharingHandlers'
      AND [SI].[index_id] IS NULL)
  BEGIN
    EXEC ('
    CREATE UNIQUE NONCLUSTERED INDEX
           [IDX_FSH_Unique]
        ON [dbo].[FileSharingHandlers]
          ([FileSharingHandlerCode],
           [FileSharingServersID])
           INCLUDE ([id],
                    [UserId],
                    [Password])
        ON [PRIMARY]')
  END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPFileSharingHandlerConnections'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPFileSharingHandlerConnections]
  END
GO

CREATE PROCEDURE
       [ky_NETSPFileSharingHandlerConnections]
      (@HandlerCode VARCHAR(10),
       @DocFolderID INT)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETSPFileSharingHandlerConnections]                                                             *
 *                                                                                                           *
 * List a handler's connection details for all File Sharing servers                                          *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.5.1.2                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2017-06-22   Pino Carafa       Created                                                                    *
 * 2017-08-03   Pino Carafa       Added Client ID and Client Secret                                          *
 * 2017-08-04   Pino Carafa       Added Site ID                                                              *
 * 2017-08-24   Pino Carafa       Added Domain and Proxy Settings KEYD-4750 and KEYD-4751                    *
 * 2017-12-05   Pino Carafa       No need to check for Personal Credentials for a Collab server              *
 *                                                                                                           *
 *************************************************************************************************************/
 BEGIN
    SELECT [FSS].[id]                          AS [ServerID],
           [FSS].[Name]                        AS [ServerName],
           [FSS].[Server]                      AS [Server],
           CONVERT(BIT,
                   CASE WHEN [FSH].[id] IS NULL
                         AND RTRIM(ISNULL([FSS].[CollabMatter], '')) = '' --2017-12-05
                             THEN 0
                        ELSE 1 END)            AS [PersonalCredentials],
           [FSS].[TypeId]                      AS [ServerTypeID],
           [FSST].[TypeName]                   AS [ServerType],
           ISNULL([FSH].[UserId],
                  [FSS].[AdminUserId])         AS [UserID],
           ISNULL([FSH].[Password],
                  [FSS].[AdminPassword])       AS [Password],
           ISNULL([FSS].[CollabMatter], '')    AS [CollabMatter],
           ISNULL([FSS].[CollabFolder], '')    AS [CollabFolder],
           [FSS].[ClientID]                    AS [ClientID],
           [FSS].[ClientSecret]                AS [ClientSecret],
           [FSS].[SiteID]                      AS [SiteID],
           [FSS].[Domain]                      AS [Domain],
           [FSS].[ProxyType]                   AS [ProxyType],
           [FSS].[ProxyServer]                 AS [ProxyServer],
           [FSS].[ProxyPort]                   AS [ProxyPort]
      FROM [dbo].[FileSharingServers] [FSS]
           INNER JOIN [dbo].[FileSharingServerTypes] [FSST]
                   ON [FSST].[id] = [FSS].[TypeId]
           LEFT OUTER JOIN [dbo].[FileSharingHandlers] [FSH]
                        ON [FSH].[FileSharingHandlerCode] = @HandlerCode
                       AND [FSH].[FileSharingServersID] = [FSS].[id]
     WHERE [FSS].[Retired] = 0
       AND ISNULL(@DocFolderID, 0) = ISNULL([FSS].[CollabDocFolderID], 0)
  ORDER BY [FSS].[Name]
 END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPFileSharingUseAdminLogon'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPFileSharingUseAdminLogon]
  END
GO

CREATE PROCEDURE
       [ky_NETSPFileSharingUseAdminLogon]
      (@ServerID INT)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETSPFileSharingUseAdminLogon]                                                                  *
 *                                                                                                           *
 * Get the Admin logon details                                                                               *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.5.1.2                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2017-06-23   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/
 BEGIN
    SELECT [FSS].[AdminUserId]         AS [UserID],
           [FSS].[AdminPassword]       AS [Password]
      FROM [dbo].[FileSharingServers] [FSS]
     WHERE [FSS].[Retired] = 0
       AND [FSS].[id] = @ServerID
 END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPAddFileSharingHandlerCredentials'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPAddFileSharingHandlerCredentials]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETSPAddFileSharingHandlerCredentials]
      (@Handler      VARCHAR(10),
       @ServerID     INT,
       @UserId       VARCHAR(100),
       @Password     VARCHAR(100))
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETSPAddFileSharingHandlerCredentials]                                                          *
 *                                                                                                           *
 * Insert/Updates a handler's File Sharing credentials                                                       *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.5.1.2                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2017-06-23   Pino Carafa       Created                                                                    *
 * 2018-10-08   Pino Carafa       Don't insert if it already exists                                          *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

INSERT
  INTO [dbo].[FileSharingHandlers]
      ([FileSharingHandlerCode],
       [FileSharingServersID])
SELECT [NEW].[FileSharingHandlerCode],
       [NEW].[FileSharingServersID]
  FROM (SELECT @Handler    AS [FileSharingHandlerCode],
               @ServerID   AS [FileSharingServersID]) [NEW]
       LEFT OUTER JOIN [dbo].[FileSharingHandlers] [FSH]
                    ON [FSH].[FileSharingHandlerCode] = [NEW].[FileSharingHandlerCode]
                   AND [FSH].[FileSharingServersID] = [NEW].[FileSharingServersID]
 WHERE [FSH].[id] IS NULL

UPDATE [FSH]
   SET [FSH].[UserId] = @UserId,
       [FSH].[Password] = @Password
  FROM [dbo].[FileSharingHandlers] [FSH]
 WHERE [FSH].[FileSharingHandlerCode] = @Handler
   AND [FSH].[FileSharingServersID] = @ServerID

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETTFGetDocFolders'
      AND SO.[type] = 'TF')
  BEGIN
    DROP FUNCTION [dbo].[ky_NETTFGetDocFolders]
  END
GO

 CREATE FUNCTION
        [dbo].[ky_NETTFGetDocFolders]
       (@parentid         INT,
        @fulltree         BIT,
        @includeparent    BIT)
RETURNS @DocFolders TABLE
       ([seq]                 INT              IDENTITY (1, 1)       NOT NULL PRIMARY KEY,
        [id]                  INT                                    NOT NULL,
        [parent]              INT                                    NOT NULL,
        [foldername]          VARCHAR(500)                           NOT NULL,
        [foldericon]          INT                                    NOT NULL)
AS
/*********************************************************************************************************
 *
 * [dbo].[ky_NETTFGetDocFolders]
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                -                                                        *   *
 *   * Last compatible software version:        - 5.5.1.1                                                *   *
 *   * Superseded by:                           - ky_NETTFGetDocFolders2                                 *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *
 * Retrieve a document folder tree
 * @parentid       - the ID of the root folder
 * @fulltree       - get the complete tree for @parentid
 *                   0 - get only the first level child nodes
 *                   1 - get all child nodes
 * @includeparent  - include the parent node itself
 *                   0 - the top level returned are the child nodes
 *                   1 - the top level returned is the parent node
 *
 * Modification History:
 * 2016-04-20    Pino Carafa      Created
 * 2016-05-12    Pino Carafa      Added Folder Icon
 *
 *********************************************************************************************************/
BEGIN
  DECLARE @DocFoldersX TABLE
         ([seq]                 INT            IDENTITY (0, 1)         NOT NULL PRIMARY KEY,
          [id]                  INT                                    NOT NULL,
          [parent]              INT                                    NOT NULL,
          [foldername]          VARCHAR(500)                           NOT NULL,
          [foldericon]          INT                                    NOT NULL)

  IF ISNULL(@parentid, 0) = 0
    BEGIN
      INSERT
          INTO @DocFoldersX
              ([id],
               [parent],
               [foldername],
               [foldericon])
        SELECT [DF].[id],
               0,
               [DF].[foldername],
               [DF].[foldericon]
          FROM [dbo].[DocFolders] [DF]
         WHERE [DF].[parent] IS NULL
      ORDER BY [DF].[foldername],
               [DF].[id]
    END
  ELSE
    BEGIN
      IF @includeparent = 1
        BEGIN
            INSERT
              INTO @DocFoldersX
                  ([id],
                   [parent],
                   [foldername],
                   [foldericon])
            SELECT [DF].[id],
                   ISNULL([DF].[parent], 0),
                   [DF].[foldername],
                   [DF].[foldericon]
              FROM [dbo].[DocFolders] [DF]
             WHERE [DF].[id] = @parentid
          ORDER BY [DF].[foldername],
                   [DF].[id]
        END
      ELSE
        BEGIN
            INSERT
              INTO @DocFoldersX
                  ([id],
                   [parent],
                   [foldername],
                   [foldericon])
            SELECT [DF].[id],
                   ISNULL([DF].[parent], 0),
                   [DF].[foldername],
                   [DF].[foldericon]
              FROM [dbo].[DocFolders] [DF]
             WHERE [DF].[parent] = @parentid
          ORDER BY [DF].[foldername],
                   [DF].[id]
        END
    END

  DECLARE @TOTAL INT
  DECLARE @ITEM INT

  SELECT @TOTAL = COUNT(1) FROM @DocFoldersX

  SET @ITEM = 0

  WHILE @ITEM < @TOTAL
    BEGIN
        DECLARE @id INT
        DECLARE @parent INT
        DECLARE @foldername VARCHAR(500)
        DECLARE @foldericon INT

        SELECT @id = [DFX].[id],
               @parent = [DFX].[parent],
               @foldername = [DFX].[foldername],
               @foldericon = [DFX].[foldericon]
          FROM @DocFoldersX [DFX]
         WHERE [DFX].[seq] = @ITEM

        INSERT
          INTO @DocFolders
              ([id],
               [parent],
               [foldername],
               [foldericon])
        SELECT @id,
               @parent,
               @foldername,
               @foldericon

        IF @fulltree = 1
          BEGIN
            INSERT
              INTO @DocFolders
                  ([id],
                   [parent],
                   [foldername],
                   [foldericon])
            SELECT [DFC].[id],
                   [DFC].[parent],
                   [DFC].[foldername],
                   [DFC].[foldericon]
              FROM [dbo].[ky_NETTFGetDocFolders](@id, 1, 0) [DFC]
          END

      SET @ITEM = @ITEM + 1
    END

  RETURN
END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETTFGetDocFolders2'
      AND SO.[type] = 'TF')
  BEGIN
    DROP FUNCTION [dbo].[ky_NETTFGetDocFolders2]
  END
GO

 CREATE FUNCTION
        [dbo].[ky_NETTFGetDocFolders2]
       (@parentid         INT,
        @fulltree         BIT,
        @includeparent    BIT)
RETURNS @DocFolders TABLE
       ([seq]                 INT              IDENTITY (1, 1)       NOT NULL PRIMARY KEY,
        [id]                  INT                                    NOT NULL,
        [parent]              INT                                    NOT NULL,
        [foldername]          VARCHAR(500)                           NOT NULL,
        [foldericon]          INT                                    NOT NULL,
        [foldershared]        BIT                                    NOT NULL)
AS
/*********************************************************************************************************
 *
 * [dbo].[ky_NETTFGetDocFolders]
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              - ky_NETTFGetDocFolders                                  *   *
 *   * First compatible version:                - 5.5.1.2                                                *   *
 *   * Last compatible software version:        - 5.5.2.1                                                *   *
 *   * Superseded by:                           - ky_NETTFGetDocFolders3                                 *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *
 * Retrieve a document folder tree
 * @parentid       - the ID of the root folder
 * @fulltree       - get the complete tree for @parentid
 *                   0 - get only the first level child nodes
 *                   1 - get all child nodes
 * @includeparent  - include the parent node itself
 *                   0 - the top level returned are the child nodes
 *                   1 - the top level returned is the parent node
 *
 * Modification History:
 * 2016-04-20    Pino Carafa      Created
 * 2016-05-12    Pino Carafa      Added Folder Icon
 * 2016-06-26    Pino Carafa      Add outer join for folder sharing
 *
 *********************************************************************************************************/
BEGIN
  DECLARE @DocFoldersX TABLE
         ([seq]                 INT            IDENTITY (0, 1)         NOT NULL PRIMARY KEY,
          [id]                  INT                                    NOT NULL,
          [parent]              INT                                    NOT NULL,
          [foldername]          VARCHAR(500)                           NOT NULL,
          [foldericon]          INT                                    NOT NULL,
          [foldershared]        BIT                                    NOT NULL)

  IF ISNULL(@parentid, 0) = 0
    BEGIN
      INSERT
          INTO @DocFoldersX
              ([id],
               [parent],
               [foldername],
               [foldericon],
               [foldershared])
        SELECT [DF].[id],
               0,
               [DF].[foldername],
               [DF].[foldericon],
               ISNULL([FSDF].[Shared], CONVERT(BIT, 0))
          FROM [dbo].[DocFolders] [DF]
               LEFT OUTER JOIN [dbo].[FileSharingDocFolders] [FSDF]
                            ON [FSDF].[DocFolderID] = [DF].[id]
         WHERE [DF].[parent] IS NULL
      ORDER BY [DF].[foldername],
               [DF].[id]
    END
  ELSE
    BEGIN
      IF @includeparent = 1
        BEGIN
            INSERT
              INTO @DocFoldersX
                  ([id],
                   [parent],
                   [foldername],
                   [foldericon],
                   [foldershared])
            SELECT [DF].[id],
                   ISNULL([DF].[parent], 0),
                   [DF].[foldername],
                   [DF].[foldericon],
                   ISNULL([FSDF].[Shared], CONVERT(BIT, 0))
              FROM [dbo].[DocFolders] [DF]
                   LEFT OUTER JOIN [dbo].[FileSharingDocFolders] [FSDF]
                                ON [FSDF].[DocFolderID] = [DF].[id]
             WHERE [DF].[id] = @parentid
          ORDER BY [DF].[foldername],
                   [DF].[id]
        END
      ELSE
        BEGIN
            INSERT
              INTO @DocFoldersX
                  ([id],
                   [parent],
                   [foldername],
                   [foldericon],
                   [foldershared])
            SELECT [DF].[id],
                   ISNULL([DF].[parent], 0),
                   [DF].[foldername],
                   [DF].[foldericon],
                   ISNULL([FSDF].[Shared], CONVERT(BIT, 0))
              FROM [dbo].[DocFolders] [DF]
                   LEFT OUTER JOIN [dbo].[FileSharingDocFolders] [FSDF]
                                ON [FSDF].[DocFolderID] = [DF].[id]
             WHERE [DF].[parent] = @parentid
          ORDER BY [DF].[foldername],
                   [DF].[id]
        END
    END

  DECLARE @TOTAL INT
  DECLARE @ITEM INT

  SELECT @TOTAL = COUNT(1) FROM @DocFoldersX

  SET @ITEM = 0

  WHILE @ITEM < @TOTAL
    BEGIN
        DECLARE @id INT
        DECLARE @parent INT
        DECLARE @foldername VARCHAR(500)
        DECLARE @foldericon INT
        DECLARE @foldershared BIT

        SELECT @id = [DFX].[id],
               @parent = [DFX].[parent],
               @foldername = [DFX].[foldername],
               @foldericon = [DFX].[foldericon],
               @foldershared = [DFX].[foldershared]
          FROM @DocFoldersX [DFX]
         WHERE [DFX].[seq] = @ITEM

        INSERT
          INTO @DocFolders
              ([id],
               [parent],
               [foldername],
               [foldericon],
               [foldershared])
        SELECT @id,
               @parent,
               @foldername,
               @foldericon,
               @foldershared

        IF @fulltree = 1
          BEGIN
            INSERT
              INTO @DocFolders
                  ([id],
                   [parent],
                   [foldername],
                   [foldericon],
                   [foldershared])
            SELECT [DFC].[id],
                   [DFC].[parent],
                   [DFC].[foldername],
                   [DFC].[foldericon],
                   [DFC].[foldershared]
              FROM [dbo].[ky_NETTFGetDocFolders2](@id, 1, 0) [DFC]
          END

      SET @ITEM = @ITEM + 1
    END

  RETURN
END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETTFGetDocFolders3'
      AND SO.[type] = 'TF')
  BEGIN
    DROP FUNCTION [dbo].[ky_NETTFGetDocFolders3]
  END
GO

 CREATE FUNCTION
        [dbo].[ky_NETTFGetDocFolders3]
       (@parentid         INT,
        @fulltree         BIT,
        @includeparent    BIT)
RETURNS @DocFolders TABLE
       ([seq]                 INT              IDENTITY (1, 1)       NOT NULL PRIMARY KEY,
        [id]                  INT                                    NOT NULL,
        [parent]              INT                                    NOT NULL,
        [foldername]          VARCHAR(500)                           NOT NULL,
        [foldericon]          INT                                    NOT NULL,
        [foldershared]        BIT                                    NOT NULL,
        [folderisours]        BIT                                    NOT NULL)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETTFGetDocFolders]                                                                             *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              - ky_NETTFGetDocFolders2                                 *   *
 *   * First compatible version:                - 5.5.1.2                                                *   *
 *   * Last compatible software version:        - 5.5.3.9                                                *   *
 *   * Superseded by:                           - ky_NETTFGetDocFolders4                                 *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Retrieve a document folder tree                                                                           *
 * @parentid       - the ID of the root folder                                                               *
 * @fulltree       - get the complete tree for @parentid                                                     *
 *                   0 - get only the first level child nodes                                                *
 *                   1 - get all child nodes                                                                 *
 * @includeparent  - include the parent node itself                                                          *
 *                   0 - the top level returned are the child nodes                                          *
 *                   1 - the top level returned is the parent node                                           *
 *                                                                                                           *
 * Modification History:                                                                                     *
 * 2016-04-20    Pino Carafa      Created                                                                    *
 * 2016-05-12    Pino Carafa      Added Folder Icon                                                          *
 * 2017-06-26    Pino Carafa      Add outer join for folder sharing                                          *
 * 2017-07-12    Pino Carafa      Add indicator to show whether the share is ours or we're linking to it     *
 * 2017-09-04    Pino Carafa      Corrected code to check whether a share is ours.                           *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN
  DECLARE @DocFoldersX TABLE
         ([seq]                 INT            IDENTITY (0, 1)         NOT NULL PRIMARY KEY,
          [id]                  INT                                    NOT NULL,
          [parent]              INT                                    NOT NULL,
          [foldername]          VARCHAR(500)                           NOT NULL,
          [foldericon]          INT                                    NOT NULL,
          [foldershared]        BIT                                    NOT NULL,
          [folderisours]        BIT                                    NOT NULL)

  IF ISNULL(@parentid, 0) = 0
    BEGIN
      INSERT
          INTO @DocFoldersX
              ([id],
               [parent],
               [foldername],
               [foldericon],
               [foldershared],
               [folderisours])
        SELECT [DF].[id],
               0,
               [DF].[foldername],
               [DF].[foldericon],
               0,
               1
          FROM [dbo].[DocFolders] [DF]
         WHERE [DF].[parent] IS NULL
      ORDER BY [DF].[foldername],
               [DF].[id]
    END
  ELSE
    BEGIN
      IF @includeparent = 1
        BEGIN
            INSERT
              INTO @DocFoldersX
                  ([id],
                   [parent],
                   [foldername],
                   [foldericon],
                   [foldershared],
                   [folderisours])
            SELECT [DF].[id],
                   ISNULL([DF].[parent], 0),
                   [DF].[foldername],
                   [DF].[foldericon],
                   0,
                   1
              FROM [dbo].[DocFolders] [DF]
             WHERE [DF].[id] = @parentid
          ORDER BY [DF].[foldername],
                   [DF].[id]
        END
      ELSE
        BEGIN
            INSERT
              INTO @DocFoldersX
                  ([id],
                   [parent],
                   [foldername],
                   [foldericon],
                   [foldershared],
                   [folderisours])
            SELECT [DF].[id],
                   ISNULL([DF].[parent], 0),
                   [DF].[foldername],
                   [DF].[foldericon],
                   0,
                   1
              FROM [dbo].[DocFolders] [DF]
             WHERE [DF].[parent] = @parentid
          ORDER BY [DF].[foldername],
                   [DF].[id]
        END
    END

  UPDATE [DFX]
     SET [DFX].[foldershared] = [FSDF].[Shared],
         [DFX].[folderisours] = [IO].[IsOurs]
    FROM @DocFoldersX [DFX]
         INNER JOIN [dbo].[FileSharingDocFolders] [FSDF]
                    CROSS APPLY (SELECT CASE WHEN COUNT(CASE WHEN ISNULL([FSS].[CollabDocFolderID], 0) = 0
                                                                  THEN 1
                                                             ELSE NULL END) = 0
                                                  THEN 0
                                             ELSE 1 END AS [IsOurs]
                                   FROM [dbo].[FileSharingMapping] [FSM]
                                        INNER JOIN [dbo].[FileSharingServers] [FSS]
                                                ON [FSS].[id] = [FSM].[FileSharingServersID]
                                  WHERE [FSM].[FileSharingDocFoldersID] = [FSDF].[DocFolderID]) [IO]
                 ON [FSDF].[DocFolderID] = [DFX].[id]

  DECLARE @TOTAL INT
  DECLARE @ITEM INT

  SELECT @TOTAL = COUNT(1) FROM @DocFoldersX

  SET @ITEM = 0

  WHILE @ITEM < @TOTAL
    BEGIN
        DECLARE @id INT
        DECLARE @parent INT
        DECLARE @foldername VARCHAR(500)
        DECLARE @foldericon INT
        DECLARE @foldershared BIT
        DECLARE @folderisours BIT

        SELECT @id = [DFX].[id],
               @parent = [DFX].[parent],
               @foldername = [DFX].[foldername],
               @foldericon = [DFX].[foldericon],
               @foldershared = [DFX].[foldershared],
               @folderisours = [DFX].[folderisours]
          FROM @DocFoldersX [DFX]
         WHERE [DFX].[seq] = @ITEM

        INSERT
          INTO @DocFolders
              ([id],
               [parent],
               [foldername],
               [foldericon],
               [foldershared],
               [folderisours])
        SELECT @id,
               @parent,
               @foldername,
               @foldericon,
               @foldershared,
               @folderisours

        IF @fulltree = 1
          BEGIN
            INSERT
              INTO @DocFolders
                  ([id],
                   [parent],
                   [foldername],
                   [foldericon],
                   [foldershared],
                   [folderisours])
            SELECT [DFC].[id],
                   [DFC].[parent],
                   [DFC].[foldername],
                   [DFC].[foldericon],
                   [DFC].[foldershared],
                   [DFC].[folderisours]
              FROM [dbo].[ky_NETTFGetDocFolders3](@id, 1, 0) [DFC]
          END

      SET @ITEM = @ITEM + 1
    END

  RETURN
END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETTFGetDocFolders4'
      AND SO.[type] = 'TF')
  BEGIN
    DROP FUNCTION [dbo].[ky_NETTFGetDocFolders4]
  END
GO

 CREATE FUNCTION
        [dbo].[ky_NETTFGetDocFolders4]
       (@parentid         INT,
        @fulltree         BIT,
        @includeparent    BIT)
RETURNS @DocFolders TABLE
       ([seq]                 INT              IDENTITY (1, 1)       NOT NULL PRIMARY KEY,
        [id]                  INT                                    NOT NULL,
        [parent]              INT                                    NOT NULL,
        [foldername]          VARCHAR(500)                           NOT NULL,
        [foldericon]          INT                                    NOT NULL,
        [foldershared]        BIT                                    NOT NULL,
        [folderisours]        BIT                                    NOT NULL,
        [documentcount]       INT                                    NOT NULL)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETTFGetDocFolders]                                                                             *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              - ky_NETTFGetDocFolders3                                 *   *
 *   * First compatible version:                - 5.5.4.1                                                *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Retrieve a document folder tree                                                                           *
 * @parentid       - the ID of the root folder                                                               *
 * @fulltree       - get the complete tree for @parentid                                                     *
 *                   0 - get only the first level child nodes                                                *
 *                   1 - get all child nodes                                                                 *
 * @includeparent  - include the parent node itself                                                          *
 *                   0 - the top level returned are the child nodes                                          *
 *                   1 - the top level returned is the parent node                                           *
 *                                                                                                           *
 * Modification History:                                                                                     *
 * 2016-04-20    Pino Carafa      Created                                                                    *
 * 2016-05-12    Pino Carafa      Added Folder Icon                                                          *
 * 2017-06-26    Pino Carafa      Add outer join for folder sharing                                          *
 * 2017-07-12    Pino Carafa      Add indicator to show whether the share is ours or we're linking to it     *
 * 2017-09-04    Pino Carafa      Corrected code to check whether a share is ours.                           *
 * 2018-04-27    Pino Carafa      Added Document Count                                                       *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN
  DECLARE @DocFoldersX TABLE
         ([seq]                 INT            IDENTITY (0, 1)         NOT NULL PRIMARY KEY,
          [id]                  INT                                    NOT NULL,
          [parent]              INT                                    NOT NULL,
          [foldername]          VARCHAR(500)                           NOT NULL,
          [foldericon]          INT                                    NOT NULL,
          [foldershared]        BIT                                    NOT NULL,
          [folderisours]        BIT                                    NOT NULL,
          [docfoldercount]      BIT                                    NOT NULL)

  IF ISNULL(@parentid, 0) = 0
    BEGIN
      INSERT
          INTO @DocFoldersX
              ([id],
               [parent],
               [foldername],
               [foldericon],
               [foldershared],
               [folderisours],
               [docfoldercount])
        SELECT [DF].[id],
               0,
               [DF].[foldername],
               [DF].[foldericon],
               0,
               1,
               0
          FROM [dbo].[DocFolders] [DF]
         WHERE [DF].[parent] IS NULL
      ORDER BY [DF].[foldername],
               [DF].[id]
    END
  ELSE
    BEGIN
      IF @includeparent = 1
        BEGIN
            INSERT
              INTO @DocFoldersX
                  ([id],
                   [parent],
                   [foldername],
                   [foldericon],
                   [foldershared],
                   [folderisours],
                   [docfoldercount])
            SELECT [DF].[id],
                   ISNULL([DF].[parent], 0),
                   [DF].[foldername],
                   [DF].[foldericon],
                   0,
                   1,
                   0
              FROM [dbo].[DocFolders] [DF]
             WHERE [DF].[id] = @parentid
          ORDER BY [DF].[foldername],
                   [DF].[id]
        END
      ELSE
        BEGIN
            INSERT
              INTO @DocFoldersX
                  ([id],
                   [parent],
                   [foldername],
                   [foldericon],
                   [foldershared],
                   [folderisours],
                   [docfoldercount])
            SELECT [DF].[id],
                   ISNULL([DF].[parent], 0),
                   [DF].[foldername],
                   [DF].[foldericon],
                   0,
                   1,
                   0
              FROM [dbo].[DocFolders] [DF]
             WHERE [DF].[parent] = @parentid
          ORDER BY [DF].[foldername],
                   [DF].[id]
        END
    END

  UPDATE [DFX]
     SET [DFX].[foldershared] = [FSDF].[Shared],
         [DFX].[folderisours] = [IO].[IsOurs]
    FROM @DocFoldersX [DFX]
         INNER JOIN [dbo].[FileSharingDocFolders] [FSDF]
                    CROSS APPLY (SELECT CASE WHEN COUNT(CASE WHEN ISNULL([FSS].[CollabDocFolderID], 0) = 0
                                                                  THEN 1
                                                             ELSE NULL END) = 0
                                                  THEN 0
                                             ELSE 1 END AS [IsOurs]
                                   FROM [dbo].[FileSharingMapping] [FSM]
                                        INNER JOIN [dbo].[FileSharingServers] [FSS]
                                                ON [FSS].[id] = [FSM].[FileSharingServersID]
                                  WHERE [FSM].[FileSharingDocFoldersID] = [FSDF].[DocFolderID]) [IO]
                 ON [FSDF].[DocFolderID] = [DFX].[id]

  DECLARE @TOTAL INT
  DECLARE @ITEM INT

  SELECT @TOTAL = COUNT(1) FROM @DocFoldersX

  SET @ITEM = 0

  WHILE @ITEM < @TOTAL
    BEGIN
        DECLARE @id INT
        DECLARE @parent INT
        DECLARE @foldername VARCHAR(500)
        DECLARE @foldericon INT
        DECLARE @foldershared BIT
        DECLARE @folderisours BIT
        DECLARE @docfoldercount INT

        SELECT @id = [DFX].[id],
               @parent = [DFX].[parent],
               @foldername = [DFX].[foldername],
               @foldericon = [DFX].[foldericon],
               @foldershared = [DFX].[foldershared],
               @folderisours = [DFX].[folderisours],
               @docfoldercount = [DFX].[docfoldercount]
          FROM @DocFoldersX [DFX]
         WHERE [DFX].[seq] = @ITEM

        INSERT
          INTO @DocFolders
              ([id],
               [parent],
               [foldername],
               [foldericon],
               [foldershared],
               [folderisours],
               [documentcount])
        SELECT @id,
               @parent,
               @foldername,
               @foldericon,
               @foldershared,
               @folderisours,
               @docfoldercount

        IF @fulltree = 1
          BEGIN
            INSERT
              INTO @DocFolders
                  ([id],
                   [parent],
                   [foldername],
                   [foldericon],
                   [foldershared],
                   [folderisours],
                   [documentcount])
            SELECT [DFC].[id],
                   [DFC].[parent],
                   [DFC].[foldername],
                   [DFC].[foldericon],
                   [DFC].[foldershared],
                   [DFC].[folderisours],
                   [DFC].[documentcount]
              FROM [dbo].[ky_NETTFGetDocFolders4](@id, 1, 0) [DFC]
          END

      SET @ITEM = @ITEM + 1
    END

  UPDATE [DF]
     SET [DF].[documentcount] = [TTD].[TOTAL]
    FROM @DocFolders [DF]
         CROSS APPLY (SELECT COUNT(1) AS [TOTAL]
                        FROM [dbo].[DocFolderDocuments] [DFD]
                       WHERE [DFD].[DocFolderID] = [DF].[id]) [TTD]


  RETURN
END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPGetDocFolderTree'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPGetDocFolderTree]
  END
GO

 CREATE PROCEDURE
        [dbo].[ky_NETSPGetDocFolderTree]
       (@root         INT)
AS
/*********************************************************************************************************
 *
 * [dbo].[ky_NETSPGetDocFolderTree]
 *
 * Retrieve a document folder tree
 * @root       - the ID of the root folder
 *
 * Modification History:
 * 2016-04-21    Pino Carafa      Created
 * 2016-05-12    Pino Carafa      Add folder icon
 * 2017-06-23    Pino Carafa      Add folder shared
 *                                (no need to create a new version of this SP as the calling software
 *                                 does not know about the new column and will not try to process it)
 * 2017-07-12    Pino Carafa      Add folder is ours
 * 2018-04-27    Pino Carafa      Add document count
 * 2018-04-30    Pino Carafa      Add missing columns in first branch
 *
 *********************************************************************************************************/
BEGIN
  IF ISNULL(@root, 0) < 1
    BEGIN
      SELECT TOP 0
             CONVERT(INT, 0)                              AS [id],
             CONVERT(INT, NULL)                           AS [parent],
             CONVERT(VARCHAR(500), '')                    AS [foldername],
             CONVERT(INT, 0)                              AS [foldericon],
             CONVERT(INT, 0)                              AS [foldershared],
             CONVERT(INT, 0)                              AS [folderisours],
             CONVERT(INT, 0)                              AS [documentcount]
      RETURN
    END

  SELECT [DF].[id]                                        AS [id],
         [DF].[parent]                                    AS [parent],
         [DF].[foldername]                                AS [foldername],
         [DF].[foldericon]                                AS [foldericon],
         [DF].[foldershared]                              AS [foldershared],
         [DF].[folderisours]                              AS [folderisours],
         [DF].[documentcount]                             AS [documentcount]
    FROM [dbo].[ky_NETTFGetDocFolders4](@root, 1, 1) [DF]
END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPDuplicateDocFolderTree'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPDuplicateDocFolderTree]
  END
GO

 CREATE PROCEDURE
        [dbo].[ky_NETSPDuplicateDocFolderTree]
       (@originalid                INT)
AS
/*********************************************************************************************************
 *
 * [dbo].[ky_NETSPDuplicateDocFolderTree]
 *
 * Duplicates a folder tree to a new root
 *
 * Modification History:
 * 2016-04-22    Pino Carafa      Created
 * 2016-05-12    Pino Carafa      Added foldericon
 * 2016-05-31    Suriya M         KEYD-3541-URGENT: Ability to COPY an existing CasePlan/WorkFlow
 * 2016-06-10    Pino Carafa      above change causes KEYD-3643. Revert to original code
 * 2017-06-23    Pino Carafa      Use new function to get doc folders, but ignore shared folders
 *                                as the duplicated tree shouldn't copy the sharing across
 * 2017-07-12    Pino Carafa      Use new function to get doc folders, but ignore the "is ours" flag
 *
 *********************************************************************************************************/
BEGIN

  DECLARE @fromid INT

  SELECT TOP 1 @fromid = [DF].[id]
    FROM [dbo].[DocFolders] [DF]
   WHERE [DF].[id] = @originalid
     AND [DF].[parent] IS NULL

  DECLARE @MAX INT

  DECLARE @data TABLE
         ([seq]                       INT              IDENTITY(1, 1)    NOT NULL      PRIMARY KEY,
          [id]                        INT                                NOT NULL,
          [parent]                    INT                                NOT NULL,
          [foldername]                VARCHAR(500)                       NOT NULL,
          [foldericon]                INT                                NOT NULL,
          [newfolder]                 BIT                                NOT NULL,
          [deletefolder]              BIT                                NOT NULL)
  
  IF (@fromid IS NOT NULL)
    BEGIN        
      INSERT
        INTO @data
            ([id],
             [parent],
             [foldername],
             [foldericon],
             [newfolder],
             [deletefolder])
      SELECT SRC.[id],
             SRC.[parent],
             SRC.[foldername],
             SRC.[foldericon],
             1,
             0
        FROM [dbo].[ky_NETTFGetDocFolders3](@fromid, 1, 1) SRC
    END

  SELECT @MAX = ISNULL(MAX([DF].[id]), 0) FROM [dbo].[DocFolders] [DF]

  DECLARE @new TABLE
         ([seq]                       INT              IDENTITY(1, 1)    NOT NULL      PRIMARY KEY,
          [oldid]                     INT                                NOT NULL      UNIQUE NONCLUSTERED,
          [newid]                     INT                                NOT NULL)

  INSERT
    INTO @new
        ([oldid],
         [newid])
  SELECT [DT].[id],
         0
    FROM @data [DT]
   WHERE [DT].[newfolder] = 1

  UPDATE [nw]
     SET [nw].[newid] = @MAX + [nw].[seq]
    FROM @new [NW]

  SET IDENTITY_INSERT [dbo].[DocFolders] ON

  INSERT
    INTO [dbo].[DocFolders]
        ([id],
         [foldername],
         [parent],
         [foldericon])
  SELECT [nw].[newid]                                            AS [id],
         [dt].[foldername]                                       AS [foldername],
         CASE ISNULL([pr].[newid], 0)
              WHEN 0 THEN CASE ISNULL([dt].[parent], 0)
                               WHEN 0 THEN NULL
                               ELSE [dt].[parent] END     
              ELSE [pr].[newid] END                              AS [parent],
         [dt].[foldericon]                                       AS [foldericon]
    FROM @data [dt]
         INNER JOIN @new [nw]
                 ON [nw].[oldid] = [dt].[id]
         LEFT OUTER JOIN @new [pr]
                      ON [pr].[oldid] = [dt].[parent]

  SET IDENTITY_INSERT [dbo].[DocFolders] OFF

  SELECT @MAX = ISNULL(MAX([DF].[id]), 0) + 1
    FROM [dbo].[DocFolders] [DF]

  DBCC CHECKIDENT('DocFolders', RESEED, @MAX) WITH NO_INFOMSGS

  DELETE [DF]
    FROM @data [dt]
         INNER JOIN [dbo].[DocFolders] [DF]
                 ON [DF].[id] = [dt].[id]
   WHERE [dt].[deletefolder] = 1

  UPDATE [DF]
     SET [DF].[foldername] = [dt].[foldername],
         [DF].[parent] = CASE ISNULL([dt].[parent], 0)
                              WHEN 0 THEN NULL
                         ELSE [dt].[parent] END,
         [DF].[foldericon] = [dt].[foldericon]
    FROM @data [dt]
         INNER JOIN [dbo].[DocFolders] [DF]
                 ON [DF].[id] = [dt].[id]
   WHERE [dt].[deletefolder] = 0
     AND [dt].[newfolder] = 0

  DECLARE @newid INT

  SELECT TOP 1
         @newid = ISNULL([nw].[newid], dt.[id])
    FROM @data [dt]
         LEFT OUTER JOIN @new [nw]
                      ON [nw].[oldid] = [dt].[id]

  SELECT @newid [id]
  RETURN @newid
END
GO

IF NOT EXISTS
  (SELECT TOP 1 1
     FROM sys.[objects] [SO]
          INNER JOIN sys.[indexes] [SI]
                  ON [SI].[object_id] = [SO].[object_id]
                 AND [SI].[name] = 'IDX_DocFolder'
    WHERE [SO].[name] = 'MatterDocFolders')
  BEGIN
    CREATE NONCLUSTERED INDEX
           [IDX_DocFolder]
        ON [dbo].[MatterDocFolders]
          ([DocFolderID])
           INCLUDE ([MatterCode])
  END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPShareFolder'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPShareFolder]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETSPShareFolder]
      (@ID    INT,
       @Share BIT)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETSPShareFolder]                                                                               *
 *                                                                                                           *
 * Share or unshare a Document Folder                                                                        *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.5.1.2                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2017-06-26   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

INSERT
  INTO [dbo].[FileSharingDocFolders]
      ([DocFolderID])
SELECT [NEW].[DocFolderID]
  FROM (SELECT @ID              AS [DocFolderID]) [NEW]
       LEFT OUTER JOIN [dbo].[FileSharingDocFolders] [FSDF]
                    ON [FSDF].[DocFolderID] = [NEW].[DocFolderID]
 WHERE [FSDF].[id] IS NULL

UPDATE [FSDF]
   SET [FSDF].[Shared] = @Share
  FROM [dbo].[FileSharingDocFolders] [FSDF]
 WHERE [FSDF].[DocFolderID] = @ID

IF (@Share = CONVERT(BIT, 0)) AND (ISNULL(@ID, 0) <> 0)
  BEGIN
    DELETE [FSM]
      FROM [dbo].[FileSharingMapping] [FSM]
     WHERE [FSM].[FileSharingDocFoldersID] = @ID

    DELETE [FSS]
      FROM [dbo].[FileSharingServers] [FSS]
     WHERE [FSS].[CollabDocFolderID] = @ID
  END

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPGetFileSharingFolderMappings'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPGetFileSharingFolderMappings]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETSPGetFileSharingFolderMappings]
      (@DocFolderID INT)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETSPGetFileSharingFolderMappings]                                                              *
 *                                                                                                           *
 * Retrieves information about File Sharing folder mappings                                                  *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.5.1.2                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2017-06-26   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

DECLARE @SharedFolders TABLE
       ([DocFolderID]             INT,
        [Shared]                  BIT,
        [ServerID]                INT,
        [ServerName]              VARCHAR(50),
        [ServerHost]              VARCHAR(50),
        [ServerTypeID]            INT,
        [ServerType]              VARCHAR(20),
        [ServerFolderID]          VARCHAR(500),
        [ServerCollabMatter]      VARCHAR(50),
        [ServerCollabFolder]      VARCHAR(50))

INSERT
  INTO @SharedFolders
      ([DocFolderID],
       [Shared],
       [ServerID],
       [ServerName],
       [ServerHost],
       [ServerTypeID],
       [ServerCollabMatter],
       [ServerCollabFolder])
SELECT [MF].[DocFolderID],
       [MF].[Shared],
       [FSS].[id],
       [FSS].[Name],
       [FSS].[Server],
       [FSS].[TypeId],
       RTRIM(ISNULL([FSS].[CollabMatter], '')),
       RTRIM(ISNULL([FSS].[CollabFolder], ''))
  FROM (SELECT [DF].[id]                                   AS [DocFolderID],
               ISNULL([FSDF].[Shared], CONVERT(BIT, 0))    AS [Shared]
          FROM [dbo].[DocFolders] [DF]
               LEFT OUTER JOIN [dbo].[FileSharingDocFolders] [FSDF]
                            ON [FSDF].[DocFolderID] = [DF].[id]
         WHERE [DF].[id] = @DocFolderID) [MF]
       CROSS APPLY [dbo].[FileSharingServers] [FSS]
 WHERE [FSS].[CollabDocFolderID] = @DocFolderID

IF NOT EXISTS
  (SELECT TOP 1 1 FROM @SharedFolders)
  BEGIN
    INSERT
      INTO @SharedFolders
          ([DocFolderID],
           [Shared],
           [ServerID],
           [ServerName],
           [ServerHost],
           [ServerTypeID],
           [ServerCollabMatter],
           [ServerCollabFolder])
    SELECT [MF].[DocFolderID],
           [MF].[Shared],
           [FSS].[id],
           [FSS].[Name],
           [FSS].[Server],
           [FSS].[TypeId],
           '',
           ''
      FROM (SELECT [DF].[id]                                   AS [DocFolderID],
                   ISNULL([FSDF].[Shared], CONVERT(BIT, 0))    AS [Shared]
              FROM [dbo].[DocFolders] [DF]
                   LEFT OUTER JOIN [dbo].[FileSharingDocFolders] [FSDF]
                                ON [FSDF].[DocFolderID] = [DF].[id]
             WHERE [DF].[id] = @DocFolderID) [MF]
           CROSS APPLY [dbo].[FileSharingServers] [FSS]
     WHERE ISNULL([FSS].[CollabDocFolderID], 0) = 0
  END

UPDATE [SF]
   SET [SF].[ServerType] = [FSST].[TypeName]
  FROM @SharedFolders [SF]
       INNER JOIN [dbo].[FileSharingServerTypes] [FSST]
               ON [FSST].[id] = [SF].[ServerTypeID]

UPDATE [SF]
   SET [SF].[ServerFolderID] = ISNULL([FSM].[FileSharingServerFolderID], '')
  FROM @SharedFolders [SF]
       LEFT OUTER JOIN [dbo].[FileSharingMapping] [FSM]
                    ON [FSM].[FileSharingDocFoldersID] = [SF].[DocFolderID]
                   AND [FSM].[FileSharingServersID] = [SF].[ServerID]

SELECT [SF].[DocFolderID],
       [SF].[ServerID],
       [SF].[ServerName],
       [SF].[ServerHost],
       [SF].[ServerTypeID],
       [SF].[ServerType],
       [SF].[ServerFolderID],
       [SF].[ServerCollabMatter],
       [SF].[ServerCollabFolder]
  FROM @SharedFolders [SF]
 WHERE [SF].[Shared] = CONVERT(BIT, 1)

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPGetFileSharingMatterFolder'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPGetFileSharingMatterFolder]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETSPGetFileSharingMatterFolder]
      (@MatterCode VARCHAR(20))
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETSPGetFileSharingMatterFolder]                                                                *
 *                                                                                                           *
 * Retrieves information about File Sharing matter folder                                                    *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.5.1.2                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2017-06-27   Pino Carafa       Created                                                                    *
 * 2017-08-29   Pino Carafa       Make sure it picks up only relevant Servers                                *
 * 2017-08-29   Pino Carafa       Server Name can be up to 50 characters                                     *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

DECLARE @SharedFolders TABLE
       ([MatterCode]              VARCHAR(20),
        [ServerID]                INT,
        [ServerName]              VARCHAR(50),
        [ServerHost]              VARCHAR(50),
        [ServerTypeID]            INT,
        [ServerType]              VARCHAR(20),
        [ServerFolderID]          VARCHAR(500))

INSERT
  INTO @SharedFolders
      ([MatterCode],
       [ServerID],
       [ServerName],
       [ServerHost],
       [ServerTypeID])
SELECT @MatterCode,
       [FSS].[id],
       [FSS].[Name],
       [FSS].[Server],
       [FSS].[TypeId]
  FROM [dbo].[FileSharingServers] [FSS]
       LEFT OUTER JOIN [dbo].[FileSharingMatterMapping] [FSMM]
                    ON [FSMM].[FileSharingMatter] = @MatterCode
                   AND [FSMM].[FileSharingServersID] = [FSS].[id]
 WHERE ISNULL([FSS].[CollabDocFolderID], 0) = 0
    OR [FSMM].[id] IS NOT NULL

UPDATE [SF]
   SET [SF].[ServerType] = [FSST].[TypeName]
  FROM @SharedFolders [SF]
       INNER JOIN [dbo].[FileSharingServerTypes] [FSST]
               ON [FSST].[id] = [SF].[ServerTypeID]

UPDATE [SF]
   SET [SF].[ServerFolderID] = ISNULL([FMM].[FileSharingServerFolderID], '')
  FROM @SharedFolders [SF]
       LEFT OUTER JOIN [dbo].[FileSharingMatterMapping] [FMM]
                    ON [FMM].[FileSharingMatter] = [SF].[MatterCode]
                   AND [FMM].[FileSharingServersID] = [SF].[ServerID]

SELECT [SF].[MatterCode],
       [SF].[ServerID],
       [SF].[ServerName],
       [SF].[ServerHost],
       [SF].[ServerTypeID],
       [SF].[ServerType],
       [SF].[ServerFolderID]
  FROM @SharedFolders [SF]

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPGetFileSharingRootFolder'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPGetFileSharingRootFolder]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETSPGetFileSharingRootFolder]
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETSPGetFileSharingRootFolder]                                                                  *
 *                                                                                                           *
 * Retrieves information about File Sharing root folder                                                      *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.5.1.2                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2017-06-27   Pino Carafa       Created                                                                    *
 * 2017-08-03   Pino Carafa       Added Client ID and Client Secret                                          *
 * 2017-08-04   Pino Carafa       Added Site ID                                                              *
 * 2017-08-24   Pino Carafa       Added Domain and Proxy Settings KEYD-4750 and KEYD-4751                    *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

DECLARE @SharedFolders TABLE
       ([ServerID]                           INT,
        [ServerName]                         VARCHAR(50),
        [ServerHost]                         VARCHAR(50),
        [ServerTypeID]                       INT,
        [ServerType]                         VARCHAR(20),
        [ServerFolderID]                     VARCHAR(500),
        [ServerClientFolderURL]              VARCHAR(500),
        [ServerClientFolderPermissionsURL]   VARCHAR(500),
        [CollabDocFolderID]                  INT,
        [CollabMatter]                       VARCHAR(50),
        [CollabFolder]                       VARCHAR(50),
        [ClientID]                           VARCHAR(100),
        [ClientSecret]                       VARCHAR(500),
        [SiteID]                             VARCHAR(100),
        [Domain]                             VARCHAR(100),
        [ProxyType]                          INT,
        [ProxyServer]                        VARCHAR(500),
        [ProxyPort]                          INT)

INSERT
  INTO @SharedFolders
      ([ServerID],
       [ServerName],
       [ServerHost],
       [ServerTypeID],
       [ServerFolderID],
       [ServerClientFolderURL],
       [ServerClientFolderPermissionsURL],
       [CollabDocFolderID],
       [CollabMatter],
       [CollabFolder],
       [ClientID],
       [ClientSecret],
       [SiteID],
       [Domain],
       [ProxyType],
       [ProxyServer],
       [ProxyPort])
SELECT [FSS].[id],
       [FSS].[Name],
       [FSS].[Server],
       [FSS].[TypeId],
       [FSS].[RootFolder],
       [FSS].[ClientFolderURL],
       [FSS].[ClientFolderPermissionsURL],
       ISNULL([FSS].[CollabDocFolderID], 0) AS [CollabDocFolderID],
       ISNULL([FSS].[CollabMatter], '')     AS [CollabMatter],
       ISNULL([FSS].[CollabFolder], '')     AS [CollabFolder],
       [FSS].[ClientID],
       [FSS].[ClientSecret],
       [FSS].[SiteID],
       [FSS].[Domain],
       [FSS].[ProxyType],
       [FSS].[ProxyServer],
       [FSS].[ProxyPort]
  FROM [dbo].[FileSharingServers] [FSS]

UPDATE [SF]
   SET [SF].[ServerType] = [FSST].[TypeName]
  FROM @SharedFolders [SF]
       INNER JOIN [dbo].[FileSharingServerTypes] [FSST]
               ON [FSST].[id] = [SF].[ServerTypeID]

SELECT [SF].[ServerID],
       [SF].[ServerName],
       [SF].[ServerHost],
       [SF].[ServerTypeID],
       [SF].[ServerType],
       [SF].[ServerFolderID],
       [SF].[ServerClientFolderURL],
       [SF].[ServerClientFolderPermissionsURL],
       [SF].[CollabDocFolderID],
       [SF].[CollabMatter],
       [SF].[CollabFolder],
       [SF].[ClientID],
       [SF].[ClientSecret],
       [SF].[SiteID],
       [SF].[Domain],
       [SF].[ProxyType],
       [SF].[ProxyServer],
       [SF].[ProxyPort]
  FROM @SharedFolders [SF]

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPGetFileSharingServer'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPGetFileSharingServer]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETSPGetFileSharingServer]
      (@ServerID INT)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETSPGetFileSharingServer]                                                                      *
 *                                                                                                           *
 * Retrieves information about a File Sharing server / root folder                                           *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.5.2.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2017-08-23   Pino Carafa       Created                                                                    *
 * 2017-08-24   Pino Carafa       Added Domain and Proxy Settings KEYD-4750 and KEYD-4751                    *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

DECLARE @SharedFolders TABLE
       ([ServerID]                           INT,
        [ServerName]                         VARCHAR(50),
        [ServerHost]                         VARCHAR(50),
        [ServerTypeID]                       INT,
        [ServerType]                         VARCHAR(20),
        [ServerFolderID]                     VARCHAR(500),
        [ServerClientFolderURL]              VARCHAR(500),
        [ServerClientFolderPermissionsURL]   VARCHAR(500),
        [CollabDocFolderID]                  INT,
        [CollabMatter]                       VARCHAR(50),
        [CollabFolder]                       VARCHAR(50),
        [ClientID]                           VARCHAR(100),
        [ClientSecret]                       VARCHAR(500),
        [SiteID]                             VARCHAR(100),
        [Domain]                             VARCHAR(100),
        [ProxyType]                          INT,
        [ProxyServer]                        VARCHAR(500),
        [ProxyPort]                          INT)

INSERT
  INTO @SharedFolders
      ([ServerID],
       [ServerName],
       [ServerHost],
       [ServerTypeID],
       [ServerFolderID],
       [ServerClientFolderURL],
       [ServerClientFolderPermissionsURL],
       [CollabDocFolderID],
       [CollabMatter],
       [CollabFolder],
       [ClientID],
       [ClientSecret],
       [SiteID],
       [Domain],
       [ProxyType],
       [ProxyServer],
       [ProxyPort])
SELECT [FSS].[id],
       [FSS].[Name],
       [FSS].[Server],
       [FSS].[TypeId],
       [FSS].[RootFolder],
       [FSS].[ClientFolderURL],
       [FSS].[ClientFolderPermissionsURL],
       ISNULL([FSS].[CollabDocFolderID], 0) AS [CollabDocFolderID],
       ISNULL([FSS].[CollabMatter], '')     AS [CollabMatter],
       ISNULL([FSS].[CollabFolder], '')     AS [CollabFolder],
       [FSS].[ClientID],
       [FSS].[ClientSecret],
       [FSS].[SiteID],
       [FSS].[Domain],
       [FSS].[ProxyType],
       [FSS].[ProxyServer],
       [FSS].[ProxyPort]
  FROM [dbo].[FileSharingServers] [FSS]
 WHERE [FSS].[id] = @ServerID

UPDATE [SF]
   SET [SF].[ServerType] = [FSST].[TypeName]
  FROM @SharedFolders [SF]
       INNER JOIN [dbo].[FileSharingServerTypes] [FSST]
               ON [FSST].[id] = [SF].[ServerTypeID]

SELECT [SF].[ServerID],
       [SF].[ServerName],
       [SF].[ServerHost],
       [SF].[ServerTypeID],
       [SF].[ServerType],
       [SF].[ServerFolderID],
       [SF].[ServerClientFolderURL],
       [SF].[ServerClientFolderPermissionsURL],
       [SF].[CollabDocFolderID],
       [SF].[CollabMatter],
       [SF].[CollabFolder],
       [SF].[ClientID],
       [SF].[ClientSecret],
       [SF].[SiteID],
       [SF].[Domain],
       [SF].[ProxyType],
       [SF].[ProxyServer],
       [SF].[ProxyPort]
  FROM @SharedFolders [SF]

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPFileSharingSaveClientFolder'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPFileSharingSaveClientFolder]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETSPFileSharingSaveClientFolder]
      (@ClientCode     VARCHAR(10),
       @ServersID      INT,
       @ServerFolderID VARCHAR(500))
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETSPFileSharingSaveClientFolder]                                                               *
 *                                                                                                           *
 * Saves information about the File Sharing Client folder                                                    *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.5.3.10                                                 *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2018-04-18   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

INSERT
  INTO [dbo].[FileSharingClientMapping]
      ([FileSharingClient],
       [FileSharingServersID])
SELECT [CON].[Code]                  AS [FileSharingClient],
       [FSS].[id]                    As [FileSharingServersID]
  FROM [dbo].[Contacts] [CON]
       INNER JOIN [dbo].[FileSharingServers] [FSS]
               ON [FSS].[id] = @ServersID
       LEFT OUTER JOIN [dbo].[FileSharingClientMapping] [FSCM]
                    ON [FSCM].[FileSharingClient] = [CON].[Code]
                   AND [FSCM].[FileSharingServersID] = [FSS].[id]
 WHERE [CON].[Code] = @ClientCode
   AND [FSCM].[id] IS NULL

UPDATE [FSCM]
   SET [FSCM].[FileSharingServerFolderID] = @ServerFolderID
  FROM [dbo].[FileSharingClientMapping] [FSCM]
 WHERE [FSCM].[FileSharingClient] = @ClientCode
   AND [FSCM].[FileSharingServersID] = @ServersID

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPFileSharingSaveMatterFolder'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPFileSharingSaveMatterFolder]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETSPFileSharingSaveMatterFolder]
      (@MatterCode     VARCHAR(20),
       @ServersID      INT,
       @ServerFolderID VARCHAR(500))
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETSPFileSharingSaveMatterFolder]                                                               *
 *                                                                                                           *
 * Retrieves information about the File Sharing matter folder                                                *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.5.1.2                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2017-06-27   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

INSERT
  INTO [dbo].[FileSharingMatterMapping]
      ([FileSharingMatter],
       [FileSharingServersID])
SELECT [MAT].[Code]                  AS [FileSharingMatter],
       [FSS].[id]                    As [FileSharingServersID]
  FROM [dbo].[SearchMatters] [MAT]
       INNER JOIN [dbo].[FileSharingServers] [FSS]
               ON [FSS].[id] = @ServersID
       LEFT OUTER JOIN [dbo].[FileSharingMatterMapping] [FSMM]
                    ON [FSMM].[FileSharingMatter] = [MAT].[Code]
                   AND [FSMM].[FileSharingServersID] = [FSS].[id]
 WHERE [MAT].[Code] = @MatterCode
   AND [FSMM].[id] IS NULL

UPDATE [FSMM]
   SET [FSMM].[FileSharingServerFolderID] = @ServerFolderID
  FROM [dbo].[FileSharingMatterMapping] [FSMM]
 WHERE [FSMM].[FileSharingMatter] = @MatterCode
   AND [FSMM].[FileSharingServersID] = @ServersID

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPFileSharingSaveDocFolder'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPFileSharingSaveDocFolder]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETSPFileSharingSaveDocFolder]
      (@DocFolderID    INT,
       @ServersID      INT,
       @ServerFolderID VARCHAR(500))
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETSPFileSharingSaveDocFolder]                                                                  *
 *                                                                                                           *
 * Retrieves information about the File Sharing folder                                                       *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.5.1.2                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2017-06-27   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

INSERT
  INTO [dbo].[FileSharingMapping]
      ([FileSharingDocFoldersID],
       [FileSharingServersID])
SELECT [DF].[id]                     AS [FileSharingDocFoldersID],
       [FSS].[id]                    As [FileSharingServersID]
  FROM [dbo].[DocFolders] [DF]
       INNER JOIN [dbo].[FileSharingServers] [FSS]
               ON [FSS].[id] = @ServersID
       LEFT OUTER JOIN [dbo].[FileSharingMapping] [FSM]
                    ON [FSM].[FileSharingDocFoldersID] = @DocFolderID
                   AND [FSM].[FileSharingServersID] = [FSS].[id]
 WHERE [DF].[id] = @DocFolderID
   AND [FSM].[id] IS NULL

UPDATE [FSM]
   SET [FSM].[FileSharingServerFolderID] = @ServerFolderID
  FROM [dbo].[FileSharingMapping] [FSM]
 WHERE [FSM].[FileSharingDocFoldersID] = @DocFolderID
   AND [FSM].[FileSharingServersID] = @ServersID

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPFileSharingSaveRootFolder'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPFileSharingSaveRootFolder]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETSPFileSharingSaveRootFolder]
      (@ServersID      INT,
       @ServerFolderID VARCHAR(500))
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETSPFileSharingSaveRootFolder]                                                                 *
 *                                                                                                           *
 * Save the File Sharing Root Folder for a File Sharing Server                                               *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.5.1.2                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2017-06-28   Pino Carafa       Created                                                                    *
 * 2017-08-03   Pino Carafa       Added Client ID and Client Secret                                          *
 * 2017-08-04   Pino Carafa       Added Site ID                                                              *
 * 2017-08-24   Pino Carafa       This SP only intended to save root folder. Remove everything else          *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

UPDATE [FSS]
   SET [FSS].[RootFolder]   = @ServerFolderID
  FROM [dbo].[FileSharingServers] [FSS]
 WHERE [FSS].[id] = @ServersID

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPFileSharingGetAllMatterFolders'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPFileSharingGetAllMatterFolders]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETSPFileSharingGetAllMatterFolders]
      (@MatterCode      VARCHAR(20))
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETSPFileSharingGetAllMatterFolders]                                                            *
 *                                                                                                           *
 * Retrieves information about File Sharing root folder                                                      *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.5.1.2                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2017-06-27   Pino Carafa       Created                                                                    *
 * 2017-07-12   Pino Carafa       Added IsOurs                                                               *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

SET NOCOUNT ON

DECLARE @DFs TABLE
       ([DocFolderID]            INT,
        [FolderName]             VARCHAR(500),
        [Shared]                 BIT,
        [IsOurs]                 BIT)

INSERT
  INTO @DFs
      ([DocFolderID],
       [FolderName],
       [Shared],
       [IsOurs])
SELECT [DF].[id],
       [DF].[foldername],
       [DF].[foldershared],
       [DF].[folderisours]
  FROM [dbo].[SearchMatters] [SM]
       INNER JOIN [dbo].[MatterDocFolders] [MDF]
               ON [MDF].[MatterCode] = [SM].[Code]
       CROSS APPLY [dbo].[ky_NETTFGetDocFolders3]([MDF].[DocFolderID], 1, 1) [DF]
 WHERE [SM].[Code] = @MatterCode

SELECT [DF].[DocFolderID],
       [DF].[FolderName],
       [FSM].[FileSharingServersID],
       [FSM].[FileSharingServerFolderID],
       [DF].[Shared],
       [DF].[IsOurs]
  FROM @DFs [DF]
       INNER JOIN [dbo].[FileSharingMapping] [FSM]
               ON [FSM].[FileSharingDocFoldersID] = [DF].[DocFolderID]
 WHERE [FSM].[FileSharingServerFolderID] <> ''
END

SET NOCOUNT OFF

GO

IF NOT EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] [SO]
    WHERE [SO].[name] = 'FileSharingDocuments')
  BEGIN
    EXEC ('
    CREATE
     TABLE [dbo].[FileSharingDocuments]
          ([id]                          INT            IDENTITY(1, 1)                                      NOT NULL,
           [FileSharingMappingID]        INT                                                                NOT NULL,
           [TrackReference]              INT                                                                NOT NULL,
           [FileSharingDocumentID]       VARCHAR(500)                                                       NOT NULL,
           CONSTRAINT [PK_FileSharingDocuments] PRIMARY KEY CLUSTERED
                     ([id])
                   ON [PRIMARY])
        ON [PRIMARY]')

    EXEC ('
    ALTER 
    TABLE [dbo].[FileSharingDocuments] 
     WITH CHECK 
      ADD CONSTRAINT 
          [FK_FileSharingDocuments_FSM]
          FOREIGN KEY([FileSharingMappingID])
          REFERENCES [dbo].[FileSharingMapping]
                    ([id])
          ON DELETE CASCADE')

    EXEC ('
    ALTER 
    TABLE [dbo].[FileSharingDocuments] 
    CHECK CONSTRAINT [FK_FileSharingDocuments_FSM]')

    EXEC ('
    ALTER 
    TABLE [dbo].[FileSharingDocuments] 
     WITH CHECK 
      ADD CONSTRAINT 
          [FK_FileSharingDocuments_DAT]
          FOREIGN KEY([TrackReference])
          REFERENCES [dbo].[DiaryAttachments]
                    ([TrackReference])
          ON DELETE CASCADE')

    EXEC ('
    ALTER 
    TABLE [dbo].[FileSharingDocuments] 
    CHECK CONSTRAINT [FK_FileSharingDocuments_DAT]')
  END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] [SO]
          LEFT OUTER JOIN SYS.[indexes] [SI]
                       ON [SI].[object_id] = [SO].[object_id]
                      AND [SI].[name] = 'IDX_TR_Unique'
    WHERE [SO].[name] = 'FileSharingDocuments'
      AND [SI].[index_id] IS NULL)
  BEGIN
    EXEC ('
    CREATE UNIQUE NONCLUSTERED INDEX
           [IDX_TR_Unique]
        ON [dbo].[FileSharingDocuments]
          ([TrackReference],
           [FileSharingMappingID])
           INCLUDE ([id],
                    [FileSharingDocumentID])
        ON [PRIMARY]')
  END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] [SO]
          LEFT OUTER JOIN SYS.[indexes] [SI]
                       ON [SI].[object_id] = [SO].[object_id]
                      AND [SI].[name] = 'IDX_DOC_Unique'
    WHERE [SO].[name] = 'FileSharingDocuments'
      AND [SI].[index_id] IS NULL)
  BEGIN
    EXEC ('
    CREATE UNIQUE NONCLUSTERED INDEX
           [IDX_DOC_Unique]
        ON [dbo].[FileSharingDocuments]
          ([FileSharingDocumentID],
           [FileSharingMappingID])
           INCLUDE ([id],
                    [TrackReference])
        ON [PRIMARY]')
  END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPFileSharingGetFSDocument'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPFileSharingGetFSDocument]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETSPFileSharingGetFSDocument]
      (@FileSharingDocumentID       VARCHAR(500),
       @ServerID                    INT)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETSPFileSharingGetFSDocument]                                                                  *
 *                                                                                                           *
 * Retrieves the Track Reference mapped to the File Sharing document                                         *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.5.1.2                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2017-06-28   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN
  SELECT [FSD].[TrackReference],
         [FSD].[FileSharingDocumentID],
         [FSM].[FileSharingServersID],
         [FSM].[FileSharingDocFoldersID],
         [FSM].[FileSharingServerFolderID]
    FROM [dbo].[FileSharingDocuments] [FSD]
         INNER JOIN [dbo].[FileSharingMapping] [FSM]
                 ON [FSM].[id] = [FSD].[FileSharingMappingID]
                AND [FSM].[FileSharingServersID] = @ServerID
   WHERE [FSD].[FileSharingDocumentID] = @FileSharingDocumentID
END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPFileSharingSaveFSDocument'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPFileSharingSaveFSDocument]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETSPFileSharingSaveFSDocument]
      (@TrackReference              INT,
       @DocFolderID                 INT,
       @ServerID                    INT,
       @FileSharingDocumentID       VARCHAR(500))
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETSPFileSharingGetFSDocument]                                                                  *
 *                                                                                                           *
 * Retrieves the Track Reference mapped to the File Sharing document                                         *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.5.1.2                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2017-06-28   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

DECLARE @MappingID INT

SELECT @MappingID = [FSM].[id]
  FROM [dbo].[FileSharingMapping] [FSM]
 WHERE [FSM].[FileSharingDocFoldersID] = @DocFolderID
   AND [FSM].[FileSharingServersID] = @ServerID

IF ISNULL(@FileSharingDocumentID, '') = ''
  BEGIN
    DELETE [FSD]
      FROM [dbo].[FileSharingDocuments] [FSD]
     WHERE [FSD].[FileSharingMappingID] = @MappingID
       AND [FSD].[TrackReference] = @TrackReference
  END
ELSE
  BEGIN
    INSERT
      INTO [dbo].[FileSharingDocuments]
          ([FileSharingMappingID],
           [TrackReference],
           [FileSharingDocumentID])
    SELECT [NEW].*
      FROM (SELECT @MappingID             AS [FileSharingMappingID],
                   @TrackReference        AS [TrackReference],
                   @FileSharingDocumentID AS [FileSharingDocumentID]) [NEW]
           LEFT OUTER JOIN [dbo].[FileSharingDocuments] [FSD]
                        ON [FSD].[FileSharingMappingID] = [NEW].[FileSharingMappingID]
                       AND [FSD].[TrackReference] = [NEW].[TrackReference]
     WHERE [FSD].[id] IS NULL

    UPDATE [FSD]
       SET [FSD].[FileSharingDocumentID] = @FileSharingDocumentID
      FROM [dbo].[FileSharingDocuments] [FSD]
     WHERE [FSD].[FileSharingMappingID] = @MappingID
       AND [FSD].[TrackReference] = @TrackReference
  END
END

GO

INSERT
  INTO [dbo].[TemplateActions]
      ([ACTIONCODE],
       [DESC],
       [WKTCODE],
       [ACTTYPE],
       [ACTIONCATEGORY],
       [STATUS],
       [PROCESSTYPE],
       [PUBLISHER],
       [UPDATEFIELD],
       [UPDATEVALUE],
       [HIGHLIGHTED],
       [PRIORITY],
       [NEXTACTIONOPTIONAL],
       [BILLABLE],
       [EMAIL],
       [BILLDESCRIPTION],
       [AssignActionTo],
       [AssignTeamActionTo],
       [ChargeValue],
       [ChargeInterest],
       [Description])
SELECT [NEW].[ACTIONCODE],
       [NEW].[DESC],
       [NEW].[WKTCODE],
       [NEW].[ACTTYPE],
       [NEW].[ACTIONCATEGORY],
       [NEW].[STATUS],
       [NEW].[PROCESSTYPE],
       [NEW].[PUBLISHER],
       [NEW].[UPDATEFIELD],
       [NEW].[UPDATEVALUE],
       [NEW].[HIGHLIGHTED],
       [NEW].[PRIORITY],
       [NEW].[NEXTACTIONOPTIONAL],
       [NEW].[BILLABLE],
       [NEW].[EMAIL],
       [NEW].[BILLDESCRIPTION],
       [NEW].[AssignActionTo],
       [NEW].[AssignTeamActionTo],
       [NEW].[ChargeValue],
       [NEW].[ChargeInterest],
       [NEW].[Description]
  FROM (SELECT 'ADMAA'                               AS [ACTIONCODE],
               'Action'                              AS [DESC],
               'SYS'                                 AS [WKTCODE],
               'A'                                   AS [ACTTYPE],
               'A'                                   AS [ACTIONCATEGORY],
               'NA'                                  AS [STATUS],
               'I'                                   AS [PROCESSTYPE],
               'N'                                   AS [PUBLISHER],
               'N'                                   AS [UPDATEFIELD],
               ''                                    AS [UPDATEVALUE],
               'N'                                   AS [HIGHLIGHTED],
               'Normal'                              AS [PRIORITY],
               'Y'                                   AS [NEXTACTIONOPTIONAL],
               'N'                                   AS [BILLABLE],
               ''                                    AS [EMAIL],
               ''                                    AS [BILLDESCRIPTION],
               'L'                                   AS [AssignActionTo],
               'L'                                   AS [AssignTeamActionTo],
               0                                     AS [ChargeValue],
               0                                     AS [ChargeInterest],
               'Action'                              AS [Description]) [NEW]
       LEFT OUTER JOIN [dbo].[TemplateActions] [TAC]
                    ON [TAC].[ACTIONCODE] = [NEW].[ACTIONCODE]
 WHERE [TAC].[ACTIONCODE] IS NULL
GO

INSERT
  INTO [dbo].[TemplateActions]
      ([ACTIONCODE],
       [DESC],
       [WKTCODE],
       [ACTTYPE],
       [ACTIONCATEGORY],
       [STATUS],
       [PROCESSTYPE],
       [PUBLISHER],
       [UPDATEFIELD],
       [UPDATEVALUE],
       [HIGHLIGHTED],
       [PRIORITY],
       [NEXTACTIONOPTIONAL],
       [BILLABLE],
       [EMAIL],
       [BILLDESCRIPTION],
       [AssignActionTo],
       [AssignTeamActionTo],
       [ChargeValue],
       [ChargeInterest],
       [Description])
SELECT [NEW].[ACTIONCODE],
       [NEW].[DESC],
       [NEW].[WKTCODE],
       [NEW].[ACTTYPE],
       [NEW].[ACTIONCATEGORY],
       [NEW].[STATUS],
       [NEW].[PROCESSTYPE],
       [NEW].[PUBLISHER],
       [NEW].[UPDATEFIELD],
       [NEW].[UPDATEVALUE],
       [NEW].[HIGHLIGHTED],
       [NEW].[PRIORITY],
       [NEW].[NEXTACTIONOPTIONAL],
       [NEW].[BILLABLE],
       [NEW].[EMAIL],
       [NEW].[BILLDESCRIPTION],
       [NEW].[AssignActionTo],
       [NEW].[AssignTeamActionTo],
       [NEW].[ChargeValue],
       [NEW].[ChargeInterest],
       [NEW].[Description]
  FROM (SELECT 'ADMBR'                               AS [ACTIONCODE],
               'Brief'                               AS [DESC],
               'SYS'                                 AS [WKTCODE],
               'A'                                   AS [ACTTYPE],
               'A'                                   AS [ACTIONCATEGORY],
               'NA'                                  AS [STATUS],
               'I'                                   AS [PROCESSTYPE],
               'N'                                   AS [PUBLISHER],
               'N'                                   AS [UPDATEFIELD],
               ''                                    AS [UPDATEVALUE],
               'N'                                   AS [HIGHLIGHTED],
               'Normal'                              AS [PRIORITY],
               'Y'                                   AS [NEXTACTIONOPTIONAL],
               'N'                                   AS [BILLABLE],
               ''                                    AS [EMAIL],
               ''                                    AS [BILLDESCRIPTION],
               'L'                                   AS [AssignActionTo],
               'L'                                   AS [AssignTeamActionTo],
               0                                     AS [ChargeValue],
               0                                     AS [ChargeInterest],
               'Brief'                               AS [Description]) [NEW]
       LEFT OUTER JOIN [dbo].[TemplateActions] [TAC]
                    ON [TAC].[ACTIONCODE] = [NEW].[ACTIONCODE]
 WHERE [TAC].[ACTIONCODE] IS NULL
GO

INSERT
  INTO [dbo].[TemplateActions]
      ([ACTIONCODE],
       [DESC],
       [WKTCODE],
       [ACTTYPE],
       [ACTIONCATEGORY],
       [STATUS],
       [PROCESSTYPE],
       [PUBLISHER],
       [UPDATEFIELD],
       [UPDATEVALUE],
       [HIGHLIGHTED],
       [PRIORITY],
       [NEXTACTIONOPTIONAL],
       [BILLABLE],
       [EMAIL],
       [BILLDESCRIPTION],
       [AssignActionTo],
       [AssignTeamActionTo],
       [ChargeValue],
       [ChargeInterest],
       [Description])
SELECT [NEW].[ACTIONCODE],
       [NEW].[DESC],
       [NEW].[WKTCODE],
       [NEW].[ACTTYPE],
       [NEW].[ACTIONCATEGORY],
       [NEW].[STATUS],
       [NEW].[PROCESSTYPE],
       [NEW].[PUBLISHER],
       [NEW].[UPDATEFIELD],
       [NEW].[UPDATEVALUE],
       [NEW].[HIGHLIGHTED],
       [NEW].[PRIORITY],
       [NEW].[NEXTACTIONOPTIONAL],
       [NEW].[BILLABLE],
       [NEW].[EMAIL],
       [NEW].[BILLDESCRIPTION],
       [NEW].[AssignActionTo],
       [NEW].[AssignTeamActionTo],
       [NEW].[ChargeValue],
       [NEW].[ChargeInterest],
       [NEW].[Description]
  FROM (SELECT 'ADMSR'                               AS [ACTIONCODE],
               'Strong Room'                         AS [DESC],
               'SYS'                                 AS [WKTCODE],
               'A'                                   AS [ACTTYPE],
               'A'                                   AS [ACTIONCATEGORY],
               'NA'                                  AS [STATUS],
               'I'                                   AS [PROCESSTYPE],
               'N'                                   AS [PUBLISHER],
               'N'                                   AS [UPDATEFIELD],
               ''                                    AS [UPDATEVALUE],
               'N'                                   AS [HIGHLIGHTED],
               'Normal'                              AS [PRIORITY],
               'Y'                                   AS [NEXTACTIONOPTIONAL],
               'N'                                   AS [BILLABLE],
               ''                                    AS [EMAIL],
               ''                                    AS [BILLDESCRIPTION],
               'L'                                   AS [AssignActionTo],
               'L'                                   AS [AssignTeamActionTo],
               0                                     AS [ChargeValue],
               0                                     AS [ChargeInterest],
               'Strong Room'                         AS [Description]) [NEW]
       LEFT OUTER JOIN [dbo].[TemplateActions] [TAC]
                    ON [TAC].[ACTIONCODE] = [NEW].[ACTIONCODE]
 WHERE [TAC].[ACTIONCODE] IS NULL
GO

/***************************************************************
 *
 * ALL DIARY STEP PROCEDURES. KEEP AND MODIFY THESE TOGETHER
 * START
 *
 ***************************************************************/
IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_CreateDiaryStep'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_CreateDiaryStep]
  END
GO

CREATE PROCEDURE [dbo].[ky_CreateDiaryStep]
                (@pStatus       Int=0,
                 @pDate         varchar(23),
                 @pCaseCode     char(17),
                 @pActionCode   char(15),
                 @pActionStatus char(3),
                 @pFnCode       char(5),
                 @pTeamCode     char(5),
                 @pText1        varchar(8000),
                 @pAttachments  char(1),
                 @pEmailAddress varchar(150),
                 @pAddressTo    varchar(2000),
                 @pccTo         varchar(2000),
                 @pbccTo        varchar(2000),
                 @pemail        char(1),
                 @pSubject      varchar(500),
                 @pProcessType  char(1),
                 @pTime         char(10),
                 @pActionId     int=0               output)
AS
/************************************************************************
*
* Create Diary Step
*
* 2015-09-30 Pino        Use ky_NETGetNextActionID
* 2016-09-13 Pino        KYC - KEYD-4040
*
*************************************************************************/
BEGIN
    SET NOCOUNT ON
    DECLARE @strcasecode        nVarChar(17)
    DECLARE @intActionID        Int
    DECLARE @intAssignno        int
    DECLARE @pPublish           char(1)
    DECLARE @KYC                char(1)
    DECLARE @SubjectLen         int
    DECLARE @CaseCodeLen        int
    DECLARE @FELen              int
    DECLARE @TeamLen            int

    select @SubjectLen = sc.[length]
      from sysobjects so
           inner join syscolumns sc
                   on sc.[id] = so.[id]
                  and sc.[name] = 'subject'
     where so.[name] = 'diary'

    IF (@SubjectLen = 16)
      SET @SubjectLen = 2000

    select @CaseCodeLen = sc.[length]
      from sysobjects so
           inner join syscolumns sc
                   on sc.[id] = so.[id]
                  and sc.[name] = 'casecode'
     where so.[name] = 'diary'

    select @FELen = sc.[length]
      from sysobjects so
           inner join syscolumns sc
                   on sc.[id] = so.[id]
                  and sc.[name] = 'fncode'
     where so.[name] = 'diary'

    select @TeamLen = sc.[length]
      from sysobjects so
           inner join syscolumns sc
                   on sc.[id] = so.[id]
                  and sc.[name] = 'teamcode'
     where so.[name] = 'diary'

-- Start Transaction -----------------------------------------------------------------------------------------------------------------------------------------

    BEGIN TRAN Key_Tran_01

    IF @@error <> 0 GOTO Tran_Fail

-- Get CaseCode From Matters Table -----------------------------------------------------------------------------------------------------------------------------------------

    SELECT @strCaseCode = (SELECT Top 1
                                  [Code]
                             FROM [dbo].[Matters]
                            WHERE [code] = @pCaseCode)

    IF @@error <> 0 GOTO Tran_Fail

    SET @strCaseCode = RTrim(Isnull(@strCaseCode, ''))
    Set @pFNCode = RTrim(@pFNCode)
    Set @pTeamCode = RTrim(@pTeamCode)

    IF LEN(@strCaseCode) < 2 GOTO Tran_Fail
    IF LEN(@strCaseCode) > @CaseCodeLen GOTO Tran_Fail
    IF LEN(@pFNCode) > @FELen GOTO Tran_Fail
    IF LEN(@pTeamCode) > @TeamLen GOTO Tran_Fail

-- Set Publish and Subject -----------------------------------------------------------------------------------------------------------------------------------

    SELECT @pPublish = ISNULL([TAC].[PUBLISHER], 'N'),
           @KYC = ISNULL([TAC].[KYC], 'N')
      FROM [dbo].[TemplateActions] [TAC]
     WHERE [TAC].[ACTIONCODE] = @pActionCode

    IF @pPublish  IS NULL SELECT @pPublish = 'N'

    SET @pSubject = LTrim(RTrim(@pSubject))

    IF LEN(@pSubject) > @SubjectLen
       BEGIN
          SET @pSubject = Substring(@pSubject, 1, @SubjectLen)
       END

    DECLARE @pdDate     DATETIME
    DECLARE @pdSentDate DATETIME
    DECLARE @pdDueDate  DATETIME
    IF RTRIM(ISNULL(@pDate, '')) <> ''
      BEGIN
        SET @pdDate = convert(DATETIME, @pDate)
        IF SERVERPROPERTY('EngineEdition') = 5
          BEGIN
            --Azure. Local time must be converted to UTC
            SET @pdDate = [dbo].[ky_NETFNConvertDateToUTC](@pdDate, NULL)
          END
      END

    IF RTRIM(ISNULL(@pDate, '')) <> ''
      BEGIN
        SET @pdDueDate = @pdDate
        IF SERVERPROPERTY('EngineEdition') = 5
          BEGIN
            -- if Due Date doesn't contain a Time component do not convert to UTC
            IF @pdDueDate <> CONVERT(DATETIME, CONVERT(VARCHAR, @pdDueDate, 112))
              BEGIN
                --Azure. Local time must be converted to UTC
                SET @pdDueDate = [dbo].[ky_NETFNConvertDateToUTC](@pdDueDate, NULL)
              END
          END
      END

-- Get Next ActionID -----------------------------------------------------------------------------------------------------------------------------------------

    --Moved this to the Insert statement to ensure that the creation of a new
    --Action ID is atomic; even though it would be rare, it is possible
    --for two people to invoke this procedure at the same time, causing the system
    --to try and generate more than one Diary entry for the same ActionID, and this
    --will cause one of these inserts to fail.

    --SELECT @intActionID = (  SELECT Top 1
    --                                ActionID
    --                           FROM Diary
    --                       ORDER BY ActionID DESC)
    --
    --IF @@error <> 0 GOTO Tran_Fail
    --
    --IF @intActionID IS NULL SELECT @intActionID = 0
    --
    --SELECT @intActionID = @intActionID + 1

-- Write Diary Record -----------------------------------------------------------------------------------------------------------------------------------------

    --PINO 2015-09-30
    EXEC @intActionID = [dbo].[ky_NETGetNextActionID]
    --PINO 2015-09-30 End

    INSERT INTO [dbo].[Diary]
               ([Status],
                [CaseCode],
                [ActionCode],
                [ActionStatus],
                [FnCode],
                [TeamCode],
                [Text1],
                [DisplayText],
                [Attachments],
                [EmailAddress],
                [AddressTo],
                [ccTo],
                [bccTo],
                [email],
                [Subject],
                [Publish],
                [ActionType],
                [ProcessType],
                [DYStartTime],
                [ActionId],
                [Date],
                [KYC])
         SELECT @pStatus,
                @pCaseCode,
                @pActionCode,
                @pActionStatus,
                @pFnCode,
                @pTeamCode,
                convert(text, @pText1),
                [dbo].[ky_RemoveSpuriousWhitespace2](ISNULL(CONVERT(VARCHAR(MAX), @pText1), ''), 200),
                @pAttachments,
                @pEmailAddress,
                @pAddressTo,
                @pccTo,
                @pbccTo,
                @pemail,
                @pSubject,
                @pPublish,
                'E',
                @pProcessType,
                @pTime,
                --IsNull(Max(ActionID), 0) + 1,  --Pino 2015-09-30
                @intActionID,                    --Pino 2015-09-30
                @pdDate,
                @KYC

    IF @@error <> 0 GOTO Tran_Fail

     --Pino 2015-09-30 START
    --SELECT @intActionID = MAX(DIA.[ActionID])
    --  FROM [dbo].[Diary] DIA
    -- WHERE RTRIM(DIA.[CaseCode]) = RTRIM(@pCaseCode)
    --   AND DIA.[DATE] = convert(datetime, @pDate)
    --Pino 2015-09-30 END

-- Write the diary delegation record ------------------------------------------------------------

    INSERT INTO [DiaryDelegations]
               ([ACTIONID],
                [HANDLER],
                [TEAM],
                [DATE],
                [TIME],
                [DATER],
                [TIMER],
                [DUEDATE],
                [DUETIME],
                [REVIEW],
                [STATUS],
                [OWNER],
                [DELEGATE],
                [DELEGATESTATUS],
                [ActionType],
                [FromHandler],
                [ReturnedBy],
                [DelType])
         VALUES(@intActionID,
                @pFnCode,
                @pTeamCode,
                @pdDate,
                @pTime,
                0,
                '',
                @pdDueDate,
                @pTime,
                ' ',
                @pStatus,
                'Y',
                @pFnCode,
                0,
                'E',
                ' ',
                ' ',
                'Created')

-- Commit Transaction ------------------------------------------------------------------------------------------------------------------

    COMMIT TRAN Key_Tran_01

    IF @@error <> 0 GOTO Tran_Fail

    Select @pActionId = @intActionID

-- End Of Processing --------------------------------------------------------------------------------------------------------------------

ExitPoint:

    Return

--  Error Processing -------------------------------------------------------------------------------------------------------------------------------------

Tran_Fail:

    rollback tran Key_Tran_01
    Select @pActionId = -1

    goto ExitPoint
END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_CreateDiaryStepWS'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_CreateDiaryStepWS]
  END
GO

CREATE PROCEDURE [dbo].[ky_CreateDiaryStepWS]
                (@pStatus       Int=0,
                 @pDate         varchar(23),
                 @pSentDate     varchar(23),
                 @pCaseCode     char(17),
                 @pActionCode   char(15),
                 @pActionStatus char(3),
                 @pFnCode       char(5),
                 @pTeamCode     char(5),
                 @pText1        varchar(8000),
                 @pAttachments  char(1),
                 @pEmailAddress varchar(150),
                 @pAddressTo    varchar(2000),
                 @pccTo         varchar(2000),
                 @pbccTo        varchar(2000),
                 @pemail        char(1),
                 @pSubject      varchar(500),
                 @pProcessType  char(1),
                 @pTime         char(10),
                 @pActionId     int=0               output)
AS
/************************************************************************
*
* Create Diary Step WS (?)
*
* 2015-09-30 Pino        Use ky_NETGetNextActionID
* 2016-09-13 Pino        KYC - KEYD-4040
* 2019-07-04 Pino        Convert input parameter dates to UTC, but keep
*                        in mind that the Due Date may not contain a
*                        HH:mm:ss part in which case we don't want to
*                        change it.
*
*************************************************************************/
BEGIN
    SET NOCOUNT ON
    DECLARE @strcasecode        nVarChar(17)
    DECLARE @intActionID        Int
    DECLARE @intAssignno        int
    DECLARE @pPublish           char(1)
    DECLARE @KYC                char(1)
    DECLARE @SubjectLen         int
    DECLARE @CaseCodeLen        int
    DECLARE @FELen              int
    DECLARE @TeamLen            int
    DECLARE @dDate              datetime

    SET @dDate = convert(datetime,   convert(varchar(4), year(convert(datetime, @pDate)))
                                   + right(convert(varchar(3), 100 + month(convert(datetime, @pDate))), 2)
                                   + right(convert(varchar(3), 100 + day(convert(datetime, @pDate))), 2))

    select @SubjectLen = sc.[length]
      from sysobjects so
           inner join syscolumns sc
                   on sc.[id] = so.[id]
                  and sc.[name] = 'subject'
     where so.[name] = 'diary'

    IF (@SubjectLen = 16)
      SET @SubjectLen = 2000

    select @CaseCodeLen = sc.[length]
      from sysobjects so
           inner join syscolumns sc
                   on sc.[id] = so.[id]
                  and sc.[name] = 'casecode'
     where so.[name] = 'diary'

    select @FELen = sc.[length]
      from sysobjects so
           inner join syscolumns sc
                   on sc.[id] = so.[id]
                  and sc.[name] = 'fncode'
     where so.[name] = 'diary'

    select @TeamLen = sc.[length]
      from sysobjects so
           inner join syscolumns sc
                   on sc.[id] = so.[id]
                  and sc.[name] = 'teamcode'
     where so.[name] = 'diary'

-- Start Transaction -----------------------------------------------------------------------------------------------------------------------------------------

    BEGIN TRAN Key_Tran_01

    IF @@error <> 0 GOTO Tran_Fail

-- Get CaseCode From Matters Table -----------------------------------------------------------------------------------------------------------------------------------------

    SELECT @strCaseCode = (SELECT Top 1
                                  [Code]
                             From [dbo].[Matters]
                            WHERE [code] = @pCaseCode)

    IF @@error <> 0 GOTO Tran_Fail

    SET @strCaseCode = RTrim(Isnull(@strCaseCode, ''))
    Set @pFNCode = RTrim(@pFNCode)
    Set @pTeamCode = RTrim(@pTeamCode)

    IF LEN(@strCaseCode) < 2 GOTO Tran_Fail
    IF LEN(@strCaseCode) > @CaseCodeLen GOTO Tran_Fail
    IF LEN(@pFNCode) > @FELen GOTO Tran_Fail
    IF LEN(@pTeamCode) > @TeamLen GOTO Tran_Fail

-- Set Publish and Subject -----------------------------------------------------------------------------------------------------------------------------------

    SELECT @pPublish = ISNULL([TAC].[PUBLISHER], 'N'),
           @KYC = ISNULL([TAC].[KYC], 'N')
      FROM [dbo].[TemplateActions] [TAC]
     WHERE [TAC].[ACTIONCODE] = @pActionCode

    IF @pPublish  IS NULL SELECT @pPublish = 'N'

    SET @pSubject = LTrim(RTrim(@pSubject))

    IF LEN(@pSubject) > @SubjectLen
       BEGIN
          SET @pSubject = Substring(@pSubject, 1, @SubjectLen)
       End

-- Get Next ActionID -----------------------------------------------------------------------------------------------------------------------------------------

    --Moved this to the Insert statement to ensure that the creation of a new
    --Action ID is atomic; even though it would be rare, it is possible
    --for two people to invoke this procedure at the same time, causing the system
    --to try and generate more than one Diary entry for the same ActionID, and this
    --will cause one of these inserts to fail.

    --SELECT @intActionID = (  SELECT Top 1
    --                                ActionID
    --                           FROM Diary
    --                       ORDER BY ActionID DESC)
    --
    --IF @@error <> 0 GOTO Tran_Fail
    --
    --IF @intActionID IS NULL SELECT @intActionID = 0
    --
    --SELECT @intActionID = @intActionID + 1

-- Write Diary Record -----------------------------------------------------------------------------------------------------------------------------------------

    --PINO 2015-09-30
    EXEC @intActionID = [dbo].[ky_NETGetNextActionID]
    --PINO 2015-09-30 End

    DECLARE @pdDate     DATETIME
    DECLARE @pdSentDate DATETIME
    DECLARE @pdDueDate  DATETIME
    IF RTRIM(ISNULL(@pDate, '')) <> ''
      BEGIN
        SET @pdDate = convert(DATETIME, @pDate)
        IF SERVERPROPERTY('EngineEdition') = 5
          BEGIN
            --Azure. Local time must be converted to UTC
            SET @pdDate = [dbo].[ky_NETFNConvertDateToUTC](@pdDate, NULL)
          END
      END

    IF RTRIM(ISNULL(@pDate, '')) <> ''
      BEGIN
        SET @pdDueDate = @pdDate
        IF SERVERPROPERTY('EngineEdition') = 5
          BEGIN
            -- if Due Date doesn't contain a Time component do not convert to UTC
            IF @pdDueDate <> CONVERT(DATETIME, CONVERT(VARCHAR, @pdDueDate, 112))
              BEGIN
                --Azure. Local time must be converted to UTC
                SET @pdDueDate = [dbo].[ky_NETFNConvertDateToUTC](@pdDueDate, NULL)
              END
          END
      END

    IF RTRIM(ISNULL(@pSentDate, '')) <> ''
      BEGIN
        SET @pdSentDate = convert(DATETIME, @pSentDate)
        IF SERVERPROPERTY('EngineEdition') = 5
          BEGIN
            --Azure. Local time must be converted to UTC
            SET @pdSentDate = [dbo].[ky_NETFNConvertDateToUTC](@pdSentDate, NULL)
          END
      END

    INSERT INTO [dbo].[Diary]
               ([Status],
                [CaseCode],
                [ActionCode],
                [ActionStatus],
                [FnCode],
                [TeamCode],
                [Text1],
                [DisplayText],
                [Attachments],
                [EmailAddress],
                [AddressTo],
                [ccTo],
                [bccTo],
                [email],
                [Subject],
                [Publish],
                [ActionType],
                [ProcessType],
                [DYStartTime],
                [ActionId],
                [Date],
                [TxmDate],
                [KYC])
         SELECT @pStatus,
                @pCaseCode,
                @pActionCode,
                @pActionStatus,
                @pFnCode,
                @pTeamCode,
                convert(text, @pText1),
                [dbo].[ky_RemoveSpuriousWhitespace2](ISNULL(CONVERT(VARCHAR(MAX), @pText1), ''), 200),
                @pAttachments,
                @pEmailAddress,
                @pAddressTo,
                @pccTo,
                @pbccTo,
                @pemail,
                @pSubject,
                @pPublish,
                'E',
                @pProcessType,
                @pTime,
                --IsNull(Max(ActionID), 0) + 1,  --Pino 2015-09-30
                @intActionID,                    --Pino 2015-09-30
                @pdDate,
                @pdSentDate,
                @KYC

    IF @@error <> 0 GOTO Tran_Fail

     --Pino 2015-09-30 START
    --SELECT @intActionID = MAX(DIA.[ActionID])
    --  FROM [dbo].[Diary] DIA
    -- WHERE RTRIM(DIA.[CaseCode]) = RTRIM(@pCaseCode)
    --   AND DIA.[DATE] = convert(datetime, @pDate)
    --Pino 2015-09-30 END

-- Write the diary delegation record ------------------------------------------------------------

    INSERT INTO [DiaryDelegations]
               ([ACTIONID],
                [HANDLER],
                [TEAM],
                [DATE],
                [TIME],
                [DATER],
                [TIMER],
                [DUEDATE],
                [DUETIME],
                [REVIEW],
                [STATUS],
                [OWNER],
                [DELEGATE],
                [DELEGATESTATUS],
                [ActionType],
                [FromHandler],
                [ReturnedBy],
                [DelType])
         VALUES(@intActionID,
                @pFnCode,
                @pTeamCode,
                @dDate,
                @pTime,
                0,
                '',
                @pdDueDate,
                @pTime,
                ' ',
                @pStatus,
                'Y',
                @pFnCode,
                0,
                'E',
                ' ',
                ' ',
                'Created')

-- Commit Transaction ------------------------------------------------------------------------------------------------------------------

    COMMIT TRAN Key_Tran_01

    IF @@error <> 0 GOTO Tran_Fail

    Select @pActionId = @intActionID

-- End Of Processing --------------------------------------------------------------------------------------------------------------------

ExitPoint:

    Return

--  Error Processing -------------------------------------------------------------------------------------------------------------------------------------

Tran_Fail:

    rollback tran Key_Tran_01
    Select @pActionId = -1

    GoTo ExitPoint

SET ANSI_NULLS ON
END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPCreateDiaryStepUTC'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPCreateDiaryStepUTC]
  END
GO

CREATE PROCEDURE [dbo].[ky_NETSPCreateDiaryStepUTC]
                (@pStatus       Int=0,
                 @pDate         DATETIME,
                 @pDueDate      DATETIME,
                 @pSentDate     DATETIME,
                 @pCaseCode     VARCHAR(20),
                 @pActionCode   CHAR(15),
                 @pActionType   CHAR(1),
                 @pActionStatus CHAR(3),
                 @pPriority     CHAR(1),
                 @pHighlighted  CHAR(1),
                 @pBillable     INT,
                 @pWorkProcess  INT,
                 @pPublish      CHAR(1),
                 @pFnCode       VARCHAR(10),
                 @pTeamCode     VARCHAR(10),
                 @pText1        VARCHAR(MAX),
                 @pAttachments  CHAR(1),
                 @pEmailAddress VARCHAR(MAX),
                 @pAddressTo    VARCHAR(MAX),
                 @pccTo         VARCHAR(MAX),
                 @pbccTo        VARCHAR(MAX),
                 @pemail        CHAR(1),
                 @pSubject      VARCHAR(MAX),
                 @pProcessType  CHAR(1),
                 @pLocation     VARCHAR(50),
                 @pDuration     INT,
                 @bAlreadyUTC   BIT)
AS
/************************************************************************
*
* Create Diary Step .NET version
*    Suitable for Add Task
*    AS well AS assigning a mail to Case, and applying a duration.
*    modifications: 2015-02-11 change NVARCHAR to VARCHAR
*                   2015-06-23 Sridharen    KEYD - 2332 - Use of LTRIM(RTRIM(
*                   2015-09-30 Pino         Use ky_NETGetNextActionID
*                   2016-09-13 Pino         KYC - KEYD-4040
*                   2018-08-08 Pino         TEXT1 is now VARCHAR(MAX)
*                   2019-07-04 Pino         Convert input parameter dates to UTC, but keep
*                                           in mind that the Due Date may not contain a
*                                           HH:mm:ss part in which case we don't want to
*                                           change it.
*
*************************************************************************/
BEGIN
    SET NOCOUNT ON
    DECLARE @strcasecode        VarChar(20)
    DECLARE @intAssignno        INT
    DECLARE @SubjectLen         INT
    DECLARE @CaseCodeLen        INT
    DECLARE @FELen              INT
    DECLARE @TeamLen            INT
    DECLARE @dDate              DATETIME
    DECLARE @pTime              CHAR(10)
    DECLARE @pEndTime           CHAR(10)
    DECLARE @dDueDate           DATETIME
    DECLARE @pDueTime           CHAR(10)
    DECLARE @continue           INT
    DECLARE @error              VARCHAR(1000)
    DECLARE @NewActionID        INT
    DECLARE @KYC                char(1)

    SET @bAlreadyUTC = ISNULL(@bAlreadyUTC, 0)

    SET @continue = 0
    SET @error = ''
    SET @NewActionID = 0

    BEGIN Try
      SET @pDuration = ISNULL(@pDuration, 0)
      IF (@pDuration <= 0)
        BEGIN
          SET @pDuration = 30
        END
        SET @pDuration = @pDuration * 60
    End Try
    BEGIN Catch
      SET @continue = 1
      SET @error = 'invalid duration information'
      SET @NewActionID = 0
    End Catch

    IF (@continue = 0)
      BEGIN
        BEGIN TRY
          -- Get only the DATE portion of the date that was passed in
          IF (@pDate IS NULL)
            BEGIN
              SET @pDate = GETDATE()
            END
          ELSE
            BEGIN
              IF @bAlreadyUTC = 0
                BEGIN
                  IF SERVERPROPERTY('EngineEdition') = 5
                    BEGIN
                      --Azure. Local time must be converted to UTC
                      SET @pDate = [dbo].[ky_NETFNConvertDateToUTC](@pDate, NULL)
                    END
                END
            END

          IF @pDueDate IS NOT NULL
            BEGIN
              IF @bAlreadyUTC = 0
                BEGIN
                  IF SERVERPROPERTY('EngineEdition') = 5
                    BEGIN
                      -- if Due Date doesn't contain a Time component do not convert to UTC
                      IF @pDueDate <> CONVERT(DATETIME, CONVERT(VARCHAR, @pDueDate, 112))
                        BEGIN
                          --Azure. Local time must be converted to UTC
                          SET @pDueDate = [dbo].[ky_NETFNConvertDateToUTC](@pDueDate, NULL)
                        END
                    END
                END
            END

          IF @pSentDate IS NOT NULL
            BEGIN
              IF @bAlreadyUTC = 0
                BEGIN
                  IF SERVERPROPERTY('EngineEdition') = 5
                    BEGIN
                      --Azure. Local time must be converted to UTC
                      SET @pSentDate = [dbo].[ky_NETFNConvertDateToUTC](@pSentDate, NULL)
                    END
                END
            END

          IF (@pDueDate IS NULL)
            SET @pDueDate = @pDate

          SET @dDate = convert(DATETIME, convert(VARCHAR, @pDate, 112))
          SET @pTime = convert(CHAR(10), (1000 * convert(INT, DATEDIFF(ms, @dDate, @pDate) / 10000)) + 1)
          SET @pEndTime = convert(CHAR(10), (@pDuration * 100) + (1000 * convert(INT, DATEDIFF(ms, @dDate, @pDate) / 10000)) + 1)

          SET @dDueDate = convert(DATETIME, convert(VARCHAR, @pDueDate, 112))
          SET @pDueTime = convert(CHAR(10), @pDuration + (10 * convert(INT, DATEDIFF(ms, @dDueDate, @pDueDate) / 10000)))
        END TRY
        BEGIN CATCH
          SET @continue = 2
          SET @error = 'error processing date parameters'
          SET @NewActionID = 0
        END CATCH
      END

    IF (@continue = 0)
      BEGIN
        BEGIN TRY
          SELECT @SubjectLen = CASE WHEN sc.xtype = 35
                                         THEN 150
                                    ELSE sc.[length] END
            FROM sysobjects so
                 INNER JOIN syscolumns sc
                         on sc.[id] = so.[id]
                        AND sc.[name] = 'subject'
           WHERE so.[name] = 'diary'

          SELECT @CaseCodeLen = sc.[length]
            FROM sysobjects so
                 INNER JOIN syscolumns sc
                         on sc.[id] = so.[id]
                        AND sc.[name] = 'casecode'
           WHERE so.[name] = 'diary'

          SELECT @FELen = sc.[length]
            FROM sysobjects so
                 INNER JOIN syscolumns sc
                         on sc.[id] = so.[id]
                        AND sc.[name] = 'fncode'
           WHERE so.[name] = 'diary'

          SELECT @TeamLen = sc.[length]
            FROM sysobjects so
                 INNER JOIN syscolumns sc
                         on sc.[id] = so.[id]
                        AND sc.[name] = 'teamcode'
           WHERE so.[name] = 'diary'
        END TRY
        BEGIN CATCH
          SET @continue = 3
          SET @error = 'invalid duration information'
          SET @NewActionID = 0
        END CATCH
      END

-- Get CaseCode From Matters Table -----------------------------------------------------------------------------------------------------------------------------------------

    IF (@continue = 0)
      BEGIN
        BEGIN TRY
          SET @strCaseCode = NULL

          SELECT Top 1
                 @strCaseCode = MAT.[Code]
            From [dbo].[Matters] MAT
           WHERE MAT.[code] = @pCaseCode

          SET @strCaseCode = RTRIM(ISNULL(@strCaseCode, ''))
          Set @pFNCode = RTRIM(@pFNCode)
          Set @pTeamCode = RTRIM(@pTeamCode)

          IF (@strCaseCode = '')
            BEGIN
              SET @continue = 4
              BEGIN TRY
                SET @pCaseCode = RTRIM(ISNULL(@pCaseCode, ''))
              END TRY
              BEGIN CATCH
                SET @pCaseCode = ''
              END CATCH
              SET @error = 'Invalid Case Code: ' + @pCaseCode
              SET @NewActionID = 0
            END

          IF (@continue = 0)
            BEGIN
              IF (LEN(@strCaseCode) < 2)
                BEGIN
                  SET @continue = 4
                  SET @error = 'Invalid parameters. Case Code too short'
                  SET @NewActionID = 0
                END
            END

          IF (@continue = 0)
            BEGIN
              IF (LEN(@strCaseCode) > @CaseCodeLen)
                BEGIN
                  SET @continue = 4
                  SET @error = 'Invalid parameters. Case Code too long'
                  SET @NewActionID = 0
                END
            END

          IF (@continue = 0)
            BEGIN
              IF (LEN(@pFNCode) > @FELen)
                BEGIN
                  SET @continue = 4
                  SET @error = 'Invalid parameters. FE code too long'
                  SET @NewActionID = 0
                END
            END

          IF (@continue = 0)
            BEGIN
              IF (LEN(@pTeamCode) > @TeamLen)
                BEGIN
                  SET @continue = 4
                  SET @error = 'Invalid parameters. TEAM code too long'
                  SET @NewActionID = 0
                END
            END

          -- Set Publish AND Subject -----------------------------------------------------------------------------------------------------------------------------------
          IF (@continue = 0)
            BEGIN
              SELECT @pPublish = CASE WHEN RTRIM(ISNULL(@pPublish, '')) <> ''
                                           THEN @pPublish
                                      ELSE ISNULL(TAC.[PUBLISHER], 'N') END,
                     @KYC = ISNULL([TAC].[KYC], 'N'),
                     @pProcessType = CASE WHEN RTRIM(ISNULL(@pProcessType, '')) <> ''
                                               THEN @pProcessType
                                          ELSE TAC.[PROCESSTYPE] END
                From TemplateActions TAC
               WHERE TAC.[ACTIONCODE] = @pActionCode

              IF (RTRIM(IsNull(@pPublish, '')) = '')
                BEGIN
                  SET @pPublish = 'N'
                END

              IF (RTRIM(IsNull(@pProcessType, '')) = '')
                BEGIN
                  SET @pProcessType = ''
                END

              SET @pSubject = RTRIM(@pSubject)

              IF LEN(@pSubject) > @SubjectLen
                BEGIN
                  SET @pSubject = Substring(@pSubject, 1, @SubjectLen)
                END
            END

          -- Truncate potentially long fields -----------------------------------------------------------------------------------------------------------------------------------
          IF (@continue = 0)
            BEGIN
              SET @pEmailAddress = CONVERT(VARCHAR(150), ISNULL(@pEmailAddress, ''))
              SET @pSubject = CONVERT(VARCHAR(500), ISNULL(@pSubject, ''))
              SET @pAddressTo = CONVERT(VARCHAR(2000), ISNULL(@pAddressTo, ''))
              SET @pccTo = CONVERT(VARCHAR(2000), ISNULL(@pccTo, ''))
              SET @pbccTo = CONVERT(VARCHAR(2000), ISNULL(@pbccTo, ''))
            END
        END TRY
        BEGIN CATCH
          SET @continue = 5
          SET @error = 'Invalid parameters Publish, ProcessType or Subject'
          SET @NewActionID = 0
        END CATCH

      END

    IF (@continue = 0)
      BEGIN
        BEGIN TRY
          -- Pino 2015-09-30 Start
          EXEC @NewActionID = [dbo].[ky_NETGetNextActionID]
          --SELECT @NewActionID = ISNULL(MAX(DIA.[ActionID]), 0) + 1
          --  FROM [dbo].[diary] DIA
          -- Pino 2015-09-30 End
        END TRY
        BEGIN CATCH
          SET @continue = 8
          SET @error = convert(VARCHAR(1000), 'Error establishing new Action ID, ' + Error_Message())
          SET @NewActionID = 0
        END CATCH
      END

-- Write Diary Record -----------------------------------------------------------------------------------------------------------------------------------------
    IF (@continue = 0)
      BEGIN
        BEGIN TRY
          INSERT INTO [dbo].[Diary]
                     ([ActionID],
                      [Status],
                      [CaseCode],
                      [ActionCode],
                      [ActionStatus],
                      [PRIORITY],
                      [HIGHLIGHTED],
                      [BILLABLE],
                      [WORKPROCESS],
                      [FnCode],
                      [TeamCode],
                      [Text1],
                      [Attachments],
                      [EmailAddress],
                      [AddressTo],
                      [Location],
                      [ccTo],
                      [bccTo],
                      [email],
                      [Subject],
                      [Publish],
                      [ActionType],
                      [ProcessType],
                      [DYSTARTTIME],
                      [DYENDTIME],
                      [Date],
                      [DUEDATE],
                      [TxmDate],
                      [DisplayText],
                      [KYC])
               VALUES(@NewActionID,
                      @pStatus,
                      @pCaseCode,
                      @pActionCode,
                      @pActionStatus,
                      @pPriority,
                      @pHighlighted,
                      @pBillable,
                      @pWorkProcess,
                      @pFnCode,
                      @pTeamCode,
                      @pText1,
                      @pAttachments,
                      @pEmailAddress,
                      @pAddressTo,
                      @pLocation,
                      @pccTo,
                      @pbccTo,
                      @pemail,
                      @pSubject,
                      @pPublish,
                      @pActionType,
                      @pProcessType,
                      @pTime,
                      @pEndTime,
                      @pDate,
                      @pDueDate,
                      @pSentDate,
                      CONVERT(VARCHAR(200), [dbo].[ky_RemoveSpuriousWhitespace2](ISNULL(@pTEXT1, ''), 200)),
                      @KYC)
        END TRY
        BEGIN CATCH
          SET @continue = 6
          SET @error = convert(VARCHAR(1000), 'Error inserting Diary Entry, ' + Error_Message())
          SET @NewActionID = 0
        END CATCH
      END

-- Write the diary delegation record ------------------------------------------------------------
    IF (@continue = 0)
      BEGIN
        BEGIN TRY
          INSERT INTO [DiaryDelegations]
                     ([ACTIONID],
                      [HANDLER],
                      [TEAM],
                      [DATE],
                      [TIME],
                      [DATER],
                      [TIMER],
                      [DUEDATE],
                      [DUETIME],
                      [REVIEW],
                      [STATUS],
                      [OWNER],
                      [DELEGATE],
                      [DELEGATESTATUS],
                      [ActionType],
                      [FromHandler],
                      [ReturnedBy],
                      [DelType])
               VALUES(@NewActionID,
                      @pFnCode,
                      @pTeamCode,
                      @dDate,
                      @pTime,
                      @dDate,
                      @pTime,
                      @dDueDate,
                      @pDueTime,
                      ' ',
                      @pStatus,
                      'Y',
                      @pFnCode,
                      0,
                      @pActionType,
                      ' ',
                      ' ',
                      'Created')
        END TRY
        BEGIN CATCH
          SET @continue = 7
          SET @error = convert(VARCHAR(1000), 'Error inserting Diary Delegation, ' + Error_Message())
          SET @NewActionID = 0
        END CATCH
      END

    IF (@continue <> 0)
      BEGIN
        IF (@NewActionID <> 0)
          BEGIN
            BEGIN TRY
              -- Instead of a "Rollback"
              DELETE [dbo].[DiaryDelegations] WHERE [ACTIONID] = @NewActionID

              DELETE [dbo].[diary] WHERE [ACTIONID] = @NewActionID

              SET @NewActionID = 0
            END TRY
            BEGIN CATCH

            END CATCH
          END
      END

    SELECT @continue AS [ErrorCode],
           @error AS [ErrorMessage],
           @NewActionID AS [NewActionID]

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPCreateDiaryStep'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPCreateDiaryStep]
  END
GO

CREATE PROCEDURE [dbo].[ky_NETSPCreateDiaryStep]
                (@pStatus       Int=0,
                 @pDate         DATETIME,
                 @pDueDate      DATETIME,
                 @pSentDate     DATETIME,
                 @pCaseCode     VARCHAR(20),
                 @pActionCode   CHAR(15),
                 @pActionType   CHAR(1),
                 @pActionStatus CHAR(3),
                 @pPriority     CHAR(1),
                 @pHighlighted  CHAR(1),
                 @pBillable     INT,
                 @pWorkProcess  INT,
                 @pPublish      CHAR(1),
                 @pFnCode       VARCHAR(10),
                 @pTeamCode     VARCHAR(10),
                 @pText1        VARCHAR(MAX),
                 @pAttachments  CHAR(1),
                 @pEmailAddress VARCHAR(MAX),
                 @pAddressTo    VARCHAR(MAX),
                 @pccTo         VARCHAR(MAX),
                 @pbccTo        VARCHAR(MAX),
                 @pemail        CHAR(1),
                 @pSubject      VARCHAR(MAX),
                 @pProcessType  CHAR(1),
                 @pLocation     VARCHAR(50),
                 @pDuration     INT)
AS
/************************************************************************
*
* Create Diary Step .NET version
*    Suitable for Add Task
*    AS well AS assigning a mail to Case, and applying a duration.
*    modifications: 2015-02-11 change NVARCHAR to VARCHAR
*                   2015-06-23 Sridharen    KEYD - 2332 - Use of LTRIM(RTRIM(
*                   2015-09-30 Pino         Use ky_NETGetNextActionID
*                   2016-09-13 Pino         KYC - KEYD-4040
*                   2018-08-08 Pino         TEXT1 is now VARCHAR(MAX)
*                   2019-07-04 Pino         Convert input parameter dates to UTC, but keep
*                                           in mind that the Due Date may not contain a
*                                           HH:mm:ss part in which case we don't want to
*                                           change it.
*                   2019-07-04 Pino         Use UTC version
*
*************************************************************************/
BEGIN

EXEC [dbo].[ky_NETSPCreateDiaryStepUTC] @pStatus,
                                        @pDate,
                                        @pDueDate,
                                        @pSentDate,
                                        @pCaseCode,
                                        @pActionCode,
                                        @pActionType,
                                        @pActionStatus,
                                        @pPriority,
                                        @pHighlighted,
                                        @pBillable,
                                        @pWorkProcess,
                                        @pPublish,
                                        @pFnCode,
                                        @pTeamCode,
                                        @pText1,
                                        @pAttachments,
                                        @pEmailAddress,
                                        @pAddressTo,
                                        @pccTo,
                                        @pbccTo,
                                        @pemail,
                                        @pSubject,
                                        @pProcessType,
                                        @pLocation,
                                        @pDuration,
                                        0
END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPCreateTemplatedDiaryStepUTC'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPCreateTemplatedDiaryStepUTC]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETSPCreateTemplatedDiaryStepUTC]
      (@pStatus       INT,
       @pDate         DATETIME,
       @pDueDate      DATETIME,
       @pSentDate     DATETIME,
       @pCaseCode     VARCHAR(20),
       @pActionCode   VARCHAR(20),
       @pFNCode       VARCHAR(10),
       @pTEAMCode     VARCHAR(10),
       @pTEXT         VARCHAR(MAX),
       @pAttachments  CHAR(1),
       @pEmailAddress VARCHAR(MAX),
       @pAddressTo    VARCHAR(MAX),
       @pccTo         VARCHAR(MAX),
       @pbccTo        VARCHAR(MAX),
       @pemail        CHAR(1),
       @pSubject      VARCHAR(MAX),
       @pLocation     VARCHAR(50),
       @pDuration     INT,
       @bAlreadyUTC   BIT)
AS
/************************************************************************
*
* ky_NETSPCreateTemplatedDiaryStepUTC
*
* Create Diary Step .NET version
*    Suitable for Add Task
*    AS well AS assigning a mail to Case, and applying a duration.
*
*    Uses the template specified through the ACTIONCODE
*
*    2015-04-29   Pino    If ActionCode does not exist, default values.
*    2015-09-30   Pino    Use ky_NETGetNextActionID
*    2016-09-13   Pino    KYC - KEYD-4040
*
*************************************************************************/
BEGIN

DECLARE @pActionType   CHAR(1)
DECLARE @pActionStatus CHAR(3)
DECLARE @pPriority     CHAR(1)
DECLARE @pHighlighted  CHAR(1)
DECLARE @pBillable     INT
DECLARE @pWorkProcess  INT
DECLARE @pPublish      CHAR(1)
DECLARE @KYC           char(1)
DECLARE @pProcessType  CHAR(1)

IF EXISTS
  (SELECT TOP 1 1
     FROM [dbo].[TemplateActions] [TAC]
    WHERE [TAC].[ACTIONCODE] = RTRIM(ISNULL(@pActionCode, '')))
  BEGIN
    SELECT @pActionType = ISNULL([TAC].[ACTTYPE], 'A'),
           @pActionStatus = ISNULL([TAC].[STATUS], 'NA'),
           @pPriority = ISNULL(CONVERT(VARCHAR(1), [TAC].[PRIORITY]), 'N'),
           @pHighlighted = CASE WHEN ISNULL([TAC].[HIGHLIGHTED], 'N') = '' THEN 'N' ELSE ISNULL([TAC].[HIGHLIGHTED], 'N') END,
           @pBillable = CASE WHEN ISNULL([TAC].[BILLABLE], 'N') = 'Y'
                                  THEN 1
                             ELSE 0 END,
           @pWorkProcess = 0,
           @pPublish = ISNULL([TAC].[PUBLISHER], 'N'),
           @KYC = ISNULL([TAC].[KYC], 'N'),
           @pProcessType = [TAC].[PROCESSTYPE]
      FROM [dbo].[TemplateActions] [TAC]
     WHERE [TAC].[ACTIONCODE] = @pActionCode
  END
ELSE
  BEGIN
    SET @pActionType = 'A'
    SET @pActionStatus = 'NA'
    SET @pPriority = 'Normal'
    SET @pHighlighted = ''
    SET @pBillable = 0
    SET @pWorkProcess = 0
    SET @pPublish = 'N'
    SET @KYC = 'N'
    SET @pProcessType = 'I'
  END

EXEC [dbo].[ky_NETSPCreateDiaryStepUTC] @pStatus,
                                        @pDate,
                                        @pDueDate,
                                        @pSentDate,
                                        @pCaseCode,
                                        @pActionCode,
                                        @pActionType,
                                        @pActionStatus,
                                        @pPriority,
                                        @pHighlighted,
                                        @pBillable,
                                        @pWorkProcess,
                                        @pPublish,
                                        @pFnCode,
                                        @pTeamCode,
                                        @pText,
                                        @pAttachments,
                                        @pEmailAddress,
                                        @pAddressTo,
                                        @pccTo,
                                        @pbccTo,
                                        @pemail,
                                        @pSubject,
                                        @pProcessType,
                                        @pLocation,
                                        @pDuration,
                                        @bAlreadyUTC

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = 'ky_NETSPCreateTemplatedDiaryStep'
      AND SO.[type] = 'P')
BEGIN
  DROP PROCEDURE [dbo].[ky_NETSPCreateTemplatedDiaryStep]
END
GO

CREATE PROCEDURE
       [dbo].[ky_NETSPCreateTemplatedDiaryStep]
      (@pStatus       INT,
       @pDate         DATETIME,
       @pDueDate      DATETIME,
       @pSentDate     DATETIME,
       @pCaseCode     VARCHAR(20),
       @pActionCode   VARCHAR(20),
       @pFNCode       VARCHAR(10),
       @pTEAMCode     VARCHAR(10),
       @pTEXT         VARCHAR(MAX),
       @pAttachments  CHAR(1),
       @pEmailAddress VARCHAR(MAX),
       @pAddressTo    VARCHAR(MAX),
       @pccTo         VARCHAR(MAX),
       @pbccTo        VARCHAR(MAX),
       @pemail        CHAR(1),
       @pSubject      VARCHAR(MAX),
       @pLocation     VARCHAR(50),
       @pDuration     INT)
AS
/************************************************************************
*
* ky_NETSPCreateTemplatedDiaryStep
* 
* Create Diary Step .NET version
*    Suitable for Add Task
*    AS well AS assigning a mail to Case, and applying a duration. 
*
*    Uses the template specified through the ACTIONCODE
*
*    2015-04-29   Pino    If ActionCode does not exist, default values.
*    2015-09-30   Pino    Use ky_NETGetNextActionID
*    2016-09-13   Pino    KYC - KEYD-4040
*
*************************************************************************/
BEGIN

EXEC [dbo].[ky_NETSPCreateTemplatedDiaryStepUTC] @pStatus,
                                                 @pDate,
                                                 @pDueDate,
                                                 @pSentDate,
                                                 @pCaseCode,
                                                 @pActionCode,
                                                 @pFNCode,
                                                 @pTEAMCode,
                                                 @pTEXT,
                                                 @pAttachments,
                                                 @pEmailAddress,
                                                 @pAddressTo,
                                                 @pccTo,
                                                 @pbccTo,
                                                 @pemail,
                                                 @pSubject,
                                                 @pLocation,
                                                 @pDuration,
                                                 0

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPCreateDefaultDiaryStepUTC'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPCreateDefaultDiaryStepUTC]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETSPCreateDefaultDiaryStepUTC]
      (@pStatus       INT,
       @pDate         DATETIME,
       @pDueDate      DATETIME,
       @pSentDate     DATETIME,
       @pCaseCode     VARCHAR(20),
       @pFNCode       VARCHAR(10),
       @pTEAMCode     VARCHAR(10),
       @pTEXT         VARCHAR(MAX),
       @pAttachments  CHAR(1),
       @pEmailAddress VARCHAR(MAX),
       @pAddressTo    VARCHAR(MAX),
       @pccTo         VARCHAR(MAX),
       @pbccTo        VARCHAR(MAX),
       @pemail        CHAR(1),
       @pSubject      VARCHAR(MAX),
       @pLocation     VARCHAR(50),
       @pDuration     INT,
       @bAlreadyUTC   BIT)
AS
/************************************************************************
*
* ky_NETSPCreateDefaultDiaryStepUTC
* 
* Create Default Diary Step .NET version
*
*    2015-09-30   Pino    Use ky_NETGetNextActionID
*    2016-09-13   Pino    KYC - KEYD-4040
*    2019-07-04   Pino    UTC version
*
*************************************************************************/
BEGIN
  DECLARE @pActionCode VARCHAR(20)

  SELECT TOP 1 @pActionCode = TA.[ACTIONCODE]
    FROM casemaster CSM
         LEFT OUTER JOIN [dbo].[Templates] TMP
                      ON TMP.[WKCODE] = CSM.[CSWKTCODE]
         LEFT OUTER JOIN [dbo].[TemplateActions] TA
                      ON (    TA.[WKTCODE] = TMP.[WKCODE]
                          AND TA.[ACTIONCATEGORY] = 'A')
                      OR TA.[ACTIONCODE] = 'ADMKD'
   WHERE CSM.[CSCODE] = @pCaseCode
ORDER BY CASE WHEN (ISNULL(TA.[WKTCODE], '') = TA.[WKTCODE]) AND (TA.[ACTTYPE] = 'A')
                   THEN 'A'
              WHEN TA.[ACTIONCODE] = 'ADMKD'
                   THEN 'B'
              WHEN (ISNULL(TA.[WKTCODE], '') = TA.[WKTCODE])
                   THEN 'C'
              ELSE 'D' END

IF (@pActionCode IS NOT NULL)
  BEGIN
    EXEC [dbo].[ky_NETSPCreateTemplatedDiaryStepUTC] @pStatus,
                                                     @pDate,
                                                     @pDueDate,
                                                     @pSentDate,
                                                     @pCaseCode,
                                                     @pActionCode,
                                                     @pFNCode,
                                                     @pTEAMCode,
                                                     @pTEXT,
                                                     @pAttachments,
                                                     @pEmailAddress,
                                                     @pAddressTo,
                                                     @pccTo,
                                                     @pbccTo,
                                                     @pemail,
                                                     @pSubject,
                                                     @pLocation,
                                                     @pDuration,
                                                     @bAlreadyUTC
  END

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPCreateDefaultDiaryStep'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPCreateDefaultDiaryStep]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETSPCreateDefaultDiaryStep]
      (@pStatus       INT,
       @pDate         DATETIME,
       @pDueDate      DATETIME,
       @pSentDate     DATETIME,
       @pCaseCode     VARCHAR(20),
       @pFNCode       VARCHAR(10),
       @pTEAMCode     VARCHAR(10),
       @pTEXT         VARCHAR(MAX),
       @pAttachments  CHAR(1),
       @pEmailAddress VARCHAR(MAX),
       @pAddressTo    VARCHAR(MAX),
       @pccTo         VARCHAR(MAX),
       @pbccTo        VARCHAR(MAX),
       @pemail        CHAR(1),
       @pSubject      VARCHAR(MAX),
       @pLocation     VARCHAR(50),
       @pDuration     INT)
AS
/************************************************************************
*
* ky_NETSPCreateDefaultDiaryStep
* 
* Create Default Diary Step .NET version
*
*    2015-09-30   Pino    Use ky_NETGetNextActionID
*    2016-09-13   Pino    KYC - KEYD-4040
*    2019-07-04   Pino    Call UTC version
*
*************************************************************************/
BEGIN

EXEC [dbo].[ky_NETSPCreateDefaultDiaryStepUTC] @pStatus,
                                               @pDate,
                                               @pDueDate,
                                               @pSentDate,
                                               @pCaseCode,
                                               @pFNCode,
                                               @pTEAMCode,
                                               @pTEXT,
                                               @pAttachments,
                                               @pEmailAddress,
                                               @pAddressTo,
                                               @pccTo,
                                               @pbccTo,
                                               @pemail,
                                               @pSubject,
                                               @pLocation,
                                               @pDuration,
                                               0
END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPCopyDiaryStep'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPCopyDiaryStep]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETSPCopyDiaryStep]
      (@pCaseCode    VARCHAR(20),
       @ActionID     INT,
       @pFNCode      VARCHAR(10),
       @pTEAMCode    VARCHAR(10))
AS
/********************************************************************************
 *
 * ky_NETSPCopyDiaryStep
 * Copies a Diary entry.
 * Note: 2014-06-19 As this procedure calls ky_NETSPCreateTemplatedDiaryStep,
 *                  it doesn't need to worry about the new DisplayText column
 *
 *       2015-04-29 Pino       Allow for non-existing Action Codes,
 *                             ky_NETSPCreateTemplatedDiaryStep has been modified
 *                             to default values in that case.
 *       2015-09-30 Pino       Use ky_NETGetNextActionID
 *       2016-09-13 Pino       KYC - KEYD-4040
 *       2019-07-04 Pino       Use UTC version with AlreadyUTC = 1 as these data
 *                             are retrieved from the database.
 *
 ********************************************************************************/
BEGIN
  DECLARE @pActionCode VARCHAR(20)

  DECLARE @pStatus       INT
  DECLARE @pDate         DATETIME
  DECLARE @pDueDate      DATETIME
  DECLARE @pSentDate     DATETIME
  DECLARE @pTEXT         VARCHAR(MAX)
  DECLARE @pAttachments  CHAR(1)
  DECLARE @pEmailAddress VARCHAR(MAX)
  DECLARE @pAddressTo    VARCHAR(MAX)
  DECLARE @pccTo         VARCHAR(MAX)
  DECLARE @pbccTo        VARCHAR(MAX)
  DECLARE @pemail        CHAR(1)
  DECLARE @pSubject      VARCHAR(MAX)
  DECLARE @pLocation     VARCHAR(50)
  DECLARE @pDuration     INT

  SELECT @pActionCode = DIA.[ACTIONCODE],
         @pStatus = 1,
         @pDate = GETDATE(),
         @pDueDate = GETDATE(),
         @pSentDate = DIA.[TxmDate],
         @pTEXT = CONVERT(VARCHAR(MAX), DIA.[TEXT1]),
         @pAttachments = DIA.[ATTACHMENTS],
         @pEmailAddress = CONVERT(VARCHAR(MAX), DIA.[EMAILADDRESS]),
         @pAddressTo = CONVERT(VARCHAR(MAX), DIA.[ADDRESSTO]),
         @pccTo = CONVERT(VARCHAR(MAX), DIA.[CCTO]),
         @pbccTo = CONVERT(VARCHAR(MAX), DIA.[BCCTO]),
         @pEMAIL = DIA.[EMAIL],
         @pSubject = CONVERT(VARCHAR(MAX), DIA.[SUBJECT]),
         @pLocation = DIA.[Location],
         @pDuration = DIA.[DURATION]
    FROM [dbo].[Diary] DIA
   WHERE DIA.[ACTIONID] = @ActionID

   SET @pActionCode = RTRIM(ISNULL(@pActionCode, ''))

  --IF (@pActionCode IS NOT NULL)
  --  BEGIN
    EXEC [dbo].[ky_NETSPCreateTemplatedDiaryStepUTC] @pStatus,
                                                     @pDate,
                                                     @pDueDate,
                                                     @pSentDate,
                                                     @pCaseCode,
                                                     @pActionCode,
                                                     @pFNCode,
                                                     @pTEAMCode,
                                                     @pTEXT,
                                                     @pAttachments,
                                                     @pEmailAddress,
                                                     @pAddressTo,
                                                     @pccTo,
                                                     @pbccTo,
                                                     @pemail,
                                                     @pSubject,
                                                     @pLocation,
                                                     @pDuration,
                                                     1
    --END

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_WDCreateDiaryStep2'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_WDCreateDiaryStep2]
  END
GO

CREATE PROCEDURE [dbo].[ky_WDCreateDiaryStep2]
                (@pStatus       Int=0,
                 @pDate         datetime,
                 @pDueDate      datetime,
                 @pSentDate     datetime,
                 @pCaseCode     varchar(20),
                 @pActionCode   char(15),
                 @pActionType   char(1),
                 @pActionStatus char(3),
                 @pPriority     char(1),
                 @pHighlighted  char(1),
                 @pBillable     int,
                 @pWorkProcess  int,
                 @pPublish      char(1),
                 @pFnCode       varchar(10),
                 @pTeamCode     varchar(10),
                 @pText1        varchar(MAX),
                 @pAttachments  char(1),
                 @pEmailAddress varchar(150),
                 @pAddressTo    varchar(2000),
                 @pccTo         varchar(2000),
                 @pbccTo        varchar(2000),
                 @pemail        char(1),
                 @pSubject      varchar(500),
                 @pProcessType  char(1),
                 @pActionID     int)
AS
/************************************************************************
*[ky_WDCreateDiaryStep2]
* Create Diary Step WinDev/WebDev version - Version 2, suitable for Add Task
*    as well as assigning a mail to Case
*      2015-06-23 Sridharen    KEYD - 2332 - Use of LTRIM(RTRIM(
*      2015-09-30 Pino         Use ky_NETGetNextActionID
*      2016-09-13 Pino         KYC - KEYD-4040
*************************************************************************/
BEGIN
    SET NOCOUNT ON
    DECLARE @strcasecode        nVarChar(17)
    DECLARE @intAssignno        int
    DECLARE @SubjectLen         int
    DECLARE @CaseCodeLen        int
    DECLARE @FELen              int
    DECLARE @TeamLen            int
    DECLARE @dDate              datetime
    DECLARE @pTime              char(10)
    DECLARE @dDueDate           datetime
    DECLARE @pDueTime           char(10)
    DECLARE @KYC                char(1)

    -- Get only the DATE portion of the date that was passed in
    IF (@pDate is null)
      SET @pDate = GETDATE()

    IF (@pDueDate is null)
      SET @pDueDate = @pDate

    SET @dDate = convert(datetime, convert(varchar, @pDate, 112))
    SET @pTime = convert(char(10), (1000 * convert(int, DATEDIFF(ms, @dDate, @pDate) / 10000)) + 1)

    SET @dDueDate = convert(datetime, convert(varchar, @pDueDate, 112))
    SET @pDueTime = convert(char(10), (1000 * convert(int, DATEDIFF(ms, @dDueDate, @pDueDate) / 10000)) + 1)

    select @SubjectLen = CASE WHEN sc.xtype = 35
                                   THEN 150
                              ELSE sc.[length] END
      from sysobjects so
           inner join syscolumns sc
                   on sc.[id] = so.[id]
                  and sc.[name] = 'subject'
     where so.[name] = 'diary'

    select @CaseCodeLen = sc.[length]
      from sysobjects so
           inner join syscolumns sc
                   on sc.[id] = so.[id]
                  and sc.[name] = 'casecode'
     where so.[name] = 'diary'

    select @FELen = sc.[length]
      from sysobjects so
           inner join syscolumns sc
                   on sc.[id] = so.[id]
                  and sc.[name] = 'fncode'
     where so.[name] = 'diary'

    select @TeamLen = sc.[length]
      from sysobjects so
           inner join syscolumns sc
                   on sc.[id] = so.[id]
                  and sc.[name] = 'teamcode'
     where so.[name] = 'diary'

-- Start Transaction -----------------------------------------------------------------------------------------------------------------------------------------

    BEGIN TRAN Key_Tran_01

    IF @@error <> 0 GOTO Tran_Fail

-- Get CaseCode From Matters Table -----------------------------------------------------------------------------------------------------------------------------------------

    SELECT @strCaseCode = (SELECT Top 1
                                  [Code]
                             From [dbo].[Matters]
                            WHERE [code] = @pCaseCode)

    IF @@error <> 0 GOTO Tran_Fail

    SET @strCaseCode = RTRIM(Isnull(@strCaseCode, ''))
    Set @pFNCode = RTRIM(@pFNCode)
    Set @pTeamCode = RTRIM(@pTeamCode)

    IF LEN(@strCaseCode) < 2 GOTO Tran_Fail
    IF LEN(@strCaseCode) > @CaseCodeLen GOTO Tran_Fail
    IF LEN(@pFNCode) > @FELen GOTO Tran_Fail
    IF LEN(@pTeamCode) > @TeamLen GOTO Tran_Fail

-- Set Publish and Subject -----------------------------------------------------------------------------------------------------------------------------------

    Select @pPublish = CASE WHEN RTRIM(ISNULL(@pPublish, '')) <> ''
                                 THEN @pPublish
                            ELSE ISNULL(TAC.[PUBLISHER], 'N') END,
           @KYC = ISNULL(TAC.[KYC], 'N'),
           @pProcessType = CASE WHEN RTRIM(ISNULL(@pProcessType, '')) <> ''
                                     THEN @pProcessType
                                ELSE TAC.[PROCESSTYPE] END
      From TemplateActions TAC
     where TAC.[ACTIONCODE] = @pActionCode

    IF (RTRIM(IsNull(@pPublish, '')) = '')
      BEGIN
        SET @pPublish = 'N'
      END

    IF (RTRIM(IsNull(@pProcessType, '')) = '')
      BEGIN
        SET @pProcessType = ''
      END

    SET @pSubject = RTRIM(@pSubject)

    IF LEN(@pSubject) > @SubjectLen
       BEGIN
          SET @pSubject = Substring(@pSubject, 1, @SubjectLen)
       End

-- Write Diary Record -----------------------------------------------------------------------------------------------------------------------------------------

    INSERT INTO [dbo].[Diary]
               ([ActionID],
                [Status],
                [CaseCode],
                [ActionCode],
                [ActionStatus],
                [PRIORITY],
                [HIGHLIGHTED],
                [BILLABLE],
                [WORKPROCESS],
                [FnCode],
                [TeamCode],
                [Text1],
                [DisplayText],
                [Attachments],
                [EmailAddress],
                [AddressTo],
                [ccTo],
                [bccTo],
                [email],
                [Subject],
                [Publish],
                [ActionType],
                [ProcessType],
                [DYStartTime],
                [Date],
                [DUEDATE],
                [TxmDate],
                [KYC])
         VALUES(@pActionID,
                @pStatus,
                @pCaseCode,
                @pActionCode,
                @pActionStatus,
                @pPriority,
                @pHighlighted,
                @pBillable,
                @pWorkProcess,
                @pFnCode,
                @pTeamCode,
                convert(text, @pText1),
                [dbo].[ky_RemoveSpuriousWhitespace2](ISNULL(CONVERT(VARCHAR(MAX), @pText1), ''), 200),
                @pAttachments,
                @pEmailAddress,
                @pAddressTo,
                @pccTo,
                @pbccTo,
                @pemail,
                @pSubject,
                @pPublish,
                @pActionType,
                @pProcessType,
                @pTime,
                @pDate,
                @dDueDate,
                @pSentDate,
                @KYC)

    IF @@error <> 0 GOTO Tran_Fail

-- Write the diary delegation record ------------------------------------------------------------

    INSERT INTO [DiaryDelegations]
               ([ACTIONID],
                [HANDLER],
                [TEAM],
                [DATE],
                [TIME],
                [DATER],
                [TIMER],
                [DUEDATE],
                [DUETIME],
                [REVIEW],
                [STATUS],
                [OWNER],
                [DELEGATE],
                [DELEGATESTATUS],
                [ActionType],
                [FromHandler],
                [ReturnedBy],
                [DelType])
         VALUES(@pActionID,
                @pFnCode,
                @pTeamCode,
                @dDate,
                @pTime,
                @dDate,
                @pTime,
                @dDueDate,
                @pDueTime,
                ' ',
                @pStatus,
                'Y',
                @pFnCode,
                0,
                @pActionType,
                ' ',
                ' ',
                'Created')

-- Commit Transaction ------------------------------------------------------------------------------------------------------------------

    COMMIT TRAN Key_Tran_01

    IF @@error <> 0 GOTO Tran_Fail

-- End Of Processing --------------------------------------------------------------------------------------------------------------------

ExitPoint:

    Return

--  Error Processing -------------------------------------------------------------------------------------------------------------------------------------

Tran_Fail:

    rollback tran Key_Tran_01

    GoTo ExitPoint
END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_WDCreateDiaryStep'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_WDCreateDiaryStep]
  END
GO

CREATE PROCEDURE [dbo].[ky_WDCreateDiaryStep]
                (@pStatus       Int=0,
                 @pDate         datetime,
                 @pSentDate     datetime,
                 @pCaseCode     char(17),
                 @pActionCode   char(15),
                 @pActionStatus char(3),
                 @pFnCode       char(5),
                 @pTeamCode     char(5),
                 @pText1        varchar(8000),
                 @pAttachments  char(1),
                 @pEmailAddress varchar(150),
                 @pAddressTo    varchar(2000),
                 @pccTo         varchar(2000),
                 @pbccTo        varchar(2000),
                 @pemail        char(1),
                 @pSubject      varchar(500),
                 @pProcessType  char(1),
                 @pActionID     int)
AS
/************************************************************************
*
* Create Diary Step WinDev/WebDev version
* As it uses ky_WDCreateDiaryStep2 it will automatically become aware
*     of the new DisplayText column
* 2015-09-30 Pino       Use ky_NETGetNextActionID
* 2016-09-13 Pino       KYC - KEYD-4040
*
*************************************************************************/
BEGIN
  EXEC [dbo].[ky_WDCreateDiaryStep2] @pStatus,
                                     @pDate,
                                     NULL,
                                     @pSentDate,
                                     @pCaseCode,
                                     @pActionCode,
                                     'E',
                                     @pActionStatus,
                                     NULL,
                                     NULL,
                                     NULL,
                                     NULL,
                                     NULL,
                                     @pFnCode,
                                     @pTeamCode,
                                     @pText1,
                                     @pAttachments,
                                     @pEmailAddress,
                                     @pAddressTo,
                                     @pccTo,
                                     @pbccTo,
                                     @pemail,
                                     @pSubject,
                                     @pProcessType,
                                     @pActionID
END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_WDCreateDiaryStep3'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_WDCreateDiaryStep3]
  END
GO

CREATE PROCEDURE [dbo].[ky_WDCreateDiaryStep3]
                (@pStatus       Int=0,
                 @pDate         datetime,
                 @pDueDate      datetime,
                 @pSentDate     datetime,
                 @pCaseCode     varchar(20),
                 @pActionCode   char(15),
                 @pActionType   char(1),
                 @pActionStatus char(3),
                 @pPriority     char(1),
                 @pHighlighted  char(1),
                 @pBillable     int,
                 @pWorkProcess  int,
                 @pPublish      char(1),
                 @pFnCode       varchar(10),
                 @pTeamCode     varchar(10),
                 @pText1        varchar(MAX),
                 @pAttachments  char(1),
                 @pEmailAddress varchar(150),
                 @pAddressTo    varchar(2000),
                 @pccTo         varchar(2000),
                 @pbccTo        varchar(2000),
                 @pemail        char(1),
                 @pSubject      varchar(500),
                 @pProcessType  char(1),
                 @pActionID     int,
                 @pDuration     int)
AS
/************************************************************************
*
* Create Diary Step WinDev/WebDev version - Version 3, suitable for Add Task
*    as well as assigning a mail to Case, and applying a duration.
* 2015-09-30 Pino       Use ky_NETGetNextActionID
* 2016-09-13 Pino       KYC - KEYD-4040
*
*************************************************************************/
BEGIN
    SET NOCOUNT ON
    DECLARE @strcasecode        nVarChar(17)
    DECLARE @intAssignno        int
    DECLARE @SubjectLen         int
    DECLARE @CaseCodeLen        int
    DECLARE @FELen              int
    DECLARE @TeamLen            int
    DECLARE @dDate              datetime
    DECLARE @pTime              char(10)
    DECLARE @pEndTime           char(10)
    DECLARE @dDueDate           datetime
    DECLARE @pDueTime           char(10)
    DECLARE @KYC                char(1)

    SET @pDuration = ISNULL(@pDuration, 0)
    IF (@pDuration <= 0)
      BEGIN
        SET @pDuration = 30
      END

    SET @pDuration = @pDuration * 60

    -- Get only the DATE portion of the date that was passed in
    IF (@pDate is null)
      SET @pDate = GETDATE()

    IF (@pDueDate is null)
      SET @pDueDate = @pDate

    SET @dDate = convert(datetime, convert(varchar, @pDate, 112))
    SET @pTime = convert(char(10), (1000 * convert(int, DATEDIFF(ms, @dDate, @pDate) / 10000)) + 1)
    SET @pEndTime = convert(char(10), (@pDuration * 100) + (1000 * convert(int, DATEDIFF(ms, @dDate, @pDate) / 10000)) + 1)

    SET @dDueDate = convert(datetime, convert(varchar, @pDueDate, 112))
    SET @pDueTime = convert(char(10), @pDuration + (10 * convert(int, DATEDIFF(ms, @dDueDate, @pDueDate) / 10000)))

    select @SubjectLen = CASE WHEN sc.xtype = 35
                                   THEN 150
                              ELSE sc.[length] END
      from sysobjects so
           inner join syscolumns sc
                   on sc.[id] = so.[id]
                  and sc.[name] = 'subject'
     where so.[name] = 'diary'

    select @CaseCodeLen = sc.[length]
      from sysobjects so
           inner join syscolumns sc
                   on sc.[id] = so.[id]
                  and sc.[name] = 'casecode'
     where so.[name] = 'diary'

    select @FELen = sc.[length]
      from sysobjects so
           inner join syscolumns sc
                   on sc.[id] = so.[id]
                  and sc.[name] = 'fncode'
     where so.[name] = 'diary'

    select @TeamLen = sc.[length]
      from sysobjects so
           inner join syscolumns sc
                   on sc.[id] = so.[id]
                  and sc.[name] = 'teamcode'
     where so.[name] = 'diary'

-- Start Transaction -----------------------------------------------------------------------------------------------------------------------------------------

    BEGIN TRAN Key_Tran_01

    IF @@error <> 0 GOTO Tran_Fail

-- Get CaseCode From Matters Table -----------------------------------------------------------------------------------------------------------------------------------------

    SELECT @strCaseCode = (SELECT Top 1
                                  [Code]
                             From [dbo].[Matters]
                            WHERE [code] = @pCaseCode)

    IF @@error <> 0 GOTO Tran_Fail

    SET @strCaseCode = RTrim(Isnull(@strCaseCode, ''))
    Set @pFNCode = RTrim(@pFNCode)
    Set @pTeamCode = RTrim(@pTeamCode)

    IF LEN(@strCaseCode) < 2 GOTO Tran_Fail
    IF LEN(@strCaseCode) > @CaseCodeLen GOTO Tran_Fail
    IF LEN(@pFNCode) > @FELen GOTO Tran_Fail
    IF LEN(@pTeamCode) > @TeamLen GOTO Tran_Fail

-- Set Publish and Subject -----------------------------------------------------------------------------------------------------------------------------------

    Select @pPublish = CASE WHEN RTRIM(ISNULL(@pPublish, '')) <> ''
                                 THEN @pPublish
                            ELSE ISNULL(TAC.[PUBLISHER], 'N') END,
           @KYC = ISNULL(TAC.[KYC], 'N'),
           @pProcessType = CASE WHEN RTRIM(ISNULL(@pProcessType, '')) <> ''
                                     THEN @pProcessType
                                ELSE TAC.[PROCESSTYPE] END
      From TemplateActions TAC
     where TAC.[ACTIONCODE] = @pActionCode

    IF (RTRIM(IsNull(@pPublish, '')) = '')
      BEGIN
        SET @pPublish = 'N'
      END

    IF (RTRIM(IsNull(@pProcessType, '')) = '')
      BEGIN
        SET @pProcessType = ''
      END

    SET @pSubject = LTrim(RTrim(@pSubject))

    IF LEN(@pSubject) > @SubjectLen
       BEGIN
          SET @pSubject = Substring(@pSubject, 1, @SubjectLen)
       End

-- Write Diary Record -----------------------------------------------------------------------------------------------------------------------------------------

    INSERT INTO [dbo].[Diary]
               ([ActionID],
                [Status],
                [CaseCode],
                [ActionCode],
                [ActionStatus],
                [PRIORITY],
                [HIGHLIGHTED],
                [BILLABLE],
                [WORKPROCESS],
                [FnCode],
                [TeamCode],
                [Text1],
                [DisplayText],
                [Attachments],
                [EmailAddress],
                [AddressTo],
                [ccTo],
                [bccTo],
                [email],
                [Subject],
                [Publish],
                [ActionType],
                [ProcessType],
                [DYSTARTTIME],
                [DYENDTIME],
                [Date],
                [DUEDATE],
                [TxmDate],
                [KYC])
         VALUES(@pActionID,
                @pStatus,
                @pCaseCode,
                @pActionCode,
                @pActionStatus,
                @pPriority,
                @pHighlighted,
                @pBillable,
                @pWorkProcess,
                @pFnCode,
                @pTeamCode,
                convert(text, @pText1),
                [dbo].[ky_RemoveSpuriousWhitespace2](ISNULL(CONVERT(VARCHAR(MAX), @pText1), ''), 200),
                @pAttachments,
                @pEmailAddress,
                @pAddressTo,
                @pccTo,
                @pbccTo,
                @pemail,
                @pSubject,
                @pPublish,
                @pActionType,
                @pProcessType,
                @pTime,
                @pEndTime,
                @pDate,
                @pDueDate,
                @pSentDate,
                @KYC)

    IF @@error <> 0 GOTO Tran_Fail

-- Write the diary delegation record ------------------------------------------------------------

    INSERT INTO [DiaryDelegations]
               ([ACTIONID],
                [HANDLER],
                [TEAM],
                [DATE],
                [TIME],
                [DATER],
                [TIMER],
                [DUEDATE],
                [DUETIME],
                [REVIEW],
                [STATUS],
                [OWNER],
                [DELEGATE],
                [DELEGATESTATUS],
                [ActionType],
                [FromHandler],
                [ReturnedBy],
                [DelType])
         VALUES(@pActionID,
                @pFnCode,
                @pTeamCode,
                @dDate,
                @pTime,
                @dDate,
                @pTime,
                @dDueDate,
                @pDueTime,
                ' ',
                @pStatus,
                'Y',
                @pFnCode,
                0,
                @pActionType,
                ' ',
                ' ',
                'Created')

-- Commit Transaction ------------------------------------------------------------------------------------------------------------------

    COMMIT TRAN Key_Tran_01

    IF @@error <> 0 GOTO Tran_Fail

-- End Of Processing --------------------------------------------------------------------------------------------------------------------

ExitPoint:

    Return

--  Error Processing -------------------------------------------------------------------------------------------------------------------------------------

Tran_Fail:

    rollback tran Key_Tran_01

    GoTo ExitPoint

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_WDCreateDiaryStep4'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_WDCreateDiaryStep4]
  END
GO

/************************************************************************
*
* Create Diary Step WinDev/WebDev version - Version 4, suitable for Add Task
*    as well as assigning a mail to Case, and applying a duration.
*                   2015-06-23 Sridharen    KEYD - 2332 - Use of LTRIM(RTRIM(
*                   2015-09-30 Pino         Use ky_NETGetNextActionID
*                   2016-09-13 Pino         KYC - KEYD-4040
*
*************************************************************************/
CREATE PROCEDURE [ky_WDCreateDiaryStep4]
                (@pStatus       Int=0,
                 @pDate         datetime,
                 @pDueDate      datetime,
                 @pSentDate     datetime,
                 @pCaseCode     varchar(20),
                 @pActionCode   char(15),
                 @pActionType   char(1),
                 @pActionStatus char(3),
                 @pPriority     char(1),
                 @pHighlighted  char(1),
                 @pBillable     int,
                 @pWorkProcess  int,
                 @pPublish      char(1),
                 @pFnCode       varchar(10),
                 @pTeamCode     varchar(10),
                 @pText1        varchar(MAX),
                 @pAttachments  char(1),
                 @pEmailAddress varchar(MAX),
                 @pAddressTo    varchar(MAX),
                 @pccTo         varchar(MAX),
                 @pbccTo        varchar(MAX),
                 @pemail        char(1),
                 @pSubject      varchar(MAX),
                 @pProcessType  char(1),
                 @pDuration     int)
AS
BEGIN
    SET NOCOUNT ON
    DECLARE @strcasecode        nVarChar(17)
    DECLARE @intAssignno        int
    DECLARE @SubjectLen         int
    DECLARE @CaseCodeLen        int
    DECLARE @FELen              int
    DECLARE @TeamLen            int
    DECLARE @dDate              datetime
    DECLARE @pTime              char(10)
    DECLARE @pEndTime           char(10)
    DECLARE @dDueDate           datetime
    DECLARE @pDueTime           char(10)
    DECLARE @continue           int
    DECLARE @error              varchar(1000)
    DECLARE @NewActionID        int
    DECLARE @KYC                char(1)

    SET @continue = 0
    SET @error = ''
    SET @NewActionID = 0

    BEGIN TRANSACTION

    Begin Try
      SET @pDuration = ISNULL(@pDuration, 0)
      IF (@pDuration <= 0)
        BEGIN
          SET @pDuration = 30
        END
        SET @pDuration = @pDuration * 60
    End Try
    Begin Catch
      SET @continue = 1
      SET @error = 'invalid duration information'
      SET @NewActionID = 0
    End Catch

    IF (@continue = 0)
      BEGIN
        BEGIN TRY
          -- Get only the DATE portion of the date that was passed in
          IF (@pDate is null)
            SET @pDate = GETDATE()

          IF (@pDueDate is null)
            SET @pDueDate = @pDate

          SET @dDate = convert(datetime, convert(varchar, @pDate, 112))
          SET @pTime = convert(char(10), (1000 * convert(int, DATEDIFF(ms, @dDate, @pDate) / 10000)) + 1)
          SET @pEndTime = convert(char(10), (@pDuration * 100) + (1000 * convert(int, DATEDIFF(ms, @dDate, @pDate) / 10000)) + 1)

          SET @dDueDate = convert(datetime, convert(varchar, @pDueDate, 112))
          SET @pDueTime = convert(char(10), @pDuration + (10 * convert(int, DATEDIFF(ms, @dDueDate, @pDueDate) / 10000)))
        END TRY
        BEGIN CATCH
          SET @continue = 2
          SET @error = 'error processing date parameters'
          SET @NewActionID = 0
        END CATCH
      END

    IF (@continue = 0)
      BEGIN
        BEGIN TRY
          select @SubjectLen = CASE WHEN sc.xtype = 35
                                         THEN 150
                                    ELSE sc.[length] END
            from sysobjects so
                 inner join syscolumns sc
                         on sc.[id] = so.[id]
                        and sc.[name] = 'subject'
           where so.[name] = 'diary'

          select @CaseCodeLen = sc.[length]
            from sysobjects so
                 inner join syscolumns sc
                         on sc.[id] = so.[id]
                        and sc.[name] = 'casecode'
           where so.[name] = 'diary'

          select @FELen = sc.[length]
            from sysobjects so
                 inner join syscolumns sc
                         on sc.[id] = so.[id]
                        and sc.[name] = 'fncode'
           where so.[name] = 'diary'

          select @TeamLen = sc.[length]
            from sysobjects so
                 inner join syscolumns sc
                         on sc.[id] = so.[id]
                        and sc.[name] = 'teamcode'
           where so.[name] = 'diary'
        END TRY
        BEGIN CATCH
          SET @continue = 3
          SET @error = 'invalid duration information'
          SET @NewActionID = 0
        END CATCH
      END

-- Get CaseCode From Matters Table -----------------------------------------------------------------------------------------------------------------------------------------

    IF (@continue = 0)
      BEGIN
        BEGIN TRY
          SET @strCaseCode = NULL

          SELECT Top 1
                 @strCaseCode = MAT.[Code]
            From [dbo].[Matters] MAT
           WHERE MAT.[code] = @pCaseCode

          SET @strCaseCode = RTRIM(Isnull(@strCaseCode, ''))
          Set @pFNCode = RTRIM(@pFNCode)
          Set @pTeamCode = RTRIM(@pTeamCode)

          IF (@strCaseCode = '')
            BEGIN
              SET @continue = 4
              BEGIN TRY
                SET @pCaseCode = RTRIM(ISNULL(@pCaseCode, ''))
              END TRY
              BEGIN CATCH
                SET @pCaseCode = ''
              END CATCH
              SET @error = 'Invalid Case Code: ' + @pCaseCode
              SET @NewActionID = 0
            END

          IF (@continue = 0)
            BEGIN
              IF (LEN(@strCaseCode) < 2)
                BEGIN
                  SET @continue = 4
                  SET @error = 'Invalid parameters. Case Code too short'
                  SET @NewActionID = 0
                END
            END

          IF (@continue = 0)
            BEGIN
              IF (LEN(@strCaseCode) > @CaseCodeLen)
                BEGIN
                  SET @continue = 4
                  SET @error = 'Invalid parameters. Case Code too long'
                  SET @NewActionID = 0
                END
            END

          IF (@continue = 0)
            BEGIN
              IF (LEN(@pFNCode) > @FELen)
                BEGIN
                  SET @continue = 4
                  SET @error = 'Invalid parameters. FE code too long'
                  SET @NewActionID = 0
                END
            END

          IF (@continue = 0)
            BEGIN
              IF (LEN(@pTeamCode) > @TeamLen)
                BEGIN
                  SET @continue = 4
                  SET @error = 'Invalid parameters. TEAM code too long'
                  SET @NewActionID = 0
                END
            END

          -- Set Publish and Subject -----------------------------------------------------------------------------------------------------------------------------------
          IF (@continue = 0)
            BEGIN
              Select @pPublish = CASE WHEN RTRIM(ISNULL(@pPublish, '')) <> ''
                                           THEN @pPublish
                                      ELSE ISNULL(TAC.[PUBLISHER], 'N') END,
                     @KYC = ISNULL(TAC.[KYC], 'N'),
                     @pProcessType = CASE WHEN RTRIM(ISNULL(@pProcessType, '')) <> ''
                                               THEN @pProcessType
                                          ELSE TAC.[PROCESSTYPE] END
                From TemplateActions TAC
               where TAC.[ACTIONCODE] = @pActionCode

              IF (RTRIM(IsNull(@pPublish, '')) = '')
                BEGIN
                  SET @pPublish = 'N'
                END

              IF (RTRIM(IsNull(@pProcessType, '')) = '')
                BEGIN
                  SET @pProcessType = ''
                END

              SET @pSubject = RTRIM(@pSubject)

              IF LEN(@pSubject) > @SubjectLen
                BEGIN
                  SET @pSubject = Substring(@pSubject, 1, @SubjectLen)
                END
            END

          -- Truncate potentially long fields -----------------------------------------------------------------------------------------------------------------------------------
          IF (@continue = 0)
            BEGIN
              SET @pEmailAddress = CONVERT(varchar(150), ISNULL(@pEmailAddress, ''))
              SET @pSubject = CONVERT(varchar(500), ISNULL(@pSubject, ''))
              SET @pAddressTo = CONVERT(varchar(2000), ISNULL(@pAddressTo, ''))
              SET @pccTo = CONVERT(varchar(2000), ISNULL(@pccTo, ''))
              SET @pbccTo = CONVERT(varchar(2000), ISNULL(@pbccTo, ''))
            END
        END TRY
        BEGIN CATCH
          SET @continue = 5
          SET @error = 'Invalid parameters Publish, ProcessType or Subject'
          SET @NewActionID = 0
        END CATCH

      END

    IF (@continue = 0)
      BEGIN
        BEGIN TRY
          -- Pino 2015-09-30 Start
          EXEC @NewActionID = [dbo].[ky_NETGetNextActionID]
          --SELECT @NewActionID = ISNULL(MAX(DIA.[ActionID]), 0) + 1
          --  FROM [dbo].[diary] DIA
          -- Pino 2015-09-30 End
        END TRY
        BEGIN CATCH
          SET @continue = 8
          SET @error = convert(varchar(1000), 'Error establishing new Action ID, ' + Error_Message())
          SET @NewActionID = 0
        END CATCH
      END

-- Write Diary Record -----------------------------------------------------------------------------------------------------------------------------------------
    IF (@continue = 0)
      BEGIN
        BEGIN TRY
          INSERT INTO [dbo].[Diary]
                     ([ActionID],
                      [Status],
                      [CaseCode],
                      [ActionCode],
                      [ActionStatus],
                      [PRIORITY],
                      [HIGHLIGHTED],
                      [BILLABLE],
                      [WORKPROCESS],
                      [FnCode],
                      [TeamCode],
                      [Text1],
                      [Attachments],
                      [EmailAddress],
                      [AddressTo],
                      [ccTo],
                      [bccTo],
                      [email],
                      [Subject],
                      [Publish],
                      [ActionType],
                      [ProcessType],
                      [DYSTARTTIME],
                      [DYENDTIME],
                      [Date],
                      [DUEDATE],
                      [TxmDate],
                      [KYC])
               VALUES(@NewActionID,
                      @pStatus,
                      @pCaseCode,
                      @pActionCode,
                      @pActionStatus,
                      @pPriority,
                      @pHighlighted,
                      @pBillable,
                      @pWorkProcess,
                      @pFnCode,
                      @pTeamCode,
                      convert(text, @pText1),
                      @pAttachments,
                      @pEmailAddress,
                      @pAddressTo,
                      @pccTo,
                      @pbccTo,
                      @pemail,
                      @pSubject,
                      @pPublish,
                      @pActionType,
                      @pProcessType,
                      @pTime,
                      @pEndTime,
                      @pDate,
                      @pDueDate,
                      @pSentDate,
                      @KYC)
        END TRY
        BEGIN CATCH
          SET @continue = 6
          SET @error = convert(varchar(1000), 'Error inserting Diary Entry, ' + Error_Message())
          SET @NewActionID = 0
        END CATCH
      END

-- Write the diary delegation record ------------------------------------------------------------
    IF (@continue = 0)
      BEGIN
        BEGIN TRY
          INSERT INTO [DiaryDelegations]
                     ([ACTIONID],
                      [HANDLER],
                      [TEAM],
                      [DATE],
                      [TIME],
                      [DATER],
                      [TIMER],
                      [DUEDATE],
                      [DUETIME],
                      [REVIEW],
                      [STATUS],
                      [OWNER],
                      [DELEGATE],
                      [DELEGATESTATUS],
                      [ActionType],
                      [FromHandler],
                      [ReturnedBy],
                      [DelType])
               VALUES(@NewActionID,
                      @pFnCode,
                      @pTeamCode,
                      @dDate,
                      @pTime,
                      @dDate,
                      @pTime,
                      @dDueDate,
                      @pDueTime,
                      ' ',
                      @pStatus,
                      'Y',
                      @pFnCode,
                      0,
                      @pActionType,
                      ' ',
                      ' ',
                      'Created')
        END TRY
        BEGIN CATCH
          SET @continue = 7
          SET @error = convert(varchar(1000), 'Error inserting Diary Delegation, ' + Error_Message())
          SET @NewActionID = 0
        END CATCH
      END
-- Commit Transaction ------------------------------------------------------------------------------------------------------------------

    IF (@continue = 0)
      BEGIN
        COMMIT
      END
    ELSE
      BEGIN
        SET @NewActionID = 0
        ROLLBACK
      END

    SELECT @continue AS [ErrorCode],
           @error AS [ErrorMessage],
           @NewActionID AS [NewActionID]

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_WDCreateDiaryStep5'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_WDCreateDiaryStep5]
  END
GO

/*************************************************************************
 *
 * Create Diary Step .NET version - Version 5, suitable for Add Task
 *    as well as assigning a mail to Case, and applying a duration.
 *                   2015-06-23 Sridharen    KEYD - 2332 - Use of LTRIM(RTRIM(
 *                   2015-09-30 Pino         Use ky_NETGetNextActionID
 *                   2016-09-13 Pino         KYC - KEYD-4040
 *                   2018-10-26 Pino         Include check for FirmDomainAddress
 *                   2019-04-19 Pino         Handle bracketed email addresses,
 *                                           e.g. "Joe Bloggs <joe@bloggs.com>"
 *
 *************************************************************************/
CREATE PROCEDURE [ky_WDCreateDiaryStep5]
                (@pStatus       Int=0,
                 @pDate         datetime,
                 @pDueDate      datetime,
                 @pSentDate     datetime,
                 @pCaseCode     varchar(20),
                 @pActionCode   char(15),
                 @pActionType   char(1),
                 @pActionStatus char(3),
                 @pPriority     char(1),
                 @pHighlighted  char(1),
                 @pBillable     int,
                 @pWorkProcess  int,
                 @pPublish      char(1),
                 @pFnCode       varchar(10),
                 @pTeamCode     varchar(10),
                 @pText1        varchar(MAX),
                 @pAttachments  char(1),
                 @pEmailAddress varchar(MAX),
                 @pAddressTo    varchar(MAX),
                 @pccTo         varchar(MAX),
                 @pbccTo        varchar(MAX),
                 @pemail        char(1),
                 @pSubject      varchar(MAX),
                 @pProcessType  char(1),
                 @pLocation     varchar(50),
                 @pDuration     int)
AS
BEGIN
    SET NOCOUNT ON
    DECLARE @strcasecode        nVarChar(20)
    DECLARE @intAssignno        int
    DECLARE @SubjectLen         int
    DECLARE @CaseCodeLen        int
    DECLARE @FELen              int
    DECLARE @TeamLen            int
    DECLARE @dDate              datetime
    DECLARE @pTime              char(10)
    DECLARE @pEndTime           char(10)
    DECLARE @dDueDate           datetime
    DECLARE @pDueTime           char(10)
    DECLARE @continue           int
    DECLARE @error              varchar(1000)
    DECLARE @NewActionID        int
    DECLARE @KYC                char(1)
    DECLARE @FirmDomainAddress  VARCHAR(50)

    SELECT @FirmDomainAddress = CONVERT(VARCHAR(50), ISNULL([SET].[KeyValue], '')) 
      FROM [dbo].[Settings] [SET]
     WHERE [SET].[KeyName] = 'FirmDomainAddress'
     
    DECLARE @CheckAddress VARCHAR(MAX)

    SET @CheckAddress = @pEmailAddress

    If CHARINDEX('@', @CheckAddress) > 1
      BEGIN
        SET @CheckAddress = SUBSTRING(@CheckAddress, CHARINDEX('@', @CheckAddress), LEN(@CheckAddress) - CHARINDEX('@', @CheckAddress) + 1)
        IF SUBSTRING(@CheckAddress, LEN(@CheckAddress), 1) IN ('>', ')', ']')
          BEGIN
            SET @CheckAddress = SUBSTRING(@CheckAddress, 1, LEN(@CheckAddress) - 1)
          END
        SET @CheckAddress = '%;' + @CheckAddress + ';%'
      END
    ELSE
      BEGIN
        SET @CheckAddress = ''
      END

    SET @FirmDomainAddress = ISNULL(@FirmDomainAddress, 'xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx')

    IF LEN(@FirmDomainAddress) > 0
      BEGIN
        IF SUBSTRING(@FirmDomainAddress, 1, 1) <> '@'
          BEGIN
            SET @FirmDomainAddress = '@' + @FirmDomainAddress
          END
      END
    
    IF @CheckAddress <> ''
      BEGIN
        IF ';' + @FirmDomainAddress + ';' LIKE @CheckAddress
          BEGIN
            SET @pProcessType = 'O'
          END
      END

    SET @continue = 0
    SET @error = ''
    SET @NewActionID = 0

    BEGIN TRANSACTION

    Begin Try
      SET @pDuration = ISNULL(@pDuration, 0)
      IF (@pDuration <= 0)
        BEGIN
          SET @pDuration = 30
        END
        SET @pDuration = @pDuration * 60
    End Try
    Begin Catch
      SET @continue = 1
      SET @error = 'invalid duration information'
      SET @NewActionID = 0
    End Catch

    IF (@continue = 0)
      BEGIN
        BEGIN TRY
          -- Get only the DATE portion of the date that was passed in
          IF (@pDate is null)
            SET @pDate = GETDATE()

          IF (@pDueDate is null)
            SET @pDueDate = @pDate

          SET @dDate = convert(datetime, convert(varchar, @pDate, 112))
          SET @pTime = convert(char(10), (1000 * convert(int, DATEDIFF(ms, @dDate, @pDate) / 10000)) + 1)
          SET @pEndTime = convert(char(10), (@pDuration * 100) + (1000 * convert(int, DATEDIFF(ms, @dDate, @pDate) / 10000)) + 1)

          SET @dDueDate = convert(datetime, convert(varchar, @pDueDate, 112))
          SET @pDueTime = convert(char(10), @pDuration + (10 * convert(int, DATEDIFF(ms, @dDueDate, @pDueDate) / 10000)))
        END TRY
        BEGIN CATCH
          SET @continue = 2
          SET @error = 'error processing date parameters'
          SET @NewActionID = 0
        END CATCH
      END

    IF (@continue = 0)
      BEGIN
        BEGIN TRY
          select @SubjectLen = CASE WHEN sc.xtype = 35
                                         THEN 150
                                    ELSE sc.[length] END
            from sysobjects so
                 inner join syscolumns sc
                         on sc.[id] = so.[id]
                        and sc.[name] = 'subject'
           where so.[name] = 'diary'

          select @CaseCodeLen = sc.[length]
            from sysobjects so
                 inner join syscolumns sc
                         on sc.[id] = so.[id]
                        and sc.[name] = 'casecode'
           where so.[name] = 'diary'

          select @FELen = sc.[length]
            from sysobjects so
                 inner join syscolumns sc
                         on sc.[id] = so.[id]
                        and sc.[name] = 'fncode'
           where so.[name] = 'diary'

          select @TeamLen = sc.[length]
            from sysobjects so
                 inner join syscolumns sc
                         on sc.[id] = so.[id]
                        and sc.[name] = 'teamcode'
           where so.[name] = 'diary'
        END TRY
        BEGIN CATCH
          SET @continue = 3
          SET @error = 'invalid duration information'
          SET @NewActionID = 0
        END CATCH
      END

-- Get CaseCode From Matters Table -----------------------------------------------------------------------------------------------------------------------------------------

    IF (@continue = 0)
      BEGIN
        BEGIN TRY
          SET @strCaseCode = NULL

          SELECT Top 1
                 @strCaseCode = MAT.[Code]
            From [dbo].[Matters] MAT
           WHERE MAT.[code] = @pCaseCode

          SET @strCaseCode = RTRIM(Isnull(@strCaseCode, ''))
          Set @pFNCode = RTRIM(@pFNCode)
          Set @pTeamCode = RTRIM(@pTeamCode)

          IF (@strCaseCode = '')
            BEGIN
              SET @continue = 4
              BEGIN TRY
                SET @pCaseCode = RTRIM(ISNULL(@pCaseCode, ''))
              END TRY
              BEGIN CATCH
                SET @pCaseCode = ''
              END CATCH
              SET @error = 'Invalid Case Code: ' + @pCaseCode
              SET @NewActionID = 0
            END

          IF (@continue = 0)
            BEGIN
              IF (LEN(@strCaseCode) < 2)
                BEGIN
                  SET @continue = 4
                  SET @error = 'Invalid parameters. Case Code too short'
                  SET @NewActionID = 0
                END
            END

          IF (@continue = 0)
            BEGIN
              IF (LEN(@strCaseCode) > @CaseCodeLen)
                BEGIN
                  SET @continue = 4
                  SET @error = 'Invalid parameters. Case Code too long'
                  SET @NewActionID = 0
                END
            END

          IF (@continue = 0)
            BEGIN
              IF (LEN(@pFNCode) > @FELen)
                BEGIN
                  SET @continue = 4
                  SET @error = 'Invalid parameters. FE code too long'
                  SET @NewActionID = 0
                END
            END

          IF (@continue = 0)
            BEGIN
              IF (LEN(@pTeamCode) > @TeamLen)
                BEGIN
                  SET @continue = 4
                  SET @error = 'Invalid parameters. TEAM code too long'
                  SET @NewActionID = 0
                END
            END

          -- Set Publish and Subject -----------------------------------------------------------------------------------------------------------------------------------
          IF (@continue = 0)
            BEGIN
              Select @pPublish = CASE WHEN RTRIM(ISNULL(@pPublish, '')) <> ''
                                           THEN @pPublish
                                      ELSE ISNULL(TAC.[PUBLISHER], 'N') END,
                     @KYC = ISNULL(TAC.[KYC], 'N'),
                     @pProcessType = CASE WHEN RTRIM(ISNULL(@pProcessType, '')) <> ''
                                               THEN @pProcessType
                                          ELSE TAC.[PROCESSTYPE] END
                From TemplateActions TAC
               where TAC.[ACTIONCODE] = @pActionCode

              IF (RTRIM(IsNull(@pPublish, '')) = '')
                BEGIN
                  SET @pPublish = 'N'
                END

              IF (RTRIM(IsNull(@pProcessType, '')) = '')
                BEGIN
                  SET @pProcessType = ''
                END

              SET @pSubject = RTRIM(@pSubject)

              IF LEN(@pSubject) > @SubjectLen
                BEGIN
                  SET @pSubject = Substring(@pSubject, 1, @SubjectLen)
                END
            END

          -- Truncate potentially long fields -----------------------------------------------------------------------------------------------------------------------------------
          IF (@continue = 0)
            BEGIN
              SET @pEmailAddress = CONVERT(varchar(150), ISNULL(@pEmailAddress, ''))
              SET @pSubject = CONVERT(varchar(500), ISNULL(@pSubject, ''))
              SET @pAddressTo = CONVERT(varchar(2000), ISNULL(@pAddressTo, ''))
              SET @pccTo = CONVERT(varchar(2000), ISNULL(@pccTo, ''))
              SET @pbccTo = CONVERT(varchar(2000), ISNULL(@pbccTo, ''))
            END
        END TRY
        BEGIN CATCH
          SET @continue = 5
          SET @error = 'Invalid parameters Publish, ProcessType or Subject'
          SET @NewActionID = 0
        END CATCH

      END

    IF (@continue = 0)
      BEGIN
        BEGIN TRY
          -- Pino 2015-09-30 Start
          EXEC @NewActionID = [dbo].[ky_NETGetNextActionID]
          --SELECT @NewActionID = ISNULL(MAX(DIA.[ActionID]), 0) + 1
          --  FROM [dbo].[diary] DIA
          -- Pino 2015-09-30 End
        END TRY
        BEGIN CATCH
          SET @continue = 8
          SET @error = convert(varchar(1000), 'Error establishing new Action ID, ' + Error_Message())
          SET @NewActionID = 0
        END CATCH
      END

-- Write Diary Record -----------------------------------------------------------------------------------------------------------------------------------------
    IF (@continue = 0)
      BEGIN
        BEGIN TRY
          INSERT INTO [dbo].[Diary]
                     ([ActionID],
                      [Status],
                      [CaseCode],
                      [ActionCode],
                      [ActionStatus],
                      [PRIORITY],
                      [HIGHLIGHTED],
                      [BILLABLE],
                      [WORKPROCESS],
                      [FnCode],
                      [TeamCode],
                      [Text1],
                      [Attachments],
                      [EmailAddress],
                      [AddressTo],
                      [Location],
                      [ccTo],
                      [bccTo],
                      [email],
                      [Subject],
                      [Publish],
                      [ActionType],
                      [ProcessType],
                      [DYSTARTTIME],
                      [DYENDTIME],
                      [Date],
                      [DUEDATE],
                      [TxmDate],
                      [KYC])
               VALUES(@NewActionID,
                      @pStatus,
                      @pCaseCode,
                      @pActionCode,
                      @pActionStatus,
                      @pPriority,
                      @pHighlighted,
                      @pBillable,
                      @pWorkProcess,
                      @pFnCode,
                      @pTeamCode,
                      convert(text, @pText1),
                      @pAttachments,
                      @pEmailAddress,
                      @pAddressTo,
                      @pLocation,
                      @pccTo,
                      @pbccTo,
                      @pemail,
                      @pSubject,
                      @pPublish,
                      @pActionType,
                      @pProcessType,
                      @pTime,
                      @pEndTime,
                      @pDate,
                      @pDueDate,
                      @pSentDate,
                      @KYC)
        END TRY
        BEGIN CATCH
          SET @continue = 6
          SET @error = convert(varchar(1000), 'Error inserting Diary Entry, ' + Error_Message())
          SET @NewActionID = 0
        END CATCH
      END

-- Write the diary delegation record ------------------------------------------------------------
    IF (@continue = 0)
      BEGIN
        BEGIN TRY
          INSERT INTO [DiaryDelegations]
                     ([ACTIONID],
                      [HANDLER],
                      [TEAM],
                      [DATE],
                      [TIME],
                      [DATER],
                      [TIMER],
                      [DUEDATE],
                      [DUETIME],
                      [REVIEW],
                      [STATUS],
                      [OWNER],
                      [DELEGATE],
                      [DELEGATESTATUS],
                      [ActionType],
                      [FromHandler],
                      [ReturnedBy],
                      [DelType])
               VALUES(@NewActionID,
                      @pFnCode,
                      @pTeamCode,
                      @dDate,
                      @pTime,
                      @dDate,
                      @pTime,
                      @dDueDate,
                      @pDueTime,
                      ' ',
                      @pStatus,
                      'Y',
                      @pFnCode,
                      0,
                      @pActionType,
                      ' ',
                      ' ',
                      'Created')
        END TRY
        BEGIN CATCH
          SET @continue = 7
          SET @error = convert(varchar(1000), 'Error inserting Diary Delegation, ' + Error_Message())
          SET @NewActionID = 0
        END CATCH
      END
-- Commit Transaction ------------------------------------------------------------------------------------------------------------------

    IF (@continue = 0)
      BEGIN
        COMMIT
      END
    ELSE
      BEGIN
        SET @NewActionID = 0
        ROLLBACK
      END

    SELECT @continue AS [ErrorCode],
           @error AS [ErrorMessage],
           @NewActionID AS [NewActionID]

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_CreateCalendarStep'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_CreateCalendarStep]
  END
GO

CREATE PROCEDURE 
       [dbo].[ky_CreateCalendarStep]
      (@pStatus         Int = 0,
       @pDate           DateTime,
       @pCaseCode       varchar(20),
       @pActionCode     char(15),
       @pActionStatus   char(3),
       @pFnCode         varchar(10),
       @pTeamCode       varchar(10),
       @pText1          varchar(8000),
       @pSubject        varchar(500),
       @pProcessType    char(1),
       @pTime           char(10),
       @pActionId       int = 0             output)
AS
/**********************************************************************************************
 *
 * [dbo].[ky_CreateCalendarStep]
 * 2015-09-30 Pino Carafa      Imported into Framework
 *            Increase parameter sizes for Casecode and Fee Earner / Team codes
 *            use ky_NETGetNextActionID
 * 2016-09-13 Pino             KYC - KEYD-4040
 *
 **********************************************************************************************/
BEGIN
    SET NOCOUNT ON
    DECLARE @strcasecode    nVarChar(17)
    DECLARE @intActionID         Int
    DECLARE @intAssignno        int

-- Start Transaction -----------------------------------------------------------------------------------------------------------------------------------------

    BEGIN TRAN Key_Tran_01

    IF @@error <> 0 GOTO Tran_Fail

-- Get CaseCode From Matters Table -----------------------------------------------------------------------------------------------------------------------------------------

    SELECT @strCaseCode = (

        SELECT Top 1 [Code]

        FROM Matters

        WHERE code = @pCaseCode )

    IF @@error <> 0 GOTO Tran_Fail

    IF LEN( @strCaseCode ) < 2 GOTO Tran_Fail

-- Get Next ActionID -----------------------------------------------------------------------------------------------------------------------------------------

    --Pino 2015-09-30 start
    --SELECT @intActionID = ( SELECT Top 1 ActionID FROM Diary ORDER BY ActionID DESC )
    EXEC @intActionID = [dbo].[ky_NETGetNextActionID]
    --Pino 2015-09-30 end

    IF @@error <> 0 GOTO Tran_Fail

    --Pino 2015-09-30 start
    --IF @intActionID IS NULL SELECT @intActionID = 0
    --SELECT @intActionID = @intActionID + 1
    --Pino 2015-09-30 end

-- Write Diary Record -----------------------------------------------------------------------------------------------------------------------------------------

    INSERT INTO Diary
        (Status,
        CaseCode,
        ActionCode,
        ActionStatus,
        FnCode,
        TeamCode,
        Text1,
    DisplayText,
        Subject,
        Publish,
        ActionType,
        ProcessType,
        DYStartTime,
        ActionId,
        [Date])
    Values
        (@pStatus,
        @pCaseCode,
        @pActionCode,
        @pActionStatus,
        @pFnCode,
        @pTeamCode,
        @pText1,
        [dbo].[ky_RemoveSpuriousWhitespace2](ISNULL(CONVERT(VARCHAR(MAX), @pText1), ''), 200),
        @pSubject,
        'N',
        'P',
        @pProcessType,
        @pTime,
         @intActionID,
        @pDate)

    IF @@error <> 0 GOTO Tran_Fail

-- Write the diary delegation record ------------------------------------------------------------

    INSERT INTO [DiaryDelegations]
                   ([ACTIONID]
                   ,[HANDLER]
                   ,[TEAM]
                   ,[DATE]
                   ,[DATER]
                   ,[TIMER]
                   ,[DUEDATE]
                   ,[DUETIME]
                   ,[REVIEW]
                   ,[STATUS]
                   ,[OWNER]
                   ,[DELEGATE]
                   ,[DELEGATESTATUS]
                   ,[ActionType]
                   ,[FromHandler]
                   ,[ReturnedBy]
                   ,[DelType])
             VALUES
                   (@intActionID,
                   @pFnCode,
                   @pTeamCode,
                   @pDate,
                   0,
                   '',
                   @pDate,
                   @pTime,
                   ' ',
                   @pStatus,
                   'Y',
                   @pFnCode,
                   0,
                   'E',
                   ' ',
                   ' ',
                   'Created')

-- Commit Transaction ------------------------------------------------------------------------------------------------------------------

    COMMIT TRAN Key_Tran_01

    IF @@error <> 0 GOTO Tran_Fail

    Select @pActionId = @intActionID

-- End Of Processing --------------------------------------------------------------------------------------------------------------------

ExitPoint:

    Return

--  Error Processing -------------------------------------------------------------------------------------------------------------------------------------

Tran_Fail:

    rollback tran Key_Tran_01
    Select @pActionId = -1

    goto ExitPoint
END
GO
/***************************************************************
 *
 * ALL DIARY STEP PROCEDURES. KEEP AND MODIFY THESE TOGETHER
 * END
 *
 ***************************************************************/

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPCreateDefaultDocumentDiaryStep'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPCreateDefaultDocumentDiaryStep]
  END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPCreateDefaultDocumentDiaryStepUTC'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPCreateDefaultDocumentDiaryStepUTC]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETSPCreateDefaultDocumentDiaryStepUTC]
      (@pStatus       INT,
       @pDate         DATETIME,
       @pDueDate      DATETIME,
       @pSentDate     DATETIME,
       @pCaseCode     VARCHAR(20),
       @pFNCode       VARCHAR(10),
       @pTEAMCode     VARCHAR(10),
       @pTEXT         VARCHAR(MAX),
       @pAttachments  CHAR(1),
       @pEmailAddress VARCHAR(MAX),
       @pAddressTo    VARCHAR(MAX),
       @pccTo         VARCHAR(MAX),
       @pbccTo        VARCHAR(MAX),
       @pemail        CHAR(1),
       @pSubject      VARCHAR(MAX),
       @pLocation     VARCHAR(50),
       @pDuration     INT,
       @bAlreadyUTC   BIT)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETSPCreateDefaultDocumentDiaryStepUTC]                                                         *
 *                                                                                                           *
 * Creates a new Diary Entry for a dropped in Document                                                       *
 * - Will use the ADMKD Action Code if it's available, otherwise it will try to pick the most appropriate    *
 *   action code based on the Case's Case Plan, the available Actions in their default display order,        *
 *   and their Action Types.                                                                                 *
 *   This should really only come into play if the customer explicitly deletes the ADMKD Action Code         *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.5.1.2                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2017-06-29   Pino Carafa       Created                                                                    *
 * 2017-08-31   Pino Carafa       Added functionality to fall back to ADMKD if the preceding statement fails *
 *                                to retrieve a valid Action Code                                            *
 * 2019-10-02   Pino Carafa       Once the ADMKD action code is available, that is what should be used       *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN
  DECLARE @pActionCode VARCHAR(20)

  IF EXISTS
    (SELECT TOP 1 1
       FROM [dbo].[TemplateActions] [TA]
      WHERE [TA].[ACTIONCODE] = 'ADMKD')
    BEGIN
      SET @pActionCode = 'ADMKD'
    END
  ELSE
    BEGIN
        SELECT TOP 1 @pActionCode = [TA].[ACTIONCODE]
          FROM [dbo].[casemaster] [CSM]
               LEFT OUTER JOIN [dbo].[Templates] [TMP]
                            ON [TMP].[WKCODE] = [CSM].[CSWKTCODE]
               LEFT OUTER JOIN [dbo].[ActionWorkTypes] [AWT]
                               LEFT OUTER JOIN [dbo].[TemplateActions] [TA]
                                            ON [TA].[ACTIONCODE] = [AWT].[ACTIONCODE]
                            ON [AWT].[WORKTYPE] = [TMP].[WKCODE]
         WHERE [CSM].[CSCODE] = @pCaseCode
      ORDER BY CASE WHEN [TA].[ACTIONCODE] = 'ADMKD'
                         THEN 'A'
                    WHEN (ISNULL([TA].[WKTCODE], '') = [TA].[WKTCODE]) AND ([TA].[ACTTYPE] = 'A')
                         THEN 'B'
                    WHEN (ISNULL([TA].[WKTCODE], '') = [TA].[WKTCODE])
                         THEN 'C'
                    ELSE 'D' END,
               ISNULL([AWT].[SEQ], 99999)
    END

IF RTRIM(ISNULL(@pActionCode, '')) = ''
  BEGIN
    SELECT TOP 1 @pActionCode = [TA].[ACTIONCODE]
      FROM [dbo].[TemplateActions] [TA]
     WHERE [TA].[ACTIONCODE] = 'ADMKD'
  END

IF (@pActionCode IS NOT NULL)
  BEGIN
    EXEC [dbo].[ky_NETSPCreateTemplatedDiaryStepUTC] @pStatus,
                                                     @pDate,
                                                     @pDueDate,
                                                     @pSentDate,
                                                     @pCaseCode,
                                                     @pActionCode,
                                                     @pFNCode,
                                                     @pTEAMCode,
                                                     @pTEXT,
                                                     @pAttachments,
                                                     @pEmailAddress,
                                                     @pAddressTo,
                                                     @pccTo,
                                                     @pbccTo,
                                                     @pemail,
                                                     @pSubject,
                                                     @pLocation,
                                                     @pDuration,
                                                     @bAlreadyUTC
  END

END

GO

CREATE PROCEDURE
       [dbo].[ky_NETSPCreateDefaultDocumentDiaryStep]
      (@pStatus       INT,
       @pDate         DATETIME,
       @pDueDate      DATETIME,
       @pSentDate     DATETIME,
       @pCaseCode     VARCHAR(20),
       @pFNCode       VARCHAR(10),
       @pTEAMCode     VARCHAR(10),
       @pTEXT         VARCHAR(MAX),
       @pAttachments  CHAR(1),
       @pEmailAddress VARCHAR(MAX),
       @pAddressTo    VARCHAR(MAX),
       @pccTo         VARCHAR(MAX),
       @pbccTo        VARCHAR(MAX),
       @pemail        CHAR(1),
       @pSubject      VARCHAR(MAX),
       @pLocation     VARCHAR(50),
       @pDuration     INT)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETSPCreateDefaultDocumentDiaryStep]                                                            *
 *                                                                                                           *
 * Creates a new Diary Entry for a dropped in Document                                                       *
 * - Will use the ADMKD Action Code if it's available, otherwise it will try to pick the most appropriate    *
 *   action code based on the Case's Case Plan, the available Actions in their default display order,        *
 *   and their Action Types.                                                                                 *
 *   This should really only come into play if the customer explicitly deletes the ADMKD Action Code         *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.5.1.2                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2017-06-29   Pino Carafa       Created                                                                    *
 * 2017-08-31   Pino Carafa       Added functionality to fall back to ADMKD if the preceding statement fails *
 *                                to retrieve a valid Action Code                                            *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

EXEC [dbo].[ky_NETSPCreateDefaultDocumentDiaryStepUTC] @pStatus,
                                                       @pDate,
                                                       @pDueDate,
                                                       @pSentDate,
                                                       @pCaseCode,
                                                       @pFNCode,
                                                       @pTEAMCode,
                                                       @pTEXT,
                                                       @pAttachments,
                                                       @pEmailAddress,
                                                       @pAddressTo,
                                                       @pccTo,
                                                       @pbccTo,
                                                       @pemail,
                                                       @pSubject,
                                                       @pLocation,
                                                       @pDuration,
                                                       0
END

GO

/**********************************************************************
 * The following block of code is used for the print tracker service
 *
 * PRINT BLOCK START
 **********************************************************************/
 IF OBJECT_ID(N'PrintJobTracker',N'U') IS NULL
BEGIN
	 CREATE TABLE [dbo].[PrintJobTracker](
	        [RecordID]         INT IDENTITY(1,1) NOT NULL,
	        [MatterCode]       NVARCHAR(20) NULL,
	        [Handler]          NVARCHAR(50) NULL,
	 	    [JobID]            NVARCHAR(100) NULL,
	 	    [FileName]         NVARCHAR(200) NULL,
	 	    [DeviceName]       NVARCHAR(500) NULL,
	 	    [NoOfPages]        INT NULL,
	 	    [LoggedInUser]     NVARCHAR(300) NULL, 
	 	    [LastUpdated]      DATETIME NULL,
	 	    [Status]           NVARCHAR(20) NULL,
	 	    [SystemIP]         NVARCHAR(30) NULL,
	 	    [Machinename]      NVARCHAR(50) NULL,
	 	    [JobUniqueID]      UNIQUEIDENTIFIER DEFAULT NEWID(),
	 	    [PageSize]         NVARCHAR(50) NULL,
	 	    [IsColorPrinter]   BIT NULL,
	 	    [ApplicationType]  VARCHAR(60) NULL,
	 	    [OriginalFileName] VARCHAR(200) NULL,
	 	    [TrackReference]   INT NULL
	 	    CONSTRAINT [PK_PrintJobTracker] PRIMARY KEY (RecordID))
END

GO

IF NOT EXISTS(SELECT TOP 1 1 FROM SYS.columns WHERE OBJECT_NAME(OBJECT_ID) = 'PrintJobTracker' and [name] = 'PageSize')
BEGIN
	ALTER TABLE [PrintJobTracker]
	ADD [PageSize] NVARCHAR(50) NULL
END

GO

IF NOT EXISTS(SELECT TOP 1 1 FROM SYS.columns WHERE OBJECT_NAME(OBJECT_ID) = 'PrintJobTracker' and [name] = 'IsColorPrinter')
BEGIN
	ALTER TABLE [PrintJobTracker]
	ADD [IsColorPrinter] BIT NULL
END

GO

IF NOT EXISTS(SELECT TOP 1 1 FROM SYS.columns WHERE OBJECT_NAME(OBJECT_ID) = 'PrintJobTracker' and [name] = 'ApplicationType')
BEGIN
	ALTER TABLE [PrintJobTracker]
	ADD [ApplicationType] NVARCHAR(60) NULL
END

GO

IF NOT EXISTS(SELECT TOP 1 1 FROM SYS.columns WHERE OBJECT_NAME(OBJECT_ID) = 'PrintJobTracker' and [name] = 'OrginalFileName')
BEGIN
	ALTER TABLE [PrintJobTracker]
	ADD [OrginalFileName] NVARCHAR(200) NULL
END

GO

IF NOT EXISTS(SELECT TOP 1 1 FROM SYS.columns WHERE OBJECT_NAME(OBJECT_ID) = 'PrintJobTracker' and [name] = 'TrackReference')
BEGIN
	ALTER TABLE [PrintJobTracker]
	ADD [TrackReference] INT NULL
END

GO

IF EXISTS(SELECT TOP 1 1 FROM SYS.columns WHERE OBJECT_NAME(OBJECT_ID) = 'PrintJobTracker' and [name] = 'OrginalFileName')
BEGIN
	IF NOT EXISTS(SELECT TOP 1 1 FROM SYS.columns WHERE OBJECT_NAME(OBJECT_ID) = 'PrintJobTracker' and [name] = 'OriginalFileName')
	BEGIN
		EXEC sp_RENAME 'PrintJobTracker.OrginalFileName' , 'OriginalFileName', 'COLUMN'
	END
END

GO

IF NOT EXISTS(SELECT TOP 1 1 FROM SYS.columns WHERE OBJECT_NAME(OBJECT_ID) = 'PrintJobTracker' and [name] = 'Cost')
BEGIN
	ALTER TABLE [PrintJobTracker]
	ADD [Cost] DECIMAL(19, 2) NULL
END

GO

IF NOT EXISTS(SELECT TOP 1 1 FROM SYS.columns WHERE OBJECT_NAME(OBJECT_ID) = 'PrintJobTracker' and [name] = 'InvoiceNo')
BEGIN
	ALTER TABLE [PrintJobTracker]
	ADD [InvoiceNo] INT NULL DEFAULT 0
END

GO

IF OBJECT_ID('PrintJobTrackerLog') IS NULL
BEGIN
	CREATE TABLE [dbo].[PrintJobTrackerLog] (
	       [RecordID] INT IDENTITY(1, 1) PRIMARY KEY,
	       [Handler]  NVARCHAR(100),
	       [Action]   NVARCHAR(100),
	       [Date]     DATETIME,
	       [Before]   NVARCHAR(MAX),
	       [After]    NVARCHAR(MAX))
END

GO

IF NOT EXISTS(SELECT TOP 1 1 FROM SYS.columns WHERE OBJECT_NAME(OBJECT_ID) = 'PrintJobTracker' and [name] = 'NoOfCopies')
BEGIN
	ALTER TABLE [PrintJobTracker]
	ADD [NoOfCopies] INT NULL DEFAULT 0
END

GO

IF OBJECT_ID(N'[ky_NETFetchCaseForPrintJob]',N'P')IS NOT NULL
	DROP PROCEDURE [ky_NETFetchCaseForPrintJob]

GO

CREATE PROCEDURE [dbo].[ky_NETFetchCaseForPrintJob] (
       @JOBNAME VARCHAR(MAX)
) 
AS
/*****************************************************************************        
        
Stored Procedure Name: ky_NETFetchCaseForPrintJob  
Description:  this procedure used to fetch Matter of a document printed.   

*****************************************************************************/
BEGIN
	SET NOCOUNT ON
    SELECT PJT.[MatterCode],
           PJT.[Handler],
           PJT.[JobID],
           PJT.[FileName],
           PJT.[DeviceName],
           PJT.[NoOfPages],
           PJT.[LoggedInUser],
           PJT.[Machinename]
      FROM [dbo].[PrintJobTracker] AS PJT
     WHERE PJT.[DeviceName] + CONVERT(VARCHAR(20), PJT.[JobID]) = @JOBNAME
	SET NOCOUNT OFF
END

GO

IF OBJECT_ID(N'FetchJobCollection',N'P') IS NOT NULL
	DROP PROCEDURE [dbo].[FetchJobCollection]

GO

CREATE PROCEDURE [dbo].[FetchJobCollection] (
	   @MachineName VARCHAR(50) 
)
AS
/*****************************************************************************

 FetchJobCollection

 This procesure will fetch all the job which status is equal to empty

 Modification History:
 *****************************************************************************/
BEGIN
	SELECT PJT.[JobUniqueID],  
	       PJT.[JobID],
	       PJT.[FileName],
	       PJT.[DeviceName],
	       CONVERT(VARCHAR(20), PJT.[NoOfPages]),
	       PJT.[LoggedInUser],
	       PJT.[Machinename],
	       PJT.[status]
	  FROM [dbo].[PrintJobTracker] AS PJT
	 WHERE PJT.[Status] = '' 
	   AND PJT.[Machinename] = @MachineName
END

GO

/**********************************************************************
 * PRINT BLOCK END
 **********************************************************************/

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] [SO]
    WHERE [SO].[name] = 'ky_NETCIAddNoteForTrackReference'
      AND [SO].[type] = 'TR')
  BEGIN
    EXEC ('
    ENABLE TRIGGER [dbo].[ky_NETCIAddNoteForTrackReference]
                ON [dbo].[DiaryAttachments]')
  END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETWizFetchContactCode'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETWizFetchContactCode]
  END
GO

CREATE PROCEDURE ky_NETWizFetchContactCode
      (@ClientCodePrefix   VARCHAR(10),
       @Result             VARCHAR(10)   OUTPUT,
       @Error              VARCHAR(100)  OUTPUT)
AS
/***************************************************************************************************************************************
 Stored Procedure Name: [ky_NETWizFetchContactCode]

 Description: This procedure generates client code AS per the settings in
     Table  : Control
    Column  : AutoAlphaClientNumbering

 Last Updated: 28MAY2014 02:19 PM

 Modification History:  21 Nov 2014 - Arun.v - Client Code generate based on setting
                        24 Nov 2014 - Pino - Pad with blanks if AutoAlphaClientNumbering = 'N'
                                             make SAM4 enabled (default length 8)
                                             Improved handling of prefixes matching or exceeding
                                             the maximum lengh (truncate if the latter is
                                             the case)
                        26 Nov 2014 - Arun.v - AutoClientCode settings added, (If its return Y
                                               then it will return client maximum code, If its N
                                               as per setting it generate client code)

                        19 Dec 2014 - Arun.v - Auto Client Number is true, then zero padding get removed issue fixed.
                        02 Jan 2015 - Arun.v - Auto Client Number is true then if auto generate number
                                               is already exist it will create new number
                        19 Jan 2015 - Pino - Auto Client Number is true - make sure Max Client No Length is taken into account
                        02 Oct 2015 - Pino - Auto Client Number is true - SAM4 - it MAY be using [dbo].[CtrlNum] for code 'NextClientNo'
                                                                                 if it does, use it.
                        03 Jul 2017 - John - Changed @NewNumber assignment so it's getting MAX(INT) rather than MAX(VARCHAR)
                                             Added additional criteria for the codes so that the numer section has to be strictly numeric
                                             See KEYD-4658

 ***************************************************************************************************************************************/
BEGIN
  SET NOCOUNT ON

  DECLARE @IsClientCodeAutoGen BIT
  DECLARE @MaxClientCode INT
  DECLARE @NewNumber INT
  DECLARE @NewSuffix VARCHAR(10)
  DECLARE @AutoClientCode VARCHAR(20)
  DECLARE @PrefixCode VARCHAR(10)

  SET @PrefixCode = '0000000000'
  SET @AutoClientCode = (SELECT ISNULL(AUTONUMBER,'N')  from [control] )

  -- Get the maximum length of the Client Code. Read Settings first...
  SELECT @MaxClientCode = [SET].[KeyValue]
    FROM [dbo].[Settings] [SET]
   WHERE [SET].[KeyName] = 'MaxClCode'

  -- ... and default if not present. 8 for SAM4, 6 for SAM3
  IF (@MaxClientCode IS NULL)
    BEGIN
      IF ([dbo].[ISSAM4]() = 0)
        BEGIN
          SET @MaxClientCode = 6
        END
      ELSE
        BEGIN
          SET @MaxClientCode = 8
        END
    END

  IF(@AutoClientCode='N')
    BEGIN
      -- Trim the prefix
      SELECT @ClientCodePrefix = RTRIM(ISNULL(@ClientCodePrefix, ''))

      -- Truncate the prefix if it's too long
      IF (LEN(@ClientCodePrefix) > @MaxClientCode)
        BEGIN
          SET @ClientCodePrefix = SUBSTRING(@ClientCodePrefix, 1, @MaxClientCode)
        END

      -- Find out how to pad the Client Code. Either add a number, or pad with blanks.
      -- Default 1 - Add a number
      SET @IsClientCodeAutoGen = 1

      -- Read from [dbo].[Control]. Value should be 'Y', translating to 1 or 'N' translating to 0.
      -- default value is 1 ('Y'), so anything other than an explicit 'N' will be read as 'Y'
      SELECT @IsClientCodeAutoGen = CASE WHEN ISNULL(CTL.[AutoAlphaClientNumbering], 'Y') = 'N' THEN 0 ELSE 1 END
        FROM [dbo].[Control] CTL

      IF (@IsClientCodeAutoGen = 1)
        BEGIN
          -- Get all the contacts that have a numeric suffix for the Prefix that was passed in, and get the largest one in order to
          -- increment by one.
		  -- Added CONVERT(INT) here to give us numerically largest value - John
		  -- Added a PATINDEX() to make sure we don't have any non-numeric characters that pass ISNUMERIC()
          SELECT @NewNumber = ISNULL(MAX(CONVERT(INT, NUF.[SUF])), 0) + 1
            FROM (SELECT SUBSTRING(CON.[Code], LEN(@ClientCodePrefix) + 1, LEN([Code]) - LEN(@ClientCodePrefix)) AS [SUF]
                    FROM [dbo].[Contacts] CON
                   WHERE CON.[Code] LIKE @ClientCodePrefix + '%'
                     AND ISNUMERIC(SUBSTRING(CON.[Code], LEN(@ClientCodePrefix) + 1, LEN([Code]) - LEN(@ClientCodePrefix))) = 1
                     AND PATINDEX('%[\,.$+-\D\E]%', SUBSTRING(CON.[Code], LEN(@ClientCodePrefix) + 1, LEN([Code]) - LEN(@ClientCodePrefix))) = 0) NUF

          -- Pad new suffix with zeros
          SET @NewSuffix = SUBSTRING(CONVERT(VARCHAR(11), CONVERT(BIGINT, POWER(CONVERT(BIGINT, 10), @MaxClientCode)) + @NewNumber), 2, @MaxClientCode)

          -- Because we truncated the Prefix if it's too long we can now be sure that the prefix is at most as long as @MaxClientCode, so we can easily append
          -- the suffix to the prefix:
          SET @NewSuffix = @ClientCodePrefix + SUBSTRING(@NewSuffix, LEN(@ClientCodePrefix) + 1, @MaxClientCode - LEN(@ClientCodePrefix))
        END
      ELSE
        BEGIN
          SET @NewSuffix = SUBSTRING(@ClientCodePrefix + '          ', 1, @MaxClientCode)
        END
    END
  ELSE
    BEGIN
      DECLARE @NextClientNo INT

      IF [dbo].[ISSAM4]() = 1
        BEGIN
          DECLARE @NCOMMAND NVARCHAR(MAX)

          SET @NCOMMAND = N'SELECT @NextClientNo = CONVERT(INT, CTN.[NumValue])
  FROM [dbo].[CtrlNum] CTN
 WHERE CTN.[Code] = ''NextClientNo'''
 
          EXEC sp_executesql @NCOMMAND, N'@NextClientNo INT OUTPUT', @NextClientNo OUTPUT
        END

      IF @NextClientNo IS NOT NULL
        WHILE EXISTS
             (SELECT TOP 1 1
                FROM [dbo].[Contacts] CNT
               WHERE CASE WHEN ISNUMERIC(CNT.[Code]) = 0
                               THEN 0
                          ELSE CONVERT(INT, CONVERT(FLOAT, CNT.[CODE])) END = @NextClientNo)
          BEGIN
            SET @NextClientNo = @NextClientNo + 1
          END

      IF (@NextClientNo IS NOT NULL)
        BEGIN
          SET @NewSuffix = SUBSTRING(CONVERT(VARCHAR(10), POWER(10, @MaxClientCode) + @NextClientNo), 2, @MaxClientCode)
        END
      ELSE
        BEGIN
          SELECT @NewSuffix = SUBSTRING(CONVERT(VARCHAR(10), CASE WHEN NCC.[NewContactNo] >= NCL.[NewClientNo]
                                                                       THEN NCC.[NewContactNo]
                                                                  ELSE NCL.[NewClientNo] END), 2, @MaxClientCode)
            FROM (SELECT POWER(10, @MaxClientCode) + ISNULL(MAX(CONVERT(INT, CON.[CODE])), 0) + 1 AS [NewContactNo]
                    FROM [dbo].[Contacts] CON
                   WHERE ISNUMERIC(CON.[CODE]) = 1
                     AND CHARINDEX('D',con.[code])=0
                     AND CHARINDEX('E',con.[Code])=0
                     AND CHARINDEX(',',con.[Code])=0
                     AND CHARINDEX('.', CON.[CODE]) = 0) NCC
                 CROSS APPLY (SELECT POWER(10, @MaxClientCode) + ISNULL(MAX(CONVERT(INT, CLT.[CLCODE])), 0) + 1 AS [NewClientNo]
                                FROM [dbo].[client] CLT
                               WHERE ISNUMERIC(CLT.[CLCODE]) = 1
                                 AND CHARINDEX('D',CLT.[CLCODE])=0
                                 AND CHARINDEX('E',CLT.[CLCODE])=0
                                 AND CHARINDEX(',',CLT.[CLCODE])=0
                                 AND CHARINDEX('.', CLT.[CLCODE]) = 0) NCL
        END

    END

  IF EXISTS
    (SELECT TOP 1 1
       FROM [dbo].[client] CLT
      WHERE RTRIM(CLT.[CLCODE]) = RTRIM(@NewSuffix))
    BEGIN
      SET @Result = '0'
      SET @Error = 'Client Code ''' + RTRIM(@NewSuffix) + ''' is already in use'
    END
  ELSE
    BEGIN
      IF(LEN(@NewSuffix)<=@MaxClientCode )
        BEGIN
          SET @Result = @NewSuffix
          SET @Error = '0'
        END
      ELSE
        BEGIN
          SET @Result = '0'
          SET @Error = 'Client Code length exceed the maximum limit, Auto Client code cannot be generate.'
        END

    END

  SET NOCOUNT OFF
END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPSetDocVersioningDate'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPSetDocVersioningDate]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETSPSetDocVersioningDate]
      (@TrackReference    INT,
       @Handler           VARCHAR(10),
       @DateTime          VARCHAR(20))
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETSPSetDocVersioningDate]                                                                      *
 *                                                                                                           *
 * Sets DateTime information on diary attachments after versioning                                           *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.5.1.2                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2017-07-04   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

UPDATE [DAT]
   SET [DAT].[DATEENTERED] = @DateTime,
       [DAT].[ENTEREDBY] = @Handler,
       [DAT].[LASTACCESSDATE] = @DateTime,
       [DAT].[LASTACCESSBY] = @Handler
  FROM [dbo].[DiaryAttachments] [DAT]
       INNER JOIN [dbo].[Handlers] [HAN]
               ON [HAN].[CODE] = @Handler
 WHERE [DAT].[TrackReference] = @TrackReference

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPSetGeneratedActionSubject'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPSetGeneratedActionSubject]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETSPSetGeneratedActionSubject]
      (@ActionID          INT,
       @NewSubject        VARCHAR(500))
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETSPSetGeneratedActionSubject]                                                                 *
 *                                                                                                           *
 * Sets Action Subject on basis of generated document name                                                   *
 * In order for this to do anything, the Action Subject must have a Question Mark in it.                     *
 * If the Action Subject doesn't have a question mark, this procedure does nothing.                          *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.5.1.2                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2017-07-04   Pino Carafa       Created                                                                    *
 * 2019-02-21   Pino Carafa       Include TEXT1 and DisplayText, but ONLY if all three columns               *
 *                                TEXT1, DisplayText and Subject contain the same value with a ?             *
 * 2019-10-01   Pino Carafa       Add code to prevent KEYD-6173                                              *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

DECLARE @OLDSUBJECT VARCHAR(500)

SELECT @OLDSUBJECT = RTRIM(ISNULL([DIA].[SUBJECT], ''))
  FROM [dbo].[Diary] [DIA]
 WHERE [DIA].[ACTIONID] = @ActionID

SET @NewSubject = RTRIM(ISNULL(@NewSubject, ''))

-- KEYD-6173 START
DECLARE @ReverseSubject VARCHAR(500)
DECLARE @FOUND BIT

SET @ReverseSubject = REVERSE(@NewSubject)
SET @FOUND = 0

IF CHARINDEX(' ', @ReverseSubject) > 0
  BEGIN
    DECLARE @NUM VARCHAR(10)
    SET @NUM = CONVERT(VARCHAR(10), SUBSTRING(@ReverseSubject, 1, CHARINDEX(' ', @ReverseSubject)))
    
    IF ISNUMERIC(@NUM) = 1
      BEGIN
        IF CONVERT(FLOAT, CONVERT(INT, FLOOR(CONVERT(FLOAT, @NUM)))) = CONVERT(FLOAT, @NUM)
          BEGIN
            SET @ReverseSubject = SUBSTRING(@ReverseSubject, CHARINDEX(' ', @ReverseSubject) + 1, LEN(@ReverseSubject) - CHARINDEX(' ', @ReverseSubject))
            SET @FOUND = 1
          ENd
      END
  END

IF @FOUND = 1
  BEGIN
    IF (CHARINDEX(' ', @ReverseSubject) > 0) AND (CHARINDEX(' ', @ReverseSubject) < 14)
      BEGIN
        IF EXISTS
          (SELECT TOP 1 1
             FROM [dbo].[AssociateTypes] [AST]
            WHERE [AST].[CODE] =  REVERSE(SUBSTRING(@ReverseSubject, 1, CHARINDEX(' ', @ReverseSubject) - 1)))
          BEGIN
            SET @NEWSUBJECT = REVERSE(SUBSTRING(@ReverseSubject, CHARINDEX(' ', @ReverseSubject) + 1, LEN(@ReverseSubject) - CHARINDEX(' ', @ReverseSubject)))
          END
      END
  END
-- KEYD-6173 END
 
UPDATE [DIA]
   SET [DIA].[SUBJECT] = @NewSubject,
       [DIA].[TEXT1] = CASE WHEN CONVERT(VARCHAR(500), [DIA].[TEXT1]) = @OLDSUBJECT
                             AND CONVERT(VARCHAR(500), [DIA].[DisplayText]) = @OLDSUBJECT
                                 THEN @NewSubject
                            ELSE [DIA].[TEXT1] END,
       [DIA].[DisplayText] = CASE WHEN CONVERT(VARCHAR(500), [DIA].[TEXT1]) = @OLDSUBJECT
                                   AND CONVERT(VARCHAR(500), [DIA].[DisplayText]) = @OLDSUBJECT
                                       THEN CONVERT(VARCHAR(200), @NewSubject)
                                  ELSE [DIA].[DisplayText] END
  FROM [dbo].[diary] [DIA]
 WHERE [DIA].[ACTIONID] = @ActionID
   AND CHARINDEX('?', [DIA].[SUBJECT]) > 0

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPWasFileDeleted'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPWasFileDeleted]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETSPWasFileDeleted]
      (@FILENAME VARCHAR(255))
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETSPWasFileDeleted]                                                                            *
 *                                                                                                           *
 * Checks whether a file is present in the DiaryAttachmentDeletionLog                                        *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.5.1.2                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2017-07-05   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

DECLARE @TRACKREF VARCHAR(255)
DECLARE @TRACKREFFLOAT FLOAT
DECLARE @TRACKREFERENCE INT

SET @TRACKREFERENCE = 0

SET @FILENAME = ISNULL(@FILENAME, '')

SET @TRACKREF = @FILENAME

-- In most places the filename will be like <filename>_<trackreference>.<extension>
-- There can be places where files will be named <trackreference>.<extension> but this first
-- statement won't touch those, so that's OK.
WHILE CHARINDEX('_', @TRACKREF) <> 0
  BEGIN
    -- take the bit up to the underscore out, keeping in mind filenames can continue multiple underscores,
    -- hence the use of WHILE so ultimately this will leave only the bit between the LAST underscore
    -- and the extension
    SET @TRACKREF = SUBSTRING(@TRACKREF, CHARINDEX('_', @TRACKREF) + 1, LEN(@TRACKREF) - CHARINDEX('_', @TRACKREF))
  END

-- if the file has an extension, strip it out, leaving only <trackreference>
IF CHARINDEX('.', @TRACKREF) <> 0
  BEGIN
    SET @TRACKREF = SUBSTRING(@TRACKREF, 1, CHARINDEX('.', @TRACKREF) - 1)
  END

-- If the remaining bit isn't numeric it can't be a Track Reference
IF ISNUMERIC(@TRACKREF) = 1
  BEGIN

    -- But if it IS numeric, it could still contain weird values. E.g. FILE_1E15.TXT would
    -- leave 1E15 which ISNUMERIC recognises as numeric. So instead of converting it to
    -- an INTEGER we'll convert it to a FLOAT first, and then check whether the resulting
    -- value is a valid integer with a value between 1 and the maximum integer value
    -- of 2147483647
    SET @TRACKREFFLOAT = CONVERT(FLOAT, @TRACKREF)

    IF (@TRACKREFFLOAT > 1) AND (@TRACKREFFLOAT <= CONVERT(FLOAT, 2147483647))
      -- So first of all the converted value needs to be within a valid INTEGER range
      BEGIN
        IF FLOOR(@TRACKREFFLOAT) = @TRACKREFFLOAT
          -- And secondly, it can't be a fraction.
          BEGIN
            -- Now we can be sure that what we stripped out is a valid integer, and proceed
            SET @TRACKREFERENCE = CONVERT(INT, @TRACKREFFLOAT)
          END
      END
  END

-- If we ended up here with a valid integer Track Reference, we can check whether a row exists
-- in [dbo].[DiaryAttachmentDeletionLog] for that Track Reference, but just to be absolutely sure
-- that we're not coming back with a False Positive, let's also ensure that the XML string contains
-- a FILENAME="....." attribute that matches the File Name we passed in.
IF @TRACKREFERENCE <> 0
  BEGIN
    DECLARE @DocumentName VARCHAR(255)

    IF PATINDEX('%_' + CONVERT(VARCHAR(10), @TRACKREFERENCE) + '.%', @FILENAME) > 0
      BEGIN
        SET @DocumentName = SUBSTRING(@FILENAME, 1, PATINDEX('%_' + CONVERT(VARCHAR(10), @TRACKREFERENCE) + '.%', @FILENAME) - 1)
      END
    ELSE
      BEGIN
        SET @DocumentName = '****'
      END

    SELECT CASE WHEN COUNT(1) = 0
                     THEN CONVERT(BIT, 0)
                ELSE CONVERT(BIT, 1) END                           AS [WasDeleted]
      FROM [dbo].[DiaryAttachmentDeletionLog] [DATL]
     WHERE [DATL].[TrackReference] = @TRACKREFERENCE
       AND (   [DATL].[deletedrow] LIKE '%FILEPATH="%' + @FILENAME + '"%'
            OR [DATL].[deletedrow] LIKE '%DOCUMENT="' + @DocumentName + '"%')
  END
ELSE
  BEGIN
    -- otherwise we cannot confirm that this is a deleted Diary Attachment so we'll just return a False
    SELECT CONVERT(BIT, 0)                                         AS [WasDeleted]
  END

END

GO

/*
 * This table will be used by Sheriff systems to automatically insert charges onto a debt ledger upon matter creation
 *
 * The reason for making the value fields into strings is because we might need formulas that are based on existing data
 * to calculate the costs of some charges (e.g. poundage)
 */
IF OBJECT_ID('[dbo].[InitialDebtCharges]', 'U') IS NULL
BEGIN
	CREATE TABLE [dbo].[InitialDebtCharges] (
	       [id]          INT IDENTITY(1, 1),
	       [Reference]   VARCHAR(20),
	       [Description] VARCHAR(200),
	       [Type]        VARCHAR(20),
	       [Value]       VARCHAR(MAX),
	       [Interest]    VARCHAR(MAX),
	       [Costs]       VARCHAR(MAX))
END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPFileSharingServerExists'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPFileSharingServerExists]
  END
GO

CREATE PROCEDURE
       [ky_NETSPFileSharingServerExists]
      (@ServerName VARCHAR(50))
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETSPFileSharingServerExists]                                                                   *
 *                                                                                                           *
 * Checks whether a File Sharing Server already exists                                                       *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.5.1.2                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2017-07-10   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/
 BEGIN
    SELECT CASE WHEN COUNT(1) = 0
                     THEN CONVERT(BIT, 0)
                ELSE CONVERT(BIT, 1) END         AS [FileSharingServerExists]
      FROM [dbo].[FileSharingServers] [FSS]
     WHERE [FSS].[Name] = @ServerName
 END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPFileSharingAddServer'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPFileSharingAddServer]
  END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPFileSharingAddServer2'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPFileSharingAddServer2]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETSPFileSharingAddServer2]
      (@ServerID                     INT,
       @TypeID                       INT,
       @Name                         VARCHAR(50),
       @Server                       VARCHAR(50),
       @AdminUserId                  VARCHAR(100),
       @AdminPassword                VARCHAR(100),
       @ServerFolderID               VARCHAR(500),
       @ClientFolderURL              VARCHAR(500),
       @ClientFolderPermissionsURL   VARCHAR(500),
       @CollabDocFolderID            INT,
       @CollabMatter                 VARCHAR(50),
       @CollabFolder                 VARCHAR(50),
       @ClientID                     VARCHAR(100),
       @ClientSecret                 VARCHAR(500),
       @SiteID                       VARCHAR(100),
       @Domain                       VARCHAR(100),
       @ProxyType                    INT,
       @ProxyServer                  VARCHAR(500),
       @ProxyPort                    INT,
       @ServerMatterMatterCode       VARCHAR(20),
       @ServerMatterFolderID         VARCHAR(500))
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETSPFileSharingAddServer2]                                                                     *
 *                                                                                                           *
 * Add details for a File Sharing Server                                                                     *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              [dbo].[ky_NETSPFileSharingAddServer]                     *   *
 *   * First compatible version:                5.5.3.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2017-07-10   Pino Carafa       Created                                                                    *
 * 2017-08-03   Pino Carafa       Added Client ID and Client Secret                                          *
 * 2017-08-04   Pino Carafa       Added Site ID                                                              *
 * 2017-08-24   Pino Carafa       Added Domain and Proxy Settings KEYD-4750 and KEYD-4751                    *
 * 2017-08-25   Pino Carafa       Allow for updating also                                                    *
 * 2017-10-25   Pino Carafa       Remove all Shares, Handler info and Mapping when a Server is updated,      *
 *                                because this no longer refers to valid folders.                            *
 * 2017-12-04   Pino Carafa       KEYD-4940 - Add code to create new FileSharingMatterMapping                *
 * 2017-12-20   Pino Carafa       KEYD-4962 - Don't remove Mappings and Shares if the only thing changed on  *
 *                                            a server are its Admin credentials.                            *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

SET @ServerID = ISNULL(@ServerID, 0)

IF @ServerID = 0
  BEGIN
    SELECT TOP 1 @ServerID = [FSS].[id]
      FROM [dbo].[FileSharingServers] [FSS]
     WHERE [FSS].[id] = @ServerID

    SET @ServerID = ISNULL(@ServerID, 0)
  END

IF @ServerID = 0
  BEGIN
    SELECT TOP 1 @ServerID = [FSS].[id]
      FROM [dbo].[FileSharingServers] [FSS]
     WHERE [FSS].[Name] = @Name

    SET @ServerID = ISNULL(@ServerID, 0)
  END

IF ISNULL(@CollabDocFolderID, 0) = 0
  BEGIN
    SET @CollabDocFolderID = null
    SET @CollabMatter = null
    SET @CollabFolder = null
  END

IF @ServerID = 0
  BEGIN
    INSERT
      INTO [dbo].[FileSharingServers]
          ([TypeId],
           [Name],
           [Server],
           [AdminUserId],
           [AdminPassword],
           [RootFolder],
           [ClientFolderURL],
           [ClientFolderPermissionsURL],
           [CollabDocFolderID],
           [CollabMatter],
           [CollabFolder],
           [ClientID],
           [ClientSecret],
           [SiteID],
           [Domain],
           [ProxyType],
           [ProxyServer],
           [ProxyPort])
    VALUES(@TypeID,
           @Name,
           @Server,
           @AdminUserId,
           @AdminPassword,
           @ServerFolderID,
           @ClientFolderURL,
           @ClientFolderPermissionsURL,
           @CollabDocFolderID,
           @CollabMatter,
           @CollabFolder,
           @ClientID,
           @ClientSecret,
           @SiteID,
           @Domain,
           @ProxyType,
           @ProxyServer,
           @ProxyPort)

    SET @ServerID = CONVERT(INT, SCOPE_IDENTITY())

  END
ELSE
  BEGIN
    DECLARE @BDelete BIT
    SET @BDelete = 0
    IF EXISTS
      (SELECT TOP 1 1
         FROM [dbo].[FileSharingServers] [FSS]
        WHERE [FSS].[id] = @ServerID
          AND (   [FSS].[TypeId] <> @TypeID
               OR [FSS].[Name] <> @Name
               OR [FSS].[Server] <> @Server
               OR [FSS].[RootFolder] <> @ServerFolderID
               OR [FSS].[ClientID] <> @ClientID
               OR [FSS].[ClientSecret] <> @ClientSecret
               OR [FSS].[SiteID] <> @SiteID
               OR [FSS].[Domain] <> @Domain))
      BEGIN
        SET @BDelete = 1
      END

    UPDATE [FSS]
       SET [FSS].[TypeId] = @TypeID,
           [FSS].[Name] = @Name,
           [FSS].[Server] = @Server,
           [FSS].[AdminUserId] = @AdminUserId,
           [FSS].[AdminPassword] = @AdminPassword,
           [FSS].[RootFolder] = @ServerFolderID,
           [FSS].[ClientFolderURL] = @ClientFolderURL,
           [FSS].[ClientFolderPermissionsURL] = @ClientFolderPermissionsURL,
           [FSS].[ClientID] = @ClientID,
           [FSS].[ClientSecret] = @ClientSecret,
           [FSS].[SiteID] = @SiteID,
           [FSS].[CollabDocFolderID] = null,
           [FSS].[CollabMatter] = null,
           [FSS].[CollabFolder] = null,
           [FSS].[Domain] = @Domain,
           [FSS].[ProxyType] = @ProxyType,
           [FSS].[ProxyServer] = @ProxyServer,
           [FSS].[ProxyPort] = @ProxyPort
      FROM [dbo].[FileSharingServers] [FSS]
     WHERE [FSS].[id] = @ServerID

    -- Remove all current share information as this is no longer correct for this Server
    IF @BDelete = 1
      BEGIN
        DELETE [FSD]
          FROM [dbo].[FileSharingMapping] [FSM]
               INNER JOIN [dbo].[FileSharingDocuments] [FSD]
                       ON [FSD].[FileSharingMappingID] = [FSM].[id]
         WHERE [FSM].[FileSharingServersID] = @ServerID

        UPDATE [FSDF]
           SET [FSDF].[Shared] = 0
          FROM [dbo].[FileSharingMapping] [FSM]
               INNER JOIN [dbo].[FileSharingDocFolders] [FSDF]
                       ON [FSDF].[DocFolderID] = [FSM].[FileSharingDocFoldersID]
         WHERE [FSM].[FileSharingServersID] = @ServerID

        DELETE [FSM]
          FROM [dbo].[FileSharingMapping] [FSM]
         WHERE [FSM].[FileSharingServersID] = @ServerID

        DELETE [FSMM]
          FROM [dbo].[FileSharingMatterMapping] [FSMM]
         WHERE [FSMM].[FileSharingServersID] = @ServerID
      END

    -- We *will* remove the File Sharing Handlers, so that ALL users will
    -- have to re-enter their credentials
    DELETE [FSH]
      FROM [dbo].[FileSharingHandlers] [FSH]
     WHERE [FSH].[FileSharingServersID] = @ServerID
  END

SET @ServerMatterFolderID = RTRIM(ISNULL(@ServerMatterFolderID, ''))

IF @ServerMatterFolderID <> ''
  BEGIN
    INSERT
      INTO [dbo].[FileSharingMatterMapping]
          ([FileSharingMatter],
           [FileSharingServersID],
           [FileSharingServerFolderID])
    VALUES(@ServerMatterMatterCode,
           @ServerID,
           @ServerMatterFolderID)
  END

SELECT @ServerID AS [NewId]

END

GO

CREATE PROCEDURE
       [dbo].[ky_NETSPFileSharingAddServer]
      (@ServerID                     INT,
       @TypeID                       INT,
       @Name                         VARCHAR(50),
       @Server                       VARCHAR(50),
       @AdminUserId                  VARCHAR(100),
       @AdminPassword                VARCHAR(100),
       @ServerFolderID               VARCHAR(500),
       @ClientFolderURL              VARCHAR(500),
       @ClientFolderPermissionsURL   VARCHAR(500),
       @CollabDocFolderID            INT,
       @CollabMatter                 VARCHAR(50),
       @CollabFolder                 VARCHAR(50),
       @ClientID                     VARCHAR(100),
       @ClientSecret                 VARCHAR(500),
       @SiteID                       VARCHAR(100),
       @Domain                       VARCHAR(100),
       @ProxyType                    INT,
       @ProxyServer                  VARCHAR(500),
       @ProxyPort                    INT)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETSPFileSharingAddServer]                                                                      *
 *                                                                                                           *
 * Add details for a File Sharing Server                                                                     *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.5.1.2                                                  *   *
 *   * Last compatible software version:        5.5.2.1                                                  *   *
 *   * Superseded by:                           [dbo].[ky_NETSPFileSharingAddServer2]                    *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2017-07-10   Pino Carafa       Created                                                                    *
 * 2017-08-03   Pino Carafa       Added Client ID and Client Secret                                          *
 * 2017-08-04   Pino Carafa       Added Site ID                                                              *
 * 2017-08-24   Pino Carafa       Added Domain and Proxy Settings KEYD-4750 and KEYD-4751                    *
 * 2017-08-25   Pino Carafa       Allow for updating also                                                    *
 * 2017-10-25   Pino Carafa       Remove all Shares, Handler info and Mapping when a Server is updated,      *
 *                                because this no longer refers to valid folders.                            *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

EXEC [dbo].[ky_NETSPFileSharingAddServer2]
     @ServerID,
     @TypeID,
     @Name,
     @Server,
     @AdminUserId,
     @AdminPassword,
     @ServerFolderID,
     @ClientFolderURL,
     @ClientFolderPermissionsURL,
     @CollabDocFolderID,
     @CollabMatter,
     @CollabFolder,
     @ClientID,
     @ClientSecret,
     @SiteID,
     @Domain,
     @ProxyType,
     @ProxyServer,
     @ProxyPort,
     '',
     ''

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPIsFileInSystem'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPIsFileInSystem]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETSPIsFileInSystem]
      (@FILENAME VARCHAR(255))
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETSPIsFileInSystem]                                                                            *
 *                                                                                                           *
 * Checks whether a file is present in the DiaryAttachments Table                                            *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.5.1.2                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2017-07-12   Pino Carafa       Created                                                                    *
 * 2017-08-16   Pino Carafa       Added functionality to handle underscored file names                       *
 * 2017-08-18   Pino Carafa       Handle Document Naming Method T                                            *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

DECLARE @TRACKREF VARCHAR(255)
DECLARE @TRACKREFFLOAT FLOAT
DECLARE @TRACKREFERENCE INT

SET @TRACKREFERENCE = 0

SET @FILENAME = ISNULL(@FILENAME, '')

SET @TRACKREF = @FILENAME

-- In most places the filename will be like <filename>_<trackreference>.<extension>
-- There can be places where files will be named <trackreference>.<extension> but this first
-- statement won't touch those, so that's OK.
WHILE CHARINDEX('_', @TRACKREF) <> 0
  BEGIN
    -- take the bit up to the underscore out, keeping in mind filenames can continue multiple underscores,
    -- hence the use of WHILE so ultimately this will leave only the bit between the LAST underscore
    -- and the extension
    SET @TRACKREF = SUBSTRING(@TRACKREF, CHARINDEX('_', @TRACKREF) + 1, LEN(@TRACKREF) - CHARINDEX('_', @TRACKREF))
  END

-- if the file has an extension, strip it out, leaving only <trackreference>
IF CHARINDEX('.', @TRACKREF) <> 0
  BEGIN
    SET @TRACKREF = SUBSTRING(@TRACKREF, 1, CHARINDEX('.', @TRACKREF) - 1)
  END

-- If the remaining bit isn't numeric it can't be a Track Reference
IF ISNUMERIC(@TRACKREF) = 1
  BEGIN

    -- But if it IS numeric, it could still contain weird values. E.g. FILE_1E15.TXT would
    -- leave 1E15 which ISNUMERIC recognises as numeric. So instead of converting it to
    -- an INTEGER we'll convert it to a FLOAT first, and then check whether the resulting
    -- value is a valid integer with a value between 1 and the maximum integer value
    -- of 2147483647
    SET @TRACKREFFLOAT = CONVERT(FLOAT, @TRACKREF)

    IF (@TRACKREFFLOAT > 1) AND (@TRACKREFFLOAT <= CONVERT(FLOAT, 2147483647))
      -- So first of all the converted value needs to be within a valid INTEGER range
      BEGIN
        IF FLOOR(@TRACKREFFLOAT) = @TRACKREFFLOAT
          -- And secondly, it can't be a fraction.
          BEGIN
            -- Now we can be sure that what we stripped out is a valid integer, and proceed
            SET @TRACKREFERENCE = CONVERT(INT, @TRACKREFFLOAT)
          END
      END
  END

-- If we ended up here with a valid integer Track Reference, we can check whether a row exists
-- in [dbo].[DiaryAttachments] for that Track Reference, but just to be absolutely sure
-- that we're not coming back with a False Positive, let's also ensure that the FILEPATH
-- matches the File Name we passed in.
IF @TRACKREFERENCE <> 0
  BEGIN
    SELECT TOP 1
           @TrackReference                                         AS [TrackReference]
      FROM [dbo].[DiaryAttachments] [DAT]
     WHERE [DAT].[TrackReference] = @TRACKREFERENCE
       AND (   REPLACE([DAT].[FILEPATH], '_', '') LIKE '%' + REPLACE(@FILENAME, '_', '') + '%'
            OR REPLACE([DAT].[FILEPATH], '_', '') LIKE '%\' + CONVERT(VARCHAR(10), @TrackReference) + '.%')
  END
ELSE
  BEGIN
    -- otherwise we cannot confirm that this is a deleted Diary Attachment so we'll just return a False
    SELECT 0                                                       AS [TrackReference]
  END

END

GO

/*
	KEYD-4663 START
*/
-- This table logs activity on a cheque request. 
-- The procedure [dbo].[ky_NETSPSetChequeRequest_v2]     inserts INTO this table
-- The procedure [dbo].[ky_NETFetchChequeRequisitionLog] gets data FROM this table
IF OBJECT_ID('[dbo].[ChequeRequisitionLog]') IS NULL
BEGIN
	CREATE TABLE [dbo].[ChequeRequisitionLog] (
	       [RecordID]          INT PRIMARY KEY IDENTITY(1, 1),
           [ID]                INT NOT NULL,
           [User]              VARCHAR(20) NOT NULL,
           [ActionDate]        DATETIME NOT NULL,
           [Action]            VARCHAR(200) NOT NULL,
           [Date]              DATETIME,
           [Bank]              VARCHAR(10),
           [Matter]            VARCHAR(20),
           [ClientCode]        VARCHAR(10),
           [Nominal]           VARCHAR(10),
           [Value]             DECIMAL(17, 2),
           [Narrative]         VARCHAR(100),
           [LedgerNarrative]   VARCHAR(100),
           [Payee]             VARCHAR(60),
           [Notes]             VARCHAR(100),
           [FeeEarner]         VARCHAR(10),
           [ThirdParty]        CHAR(1),
           [Supplier]          VARCHAR(10),
           [OutlayCode]        VARCHAR(10),
           [PayClient]         CHAR(1),
           [EFTYorN]           CHAR(1),
           [OnceOffPayment]    CHAR(1),
           [CaseAssociateCode] VARCHAR(6))
END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = 'ky_NETFetchChequeRequisitionLog'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETFetchChequeRequisitionLog]
  END
GO

CREATE PROCEDURE [dbo].[ky_NETFetchChequeRequisitionLog] (
       @ID INT)
AS
/*************************************************************
 *
 * ky_NETFetchChequeRequisitionLog - Fetches the log for a cheque requisition
 *
 * Modification History
 * 2017-07-13 John      Created
 *
 *************************************************************/
BEGIN
	SELECT [ID],
           [User],
           [ActionDate],
           [Action],
           [Date],
           [Bank],
           [Matter],
           [ClientCode],
           [Nominal],
           [Value],
           [Narrative],
           [LedgerNarrative],
           [Payee],
           [Notes],
           [FeeEarner],
           [ThirdParty],
           [Supplier],
           [OutlayCode],
           [PayClient],
           [EFTYorN],
           [OnceOffPayment],
           [CaseAssociateCode]
	  FROM [dbo].[ChequeRequisitionLog] AS CRL
	 WHERE CRL.[ID] = @ID
END

GO

IF  EXISTS (SELECT * 
              FROM sys.objects 
             WHERE [object_id] = OBJECT_ID(N'[dbo].[ky_NETSPReleaseChequeRequest]') 
               AND [type] in (N'P', N'PC'))
	DROP PROCEDURE [dbo].[ky_NETSPReleaseChequeRequest]
GO

CREATE PROCEDURE [dbo].[ky_NETSPReleaseChequeRequest] (
       @IDV VARCHAR(10))
AS
/*************************************************************
 *
 * ky_NETSPReleaseChequeRequest
 *
 * 2017-07-17 John Ginnane      Added missing SP to SVN
 *
 *************************************************************/
BEGIN
	DECLARE @ID              INT
	DECLARE @COMMAND         VARCHAR(MAX)

	BEGIN TRY
	  SET @ID = CONVERT(INT, @IDV)
	END TRY
	BEGIN CATCH
	  SET @ID = 0
	END CATCH

	IF EXISTS
	  (SELECT TOP 1 1 
		 FROM SYS.[objects] SO
		WHERE SO.[name] = 'ChequeRequisitions'
		  AND SO.[type] = 'U')
	BEGIN
	  SET @COMMAND = 'UPDATE [dbo].[ChequeRequisitions] SET [ChequeStatus] = ''Released'' WHERE [RecordId] = ' + CONVERT(VARCHAR(10), @ID) + ' AND [ChequeStatus]=''Request'''
	  EXEC (@COMMAND)
	END

	IF EXISTS
	  (SELECT TOP 1 1 
		 FROM SYS.[objects] SO
		WHERE SO.[name] = 'ChequeRequisitionHeader'
		  AND SO.[type] = 'U')
	BEGIN
	  SET @COMMAND = 'UPDATE [dbo].[ChequeRequisitionHeader] SET [STATUS] = ''1'' WHERE [REQNO] = ' + CONVERT(VARCHAR(10), @ID) + ' AND [STATUS] = ''0'''
	  EXEC (@COMMAND)
	END
END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETGetHandlerLogon'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETGetHandlerLogon]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETGetHandlerLogon]
      (@HandlerCode VARCHAR(10))
AS
/*****************************************************************************
 *
 * ky_NETGetHandlerLogon
 *
 * Get the logon details for a handler
 *
 * Modification History:
 * 2015-01-29 Pino Carafa    - imported into Framework Release 2
 *                           - Added IsPartner
 * 2016-04-12 Suriya         - Added Retired - KEYD-3259-System throwing an exception error when we change the ADM handler to retired
 * 2017-01-05 Arun           - Have include password expiry date condition.
 * 2017-07-18 John           - Added handler type
 * 2019-02-25 Pino Carafa    - Added handler check
 * 2019-07-08 John Ginnane   - KEYD-6012 wrapped HLO fields in ISNULL
 *****************************************************************************/
BEGIN
DECLARE @PasswordExpiryDate INT
SET @PasswordExpiryDate = (
                            SELECT ISNULL(KeyValue,60)
                              FROM Settings
                             WHERE [KeyName] = 'PasswordExpiryDays')


DECLARE @COMMAND VARCHAR(MAX)

SET @COMMAND = 'SELECT [HLO].[HANDLER],
       ISNULL([HLO].[Password], '''')    AS [Password],
       ISNULL([HLO].[Active], '''')      AS [Active],
       ISNULL([HLO].[Supervisor], '''')  AS [Supervisor],
       RTRIM(ISNULL([HAN].[CODE], '''')) AS [CODE],
       RTRIM(ISNULL([HAN].[TEAM], '''')) AS [TEAM],
       RTRIM(ISNULL([HAN].[NAME], '''')) AS [NAME],
       RTRIM(ISNULL([HAN].[EMAIL], '''')) AS [EMAIL],
       REPLACE(ISNULL([HAN].[RETIRED], ''N''),'' '',''N'') AS [RETIRED],
       CASE WHEN RTRIM(ISNULL([HAN].[TYPE], '''')) = ''P''
                 THEN CONVERT(BIT, 1)
            WHEN RTRIM(ISNULL([HAN].[CODE], '''')) = ''ADM''
                 THEN CONVERT(BIT, 1)
            ELSE CONVERT(BIT, 0) END AS [IsPartner],
       ISNULL([GM].[BitWiseID], [dbo].[ky_NETFNGetOldPermissions](RTRIM(ISNULL([HAN].[CODE], '''')))) AS [BitWiseID],
       UPPER(ISNULL([HAN].[AutoLaunchTimer], ''N'')) AS [AutoLaunchTimer],
       CHECKSUM(''KeyhouseHCS'', [HAN].[CODE] + REVERSE([HAN].[CODE]), [HAN].[NAME]) AS [HandlerCheck]'

SELECT @COMMAND = @COMMAND + ',
       CASE WHEN [GM].[BitWiseID] IS NULL
                 THEN CASE WHEN CONVERT(BIGINT, CONVERT(BIGINT, [dbo].[ky_NETFNGetOldPermissions](RTRIM(ISNULL([HAN].[CODE], '''')))) & CONVERT(BIGINT, ' + CONVERT(VARCHAR, SNG.[BitWiseID]) + ')) > CONVERT(BIGINT, 0)
                                THEN 1
                           ELSE 0 END
            WHEN CONVERT(BIGINT, CONVERT(BIGINT, [GM].[BitWiseID]) & CONVERT(BIGINT, ' + CONVERT(VARCHAR, SNG.[BitWiseID]) + ')) > CONVERT(BIGINT, 0)
                 THEN 1
            ELSE 0 END AS [' + Replace(SNG.[GroupName], '''', '''''') + ']'
  FROM [dbo].[SecurityNETGroups] SNG

SET @COMMAND = @COMMAND + ',
  [HLO].[Date] + '+ CONVERT(VARCHAR(5),ISNULL(@PasswordExpiryDate,'60')) +' AS [Date],
       ISNULL([HAN].[TYPE], '''') AS [TYPE]
  FROM [dbo].[Handlers] [HAN]
       LEFT OUTER JOIN [dbo].[HandlerNETLogons] [HLO]
                    ON [HLO].[HANDLER] = [HAN].[CODE]
       LEFT OUTER JOIN [dbo].[HandlerNETGroupMembership] GM
                    ON [GM].[HANDLER] = [HAN].[CODE]
 WHERE [HAN].[CODE] = ''' + Replace(@HandlerCode, '''', '''''') + ''''

EXEC (@COMMAND)

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETFNGetLoggedOnHandler'
      AND SO.[type] = 'FN')
  BEGIN
    DROP FUNCTION [dbo].[ky_NETFNGetLoggedOnHandler]
  END
GO

CREATE FUNCTION
       [dbo].[ky_NETFNGetLoggedOnHandler]
      ()
RETURNS VARCHAR(10)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETFNGetLoggedOnHandler]                                                                        *
 *                                                                                                           *
 * Get the Code for the Currently Logged on Handler                                                          *
 * This requires the software using the Connection to set its CONTEXT_INFO                                   *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.5.3.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2017-10-13   Pino Carafa       Created                                                                    *
 * 2019-03-21   Pino Carafa       Azure compatibility                                                        *
 * 2019-10-09   John Ginnane      KEYD-6197 - Now referring to sysprocesses as [sys].[sysprocesses]          *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

DECLARE @Handler VARCHAR(10)

SELECT @Handler = CONVERT(VARCHAR(10),
                          CASE WHEN CHARINDEX(CHAR(0),
                                              CONVERT(VARCHAR(MAX),
                                                      [P].[context_info])) > 0
                                    THEN SUBSTRING(CONVERT(VARCHAR(MAX),
                                                            [P].[context_info]),
                                                    1,
                                                    CHARINDEX(CHAR(0),
                                                              CONVERT(VARCHAR(MAX),
                                                              [P].[context_info])) - 1)
                                ELSE CONVERT(VARCHAR(MAX),
                                            [P].[context_info]) END)
  FROM [sys].[sysprocesses] P
  WHERE P.[spid] = @@SPID

SET @Handler = RTRIM(ISNULL((SELECT [HAN].[Code] FROM [dbo].[Handlers] [HAN] WHERE [HAN].[CODE] = @Handler), 'ADM'))

RETURN @Handler
END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETFrameworkLicenseCount'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETFrameworkLicenseCount]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETFrameworkLicenseCount]
      (@Database      NVARCHAR(20),
       @Result        INT                OUTPUT)
AS
/*********************************************************************************************
 *
 * Stored Procedure Name: [dbo].[ky_NETFrameworkLicenseCount]
 * Description: This procedure is used to Get the instance count running on server,
 *              the count will be exclude the ADMIN login
 *
 * Created By: Arun.V
 * Modification History:
 * ????-??-??   Arun V.           Created
 * 2017-07-20   Pino Carafa       Modified it to prevent duplicates for same hostname/handler
 * 2017-11-15   Pino Carafa       Exclude Connections for Matter Search
 * 2018-11-27   Pino Carafa       No longer count process ids as there are too many
 *                                different possible connections going on
 * 2019-03-21   Pino Carafa       Azure compatibility
 * 2019-10-09   John Ginnane      KEYD-6197 - Prepended [sysprocesses] with [sys].
 *
 *********************************************************************************************/
BEGIN

SET NOCOUNT ON

DECLARE @RESULTTBL TABLE
       ([HostName]              NCHAR(128),
        [ProgramName]           NCHAR(128),
        [HandlerCode]           VARCHAR(10),
        [HandlerName]           VARCHAR(30),
        [Status]                NVARCHAR(60),
        [login_time]            DATETIME,
        [hostprocess]           NCHAR(10),
        [nt_domain]             NCHAR(128),
        [nt_username]           NCHAR(128))

INSERT
  INTO @RESULTTBL
      ([HostName],
       [ProgramName],
       [HandlerCode],
       [HandlerName],
       [Status],
       [login_time],
       [hostprocess],
       [nt_domain],
       [nt_username])
SELECT [P].[hostname]                                 AS [HostName],
       [P].[program_name]                             AS [ProgramName],
       RTRIM(ISNULL(HAN.[CODE], ''))                  AS [HandlerCode],
       RTRIM(ISNULL(HAN.[Name], ''))                  AS [HandlerName],
       [DB].[state_desc]                              AS [Status],
       CONVERT(NVARCHAR(10),[P].[login_time],108)     AS [LoginTime],
       [P].[hostprocess]                              AS [HostProcessID],
       CASE WHEN RTRIM([P].[nt_domain]) <> N''
                 THEN [P].[nt_domain]
            ELSE '' END,
       CASE WHEN RTRIM([P].[nt_username]) <> N''
                 THEN [P].[nt_username]
            ELSE '' END  
  FROM SYS.[databases] [DB]
       INNER JOIN [sys].[sysprocesses] [P]
               ON [P].[dbid] = DB.[database_id]
              AND [P].[program_name] LIKE 'KHDesktop%'
              AND [P].[program_name] NOT LIKE '%MatterSearch%'
              AND [P].[program_name] NOT LIKE '%OutlookAddin%'
              AND [P].[program_name] NOT LIKE '%OfficeAddin%'
              AND CASE WHEN CHARINDEX(CHAR(0),
                                      CONVERT(VARCHAR(MAX),
                                              [P].[context_info])) > 0
                            THEN SUBSTRING(CONVERT(VARCHAR(MAX),
                                                   [P].[context_info]),
                                           1,
                                           CHARINDEX(CHAR(0),
                                                     CONVERT(VARCHAR(MAX),
                                                     [P].[context_info])) - 1)
                       ELSE CONVERT(VARCHAR(MAX),
                                    [P].[context_info]) END NOT IN ('', 'ADM')
       LEFT OUTER JOIN [Handlers] HAN
                    ON HAN.[CODE] = CASE WHEN CHARINDEX(CHAR(0),
                                                        CONVERT(VARCHAR(MAX),
                                                                [P].[context_info])) > 0
                                              THEN SUBSTRING(CONVERT(VARCHAR(MAX),
                                                                     [P].[context_info]),
                                                             1,
                                                             CHARINDEX(CHAR(0),
                                                                       CONVERT(VARCHAR(MAX),
                                                                               [P].[context_info])) - 1)
                                         ELSE CONVERT(VARCHAR(MAX),
                                                      [P].[context_info]) END
 WHERE [DB].[name] = @Database

  SELECT @Result = COUNT(1)
    FROM (  SELECT [RES].[HostName],
                   [RES].[HandlerCode],
                   [RES].[HandlerName],
                   [RES].[Status],
                   [RES].[nt_domain],
                   [RES].[nt_username]
              FROM @RESULTTBL [RES]
          GROUP BY [RES].[HostName],
                   [RES].[HandlerCode],
                   [RES].[HandlerName],
                   [RES].[Status],
                   [RES].[nt_domain],
                   [RES].[nt_username]) [ORGRES]

SET NOCOUNT OFF

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETFetchCurrentLogonUsers'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETFetchCurrentLogonUsers]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETFetchCurrentLogonUsers]
      (@Database NVARCHAR(20))
AS
/*********************************************************************************************
 *
 * Stored Procedure Name: [dbo].[ky_NETFetchCurrentLogonUsers]
 * Description: This procedure is used to fetch current logon users
 *
 * Created By: Arun.V
 * Modification History:
 * ????-??-??   Arun V.           Created
 * 2017-07-20   Pino Carafa       Modified it to prevent duplicates for same hostname/handler
 *                                Please note that we can count individual logins if the
 *                                customer caters for using Windows Logons for their SQL
 *                                connections. Otherwise we must count individual host process
 *                                ids, to prevent multiple logins with the same handler code
 *                                through something like a Citrix server
 * 2017-11-15   Pino Carafa       Exclude Connections for Matter Search
 * 2019-03-21   Pino Carafa       Azure compatibility
 * 2019-10-09   John Ginnane      KEYD-6197 - Prepended [sysprocesses] with [sys].
 *
 *********************************************************************************************/
BEGIN

SET NOCOUNT ON

DECLARE @RESULT TABLE
       ([HostName]              NCHAR(128),
        [ProgramName]           NCHAR(128),
        [HandlerCode]           VARCHAR(10),
        [HandlerName]           VARCHAR(30),
        [Status]                NVARCHAR(60),
        [login_time]            DATETIME,
        [hostprocess]           NCHAR(10),
        [nt_domain]             NCHAR(128),
        [nt_username]           NCHAR(128))

INSERT
  INTO @RESULT
      ([HostName],
       [ProgramName],
       [HandlerCode],
       [HandlerName],
       [Status],
       [login_time],
       [hostprocess],
       [nt_domain],
       [nt_username])
SELECT [P].[hostname]                                 AS [HostName],
       [P].[program_name]                             AS [ProgramName],
       RTRIM(ISNULL(HAN.[CODE], ''))                  AS [HandlerCode],
       RTRIM(ISNULL(HAN.[Name], ''))                  AS [HandlerName],
       [DB].[state_desc]                              AS [Status],
       CONVERT(NVARCHAR(10),[P].[login_time],108)     AS [LoginTime],
       [P].[hostprocess]                              AS [HostProcessID],
       CASE WHEN RTRIM([P].[nt_domain]) <> N''
                 THEN [P].[nt_domain]
            ELSE [P].[hostprocess] END,
       CASE WHEN RTRIM([P].[nt_username]) <> N''
                 THEN [P].[nt_username]
            ELSE [P].[hostprocess] END                                  
  FROM SYS.[databases] [DB]
       INNER JOIN [sys].[sysprocesses] [P]
               ON [P].[dbid] = DB.[database_id]
              AND [P].[program_name] LIKE 'KHDesktop%'
              AND [P].[program_name] NOT LIKE '%MatterSearch%'
              AND [P].[program_name] NOT LIKE '%OutlookAddin%'
              AND [P].[program_name] NOT LIKE '%OfficeAddin%'
              AND CASE WHEN CHARINDEX(CHAR(0),
                                      CONVERT(VARCHAR(MAX),
                                              [P].[context_info])) > 0
                            THEN SUBSTRING(CONVERT(VARCHAR(MAX),
                                                   [P].[context_info]),
                                           1,
                                           CHARINDEX(CHAR(0),
                                                     CONVERT(VARCHAR(MAX),
                                                     [P].[context_info])) - 1)
                       ELSE CONVERT(VARCHAR(MAX),
                                    [P].[context_info]) END NOT IN ('', 'ADM')
       LEFT OUTER JOIN [Handlers] HAN
                    ON HAN.[CODE] = CASE WHEN CHARINDEX(CHAR(0),
                                                        CONVERT(VARCHAR(MAX),
                                                                [P].[context_info])) > 0
                                              THEN SUBSTRING(CONVERT(VARCHAR(MAX),
                                                                     [P].[context_info]),
                                                             1,
                                                             CHARINDEX(CHAR(0),
                                                                       CONVERT(VARCHAR(MAX),
                                                                               [P].[context_info])) - 1)
                                         ELSE CONVERT(VARCHAR(MAX),
                                                      [P].[context_info]) END
 WHERE [DB].[name] = @Database

  SELECT [ORGRES].[HostName],
         [TOPRES].[ProgramName],
         [ORGRES].[HandlerCode],
         [ORGRES].[HandlerName],
         [ORGRES].[Status],
         [TOPRES].[LoginTime],
         [TOPRES].[HostProcessID]
    FROM (  SELECT [RES].[HostName],
                   [RES].[HandlerCode],
                   [RES].[HandlerName],
                   [RES].[Status],
                   [RES].[nt_domain],
                   [RES].[nt_username]
              FROM @RESULT [RES]
          GROUP BY [RES].[HostName],
                   [RES].[HandlerCode],
                   [RES].[HandlerName],
                   [RES].[Status],
                   [RES].[nt_domain],
                   [RES].[nt_username]) [ORGRES]
         CROSS APPLY (  SELECT TOP 1 
                               [RES2].[ProgramName]                            AS [ProgramName],
                               CONVERT(NVARCHAR(10),[RES2].[login_time],108)   AS [LoginTime],
                               [RES2].[hostprocess]                            AS [HostProcessID]
                          FROM @RESULT [RES2]
                         WHERE [RES2].[HostName] = [ORGRES].[HostName]
                           AND [RES2].[HandlerCode] = [ORGRES].[HandlerCode]
                           AND [RES2].[nt_domain] = [ORGRES].[nt_domain]
                           AND [RES2].[nt_username] = [ORGRES].[nt_username]
                      ORDER BY [RES2].[login_time],
                               [RES2].[Status]) [TOPRES]
ORDER BY [ORGRES].[HandlerName]

SET NOCOUNT OFF

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPGetFileShareDocumentID'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPGetFileShareDocumentID]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETSPGetFileShareDocumentID]
      (@MatterCode           VARCHAR(20),
       @TrackReference       INT,
       @FilePath             VARCHAR(255),
       @FileSharingServerID  INT)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETSPGetFileShareDocumentID]                                                                    *
 *                                                                                                           *
 * Checks whether a File Share Document is already stored for this item. If so, return the file sharing id   *
 * As additional security measures we will check that the Matter Code, Track Reference and File Path match   *
 * the values that are passed in before retrieving the File Sharing documents ID.                            *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.5.1.2                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2017-07-21   Pino Carafa       Created                                                                    *
 * 2017-08-15   Pino Carafa       Added Document Folder ID                                                   *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

SELECT @FilePath =
       CASE WHEN UNC.[UNC] IS NULL
                 THEN RTRIM(ISNULL([DAT].[FilePath], ''))
            ELSE   UNC.[UNC]
                 + SUBSTRING(RTRIM(ISNULL([DAT].[FilePath],
                                          '')),
                             3,
                             LEN(RTRIM(ISNULL([DAT].[FilePath],
                                              ''))) - 2) END
  FROM (SELECT @FilePath AS [FilePath]) [DAT]
       LEFT OUTER JOIN [dbo].[UNCAlias] UNC
                    ON SUBSTRING([DAT].[FilePath],
                                 2,
                                 1) = ':'
                   AND UNC.[Drive] = SUBSTRING([DAT].[FilePath],
                                               1,
                                               1)

SELECT [FSD].[FileSharingDocumentID],
       [FSM].[FileSharingDocFoldersID]
  FROM [dbo].[DiaryAttachments] [DAT]
       INNER JOIN [dbo].[diary] [DIA]
               ON [DIA].[ACTIONID] = [DAT].[DiaryID]
              AND [DIA].[CASECODE] = @MatterCode
       INNER JOIN [dbo].[FileSharingDocuments] [FSD]
                  INNER JOIN [dbo].[FileSharingMapping] [FSM]
                          ON [FSM].[id] = [FSD].[FileSharingMappingID]
                         AND [FSM].[FileSharingServersID] = @FileSharingServerID
               ON [FSD].[TrackReference] = [DAT].[TrackReference]
       LEFT OUTER JOIN [dbo].[UNCAlias] UNC
                    ON SUBSTRING([DAT].[FilePath],
                                 2,
                                 1) = ':'
                   AND UNC.[Drive] = SUBSTRING([DAT].[FilePath],
                                               1,
                                               1)
 WHERE [DAT].[TrackReference] = @TrackReference
   AND @FilePath =
       CASE WHEN UNC.[UNC] IS NULL
                 THEN RTRIM(ISNULL([DAT].[FilePath], ''))
            ELSE   UNC.[UNC]
                 + SUBSTRING(RTRIM(ISNULL([DAT].[FilePath],
                                          '')),
                             3,
                             LEN(RTRIM(ISNULL([DAT].[FilePath],
                                              ''))) - 2) END

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPRemoveDocsFromDocFolder'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPRemoveDocsFromDocFolder]
  END
GO

 CREATE PROCEDURE
        [dbo].[ky_NETSPRemoveDocsFromDocFolder]
       (@removexml         VARCHAR(MAX))
AS
/*********************************************************************************************************
 *
 * [dbo].[ky_NETSPRemoveDocsFromDocFolder]
 *
 * Flags the documents passed in through @removexml as no longer being "in" the document
 * folder(s) in the XML document
 *
 * @removexml       - an xml string outlining the nature of the changes
 *
 * Modification History:
 * 2016-04-29    Pino Carafa      Created
 * 2017-07-21    Pino Carafa      Added functionality to clear out FileSharingDocuments
 *
 *********************************************************************************************************/
BEGIN
  SET NOCOuNT ON

  DECLARE @iSL INT
  DECLARE @MAX INT

  DECLARE @data TABLE
         ([seq]                       INT              IDENTITY(1, 1)    NOT NULL      PRIMARY KEY,
          [folderid]                  INT                                NOT NULL,
          [trackreference]            INT                                NOT NULL)

  BEGIN TRY
  EXEC sp_xml_preparedocument @iSL OUTPUT, @removexml
  END TRY
  BEGIN CATCH
  EXEC sp_xml_preparedocument @iSL OUTPUT, N'<removedocsfromfolders/>'
  END CATCH

  INSERT
    INTO @data
        ([folderid],
         [trackreference])
  SELECT SRC.[folderid],
         SRC.[trackreference]
    FROM OPENXML(@iSL, N'removedocsfromfolders/remove')
    WITH ([folderid]                  INT            '@fromfolder',
          [trackreference]            INT            '@trackreference') SRC

  EXEC sp_xml_removedocument @iSL

  DELETE [FSD]
    FROM @data [DT]
         INNER JOIN [dbo].[DocFolderDocuments] [DFD]
                    INNER JOIN [dbo].[FileSharingMapping] [FSM]
                               INNER JOIN [dbo].[FileSharingDocuments] [FSD]
                                       ON [FSD].[FileSharingMappingID] = [FSM].[id]
                            ON [FSM].[FileSharingDocFoldersID] = [DFD].[DocFolderID]
                 ON [DFD].[DocFolderID] = [DT].[folderid]
                AND [DFD].[TrackReference] = [DT].[trackreference]
   WHERE [FSD].[TrackReference] = [DT].[trackreference]

  DELETE [DFD]
    FROM @data [DT]
         INNER JOIN [dbo].[DocFolderDocuments] [DFD]
                 ON [DFD].[DocFolderID] = [DT].[folderid]
                AND [DFD].[TrackReference] = [DT].[trackreference]

  SET NOCOUNT OFF

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPGetFileSharingActionDocuments'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPGetFileSharingActionDocuments]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETSPGetFileSharingActionDocuments]
      (@ActionID INT)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETSPGetFileSharingActionDocuments]                                                             *
 *                                                                                                           *
 * List the File Sharing IDs for all documents attached to this action                                       *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.5.1.2                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2017-07-21   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

SELECT [FSM].[FileSharingServersID],
       [FSD].[FileSharingDocumentID],
       [FSM].[FileSharingDocFoldersID],
       [FSD].[TrackReference]
  FROM [dbo].[diary] [DIA]
       INNER JOIN [dbo].[DiaryAttachments] [DAT]
                  INNER JOIN [dbo].[FileSharingDocuments] [FSD]
                             INNER JOIN [dbo].[FileSharingMapping] [FSM]
                                     ON [FSM].[id] = [FSD].[FileSharingMappingID]
                          ON [FSD].[TrackReference] = [DAT].[TrackReference]
               ON [DAT].[DiaryID] = [DIA].[ACTIONID]
 WHERE [DIA].[ACTIONID] = @ActionID

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPGetFileSharingTrackReferenceDocuments'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPGetFileSharingTrackReferenceDocuments]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETSPGetFileSharingTrackReferenceDocuments]
      (@TrackReference INT)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETSPGetFileSharingTrackReferenceDocuments]                                                     *
 *                                                                                                           *
 * List the File Sharing IDs for the document with this Track Reference                                      *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.5.1.2                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2017-07-21   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

SELECT [FSM].[FileSharingServersID],
       [FSD].[FileSharingDocumentID],
       [FSM].[FileSharingDocFoldersID],
       [FSD].[TrackReference]
  FROM [dbo].[DiaryAttachments] [DAT]
       INNER JOIN [dbo].[FileSharingDocuments] [FSD]
                  INNER JOIN [dbo].[FileSharingMapping] [FSM]
                          ON [FSM].[id] = [FSD].[FileSharingMappingID]
               ON [FSD].[TrackReference] = [DAT].[TrackReference]
 WHERE [DAT].[TrackReference] = @TrackReference

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPCreateDiaryAttachmentXML'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPCreateDiaryAttachmentXML]
  END
GO

CREATE PROCEDURE [dbo].[ky_NETSPCreateDiaryAttachmentXML]
                (@pXML      NVARCHAR(MAX))
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETSPCreateDiaryAttachmentXML]                                                                  *
 *                                                                                                           *
 * Create a Diary Attachment for a particular Action                                                         *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              - [dbo].[ky_NETSPCreateDiaryAttachment]                  *   *
 *   * First compatible version:                - 5.7.2.1                                                *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2015-09-16 Pino Carafa          Imported from Release 1                                                   *
 * 2015-10-19 Sridharan            KEYD-2789-Copy to Another File doesn't work correctly                     *
 *                                 when using the Document Manager                                           *
 * 2016-05-01 Pino Carafa          Add DocFolders parameter                                                  *
 *                                 Class can now be up to 10 characters                                      *
 *                                 @strType can now be up to 10 characters                                   *
 * 2016-10-24 Pino Carafa          Remove invalid filename characters in filename                            *
 * 2016-10-27 Pino Carafa          Handle full file paths through @pDocument                                 *
 * 2017-03-10 Pino Carafa          Use ky_NETGetNextTrackReference                                           *
 * 2017-07-24 Pino Carafa          Add support for Document Naming Method and Storage Method                 *
 * 2019-07-29 Pino Carafa          XML Version including support to add DiaryAttachmentText                  *
 * 2019-07-29 Pino Carafa          Added functionality to write to the Undertakings log as had been          *
 *                                 implemented in [dbo].[ky_NETAAInsertDiaryAttchement]                      *
 * 2019-07-30 Pino Carafa          Added Trackreference to the input XML. If left out procedure will         *
 *                                 get the next Track Reference itself, otherwise it will use what is        *
 *                                 passed in                                                                 *
 * 2019-08-21 Pino Carafa          Added script to write/update the [ReindexStatus] table. Only affects      *
 *                                 On Premise databases                                                      *
 * 2019-09-13 Pino Carafa          KEYD-6133                                                                 *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

DECLARE @pDiaryID            INT
DECLARE @pName               VARCHAR(10)
DECLARE @pDocName            VARCHAR(100)
DECLARE @pDocument           VARCHAR(255)
DECLARE @pType               VARCHAR(10)
DECLARE @pDocClass           VARCHAR(10)
DECLARE @pDateEntered        VARCHAR(17)
DECLARE @pEnterBy            VARCHAR(8)
DECLARE @pLastAccessed       VARCHAR(17)
DECLARE @pLastAccessedby     VARCHAR(8)
DECLARE @pSystem             VARCHAR(1)
DECLARE @pDictationFile      VARCHAR(255)
DECLARE @pSource             VARCHAR(20)
DECLARE @pDocumentText       VARCHAR(MAX)
DECLARE @Fees                INT
DECLARE @Outlay              INT
DECLARE @inTrackRef          INT
DECLARE @DocFolders          TABLE
       ([id]                 INT)

SET NOCOUNT ON

DECLARE @iSL INT

BEGIN TRY
EXEC sp_xml_preparedocument @iSL OUTPUT, @pXML
END TRY
BEGIN CATCH
EXEC sp_xml_preparedocument @iSL OUTPUT, N'<CreateDiaryAttachment />'
END CATCH

SELECT @pDiaryID        = [XML].[DiaryID],
       @pName           = [XML].[Name],
       @pDocName        = [XML].[DocName],
       @pDocument       = [XML].[Document],
       @pType           = [XML].[Type],
       @pDocClass       = [XML].[DocClass],
       @pDateEntered    = [XML].[DateEntered],
       @pEnterBy        = [XML].[EnterBy],
       @pLastAccessed   = [XML].[LastAccessed],
       @pLastAccessedBy = [XML].[LastAccessedBy],
       @pSystem         = [XML].[System],
       @pDictationFile  = [XML].[DictationFile],
       @pSource         = [XML].[Source],
       @pDocumentText   = [XML].[DocumentText],
       @Fees            = [XML].[Fees],
       @Outlay          = [XML].[Outlay],
       @inTrackRef      = [XML].[TrackReference]
FROM OPENXML(@iSL, N'CreateDiaryAttachment')
WITH ([DiaryID]                  INT             '@diaryid',
      [Name]                     VARCHAR(10)     '@name',
      [DocName]                  VARCHAR(100)    '@docname',
      [Document]                 VARCHAR(255)    '@document',
      [Type]                     VARCHAR(10)     '@type',
      [DocClass]                 VARCHAR(10)     '@docclass',
      [DateEntered]              VARCHAR(17)     '@dateentered',
      [EnterBy]                  VARCHAR(8)      '@enterby',
      [LastAccessed]             VARCHAR(17)     '@lastaccessed',
      [LastAccessedBy]           VARCHAR(8)      '@lastaccessedby',
      [System]                   VARCHAR(1)      '@system',
      [DictationFile]            VARCHAR(255)    '@dictationfile',
      [Source]                   VARCHAR(20)     '@source',
      [Fees]                     INT             '@fees',
      [Outlay]                   INT             '@outlay',
      [TrackReference]           INT             '@trackreference',
      [DocumentText]             VARCHAR(MAX)    'documenttext') [XML]


DECLARE @docfoldercount INT

SELECT @docfoldercount = COUNT(1)
FROM OPENXML(@iSL, N'CreateDiaryAttachment/selectedfolders/folder')
WITH ([id]                       INT             '@id') [XML]

IF @docfoldercount > 0
  BEGIN
    INSERT
      INTO @DocFolders
          ([id])
    SELECT TOP (@docfoldercount) [id]
      FROM OPENXML(@iSL, N'CreateDiaryAttachment/selectedfolders/folder')
      WITH ([id]                       INT             '@id') [XML]
  END

BEGIN TRY
EXEC sp_xml_removedocument @iSL
END TRY
BEGIN CATCH
END CATCH

SET @pDiaryID        = ISNULL(@pDiaryID, 0)
SET @pName           = ISNULL(@pName, '')
SET @pDocument       = ISNULL(@pDocument, '')
SET @pDocClass       = ISNULL(@pDocClass, '')
SET @pEnterBy        = ISNULL(@pEnterBy, '')
SET @pLastAccessedBy = ISNULL(@pLastAccessedBy, '')
SET @pSystem         = CASE WHEN ISNULL(@pSystem, 'N') = 'Y' THEN 'Y' ELSE 'N' END
SET @pDictationFile  = ISNULL(@pDictationFile, '')
SET @Fees            = ISNULL(@Fees, 0)
SET @Outlay          = ISNULL(@Outlay, 0)

DECLARE @strcasecode              VARCHAR(20)
DECLARE @strFileName              VARCHAR(255)
DECLARE @strUNCFileName           VARCHAR(500)
DECLARE @strMatterNo              VARCHAR(10)
DECLARE @strClientCode            VARCHAR(10)
DECLARE @strDefaultFolder         VARCHAR(255)
DECLARE @strSource                VARCHAR(20)
DECLARE @strType                  VARCHAR(10)
DECLARE @continue                 INT
DECLARE @error                    VARCHAR(1000)
DECLARE @DocNamingMethod          CHAR(1)
DECLARE @StorageMethod            CHAR(1)

SELECT @DocNamingMethod = [CON].[DocumentNamingMethod] FROM [dbo].[control] [CON]
SET @DocNamingMethod = CASE WHEN ISNULL(@DocNamingMethod, ' ') = 'T' THEN 'T' ELSE ' ' END

SELECT @StorageMethod = [SCN].[STORAGEMETHOD] FROM [dbo].[SystemConfig] [SCN]
SET @StorageMethod = CASE WHEN ISNULL(@StorageMethod, 'C') = 'D' THEN 'D' ELSE 'C' END

SET @continue = 0
SET @error = ''

SET @pDiaryID = ISNULL(@pDiaryID, 0)

IF (@pDiaryID = 0)
  BEGIN
    SET @continue = 12
    SET @error = 'invalid or missing Diary ID'
  END

-- Get Diary and Matter information
IF (@continue = 0)
  BEGIN
    BEGIN TRY
      SELECT @strcasecode   = RTRIM(ISNULL([MAT].[code], '')),
             @strClientCode = RTRIM(ISNULL([MAT].[clientcode], '')),
             @strMatterNo   = RTRIM(ISNULL([MAT].[matter], '')),
             @strSource     = RTRIM(ISNULL([DIA].[PROCESSTYPE], ''))
        FROM [dbo].[diary] [DIA]
             INNER JOIN [dbo].[matters] [MAT]
                     ON [MAT].[code] = [DIA].[CASECODE]
       WHERE [DIA].[ACTIONID] = @pDiaryID

      SET @strcasecode = ISNULL(@strcasecode, '')
      SET @strClientCode = ISNULL(@strClientCode, '')
      SET @strMatterNo = ISNULL(@strMatterNo, '')
      SET @strSource = CASE WHEN ISNULL(@pSource, '') = ''
                                 THEN CASE WHEN ISNULL(@strSource, 'I') = 'I'
                                                THEN 'Received E-mail'
                                           ELSE 'Sent E-mail' END
                            ELSE @pSource END

      IF (@strcasecode = '')
        BEGIN
          SET @continue = 2
          SET @error = 'invalid or missing Diary ID'
        END
    END TRY
    BEGIN CATCH
      SET @continue = 1
      SET @error = 'Error retrieving Diary information for Diary ID: ' + ISNULL(convert(VARCHAR(10), @pDiaryID), 'unknown') + ' ' + Error_Message()
    END CATCH
  END

-- Get Next TrackReference  -----------------------------------------------------------------------------------------------------------------------------------------
IF (@continue = 0)
  BEGIN
    -- If Next TrackReference was passed in, just use what was passed in.
    IF @inTrackRef IS NULL
      BEGIN
        BEGIN TRY
          EXEC @inTrackRef = [dbo].[ky_NETGetNextTrackReference]
        END TRY
        BEGIN CATCH
          SET @continue = 3
          SET @error = 'Error calculating next Track Reference: ' + Error_Message()
        END CATCH
      END
  END

-- Write DiaryAttachment  Record -----------------------------------------------------------------------------------------------------------------------------------------
IF (@continue = 0)
  BEGIN
    BEGIN TRY
      SELECT @strDefaultFolder = (SELECT RTRIM(ISNULL([SC].[SCWPFOLDER], ''))
                                    FROM [dbo].[SystemConfig] [SC])
      SET @strDefaultFolder = ISNULL(@strDefaultFolder, '')

      IF (@strDefaultFolder = '')
        BEGIN
          SET @continue = 4
          SET @error = 'SCWPFOLDER not SET in System Config'
        END
    END TRY
    BEGIN CATCH
      SET @continue = 5
      SET @error = 'Error retrieving SCWPFOLDER: ' + Error_Message()
    END CATCH
  END

IF (@continue = 0)
  BEGIN
    BEGIN TRY
      IF @StorageMethod = 'C'
        BEGIN
          SELECT @strFileName = RTRIM(ISNULL([CLT].[DCOUMENTFOLDER], ''))
            FROM [dbo].[client] [CLT]
           WHERE [CLT].[CLCODE] = @strClientCode

          SET @strFileName = RTRIM(ISNULL(@strFileName, ''))

          IF (@strFileName = '')
            BEGIN
              SET @strFileName = @strDefaultFolder + '\' + @strClientCode

              UPDATE [CLT]
                 SET [CLT].[DCOUMENTFOLDER] = @strFileName
                FROM [dbo].[client] [CLT]
               WHERE [CLT].[CLCODE] = @strClientCode
            END
        END
      ELSE
        BEGIN
          SET @strFileName = @strDefaultFolder + '\' + CONVERT(VARCHAR(6), GETDATE(), 112)
        END
    END TRY
    BEGIN CATCH
      SET @continue = 6
      SET @error = 'Error establishing Document Folder: ' + Error_Message()
    END CATCH
  END

IF (@continue = 0)
  BEGIN
    SET @pDocument = RTRIM(ISNULL(@pDocument, ''))
    SET @pDocument = REPLACE(@pDocument, CHAR(13) + CHAR(10), '_')
    IF (@pDocument = '')
      BEGIN
        SET @continue = 7
        SET @error = 'missing document name'
      END
  END

IF (@continue = 0)
  BEGIN
    SET @pDocName = RTRIM(ISNULL(@pDocName, ''))
    SET @pDocName = REPLACE(@pDocName, CHAR(13) + CHAR(10), '_')
    IF (@pDocName = '')
      BEGIN
        SET @continue = 7
        SET @error = 'missing document name'
      END
  END

IF (@continue = 0)
  BEGIN
    BEGIN TRY
      SET @pType = RTRIM(ISNULL(@pType, ''))
      SET @strType = convert(VARCHAR(10), @pType)
    END TRY
    BEGIN CATCH
      SET @continue = 8
      SET @error = 'Error interpreting Document Type: ' + Error_Message()
    END CATCH
  END

IF (@continue = 0)
  BEGIN
    BEGIN TRY
      DECLARE @pDocName2 VARCHAR(100)
      DECLARE @pDocumentWork VARCHAR(255)

      SET @pDocumentWork = REVERSE(@pDocument)
      IF CHARINDEX('\', @pDocumentWork) > 0
        BEGIN
          SET @pDocumentWork = SUBSTRING(@pDocumentWork, 1, CHARINDEX('\', @pDocumentWork) - 1)
        END

      IF CHARINDEX('.', @pDocumentWork) > 0
        BEGIN
          SET @pDocumentWork = SUBSTRING(@pDocumentWork, CHARINDEX('.', @pDocumentWork) + 1, LEN(@pDocumentWork) - CHARINDEX('.', @pDocumentWork))
        END

      SET @pDocumentWork = REVERSE(@pDocumentWork)

      IF @DocNamingMethod <> 'T'
        BEGIN
          IF ISNULL(@inTrackRef, 0) <> 0
            BEGIN
              IF LEN(@pDocumentWork) > LEN(CONVERT(VARCHAR(10), @inTrackRef)) + 1
                BEGIN
                  IF SUBSTRING(@pDocumentWork, LEN(@pDocumentWork) - LEN(CONVERT(VARCHAR(10), @inTrackRef)), LEN(CONVERT(VARCHAR(10), @inTrackRef)) + 1) = '_' + CONVERT(VARCHAR(10), @inTrackRef)
                    BEGIN
                      SET @pDocumentWork = SUBSTRING(@pDocumentWork, 1, LEN(@pDocumentWork) - LEN(CONVERT(VARCHAR(10), @inTrackRef)) - 1)
                    END
                END
            END

          SET @pDocName2 = REPLACE(REPLACE(REPLACE([dbo].[ky_NETFNRemovePatternFromString](@pDocumentWork, '%[:*<>$&.!?@#~/\\\%|`{}\+=_^",]%'), CHAR(13), ''), CHAR(10), ''), CHAR(9), '')
          IF @StorageMethod = 'D'
            BEGIN
              SET @strFileName = @strFileName
                               + '\'
                               + @pDocName2
                               + '_'
                               + convert(VARCHAR(10), @inTrackRef)
            END
          ELSE
            BEGIN
              SET @strFileName = @strFileName
                               + '\'
                               + @strMatterNo
                               + '\'
                               + @pDocName2
                               + '_'
                               + convert(VARCHAR(10), @inTrackRef)
            END
        END
      ELSE
        BEGIN
          IF @StorageMethod = 'D'
            BEGIN
              SET @strFileName = @strFileName
                               + '\'
                               + convert(VARCHAR(10), @inTrackRef)
            END
          ELSE
            BEGIN
              SET @strFileName = @strFileName
                               + '\'
                               + @strMatterNo
                               + '\'
                               + convert(VARCHAR(10), @inTrackRef)
            END
        END

      IF (@pType <> '')
        BEGIN
          SET @strFileName = @strFileName
                           + '.'
                           + @pType
        END
    END TRY
    BEGIN CATCH
      SET @continue = 9
      SET @error = 'Error establishing filename: ' + Error_Message()
    END CATCH
  END

SET @strFileName = ISNULL(@strFileName, '')
SET @strUNCFileName = @strFileName

IF (SUBSTRING(@strUNCFileName, 2, 1) = ':')
  BEGIN
    SELECT @strUNCFileName = ISNULL(MAX([UNC].[UNC]) + SUBSTRING(@strUNCFileName, 3, LEN(@strUNCFileName) - 2), @strUNCFileName)
      FROM [dbo].[UNCAlias] [UNC]
     WHERE SUBSTRING(@strUNCFileName, 1, 1) = [UNC].[Drive]

    SET @strUNCFileName = LEFT(@strUNCFileName, 2) + REPLACE(RIGHT(@strUNCFileName, LEN(@strUNCFileName) - 2), '\\', '\')
  END

IF(@strType = 'FOL')
  BEGIN
    SET @strFileName = @pDocument
  END

IF (@continue = 0)
  BEGIN
    BEGIN TRY
      INSERT INTO DiaryAttachments
                 ([DiaryID],
                  [TrackReference],
                  [CASECODE],
                  [NAME],
                  [DOCUMENT],
                  [FILEPATH],
                  [TYPE],
                  [DOCCLASS],
                  [DATEENTERED],
                  [ENTEREDBY],
                  [LASTACCESSDATE],
                  [LASTACCESSBY],
                  [SYSTEM],
                  [DICTATIONFILE],
                  [SOURCE],
                  [Fees],
                  [Outlay])
           VALUES(@pDiaryID,
                  @inTrackRef,
                  @strcasecode,
                  @pName,
                  @pDocName,
                  @strFileName,
                  @strType,
                  @pDocClass,
                  convert(DATETIME, @pDateEntered),
                  @pEnterBy,
                  convert(DATETIME, @pLastAccessed),
                  @pLastAccessedby,
                  @pSystem,
                  @pDictationFile,
                  @strSource,
                  @Fees,
                  @Outlay)
    END TRY
    BEGIN CATCH
      SET @continue = 10
      SET @error = 'Error inserting Diary Attachment record: ' + Error_Message()
    END CATCH
  END

IF (@continue = 0)
  BEGIN
    BEGIN TRY
      DECLARE @data TABLE
             ([seq]                       INT              IDENTITY(1, 1)    NOT NULL      PRIMARY KEY,
              [folderid]                  INT                                NOT NULL,
              [trackreference]            INT                                NOT NULL)
          
      INSERT
        INTO @data
            ([folderid],
             [trackreference])
      SELECT [SRC].[id],
             @inTrackRef
        FROM @DocFolders [SRC]

      INSERT
        INTO [dbo].[DocFolderDocuments]
            ([DocFolderID],
             [TrackReference])
      SELECT [DT].[folderid],
             [DT].[trackreference] 
        FROM @data [DT]
             LEFT OUTER JOIN [dbo].[DocFolderDocuments] [DFD]
                          ON [DFD].[DocFolderID] = [DT].[folderid]
                         AND [DFD].[TrackReference] = [DT].[trackreference]
       WHERE [DFD].[TrackReference] IS NULL
    END TRY
    BEGIN CATCH
      SET @continue = 11
      SET @error = 'Error copying documents to document folder: ' + Error_Message()
    END CATCH
  END

IF @continue = 0
  BEGIN
    SET @pDocumentText = RTRIM(ISNULL(@pDocumentText, ''))
    BEGIN TRY
      BEGIN
        IF EXISTS
          (SELECT TOP 1 1
             FROM [dbo].[DiaryAttachmentText] [DAX]
            WHERE [DAX].[TrackReference] = @inTrackRef)
          BEGIN
            UPDATE [DAX]
               SET [DAX].[AccessDate]  = @pLastAccessed,
                   [DAX].[ModifyDate]  = @pLastAccessed,
                   [DAX].[CreatedDate] = @pDateEntered,
                   [DAX].[TEXT]        = @pDocumentText
              FROM [dbo].[DiaryAttachmentText] [DAX]
             WHERE [DAX].[TrackReference] = @inTrackRef
          END
        ELSE
          BEGIN
            INSERT
              INTO [dbo].[DiaryAttachmentText]
                  ([TrackReference],
                   [AccessDate],
                   [CreatedDate],
                   [ModifyDate],
                   [TEXT])
            VALUES(@inTrackRef,
                   @pLastAccessed,
                   @pDateEntered,
                   @pLastAccessed,
                   @pDocumentText)
          END
      END
    END TRY
    BEGIN CATCH
    END CATCH

    BEGIN TRY
      BEGIN
        IF EXISTS
          (SELECT TOP 1 1
             FROM [dbo].[ReindexStatus] [RIS]
            WHERE [RIS].[TrackReference] = @inTrackRef)
          BEGIN
            UPDATE [RIS]
               SET [RIS].[STATUS] = 1
              FROM [dbo].[ReindexStatus] [RIS]
             WHERE [RIS].[TrackReference] = @inTrackRef
          END
        ELSE
          BEGIN
            INSERT
              INTO [dbo].[ReindexStatus]
                  ([TrackReference],
                   [FILEPATH],
                   [STATUS])
            VALUES(@inTrackRef,
                   @strFileName,
                   1)
          END
      END
    END TRY
    BEGIN CATCH
    END CATCH
  END

IF EXISTS
  (SELECT TOP 1 1
     FROM [dbo].[diary] [DIA] WITH (NOLOCK)
    WHERE [DIA].[ACTIONID] = @pDiaryID
      AND [DIA].[ACTIONTYPE] = 'U')
  BEGIN
    DECLARE @UserName VARCHAR(20)
    DECLARE @Description VARCHAR(500)
    
    SET @UserName = ISNULL((SELECT [HAN].[NAME]
                              FROM [dbo].[Handlers] [HAN]
                             WHERE [HAN].[CODE] = @pEnterBy),
                           CASE WHEN @pEnterBy = ''
                                     THEN 'Unknown'
                                ELSE @pEnterBy END)

    SET @Description = 'Attachment Added: ' + CHAR(13) + CHAR(10)
                     + 'Date:  ' + CONVERT(VARCHAR(11), CONVERT(DATETIME, @pDateEntered), 106) + CHAR(13) + CHAR(10)
                     + 'Document Name: ' + @pDocName + CHAR(13) + CHAR(10)
                     + 'Document Class: ' + @pDocClass + CHAR(13) + CHAR(10)

    INSERT
      INTO [dbo].[UndertakingLog]
          ([UndertakingID],
           [FieldChanges],
           [FromValue],
           [ToValue],
           [ByWho],
           [ModifiedDate])
    VALUES(@pDiaryID,
           'New Documents Added',
           @Description,
           ' - NIL - ',
           @UserName,
           GETDATE())
  END

SELECT @strcasecode    AS [CaseCode],
       @inTrackRef     AS [NewTrackReference],
       @strFileName    AS [Path],
       @strUNCFileName AS [UNCPath],
       @continue       AS [ErrorCode],
       @error          AS [ErrorMessage]
END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPCreateDiaryAttachment'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPCreateDiaryAttachment]
  END
GO

CREATE PROCEDURE [dbo].[ky_NETSPCreateDiaryAttachment]
                (@pDiaryID            Int=0,
                 @pName               VARCHAR(10),
                 @pDocName            VARCHAR(100),
                 @pDocument           VARCHAR(255),
                 @pType               VARCHAR(10),
                 @pDocClass           VARCHAR(10),
                 @pDateEntered        VARCHAR(17),
                 @pEnterBy            VARCHAR(8),
                 @pLastAccessed       VARCHAR(17),
                 @pLastAccessedby     VARCHAR(8),
                 @pSystem             VARCHAR(1),
                 @pDictationFile      VARCHAR(255),
                 @pSource             VARCHAR(20),
                 @pDocFolders         VARCHAR(MAX))
AS
/*************************************************************************************************************
 *                                                                                                           *
 * ky_NETSPCreateDiaryAttachment                                                                             *
 *                                                                                                           *
 * Create a Diary Attachment for a particular Action                                                         *
 *                                                                                                           *
 * PROCEDURE IS OBSOLETE but kept backward compatible by calling new procedure                               *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                -                                                        *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           - [dbo].[ky_NETSPCreateDiaryAttachmentXML]               *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2015-09-16 Pino Carafa          Imported from Release 1                                                   *
 * 2015-10-19 Sridharan            KEYD-2789-Copy to Another File doesn't work correctly                     *
 *                                 when using the Document Manager                                           *
 * 2016-05-01 Pino Carafa          Add DocFolders parameter                                                  *
 *                                 Class can now be up to 10 characters                                      *
 *                                 @strType can now be up to 10 characters                                   *
 * 2016-10-24 Pino Carafa          Remove invalid filename characters in filename                            *
 * 2016-10-27 Pino Carafa          Handle full file paths through @pDocument                                 *
 * 2017-03-10 Pino Carafa          Use ky_NETGetNextTrackReference                                           *
 * 2017-07-24 Pino Carafa          Add support for Document Naming Method and Storage Method                 *
 * 2019-07-29 Pino Carafa          Procedure is obsolete and it has been replaced by                         *
 *                                 [dbo].[ky_NETSPCreateDiaryAttachmentXML]                                  *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

DECLARE @XML NVARCHAR(MAX)

SET @XML = CONVERT(NVARCHAR(MAX), 
                   (SELECT *
                      FROM (SELECT @pDiaryID           AS [diaryid],
                                   @pName              AS [name],
                                   @pDocName           AS [docname],
                                   @pDocument          AS [document],
                                   @pType              AS [type],
                                   @pDocClass          AS [docclass],
                                   @pDateEntered       AS [dateentered],
                                   @pEnterBy           AS [enterby],
                                   @pLastAccessed      AS [lastaccessed],
                                   @pLastAccessedby    AS [lastaccessedby],
                                   @pSystem            AS [system],
                                   @pDictationFile     AS [dictationfile],
                                   @pSource            AS [source],
                                   'df'                AS [docfolders]) [CreateDiaryAttachment]
                       FOR XML AUTO))

IF RTRIM(ISNULL(@pDocFolders, '')) <> ''
  BEGIN
    SET @XML = REPLACE(@XML, N'docfolders="df"/>', N'>' + CONVERT(NVARCHAR(MAX), @pDocFolders) + N'</CreateDiaryAttachment>')
  END

EXEC [dbo].[ky_NETSPCreateDiaryAttachmentXML] @XML

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETAAInsertDiaryAttchement'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETAAInsertDiaryAttchement]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETAAInsertDiaryAttchement]
      (@DIARYID            INT,
       @TRACKREFERENCE     INT,
       @CASECODE           VARCHAR(20),
       @NAME               VARCHAR(10),
       @DOCUMENT           VARCHAR(100),
       @FILEPATH           VARCHAR(255),
       @TYPE               VARCHAR(10),
       @DOCCLASS           VARCHAR(10),
       @DATEENTERED        DATETIME,
       @ENTEREDBY          VARCHAR(8),
       @LASTACCESSDATE     DATETIME,
       @LASTACCESSBY       VARCHAR(8),
       @SYSTEM             VARCHAR(1),
       @DICTATIONFILE      VARCHAR(255),
       @Source             VARCHAR(20),
       @Fees               INT,
       @Outlay             INT,
       @Result             VARCHAR(255) OUTPUT)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETAAInsertDiaryAttchement]                                                                     *
 *                                                                                                           *
 * This stored procedure is used to insert or update the Client Case Action Attachment information.          *
 * IF ID already exists then it will update that record, otherwise it will insert a new one                  *
 * (Pino 2019-07-30 the "update" part seems to have been disabled for quite a while so I won't recreate      *
 *  that functionality)                                                                                      *
 *                                                                                                           *
 * PROCEDURE IS OBSOLETE but kept backward compatible by calling new procedure                               *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                -                                                        *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           - [dbo].[ky_NETSPCreateDiaryAttachmentXML]               *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 * Modification History                                                                                      *
 * 2014-??-?? ???????????????????  Created                                                                   *
 * 2014-08-20 Pino Carafa          allow for 20 characters in Case Code as this field will be extended       *
 * 2014-10-23 Pino Carafa          Don't set NOCOUNT ON as it's useful to know how many rows were affected.  *
 * 2015-04-17 Sridharan            Added a result output parameter and check the filepath length             *
 * 2016-03-22 Sridharan            KEYD-3422 Changes to the Firm Undertakings Grid and Form                  *
 * 2019-07-30 Pino Carafa          Call [dbo].[ky_NETSPCreateDiaryAttachmentXML]                             *
 * 2019-07-30 Pino Carafa          Increased TYPE and DOCCLASS to 10 characters                              *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

SET NOCOUNT ON

--Original procedure had been modified to INSERT only. So let's make sure that doesn't change.
IF EXISTS
  (SELECT 1
     FROM [dbo].[DiaryAttachments] DAT
    WHERE DAT.[TRACKREFERENCE] = @TRACKREFERENCE)
  BEGIN
    SET @Result = ''
    RETURN
  END

-- This code appears to be an obsolete attempt to restrict a filename to 255 characters or less
-- we will keep it in for backward compatibility
-- START
IF LEN(@FILEPATH) > 255
  BEGIN
    DECLARE @FILENAME VARCHAR(255)
    DECLARE @NPOS     INT

    SET @FILENAME = REVERSE(@FILEPATH)
    SET @NPOS = CHARINDEX('\', @FILENAME)
    IF @NPOS > 0
      BEGIN
        SET @FILENAME = SUBSTRING(@FILENAME, 1, @NPOS - 1)
      END
    SET @FILENAME = REVERSE(@FILENAME)

    SET @FILEPATH = [dbo].[ky_NETFNGETFILEPATH] (@FILENAME,
                                                 @FILEPATH )
    IF @FILEPATH = 'NOCHANGE'
      BEGIN
        SET @Result = 'NOCHANGE'
        RETURN
      END
  END
-- This code appears to be an obsolete attempt to restrict a filename to 255 characters or less
-- we will keep it in for backward compatibility
-- END

-- KEYD-6133 - older versions of the software might pass in DOC for DOCX or XLS for XLSX documents
--             this block corrects that - START
DECLARE @RECALCTYPE VARCHAR(255)

SET @RECALCTYPE = REVERSE(@FILEPATH)

IF @RECALCTYPE IS NOT NULL
  BEGIN
    IF CHARINDEX('.', @RECALCTYPE) > 0
      BEGIN
        SET @RECALCTYPE = SUBSTRING(@RECALCTYPE, 1, CHARINDEX('.', @RECALCTYPE) - 1)
        SET @RECALCTYPE = UPPER(REVERSE(@RECALCTYPE))

        IF LEN(@RECALCTYPE) > LEN(@TYPE)
          BEGIN
            IF SUBSTRING(@RECALCTYPE, 1, LEN(@TYPE)) = @TYPE
              BEGIN
                SET @TYPE = @RECALCTYPE
              END
          END
      END
  END
-- KEYD-6133 - older versions of the software might pass in DOC for DOCX or XLS for XLSX documents
--             this block corrects that - END

DECLARE @XML NVARCHAR(MAX)

DECLARE @DATEENTEREDSTR    VARCHAR(17)
DECLARE @LASTACCESSDATESTR VARCHAR(17)

SET @DATEENTEREDSTR    = CONVERT(VARCHAR, @DATEENTERED, 112)    + ' ' + CONVERT(VARCHAR, @DATEENTERED, 108)
SET @LASTACCESSDATESTR = CONVERT(VARCHAR, @LASTACCESSDATE, 112) + ' ' + CONVERT(VARCHAR, @LASTACCESSDATE, 108)

SET @XML = CONVERT(NVARCHAR(MAX),
                   (SELECT *
                      FROM (SELECT @DIARYID            AS [diaryid],
                                   @TRACKREFERENCE     AS [trackreference],
                                   @NAME               AS [name],
                                   @DOCUMENT           AS [docname],
                                   @FILEPATH           AS [document],
                                   @TYPE               AS [type],
                                   @DOCCLASS           AS [docclass],
                                   @DATEENTEREDSTR     AS [dateentered],
                                   @ENTEREDBY          AS [enterby],
                                   @LASTACCESSDATESTR  AS [lastaccessed],
                                   @LASTACCESSBY       AS [lastaccessedby],
                                   @SYSTEM             AS [system],
                                   @DICTATIONFILE      AS [dictationfile],
                                   @Source             AS [source],
                                   @Fees               AS [fees],
                                   @Outlay             AS [outlay]) [CreateDiaryAttachment]
                       FOR XML AUTO))

EXEC [dbo].[ky_NETSPCreateDiaryAttachmentXML] @XML

SET @Result = @FILEPATH

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPCopyDiaryAttachmentXML'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPCopyDiaryAttachmentXML]
  END
GO

CREATE PROCEDURE   [dbo].[ky_NETSPCopyDiaryAttachmentXML]
      (@pXML      NVARCHAR(MAX))
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETSPCopyDiaryAttachmentXML]                                                                    *
 *                                                                                                           *
 * Create a Diary Attachment for a particular Action                                                         *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              - [dbo].[ky_NETSPCopyDiaryAttachment]                    *   *
 *   * First compatible version:                - 5.7.2.1                                                *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * ????-??-?? ???????????????????  Created                                                                   *
 * 2019-07-29 Pino Carafa          Converted to XML procedure and added functionality to handle              *
 *                                 document text                                                             *
 * 2019-08-21 Pino Carafa          Added code to write to [ReindexStatus] table                              *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

DECLARE @TrackReference            int
DECLARE @NewDiaryID                int
DECLARE @NewTrackReference         int
DECLARE @NewLocation               VARCHAR(255)
DECLARE @Handler                   VARCHAR(10)
DECLARE @Status                    VARCHAR(5)
DECLARE @pDocumentText             VARCHAR(MAX)

DECLARE @iSL INT

BEGIN TRY
EXEC sp_xml_preparedocument @iSL OUTPUT, @pXML
END TRY
BEGIN CATCH
EXEC sp_xml_preparedocument @iSL OUTPUT, N'<CopyDiaryAttachment />'
END CATCH

SELECT @TrackReference    = [XML].[TrackReference],
       @NewDiaryID        = [XML].[NewDiaryID],
       @NewTrackReference = [XML].[NewTrackReference],
       @NewLocation       = [XML].[NewLocation],
       @Handler           = [XML].[Handler],
       @Status            = [XML].[Status],
       @pDocumentText     = [XML].[DocumentText]
FROM OPENXML(@iSL, N'CopyDiaryAttachment')
WITH ([TrackReference]           INT             '@trackreference',
      [NewDiaryID]               INT             '@newdiaryid',
      [NewTrackReference]        INT             '@newtrackreference',
      [NewLocation]              VARCHAR(255)    '@newlocation',
      [Handler]                  VARCHAR(10)     '@handler',
      [Status]                   VARCHAR(5)      '@status',
      [DocumentText]             VARCHAR(MAX)    'documenttext') [XML]

BEGIN TRY
EXEC sp_xml_removedocument @iSL
END TRY
BEGIN CATCH
END CATCH

DECLARE @CASECODE                  VARCHAR(20)

SET @CASECODE = (SELECT [DIA].[Casecode]
                   FROM [dbo].[diary] [DIA]
                  WHERE [DIA].[ACTIONID] = @NewDiaryID)

IF RTRIM(ISNULL(@CASECODE, '')) = ''
  BEGIN
    RETURN
  END

INSERT
  INTO [dbo].[DiaryAttachments]
      ([TRACKREFERENCE],
       [CASECODE],
       [DATEENTERED],
       [DIARYID],
       [DICTATIONFILE],
       [DOCCLASS],
       [DOCUMENT],
       [ENTEREDBY],
       [FILEPATH],
       [LASTACCESSBY],
       [LASTACCESSDATE],
       [NAME],
       [Source],
       [SYSTEM],
       [Type])
SELECT @NewTrackReference,
       @CASECODE,
       CASE WHEN @Status = ''
                 THEN [DAT].[DATEENTERED]
            ELSE GETDATE() END,
       @NewDiaryID,
       [DAT].[DICTATIONFILE],
       [DAT].[DOCCLASS],
       [DAT].[DOCUMENT],
       [HAN].[CODE],
       @NewLocation,
       [HAN].[CODE],
       CASE WHEN @Status = ''
                 THEN [DAT].[LASTACCESSDATE]
            ELSE GETDATE() END,
       [DAT].[NAME],
       [DAT].[SOURCE],
       [DAT].[SYSTEM],
       [DAT].[TYPE]
  FROM [dbo].[DiaryAttachments] [DAT]
       INNER JOIN [dbo].[diary] [DIA]
               ON DIA.[ACTIONID] = [DAT].[DIARYID]
       INNER JOIN [dbo].[Handlers] [HAN]
               ON [HAN].[CODE] = @Handler
 WHERE [DAT].[TRACKREFERENCE] = @TrackReference

IF @pDocumentText IS NULL
  BEGIN
    SELECT @pDocumentText = [DAX].[TEXT]
      FROM [dbo].[DiaryAttachmentText] [DAX]
     WHERE [DAX].[TrackReference] = @TrackReference
  END

SET @pDocumentText = RTRIM(ISNULL(@pDocumentText, ''))
IF EXISTS
  (SELECT TOP 1 1
      FROM [dbo].[DiaryAttachmentText] [DAX]
    WHERE [DAX].[TrackReference] = @NewTrackReference)
  BEGIN
    UPDATE [DAX]
       SET [DAX].[AccessDate]  = [DAT].[LASTACCESSDATE],
           [DAX].[ModifyDate]  = [DAT].[LASTACCESSDATE],
           [DAX].[CreatedDate] = [DAT].[LASTACCESSDATE],
           [DAX].[TEXT]        = @pDocumentText
      FROM [dbo].[DiaryAttachments] [DAT]
           INNER JOIN [dbo].[DiaryAttachmentText] [DAX]
                   ON [DAX].[TrackReference] = [DAT].[TrackReference]
     WHERE [DAT].[TrackReference] = @NewTrackReference
  END
ELSE
  BEGIN
    INSERT
      INTO [dbo].[DiaryAttachmentText]
          ([TrackReference],
           [AccessDate],
           [CreatedDate],
           [ModifyDate],
           [TEXT])
    SELECT [DAT].[TrackReference],
           [DAT].[LASTACCESSDATE],
           [DAT].[LASTACCESSDATE],
           [DAT].[LASTACCESSDATE],
           @pDocumentText
      FROM [dbo].[DiaryAttachments] [DAT]
     WHERE [DAT].[TrackReference] = @NewTrackReference
  END

IF EXISTS
  (SELECT TOP 1 1
     FROM [dbo].[ReindexStatus] [RIS]
    WHERE [RIS].[TrackReference] = @NewTrackReference)
  BEGIN
    UPDATE [RIS]
       SET [RIS].[STATUS] = 1
      FROM [dbo].[ReindexStatus] [RIS]
     WHERE [RIS].[TrackReference] = @NewTrackReference
  END
ELSE
  BEGIN
    INSERT
      INTO [dbo].[ReindexStatus]
          ([TrackReference],
           [FILEPATH],
           [STATUS])
    VALUES(@NewTrackReference,
           @NewLocation,
           1)
  END

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPCopyDiaryAttachment'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPCopyDiaryAttachment]
  END
GO

CREATE PROCEDURE   [dbo].[ky_NETSPCopyDiaryAttachment]
      (@TrackReference            int,
       @NewDiaryID                int,
       @NewTrackReference         int,
       @NewLocation               VARCHAR(255),
       @Handler                   VARCHAR(10),
       @Status                    VARCHAR(5))
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETSPCopyDiaryAttachment]                                                                       *
 *                                                                                                           *
 * Copies a Diary Attachment to another action id.                                                           *
 *                                                                                                           *
 * PROCEDURE IS OBSOLETE but kept backward compatible by calling new procedure                               *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                -                                                        *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           - [dbo].[ky_NETSPCopyDiaryAttachmentXML]                 *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * ????-??-?? ???????????????????  Created                                                                   *
 * 2019-07-30 Pino Carafa          Procedure is obsolete and it has been replaced by                         *
 *                                 [dbo].[ky_NETSPCopyDiaryAttachmentXML]                                    *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

DECLARE @XML NVARCHAR(MAX)

SET @XML = CONVERT(NVARCHAR(MAX),
                   (SELECT *
                      FROM (SELECT @TrackReference        AS [trackreference],
                                   @NewDiaryID            AS [newdiaryid],
                                   @NewTrackReference     AS [newtrackreference],
                                   @NewLocation           AS [newlocation],
                                   @Handler               AS [handler],
                                   @Status                AS [status]) [CopyDiaryAttachment]
                       FOR XML AUTO))

EXEC [dbo].[ky_NETSPCopyDiaryAttachmentXML] @XML

END

GO

-- There are different variations on this table "out there". If the table does not look as expected, let's save the original table
-- into a copy named "InterestRatesObsolete", just in case some piece of software surfaces that expects the original table with its
-- original layout. One layout we have come across is one with a DATE (or RateDate in SAM4), 6 Rate values RATE1 through RATE6,
-- and 6 "bands", BAND1 through BAND6
IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] [SO]
          LEFT OUTER JOIN SYS.[objects] [SO2]
                       ON [SO2].[name] = 'InterestRatesObsolete'
                      AND [SO2].[type] = 'U'
          CROSS APPLY (SELECT SUM(CASE WHEN [SC2].[name] IN ('RECORDID', 'DATEFROM', 'DATETO', 'RATE') THEN 1 ELSE 0 END) AS [GoodColumns],
                              SUM(CASE WHEN [SC2].[name] IN ('RECORDID', 'DATEFROM', 'DATETO', 'RATE') THEN 0 ELSE 1 END) AS [BadColumns]
                         FROM SYS.[columns] [SC2]
                        WHERE [SC2].[object_id] = [SO].[object_id]) [CC]
    WHERE [SO].[name] = 'InterestRates'
      AND [SO].[type] = 'U'
      AND (   [CC].[GoodColumns] <> 4
           OR [CC].[BadColumns] > 0)
      AND [SO2].[object_id] IS NULL)
  BEGIN
    EXEC ('
    SELECT *
      INTO [dbo].[InterestRatesObsolete]
      FROM [dbo].[InterestRates]')

    EXEC ('DROP TABLE [dbo].[InterestRates]')
  END
GO

-- If the InterestRates table has the right columns but the wrong definition, e.g. RECORDID is not
-- defined as an IDENTITY column, save the data and drop the original table
IF EXISTS
  (SELECT *
     FROM SYS.[objects] [SO]
          LEFT OUTER JOIN SYS.[key_constraints] [KC]
                       ON [KC].[parent_object_id] = [SO].[object_id]
                      AND [KC].[type] = 'PK'
    WHERE [SO].[name] = 'InterestRates'
      AND ISNULL([KC].[name], '') <> 'PK_InterestRates')
  BEGIN
    EXEC ('
    SELECT *
      INTO [dbo].[InterestRatesSaved]
      FROM [dbo].[InterestRates]')

    EXEC ('DROP TABLE [dbo].[InterestRates]')
  END
GO

-- If the InterestRates table does not exist at this point, create it now with the correct definition
IF NOT EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] [SO]
    WHERE [SO].[name] = 'InterestRates'
      AND [SO].[type] = 'U')
  BEGIN
    EXEC ('
    CREATE TABLE
           [dbo].[InterestRates]
          ([RECORDID]              INT            IDENTITY(1, 1)             NOT NULL,
           [DATEFROM]              INT                                       NOT NULL,
           [DATETO]                INT                                       NOT NULL,
           [RATE]                  DECIMAL(7, 2)                                 NULL,
           CONSTRAINT [PK_InterestRates]
                      PRIMARY KEY CLUSTERED
                     ([RECORDID])
                   ON [PRIMARY],
           CONSTRAINT [UQ_InterestRates]
                      UNIQUE NONCLUSTERED
                     ([DATEFROM],
                      [RECORDID])
                   ON [PRIMARY])
        ON [PRIMARY]')
  END
GO

-- If old data were present, re-populate the corrected table
IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] [SO]
    WHERE [SO].[name] = 'InterestRatesSaved'
      AND [SO].[type] = 'U')
  BEGIN
    EXEC ('
    SET IDENTITY_INSERT [dbo].[InterestRates] ON

    INSERT
      INTO [dbo].[InterestRates]
          ([RECORDID],
           [DATEFROM],
           [DATETO],
           [RATE])
    SELECT [INR].[RECORDID],
           [INR].[DATETO],
           [INR].[DATETO],
           [INR].[RATE]
      FROM [dbo].[InterestRatesSaved] [INR]
      
    SET IDENTITY_INSERT [dbo].[InterestRates] OFF
    
    DECLARE @maxVal INT
    SELECT @maxVal = ISNULL(max([RECORDID]),0)+1 from [dbo].[InterestRates]
    DBCC CHECKIDENT(''InterestRates'', RESEED, @maxVal) WITH NO_INFOMSGS')

    EXEC ('DROP TABLE [dbo].[InterestRatesSaved]')
  END
GO

-- Part of KEYD-4599
-- Used to log who deleted what diary item - John Ginnane
IF NOT EXISTS(SELECT TOP 1 1 
                FROM sys.columns AS COL
               WHERE OBJECT_NAME(COL.[object_id]) = 'ItemDiary' 
                 AND COL.[name] = 'Deleted')
BEGIN
	ALTER TABLE [dbo].[ItemDiary]
	  ADD [Deleted] VARCHAR(1) NULL DEFAULT 'N'
END

GO

IF NOT EXISTS(SELECT TOP 1 1 
                FROM sys.columns AS COL
               WHERE OBJECT_NAME(COL.[object_id]) = 'ItemDiary' 
                 AND COL.[name] = 'DeletedBy')
BEGIN
	ALTER TABLE [dbo].[ItemDiary]
	  ADD [DeletedBy] VARCHAR(30) NULL DEFAULT ''
END

GO

IF NOT EXISTS(SELECT TOP 1 1 
                FROM sys.columns AS COL
               WHERE OBJECT_NAME(COL.[object_id]) = 'ItemDiary' 
                 AND COL.[name] = 'DeletedDate')
BEGIN
	ALTER TABLE [dbo].[ItemDiary]
	  ADD [DeletedDate] DATETIME NULL
END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETStrongRoomDeleteDiary'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETStrongRoomDeleteDiary]
  END
GO

CREATE PROCEDURE [dbo].[ky_NETStrongRoomDeleteDiary]
      (@RecordID INT,
       @User     VARCHAR(30),
       @Result   INT OUTPUT)
AS
/************************************************************************************************
 * ky_NETStrongRoomDeleteDiary
 *
 * This procedure is used to delete a diary item from the strong room diary
 *
 * Modification History:
 * 2017-07-25 John         Created
 *
 ************************************************************************************************/
BEGIN
	SET NOCOUNT ON

	SELECT @Result = -1

	IF EXISTS(SELECT TOP 1 1
				FROM sys.columns COL
			   WHERE OBJECT_NAME(COL.[object_id]) = 'ItemDiary'
				 AND COL.[name] = 'Deleted') AND
	   EXISTS(SELECT TOP 1 1
				FROM sys.columns COL
			   WHERE OBJECT_NAME(COL.[object_id]) = 'ItemDiary'
				 AND COL.[name] = 'DeletedBy') AND
	   EXISTS(SELECT TOP 1 1
	            FROM sys.columns COL
	           WHERE OBJECT_NAME(COL.[object_id]) = 'ItemDiary'
	             AND COL.[name] = 'DeletedDate')
	BEGIN
		DECLARE @SQL NVARCHAR(MAX)
		
		SELECT @SQL = '
		UPDATE ID
		   SET ID.[Deleted] = ''Y'',
		       ID.[DeletedBy] = @User,
		       ID.[DeletedDate] = GETDATE()
		  FROM [dbo].[ItemDiary] AS ID
		 WHERE ID.[RECORDID] = @RecordID'
		
		EXEC sp_executesql @SQL, N'@User VARCHAR(30), @RecordID INT', @User = @User, @RecordID = @RecordID
		
		SELECT @Result = 0 
	END

	SET NOCOUNT OFF
END

GO

IF OBJECT_ID(N'[dbo].[ky_NETStrongRoomFetchDiaryDetails]',N'P')IS NOT NULL
    DROP PROCEDURE [dbo].[ky_NETStrongRoomFetchDiaryDetails]
GO

CREATE PROCEDURE [dbo].[ky_NETStrongRoomFetchDiaryDetails] 
      (@Number INT)
AS
/*
	ky_NETStrongRoomFetchDiaryDetails

	This stored procedure is used to FETCH diary details
	Last Updated: 2015-02-12 Arun          - Change NVARCHAR to VARCHAR
				  2015-03-12 Pino          - Use the actual Primary Key "NUMBER", not "ITEMCODE"
				  2017-07-25 John          - Updated procedure to match current standards. Added support for new columns
*/
BEGIN
	SET NOCOUNT ON

	DECLARE @ITEMDIARY_SQL NVARCHAR(MAX)

	SELECT @ITEMDIARY_SQL = '
	SELECT CONVERT(VARCHAR(11), IDI.[DATE], 106)                         AS [DATE],
		   RTRIM(ISNULL(IDI.[DETAILS], ''''))                              AS [DETAILS],
		   CONVERT(VARCHAR(11),
				   DATEADD(dd,
						   IDI.[COMPLETE],
						   ''18001228''),
				   106)                                                  AS [COMPLETE],
		   IDI.[RECORDID]
	  FROM [dbo].[ItemDiary] IDI
	 WHERE IDI.[NUMBER] = @Number'

	-- If the column exists then don't show records that are deleted
	IF EXISTS(SELECT TOP 1 1 
				FROM sys.columns AS COL
			   WHERE OBJECT_NAME(COL.[object_id]) = 'ItemDiary' 
				 AND COL.[name] = 'Deleted')
	BEGIN
		SELECT @ITEMDIARY_SQL = @ITEMDIARY_SQL + '
	  AND ISNULL(IDI.[Deleted], ''N'') <> ''Y'''
	END

	EXEC sp_executesql @ITEMDIARY_SQL, N'@Number INT', @Number = @Number

	SET NOCOUNT OFF
END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETStrongRoomFetchFileForEdit'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETStrongRoomFetchFileForEdit]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETStrongRoomFetchFileForEdit]
      (@Number int)
AS
/****************************************************************************************************
 * ky_NETStrongRoomFetchFileForEdit
 *
 * This stored procedrue is used to FETCH file information for edit
 *
 * Modification History:
 * 2014-05-28 02:58 PM
 * 2015-02-12 Pino NVARCHAR to VARCHAR
 * 2015-03-12 Pino Replace fetch item ITEMCODE with actual Primary Key value NUMBER
 *            SP returned a Table 3 that only contained the CLIENTCODE but that
 *            wasn't referred to in the software. Deleted.
 * 2015-11-26 Pino Add "Needs Attention" column
 * 2015-12-02 Pino Added Executors 3 and 4
 *                 Add "Property Owner"
 * 2015-12-03 Pino Added From and To WHO codes and names
 * 2015-12-07 Pino Added Due Date
 * 2015-12-08 Pino Added Deceased Date
 *                 Added Comment to Item History result set
 *                 Resolve names for WHO and HANDLER in Item History
 *                 Add DeedType and WillType columns from [dbo].[ItemTypes]
 * 2016-04-22 Suriya M KEYD-2666-Allow the Strong Room to hold a scan capture of documents
 * 2017-07-25 John Added criteria to exclude deleted records from item diary - KEYD-4599
 ****************************************************************************************************/
BEGIN
SET NOCOUNT ON

--Table 0
SELECT RTRIM(ISNULL(FMI.[ITEMTYPE], ''))                             AS [ITEMTYPE],
       RTRIM(ISNULL(FMI.[ITEMCODE], ''))                             AS [ITEMCODE],
       RTRIM(ISNULL(FMI.[CLIENTCODE], ''))                           AS [CLIENTCODE],
       RTRIM(ISNULL(FMI.[MATTERCODE], ''))                           AS [MATTERCODE],
       RTRIM(ISNULL(FMI.[DESCRIPTION], ''))                          AS [DESCRIPTION],
       RTRIM(ISNULL(FMI.[LOCATION], ''))                             AS [LOCATION],
       RTRIM(ISNULL(FMI.[BOXNO], ''))                                AS [BOXNO],
       RTRIM(ISNULL(FMI.[SEARCHCODE], ''))                           AS [SEARCHCODE],
       RTRIM(ISNULL(FMI.[FEEARNER], ''))                             AS [FEEARNER],
       CONVERT(VARCHAR(11), FMI.[ENTRYDATE], 112)                    AS [ENTRYDATE],
       CONVERT(VARCHAR(11), FMI.[DESTROYDATE], 112)                  AS [DESTROYDATE],
       FMI.[STATUS]                                                  AS [STATUS],
       FMI.[OPENCLOSED]                                              AS [OPENCLOSED],
       RTRIM(ISNULL(FMI.[CLOSEDFILENO], ''))                         AS [CLOSEDFILENO],
       CONVERT(VARCHAR(11), FMI.[CLOSEDDATE], 112)                   AS [CLOSEDDATE],
       RTRIM(ISNULL(FMI.[LASTACTIONWHO], ''))                        AS [LASTACTIONWHO],
       FMI.[NUMBER]                                                  AS [NUMBER],
       RTRIM(ISNULL(FMI.[DEEDADDRESS], ''))                          AS [DEEDADDRESS],
       FMI.[ARDATE]                                                  AS [ARDATE],
       FMI.[ACCOUNTABLERECEIPT]                                      AS [ACCOUNTABLERECEIPT],
       RTRIM(ISNULL(FMI.[ARNARRATIVE], ''))                          AS [ARNARRATIVE],
       RTRIM(ISNULL(FMI.[TESTATOR], ''))                             AS [TESTATOR],
       RTRIM(ISNULL(FMI.[TESTATORADDRESS], ''))                      AS [TESTATORADDRESS],
       FMI.[WILLDATE]                                                AS [WILLDATE],
       FMI.[ESTIMATEESTATE]                                          AS [ESTIMATEESTATE],
       RTRIM(ISNULL(FMI.[EXECUTORNAME1], ''))                        AS [EXECUTORNAME1],
       RTRIM(ISNULL(FMI.[EXECUTORADDRESS1], ''))                     AS [EXECUTORADDRESS1],
       RTRIM(ISNULL(FMI.[EXECUTORNAME2], ''))                        AS [EXECUTORNAME2],
       RTRIM(ISNULL(FMI.[EXECUTORADDRESS2], ''))                     AS [EXECUTORADDRESS2],
       RTRIM(ISNULL(FMI.[EXECUTORNAME3], ''))                        AS [EXECUTORNAME3],
       RTRIM(ISNULL(FMI.[EXECUTORADDRESS3], ''))                     AS [EXECUTORADDRESS3],
       RTRIM(ISNULL(FMI.[EXECUTORNAME4], ''))                        AS [EXECUTORNAME4],
       RTRIM(ISNULL(FMI.[EXECUTORADDRESS4], ''))                     AS [EXECUTORADDRESS4],
       RTRIM(ISNULL(FMI.[Property Owner], ''))                       AS [Property Owner],
       FMI.[NeedsAttention]                                          AS [NeedsAttention],
       RTRIM(ISNULL(FMI.[FROMWHOCODE], ''))                          AS [FROMWHOCODE],
       RTRIM(ISNULL(FMI.[FROMWHONAME], ''))                          AS [FROMWHONAME],
       RTRIM(ISNULL(FMI.[TOWHOCODE], ''))                            AS [TOWHOCODE],
       RTRIM(ISNULL(FMI.[TOWHONAME], ''))                            AS [TOWHONAME],
       FMI.[DueDate]                                                 AS [DueDate],
       FMI.[DeceasedDate]                                            AS [DeceasedDate],
       ISNULL(ITP.[DeedType], 0)                                     AS [IsDeed],
       ISNULL(ITP.[WillType], 0)                                     AS [IsWill],
       ISNULL(FMI.ActionID,0)										 AS [ActionID]
  FROM [dbo].[FileManagerItems] FMI
       LEFT OUTER JOIN [dbo].[ItemTypes] ITP
                    ON ITP.TYPECODE = FMI.[ITEMTYPE]
 WHERE FMI.[NUMBER] = @Number

--Table 1
SELECT IHI.[ITEMCODE]                                                AS [ItemCode],
       RTRIM(ISNULL(FWH.[NAME],
                    RTRIM(ISNULL(IHI.[WHO], ''))))                   AS [ForWho],
       IHI.[OUTDATE]                                                 AS [OutDate],
       CONVERT(VARCHAR,
               DATEADD(ms,
                       10 * (IHI.[OUTTIME] - 1),
                       CONVERT(VARCHAR, GETDATE(), 112)),
               108)                                                  AS [OutTime],
       IHI.[INDATE]                                                  AS [InDate],
       CONVERT(VARCHAR,
               DATEADD(ms,
                       10 * (IHI.[INTIME] - 1),
                       CONVERT(VARCHAR, GETDATE(), 112)),
               108)                                                  AS [InTime],
       RTRIM(ISNULL(WHO.[NAME],
                    RTRIM(ISNULL(IHI.[HANDLER], ''))))               AS [Who],
       [dbo].[ky_RemoveSpuriousWhitespace2](IHI.[COMMENT], 300)      AS [Comment]
  FROM [dbo].[ItemHistory] IHI
       LEFT OUTER JOIN [dbo].[Handlers] FWH
                    ON FWH.[CODE] = IHI.[WHO]
       LEFT OUTER JOIN [dbo].[Handlers] WHO
                    ON WHO.[CODE] = IHI.[HANDLER]
 WHERE IHI.[NUMBER] = @Number

--Table 2
-- Added check for new "Deleted" columns
-- All the hard work is done in the follow procedure
EXEC [dbo].[ky_NETStrongRoomFetchDiaryDetails] @Number

SET NOCOUNT OFF
END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPGetMissingOrMustPromptUDFs'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPGetMissingOrMustPromptUDFs]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETSPGetMissingOrMustPromptUDFs]
      (@XML NVARCHAR(MAX))
AS
/*********************************************************************************************************************
 *
 * [ky_NETSPGetMissingOrMustPromptUDFs]
 * Processes the Fields that need to be generated, and returns a list of UDFs that are missing or "Always Prompt"
 *
 * Example XML string
 * <UDFS case="000001/0007">
 *   <UDF name="addwitnesses" />
 *   <UDF name="traineeref" />
 *   <UDF name="boohiss" />
 *   <UDF name="SummonsServer"/>
 * </UDFS>
 *
 * Modification History
 * 2016-10-12   Pino Carafa            Created
 * 2017-07-26   Pino Carafa            Ignore those that have a SQL statement
 *
 *********************************************************************************************************************/
BEGIN


DECLARE @iSL INT
DECLARE @UDFS TABLE
       ([CASECODE]         VARCHAR(20)                         NOT NULL,
        [FIELDNAME]        VARCHAR(30)        PRIMARY KEY      NOT NULL,
        UNIQUE NONCLUSTERED ([CASECODE],
                             [FIELDNAME]))

BEGIN TRY
EXEC sp_xml_preparedocument @iSL OUTPUT, @XML
END TRY
BEGIN CATCH
EXEC sp_xml_preparedocument @iSL OUTPUT, N'<UDFS/>'
END CATCH

INSERT
  INTO @UDFS
      ([CASECODE],
       [FIELDNAME])
SELECT SRC.[CASECODE],
       SRC.[FIELDNAME]
FROM OPENXML(@iSL, N'UDFS/UDF')
WITH ([CASECODE]       VARCHAR(20)     '../@case',
      [FIELDNAME]      VARCHAR(30)     '@name') SRC

BEGIN TRY
EXEC sp_xml_removedocument @iSL
END TRY
BEGIN CATCH
END CATCH

SELECT [UDFS].[FIELDNAME]
  FROM @UDFS [UDFS]
       INNER JOIN [dbo].[SystemUserDefinedFields] [SUF]
               ON [SUF].[FILEPREFIX] = 'UDF:'
              AND [SUF].[FIELDNAME] = [UDFS].[FIELDNAME]
              AND RTRIM(ISNULL([SUF].[SQLSTATEMENT], '')) = ''
       LEFT OUTER JOIN [dbo].[CaseUDFAnswers] [CUF]
                    ON [CUF].[CASECODE] = [UDFS].[CASECODE]
                   AND [CUF].[UDFFILE] = [SUF].[FILEPREFIX]
                   AND [CUF].[UDFNAME] = [SUF].[FIELDNAME]
 WHERE (    RTRIM(ISNULL([CUF].[TEXT1], '')) = ''
        AND ISNULL([CUF].[NumberValue], 0) = 0)
    OR [SUF].[HEIGHT] = 1
END

GO

IF NOT EXISTS
  (SELECT TOP 1 1
     FROM sys.objects [SO]
    WHERE [SO].[name] = 'IgnoreClientContacts'
      AND [SO].[type] = 'U')
  BEGIN
    CREATE TABLE
           [dbo].[IgnoreClientContacts]
          ([ClientCode] VARCHAR(10),
           CONSTRAINT [PK_IgnoreClientContacts]
                      PRIMARY KEY CLUSTERED
                     ([ClientCode]))
  END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPListCaseContacts'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPListCaseContacts]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETSPListCaseContacts]
      (@CaseCode VARCHAR(20),
       @ForEmail BIT          = 1,
       @Number   INT          = 0,
       @Status   VARCHAR(1)   = '')
AS
/*************************************************************************************************************
 * [ky_NETSPListCaseContacts]
 * List Case Contacts for Outlook Add-in
 *
 *   *******************
 *   *                 *
 *   * PLEASE NOTE:    * If you make any changes to this procedure's parameters, please inform Pino Carafa as
 *   * ============    * this will impact the Outlook Add-in!
 *   *                 *
 *   *******************
 *
 * Modification History
 * 2015-12-03    Pino              Make it usable for Framework, too
 * 2016-07-21    Suriya M          KEYD-3783 - When you add a new strong room Deed file to a case, the Retired
 *                                 Associates are showing in the "From Whom" & "To Whom" fields of "DEED" tab.
 * 2016-07-22    Pino              Set default value for Status to ensure Outlook Add-in remains unaffected
 * 2018-05-01    Pino Carafa       KEYD-4573 Add Secondary Email
 * 2019-12-04    Pino Carafa       Return CC CONTYPE rather than CAN TYPE in first query
 * 2019-12-16    Pino Carafa       Return Occupation
 *
 *************************************************************************************************************/
BEGIN
SET NOCOUNT ON
  DECLARE @FromWhom VARCHAR(10)

  IF(@Status = 'T')
    BEGIN
      SET @FromWhom = RTRIM(ISNULL((SELECT [FMI].[TOWHOCODE]
                                      FROM [dbo].[FileManagerItems] [FMI]
                                     WHERE [FMI].[NUMBER] = @Number), ''))
    END
  ELSE
    BEGIN
      SET @FromWhom = RTRIM(ISNULL((SELECT [FMI].[FROMWHOCODE]
                                      FROM [dbo].[FileManagerItems] [FMI]
                                     WHERE [FMI].[NUMBER] = @Number), ''))
    END

  SELECT [RES].[CODE],
         CASE WHEN [RES].[NAME] = ''
                   THEN [RES].[COMPANY]
              ELSE [RES].[NAME] END                                              AS [NAME],
         [RES].[COMPANY],
         [RES].[EMAIL],
         [RES].[NAMECODE],
         [RES].[CASECODE],
         [RES].[TYPE],
         [RES].[Occupation]                                                      AS [Occupation],
         CASE WHEN RTRIM(ISNULL([AST].[DESCRIPTION], '')) = ''
                   THEN [RES].[TYPE]
              ELSE RTRIM(ISNULL([AST].[DESCRIPTION], '')) END                    AS [TYPEDESC],
         LTRIM(  CASE WHEN [RES].[NAME] = ''
                           THEN ''
                      WHEN [RES].[NAME] = [RES].[COMPANY]
                           THEN ''
                      ELSE [RES].[NAME] END
               + CASE WHEN [RES].[COMPANY] = ''
                           THEN ''
                      ELSE ' (' + [RES].[COMPANY] + ')' END)                     AS [NAMECOMP]
    FROM (SELECT CONVERT(VARCHAR(10), RTRIM(ISNULL([CAN].[CODE], '')))           AS [CODE],
                 CONVERT(VARCHAR(300), RTRIM(ISNULL([CAN].[NAME], '')))          AS [NAME],
                 CONVERT(VARCHAR(50), RTRIM(ISNULL([CAN].[COMPANY], '')))        AS [COMPANY],
                 CONVERT(VARCHAR(100), RTRIM(ISNULL([CAN].[EMAIL], '')))         AS [EMAIL],
                 CONVERT(VARCHAR(100), RTRIM(ISNULL([CAN].[OCCUPATION], '')))    AS [Occupation],
                 CONVERT(VARCHAR(10), RTRIM(ISNULL([CC].[NAMECODE], '')))        AS [NAMECODE],
                 CONVERT(VARCHAR(20), RTRIM(ISNULL([CC].[CASECODE], '')))        AS [CASECODE],
                 CONVERT(VARCHAR(20), RTRIM(ISNULL([CC].[CONTYPE], '')))         AS [TYPE] --PINO 2019-12-04, WAS [CAN].[TYPE]
            FROM [dbo].[CaseContacts] [CC]
                 INNER JOIN [dbo].[CaseAssociatesNames] [CAN]
                         ON [CAN].[CODE] = [CC].[NameCODE]
                        AND (   @ForEmail = 0
                             OR RTRIM(ISNULL([CAN].[EMAIL], '')) <> '')
           WHERE [CC].[CASECODE] = @CaseCode
             AND (   ISNULL([CC].Retired,'N') <> 'Y'
                  OR [CC].NAMECODE = CASE @Number
                                        WHEN 0 THEN ''
                                   ELSE @FromWhom END)
          UNION
          SELECT CONVERT(VARCHAR(10), RTRIM(ISNULL([CTC].[CLCODE], '')))         AS [CODE],
                 CONVERT(VARCHAR(300),   RTRIM(ISNULL([CTC].[FIRSTNAME], ''))
                                       + ' '
                                       + RTRIM(ISNULL([CTC].[SURNAME], '')))     AS [NAME],
                 CONVERT(VARCHAR(50), RTRIM(ISNULL([CTC].[CLNAMECON], '')))      AS [COMPANY],
                 CONVERT(VARCHAR(100), RTRIM(ISNULL([CTC].[EmailAddress], '')))  AS [EMAIL],
                 CONVERT(VARCHAR(100), RTRIM(ISNULL([CTC].[Occupation], '')))    AS [Occupation],
                 CONVERT(VARCHAR(10), '')                                        AS [NAMECODE],
                 CONVERT(VARCHAR(20), @CaseCode)                                 AS [CASECODE],
                 CONVERT(VARCHAR(20), 'Client Contact')                          AS [TYPE]
            FROM [dbo].[matters] [MAT]
                 INNER JOIN [dbo].[ClientContacts] [CTC]
                         ON [CTC].[CLCODE] = [MAT].[ClientCode]
                        AND (   @ForEmail = 0
                             OR RTRIM(ISNULL([CTC].[EmailAddress], '')) <> '')
           WHERE [MAT].[CODE] = @CaseCode
          UNION
          SELECT CONVERT(VARCHAR(10), RTRIM(ISNULL([CAN].[CODE], '')))           AS [CODE],
                 CONVERT(VARCHAR(300), RTRIM(ISNULL([CAC].[NAME], '')))          AS [NAME],
                 CONVERT(VARCHAR(50), RTRIM(ISNULL([CAN].[COMPANY], '')))        AS [COMPANY],
                 CONVERT(VARCHAR(100), RTRIM(ISNULL([CAC].[EMAIL], '')))         AS [EMAIL],
                 CONVERT(VARCHAR(100), RTRIM(ISNULL([CAN].[OCCUPATION], '')))    AS [Occupation],
                 CONVERT(VARCHAR(10), RTRIM(ISNULL([CC].[NAMECODE], '')))        AS [NAMECODE],
                 CONVERT(VARCHAR(20), RTRIM(ISNULL([CC].[CASECODE], '')))        AS [CASECODE],
                 CONVERT(VARCHAR(20), RTRIM(ISNULL([CC].[CONTYPE], '')))         AS [TYPE]
            FROM [dbo].[CaseContacts] [CC]
                 INNER JOIN [dbo].[CaseAssociatesNames] [CAN]
                            INNER JOIN [dbo].[CaseAssociatesContacts] [CAC]
                                    ON [CAC].[NAMECODE] = [CAN].[CODE]
                                   AND (   @ForEmail = 0
                                        OR RTRIM(ISNULL([CAC].[EMAIL], '')) <> '')
                         ON [CAN].[CODE] = [CC].[NameCODE]
           WHERE [CC].[CASECODE] = @CaseCode
          UNION
          SELECT CONVERT(VARCHAR(10), RTRIM(ISNULL([CTC].[CODE], '')))           AS [CODE],
                 CONVERT(VARCHAR(300), RTRIM(ISNULL([CTC].[Name], '')))          AS [NAME],
                 CONVERT(VARCHAR(50), '')                                        AS [COMPANY],
                 CONVERT(VARCHAR(100), RTRIM(ISNULL([CTC].[email], '')))         AS [EMAIL],
                 CONVERT(VARCHAR(100), RTRIM(ISNULL([CTC].[Occupation], '')))    AS [Occupation],
                 CONVERT(VARCHAR(10), '')                                        AS [NAMECODE],
                 CONVERT(VARCHAR(20), @CaseCode)                                 AS [CASECODE],
                 CONVERT(VARCHAR(20), 'Client')                                  AS [TYPE]
            FROM [dbo].[matters] [MAT]
                 INNER JOIN [dbo].[Contacts] [CTC]
                         ON [CTC].[CODE] = [MAT].[ClientCode]
                        AND (   @ForEmail = 0
                             OR RTRIM(ISNULL([CTC].[email], '')) <> '')
           WHERE [MAT].[CODE] = @CaseCode
          UNION
          SELECT CONVERT(VARCHAR(10), RTRIM(ISNULL([CTC].[CODE], '')))           AS [CODE],
                 CONVERT(VARCHAR(300), RTRIM(ISNULL([CTC].[Name], '')))          AS [NAME],
                 CONVERT(VARCHAR(50), '')                                        AS [COMPANY],
                 CONVERT(VARCHAR(100), RTRIM(ISNULL([CTC].[SecondaryEmail],
                                                    '')))                        AS [EMAIL],
                 CONVERT(VARCHAR(100), RTRIM(ISNULL([CTC].[Occupation], '')))    AS [Occupation],
                 CONVERT(VARCHAR(10), '')                                        AS [NAMECODE],
                 CONVERT(VARCHAR(20), @CaseCode)                                 AS [CASECODE],
                 CONVERT(VARCHAR(20), 'Client')                                  AS [TYPE]
            FROM [dbo].[matters] [MAT]
                 INNER JOIN [dbo].[Contacts] [CTC]
                         ON [CTC].[CODE] = [MAT].[ClientCode]
                        AND (   @ForEmail = 0
                             OR RTRIM(ISNULL([CTC].[SecondaryEmail], '')) <> '')
           WHERE [MAT].[CODE] = @CaseCode) [RES]
         LEFT OUTER JOIN [dbo].[AssociateTypes] [AST]
                      ON [AST].[CODE] = [RES].[TYPE]
ORDER BY CASE WHEN [RES].[NAME] = ''
                   THEN [RES].[COMPANY]
              ELSE [RES].[NAME] END,
         [RES].[TYPE]
SET NOCOUNT OFF
END

GO

INSERT
  INTO [dbo].[TemplateActions]
      ([ACTIONCODE],
       [DESC],
       [WKTCODE],
       [ACTTYPE],
       [ACTIONCATEGORY],
       [STATUS],
       [PROCESSTYPE],
       [PUBLISHER],
       [UPDATEFIELD],
       [UPDATEVALUE],
       [HIGHLIGHTED],
       [MILESTONETYPE],
       [PRIORITY],
       [NEXTACTIONOPTIONAL],
       [BILLABLE],
       [EMAIL],
       [BILLDESCRIPTION],
       [AssignActionTo],
       [ASSIGNHANDLER],
       [AssignTeamActionTo],
       [ASSIGNTEAM],
       [Description])
SELECT [NEW].[ACTIONCODE],
       [NEW].[DESC],
       [NEW].[WKTCODE],
       [NEW].[ACTTYPE],
       [NEW].[ACTIONCATEGORY],
       [NEW].[STATUS],
       [NEW].[PROCESSTYPE],
       [NEW].[PUBLISHER],
       [NEW].[UPDATEFIELD],
       [NEW].[UPDATEVALUE],
       [NEW].[HIGHLIGHTED],
       [NEW].[MILESTONETYPE],
       [NEW].[PRIORITY],
       [NEW].[NEXTACTIONOPTIONAL],
       [NEW].[BILLABLE],
       [NEW].[EMAIL],
       [NEW].[BILLDESCRIPTION],
       [NEW].[AssignActionTo],
       [NEW].[ASSIGNHANDLER],
       [NEW].[AssignTeamActionTo],
       [NEW].[ASSIGNTEAM],
       [NEW].[Description]
  FROM (SELECT 'ADMFS'                                     AS [ACTIONCODE],
               'File Sharing Request'                      AS [DESC],
               'SYS'                                       AS [WKTCODE],
               'A'                                         AS [ACTTYPE],
               'A'                                         AS [ACTIONCATEGORY],
               'NA'                                        AS [STATUS],
               'I'                                         AS [PROCESSTYPE],
               'N'                                         AS [PUBLISHER],
               ''                                          AS [UPDATEFIELD],
               ''                                          AS [UPDATEVALUE],
               'N'                                         AS [HIGHLIGHTED],
               ''                                          AS [MILESTONETYPE],
               'Normal'                                    AS [PRIORITY],
               'Y'                                         AS [NEXTACTIONOPTIONAL],
               'N'                                         AS [BILLABLE],
               ''                                          AS [EMAIL],
               ''                                          AS [BILLDESCRIPTION],
               'L'                                         AS [AssignActionTo],
               ''                                          AS [ASSIGNHANDLER],
               'L'                                         AS [AssignTeamActionTo],
               ''                                          AS [ASSIGNTEAM],
               'File Sharing Request'                      AS [Description]) [NEW]
       LEFT OUTER JOIN [dbo].[TemplateActions] [TAC]
                    ON [TAC].[ACTIONCODE] = [NEW].[ACTIONCODE]
 WHERE [TAC].[ACTIONCODE] IS NULL
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPCheckExistingCases'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPCheckExistingCases]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETSPCheckExistingCases]
      (@SENT                  VARCHAR(17),
       @HASH                  VARCHAR(32),
       @MATTER                VARCHAR(20),
       @SenderSMTPAddress     VARCHAR(MAX),
       @TrackReference        INT = 0)
AS
/******************************************************************************
 *
 * ky_NETSPCheckExistingCases
 * Checks whether an e-mail may already have been saved to case
 *
 * Modification History
 * 2015-05-19  Pino  Check message hashes first. Add [id] column to ensure
 *                   correct ordering
 * 2015-05-28  Pino  Remove duplicates
 * 2015-06-12  Pino  If Hash is passed in, use it!
 * 2016-10-24  Pino  Use Track Reference
 * 2017-03-13  Pino  Extend time span to 10 minutes
 * 2017-05-02  Pino  Include a check on [dbo].[diary].[TxmDate] when checking
 *                   a Hash code
 * 2017-11-23  Pino  Include a check on [dbo].[diary].[Date] when checking
 *                   a Hash code
 *
 ******************************************************************************/
BEGIN

DECLARE @cases TABLE
       ([id]                     INT                  IDENTITY(1, 1),
        [MatterCode]             VARCHAR(20),
        [ClientCode]             VARCHAR(20),
        [MatterDescription]      VARCHAR(500),
        [ClientName]             VARCHAR(500),
        [ActionID]               INT,
        [Subject]                VARCHAR(1000))

SET @SenderSMTPAddress = RTRIM(ISNULL(@SenderSMTPAddress, ''))
SET @MATTER = RTRIM(ISNULL(@MATTER, ''))

SET @HASH = RTRIM(ISNULL(@HASH, ''))

SET @TrackReference = ISNULL(@TrackReference, 0)

IF (@TrackReference <> 0)
  BEGIN
    INSERT
      INTO @cases
          ([MatterCode],
           [ClientCode],
           [MatterDescription],
           [ClientName],
           [ActionID],
           [Subject])
    SELECT RTRIM(ISNULL(DIA.[CASECODE], '')) AS [MatterCode],
           RTRIM(ISNULL(MAT.[ClientCode], '')) As [ClientCode],
           RTRIM(ISNULL(MAT.[Description], '')) As [MatterDescription],
           RTRIM(ISNULL(CON.[name], '')) As [ClientName],
           ISNULL(DIA.[ActionID], 0) AS [ActionID],
           RTRIM(ISNULL(CONVERT(VARCHAR(1000), DIA.[Subject]), '')) As [Subject]
      FROM [dbo].[MessageHashXRef] H
           INNER JOIN [dbo].[DiaryAttachments] DAT
                      INNER JOIN [dbo].[diary] DIA
                                 INNER JOIN [dbo].[matters] MAT
                                            INNER JOIN [dbo].[contacts] CON
                                                    ON CON.[code] = MAT.[ClientCode]
                                         ON MAT.[code] = DIA.[CASECODE]
                              ON DIA.[ACTIONID] = DAT.[DIARYID]
                   ON DAT.[TRACKREFERENCE] = H.[TrackReference]
     WHERE H.[TrackReference] = @TrackReference
  END

IF (@HASH <> '')
  BEGIN
    INSERT
      INTO @cases
          ([MatterCode],
           [ClientCode],
           [MatterDescription],
           [ClientName],
           [ActionID],
           [Subject])
    SELECT RTRIM(ISNULL(DIA.[CASECODE], '')) AS [MatterCode],
           RTRIM(ISNULL(MAT.[ClientCode], '')) As [ClientCode],
           RTRIM(ISNULL(MAT.[Description], '')) As [MatterDescription],
           RTRIM(ISNULL(CON.[name], '')) As [ClientName],
           ISNULL(DIA.[ActionID], 0) AS [ActionID],
           RTRIM(ISNULL(CONVERT(VARCHAR(1000), DIA.[Subject]), '')) As [Subject]
      FROM [dbo].[MessageHashXRef] H
           INNER JOIN [dbo].[DiaryAttachments] DAT
                      INNER JOIN [dbo].[diary] DIA
                                 INNER JOIN [dbo].[matters] MAT
                                            INNER JOIN [dbo].[contacts] CON
                                                    ON CON.[code] = MAT.[ClientCode]
                                         ON MAT.[code] = DIA.[CASECODE]
                              ON DIA.[ACTIONID] = DAT.[DIARYID]
                   ON DAT.[TRACKREFERENCE] = H.[TrackReference]
     WHERE H.[Hash] = @Hash
       AND (   (    H.[Sent] >= DATEADD(ss, -600, @SENT)
                AND H.[Sent] <= DATEADD(ss, 600, @SENT))
            OR (    ISNULL([DIA].[TxmDate], [H].[Sent]) >= DATEADD(ss, -600, @SENT)
                AND ISNULL([DIA].[TxmDate], [H].[Sent]) <= DATEADD(ss, 600, @SENT))
            OR (    ISNULL([DIA].[Date], [H].[Sent]) >= DATEADD(ss, -600, @SENT)
                AND ISNULL([DIA].[Date], [H].[Sent]) <= DATEADD(ss, 600, @SENT)))
       AND H.[TrackReference] <> @TrackReference
  END

IF (@SenderSMTPAddress <> '')
  BEGIN
    IF (@MATTER <> '')
      BEGIN
        INSERT
          INTO @cases
              ([MatterCode],
               [ClientCode],
               [MatterDescription],
               [ClientName],
               [ActionID],
               [Subject])
          SELECT RTRIM(ISNULL(MAT.[Code], '')) As [MatterCode],
                 RTRIM(ISNULL(MAT.[ClientCode], '')) As [ClientCode],
                 RTRIM(ISNULL(MAT.[Description], '')) As [MatterDescription],
                 RTRIM(ISNULL(CON.[name], '')) As [ClientName],
                 ISNULL(DIA.[ActionID], 0) AS [ActionID],
                 RTRIM(ISNULL(CONVERT(VARCHAR(1000), DIA.[Subject]), '')) As [Subject]
            FROM [dbo].[matters] MAT
                 INNER JOIN [dbo].[contacts] CON
                         ON CON.[code] = MAT.[ClientCode]
                 OUTER APPLY (SELECT TOP 1 DI2.[ActionID],
                                           DI2.[Subject]
                                FROM [dbo].[diary] DI2
                               WHERE DI2.[CASECODE] = MAT.[CODE]
                                 AND DI2.[TxmDate] >= DATEADD(ss, -120, @SENT)
                                 AND DI2.[TxmDate] <= DATEADD(ss, 120, @SENT)
                                 AND (   DI2.[EMAILADDRESS] LIKE '%<' + @SenderSMTPAddress + '>%'
                                      OR DI2.[EMAILADDRESS] LIKE '%(' + @SenderSMTPAddress + ')%'
                                      OR DI2.[EMAILADDRESS] LIKE '%' + @SenderSMTPAddress + ' <%>%'
                                      OR DI2.[EMAILADDRESS] LIKE '%' + @SenderSMTPAddress + ' (%)%'
                                      OR DI2.[EMAILADDRESS] = @SenderSMTPAddress
                                      OR DI2.[EMAILADDRESS] LIKE [dbo].[ky_InterpretExchangeAddress](@SenderSMTPAddress))
                                 AND DI2.[EMAIL] = 'Y') DIA
                 CROSS APPLY (SELECT COUNT(1) AS [TOTAL]
                                FROM @cases CAS
                               WHERE CAS.[MatterCode] = MAT.[Code]
                                 AND CAS.[ActionID] = DIA.[ACTIONID]) ALI
           WHERE MAT.[code] = @MATTER
             AND ALI.[TOTAL] = 0
      END
    ELSE
      BEGIN
        IF (@HASH = '')
          BEGIN
            INSERT
              INTO @cases
                  ([MatterCode],
                   [ClientCode],
                   [MatterDescription],
                   [ClientName],
                   [ActionID],
                   [Subject])
              SELECT RTRIM(ISNULL(MAT.[Code], '')) As [MatterCode],
                     RTRIM(ISNULL(MAT.[ClientCode], '')) As [ClientCode],
                     RTRIM(ISNULL(MAT.[Description], '')) As [MatterDescription],
                     RTRIM(ISNULL(CON.[name], '')) As [ClientName],
                     ISNULL(MAX(DIA.[ActionID]), 0) AS [ActionID],
                     RTRIM(ISNULL(CONVERT(VARCHAR(1000), DIA.[Subject]), '')) As [Subject]
                FROM [dbo].[diary] DIA
                     INNER JOIN [dbo].[matters] MAT
                                INNER JOIN [dbo].[contacts] CON
                                        ON CON.[code] = MAT.[ClientCode]
                             ON MAT.[code] = DIA.[CASECODE]
                     CROSS APPLY (SELECT COUNT(1) AS [TOTAL]
                                    FROM @cases CAS
                                   WHERE CAS.[ActionID] = DIA.[ACTIONID]) ALI
               WHERE DIA.[TxmDate] >= DATEADD(ss, -120, @SENT)
                 AND DIA.[TxmDate] <= DATEADD(ss, 120, @SENT)
                 AND (   DIA.[EMAILADDRESS] LIKE '%<' + @SenderSMTPAddress + '>%'
                      OR DIA.[EMAILADDRESS] LIKE '%(' + @SenderSMTPAddress + ')%'
                      OR DIA.[EMAILADDRESS] LIKE '%' + @SenderSMTPAddress + ' <%>%'
                      OR DIA.[EMAILADDRESS] LIKE '%' + @SenderSMTPAddress + ' (%)%'
                      OR DIA.[EMAILADDRESS] = @SenderSMTPAddress
                      OR DIA.[EMAILADDRESS] LIKE [dbo].[ky_InterpretExchangeAddress](@SenderSMTPAddress))
                 AND DIA.[EMAIL] = 'Y'
                 AND ALI.[TOTAL] = 0
            GROUP BY RTRIM(ISNULL(MAT.[Description], '')),
                     RTRIM(ISNULL(MAT.[Code], '')),
                     RTRIM(ISNULL(MAT.[ClientCode], '')),
                     RTRIM(ISNULL(CON.[name], '')),
                     RTRIM(ISNULL(CONVERT(VARCHAR(1000), DIA.[Subject]), ''))
          END
      END
  END

  SELECT CS.[MatterCode],
         CS.[ClientCode],
         CS.[MatterDescription],
         CS.[ClientName],
         CS.[ActionID],
         CS.[Subject]
    FROM @cases CS
ORDER BY CS.[id]

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETUpdateSubjectandAttachment2'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETUpdateSubjectandAttachment2]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETUpdateSubjectandAttachment2]
      (@ActionID            INT,
       @Text                VARCHAR (MAX),
       @From                VARCHAR (50),
       @settings            CHAR (1),
       @Handler             NVARCHAR (10),      --@Handler MUST be NVARCHAR as it's defined as such in [dbo].[UserSettings]
       @SendToNewAction     VARCHAR(30) = '')
AS
/******************************************************************************************************
 *
 * Created By :
 * Date       : 25MAR2015 2:37 PM
 * Description: This stored procedure is used to Update subject and attachement details.
 *              Originally written as ky_NETUpdateSubjectandAttachment
 *              Had to create new version as the DocSyncWithSubject handling was completely wrong
 *
 * Modification History
 * ----------------------------------------------------------------------------------------------
 * Date        | Modified By   | Description
 * ----------------------------------------------------------------------------------------------
 * 2015-02-12    Arun            Change NVARCHAR to VARCHAR
 * 2016-08-31    Sridharan       KEYD:3793-Send to in Case Diary Attachment is closing new action and
 *                               renaming old (Added a new parameter SendToNewAction)
 * 2016-09-19    Pino            Looking into KEYD-4060 I found a few minor issues with this procedure,
 *                               it was not easy to read, and the @Text input parameter could cause
 *                               truncation issues on the columns it's updating.
 * 2017-04-05    Pino            Allow for saving No value for Setting. Blank to Reset.
 *                               Y = use saved Sync setting, Yes picked
 *                               N = use saved Sync setting, No picked
 *                                 = Don't use a saved Sync setting. Prompt the user.
 *
 ******************************************************************************************************/
BEGIN

SET NOCOUNT ON;

IF ISNULL(@settings, '') <> 'N'
  BEGIN
    IF @From = 'Document Manager'
      BEGIN
        IF @SendToNewAction = ''
          BEGIN
            UPDATE [DIA]
               SET [DIA].[subject] = RTRIM(CONVERT(VARCHAR(500), @Text))
              FROM [dbo].[diary] [DIA]
             WHERE [DIA].[ACTIONID] = @ActionID;
          END
      END
    ELSE
      IF @From = 'Case Diary'
        BEGIN
          UPDATE [DAT]
             SET [DAT].[DOCUMENT] = RTRIM(CONVERT(VARCHAR(100), @Text))
            FROM [dbo].[DiaryAttachments] [DAT]
           WHERE [DAT].[DiaryID] = @ActionID

          UPDATE [DIA]
             SET [DIA].[SUBJECT] = RTRIM(CONVERT(VARCHAR(500), @Text))
            FROM [dbo].[diary] [DIA]
           WHERE [DIA].[ACTIONID] = @ActionID
        END
  END

IF @settings = 'Y'
  BEGIN
    IF EXISTS
      (SELECT TOP 1
              [USS].[SettingID]
         FROM [dbo].[UserSettings] [USS]
        WHERE [USS].[Handler] = @Handler
          AND [USS].[key] = 'DocSyncWithSubject')
      BEGIN
        UPDATE [USS]
           SET [USS].[Value] = 'True'
          FROM [dbo].[UserSettings] [USS]
         WHERE [USS].[Handler] = @Handler
           AND [USS].[key] = 'DocSyncWithSubject'
      END
    ELSE
      BEGIN
        INSERT
          INTO [dbo].[UserSettings]
              ([Handler],
               [Key],
               [Value])
        VALUES(@Handler,
               'DocSyncWithSubject',
               'True')
      END

    IF EXISTS
      (SELECT TOP 1
              [USS].[SettingID]
         FROM [dbo].[UserSettings] [USS]
        WHERE [USS].[Handler] = @Handler
          AND [USS].[key] = 'DocSyncWithSubjectYesNo')
      BEGIN
        UPDATE [USS]
           SET [USS].[Value] = 'True'
          FROM [dbo].[UserSettings] [USS]
         WHERE [USS].[Handler] = @Handler
           AND [USS].[key] = 'DocSyncWithSubjectYesNo'
      END
    ELSE
      BEGIN
        INSERT
          INTO [dbo].[UserSettings]
              ([Handler],
               [Key],
               [Value])
        VALUES(@Handler,
               'DocSyncWithSubjectYesNo',
               'True')
      END
  END
ELSE IF @settings = 'N'
  BEGIN
    IF EXISTS
      (SELECT TOP 1
              [USS].[SettingID]
         FROM [dbo].[UserSettings] [USS]
        WHERE [USS].[Handler] = @Handler
          AND [USS].[key] = 'DocSyncWithSubject')
      BEGIN
        UPDATE [USS]
           SET [USS].[Value] = 'True'
          FROM [dbo].[UserSettings] [USS]
         WHERE [USS].[Handler] = @Handler
           AND [USS].[key] = 'DocSyncWithSubject'
      END
    ELSE
      BEGIN
        INSERT
          INTO [dbo].[UserSettings]
              ([Handler],
               [Key],
               [Value])
        VALUES(@Handler,
               'DocSyncWithSubject',
               'True')
      END

    IF EXISTS
      (SELECT TOP 1
              [USS].[SettingID]
         FROM [dbo].[UserSettings] [USS]
        WHERE [USS].[Handler] = @Handler
          AND [USS].[key] = 'DocSyncWithSubjectYesNo')
      BEGIN
        UPDATE [USS]
           SET [USS].[Value] = 'False'
          FROM [dbo].[UserSettings] [USS]
         WHERE [USS].[Handler] = @Handler
           AND [USS].[key] = 'DocSyncWithSubjectYesNo'
      END
    ELSE
      BEGIN
        INSERT
          INTO [dbo].[UserSettings]
              ([Handler],
               [Key],
               [Value])
        VALUES(@Handler,
               'DocSyncWithSubjectYesNo',
               'False')
      END
  END
ELSE
  BEGIN
    IF EXISTS
      (SELECT TOP 1
              [USS].[SettingID]
         FROM [dbo].[UserSettings] [USS]
        WHERE [USS].[Handler] = @Handler
          AND [USS].[key] = 'DocSyncWithSubject')
      BEGIN
        UPDATE [USS]
           SET [USS].[Value] = 'False'
          FROM [dbo].[UserSettings] [USS]
         WHERE [USS].[Handler] = @Handler
           AND [USS].[key] = 'DocSyncWithSubject'
      END
    ELSE
      BEGIN
        INSERT
          INTO [dbo].[UserSettings]
              ([Handler],
               [Key],
               [Value])
        VALUES(@Handler,
               'DocSyncWithSubject',
               'False')
      END

    IF EXISTS
      (SELECT TOP 1
              [USS].[SettingID]
         FROM [dbo].[UserSettings] [USS]
        WHERE [USS].[Handler] = @Handler
          AND [USS].[key] = 'DocSyncWithSubjectYesNo')
      BEGIN
        UPDATE [USS]
           SET [USS].[Value] = 'False'
          FROM [dbo].[UserSettings] [USS]
         WHERE [USS].[Handler] = @Handler
           AND [USS].[key] = 'DocSyncWithSubjectYesNo'
      END
    ELSE
      BEGIN
        INSERT
          INTO [dbo].[UserSettings]
              ([Handler],
               [Key],
               [Value])
        VALUES(@Handler,
               'DocSyncWithSubjectYesNo',
               'False')
      END
  END

SET NOCOUNT OFF;

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPUpdateDocManagerDoc'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPUpdateDocManagerDoc]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETSPUpdateDocManagerDoc]
      (@TrackReference        VARCHAR(10),
       @Document              VARCHAR(2000),
       @DocumentDate          VARCHAR(17),
       @ClassCode             VARCHAR(10),
       @Subject               VARCHAR(2000),
       @DocumentPath          VARCHAR(1000),
       @Result                VARCHAR(255)      OUTPUT)
AS
/************************************************************************************************************
 *
 * ky_NETSPUpdateDocManagerDoc
 *
 * Updates a Diary Attachment entry FROM the Document Manager screen in the Framework
 *
 * Modification History
 * ????-??-??   Sridharan         Created
 * 2015-04-08   Sridharan         KEYD-2109 Added a new parameter to get modified filepath
 *                                          and change the relevant column in the Table
 * 2015-04-15   Sridharan         Added a new output parameter to get the File path
 * 2016-03-22   Sridharan         KEYD-3422 Changes to the Firm Undertakings Grid and Form
 * 2017-08-15   Pino Carafa       a) Procedure was practically unreadable
 * 2017-08-15   Pino Carafa       b) Remove irrelevant SELECT statement (proc always executed as NonQuery)
 * 2017-08-15   Pino Carafa       c) Replace repeated SET @var = (SELECT ...) statements with a single SELECT
 * 2017-08-15   Pino Carafa       d) Make description larger. VARCHAR(100) may not be enough.
 *
 ************************************************************************************************************/
BEGIN

DECLARE @LEN INT
DECLARE @FPATH VARCHAR(1000)
DECLARE @ActionID INT
DECLARE @ActionType VARCHAR(1)
DECLARE @UserName VARCHAR(20)
DECLARE @OLDDocument VARCHAR(500)
DECLARE @OLDClass VARCHAR(5)
DECLARE @OLDDate  VARCHAR(11)
DECLARE @Description VARCHAR(500)
DECLARE @DescriptionTo VARCHAR(500)

-- 2017-08-15 c)
SELECT @FPATH          = [DAT].[FILEPATH],
       @Actionid       = [DAT].[DiaryID],
       @ActionType     = ISNULL([DIA].[ACTIONTYPE], ''),
       @UserName       = RTRIM(ISNULL([HAN].[NAME], ISNULL([DIA].[FNCODE], ''))),
       @OLDDocument    = [DAT].[DOCUMENT],
       @OLDClass       = [DAT].[DOCCLASS],
       @OLDDate        = CONVERT(VARCHAR(11), CONVERT(DATETIME, [DAT].[DATEENTERED]), 106),
       @Description    = 'Attachment Changed From: ' + CHAR(13) + CHAR(10)
                       + 'Date:  ' + CONVERT(VARCHAR(11), CONVERT(DATETIME, [DAT].[DATEENTERED]), 106) + CHAR(13) + CHAR(10)
                       + 'Document Name: ' + [DAT].[DOCUMENT] + CHAR(13) + CHAR(10)
                       + 'Document Class: ' + [DAT].[DOCCLASS] + CHAR(13) + CHAR(10)
  FROM [dbo].[DiaryAttachments] [DAT]
       LEFT OUTER JOIN [dbo].[diary] [DIA]
                       LEFT OUTER JOIN [dbo].[Handlers] [HAN]
                                    ON [HAN].[CODE] = [DIA].[FNCODE]
                    ON [DIA].[ACTIONID] = [DAT].[DiaryID]
 WHERE [DAT].[TrackReference] = @TrackReference

SET @DescriptionTo = 'Attachment Changed To: ' + CHAR(13) + CHAR(10)
                   + 'Date:  ' + CONVERT(VARCHAR(11),CONVERT(DATETIME, @DocumentDate), 106) + CHAR(13) + CHAR(10)
                   + 'Document Name: ' + RTRIM(ISNULL(@DOCUMENT, '')) + CHAR(13) + CHAR(10)
                   + 'Document Class: ' + RTRIM(ISNULL(@ClassCode, '')) + CHAR(13) + CHAR(10)

IF NOT (@DocumentPath = '***NOCHANGE***')
  BEGIN
    SET @FPATH = [dbo].[ky_NETFNGETFILEPATH](@DocumentPath, @FPATH)
  END

IF NOT (@FPATH = 'NOCHANGE')
  BEGIN

    --2017-08-15 b) what???
    --SELECT * FROM diary WHERE ACTIONID = @ActionID AND ACTIONTYPE = 'U'

    IF @ActionType = 'U' --2017-08-15 c)
      BEGIN
        IF NOT (    RTRIM(@Document)  = RTRIM(@OLDDocument)
                AND RTRIM(@ClassCode) = RTRIM(@OLDClass)
                AND RTRIM(@OLDDate)   = CONVERT(VARCHAR(11), CONVERT(DATETIME, @DocumentDate), 106))
          BEGIN
            INSERT
              INTO [dbo].[UndertakingLog]
                  ([UndertakingID],
                   [FieldChanges],
                   [FromValue],
                   [ToValue],
                   [ByWho],
                   [ModifiedDate])
            VALUES(@ActionID,
                   'Documents Edited',
                   @Description,
                   @DescriptionTo,
                   @UserName,
                   GETDATE())
          END
      END

    SELECT @LEN = CASE WHEN [SC].[system_type_id] = 35
                            THEN 2000
                       ELSE [SC].[max_length] END
      FROM SYS.[objects] [SO]
           INNER JOIN SYS.[columns] [SC]
                   ON [SC].[object_id] = [SO].[object_id]
                  AND [SC].[name] = 'Document'
     WHERE [SO].[name] = 'DiaryAttachments'

     UPDATE [DAT]
        SET [DAT].[DATEENTERED] = @DocumentDate,
            [DAT].[DOCUMENT]    = CASE WHEN @Document = '***NOCHANGE***' 
                                            THEN [DAT].[DOCUMENT]
                                       ELSE SUBSTRING(@Document, 1, @LEN) END,
            [DAT].[FILEPATH]    = CASE WHEN @DocumentPath  = '***NOCHANGE***'
                                            THEN [DAT].[FILEPATH]
                                       ELSE @FPATH END,
            [DAT].[DOCCLASS] = @ClassCode
       FROM [dbo].[DiaryAttachments] [DAT]
      WHERE [DAT].[TrackReference] = CONVERT(INT, @TrackReference)

     IF (@Subject <> '***NOCHANGE***')
       BEGIN
         SELECT @LEN = CASE WHEN SC.[system_type_id] = 35 
                                 THEN 2000
                            ELSE SC.[max_length] END
           FROM SYS.[objects] [SO]
                INNER JOIN SYS.[columns] [SC]
                        ON [SC].[object_id] = [SO].[object_id]
                       AND [SC].[name] = 'SUBJECT'
          WHERE [SO].[name] = 'diary'

         UPDATE [DIA]
            SET [DIA].[SUBJECT] = SUBSTRING(@Subject, 1, @LEN)
           FROM [dbo].[diary] [DIA]
          WHERE [DIA].[ACTIONID] = @ActionID
       END

     SET @Result = SUBSTRING(@FPATH,LEN(@FPATH)-CHARINDEX ('\',REVERSE(@FPATH))+2,LEN(@FPATH))
  END
ELSE
  BEGIN
    SET @Result = 'NOCHANGE'
  END
END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPGetAllFileSharesForTrackReference'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPGetAllFileSharesForTrackReference]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETSPGetAllFileSharesForTrackReference]
      (@TrackReference       INT)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETSPGetAllFileSharesForTrackReference]                                                         *
 *                                                                                                           *
 * Gets all File Sharing information for a document                                                          *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.5.1.2                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2017-08-16   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

SELECT [FSD].[FileSharingDocumentID]                                   AS [FileSharingDocumentID],
       [FSM].[FileSharingDocFoldersID]                                 AS [FileSharingDocFoldersID],
       [FSM].[FileSharingServersID]                                    AS [FileSharingServersID],
       CASE WHEN UNC.[UNC] IS NULL
                 THEN RTRIM(ISNULL([DAT].[FilePath], ''))
            ELSE   UNC.[UNC]
                 + SUBSTRING(RTRIM(ISNULL([DAT].[FilePath],
                                          '')),
                             3,
                             LEN(RTRIM(ISNULL([DAT].[FilePath],
                                              ''))) - 2) END           AS [FilePath]
  FROM [dbo].[DiaryAttachments] [DAT]
       INNER JOIN [dbo].[diary] [DIA]
               ON [DIA].[ACTIONID] = [DAT].[DiaryID]
       INNER JOIN [dbo].[FileSharingDocuments] [FSD]
                  INNER JOIN [dbo].[FileSharingMapping] [FSM]
                          ON [FSM].[id] = [FSD].[FileSharingMappingID]
               ON [FSD].[TrackReference] = [DAT].[TrackReference]
       LEFT OUTER JOIN [dbo].[UNCAlias] UNC
                    ON SUBSTRING([DAT].[FilePath],
                                 2,
                                 1) = ':'
                   AND UNC.[Drive] = SUBSTRING([DAT].[FilePath],
                                               1,
                                               1)
 WHERE [DAT].[TrackReference] = @TrackReference

END

GO

SET IDENTITY_INSERT [dbo].[KHCtlMaster] ON

DECLARE @id int
DECLARE @FormName VARCHAR(100)
DECLARE @ControlName VARCHAR(100)
DECLARE @FriendlyName VARCHAR(100)
DECLARE @IsConfigurable bit
DECLARE @DisableMode int
DECLARE @Deleted bit

SET @ID = 7752
SET @FormName = 'PrintLedgerPage'
SET @ControlName = 'bWriteOff'
SET @FriendlyName = 'Write Off Records'
SET @IsConfigurable = 1
SET @DisableMode = 2
SET @Deleted = 0

IF EXISTS
  (SELECT TOP 1 1
     FROM [dbo].[KHCtlMaster] CTM
    WHERE CTM.[id] = @id)
  BEGIN
    UPDATE CTM
       SET CTM.[FormName] = @FormName,
           CTM.[ControlName] = @ControlName,
           CTM.[FriendlyName] = @FriendlyName,
           CTM.[IsConfigurable] = @IsConfigurable,
           CTM.[DisableMode] = @DisableMode,
           CTM.[Deleted] = @Deleted
      FROM [dbo].[KHCtlMaster] CTM
     WHERE CTM.[id] = @id
  END
ELSE
  BEGIN
    INSERT
      INTO [dbo].[KHCtlMaster]
          ([id],
           [FormName],
           [ControlName],
           [FriendlyName],
           [IsConfigurable],
           [DisableMode],
           [Deleted])
    VALUES(@id,
           @FormName,
           @ControlName,
           @FriendlyName,
           @IsConfigurable,
           @DisableMode,
           @Deleted)
  END

SET @ID = 7451
SET @FormName = 'PrintLedgerPage'
SET @ControlName = 'bDelete'
SET @FriendlyName = 'Delete Records'
SET @IsConfigurable = 1
SET @DisableMode = 2
SET @Deleted = 0

IF EXISTS
  (SELECT TOP 1 1
     FROM [dbo].[KHCtlMaster] CTM
    WHERE CTM.[id] = @id)
  BEGIN
    UPDATE CTM
       SET CTM.[FormName] = @FormName,
           CTM.[ControlName] = @ControlName,
           CTM.[FriendlyName] = @FriendlyName,
           CTM.[IsConfigurable] = @IsConfigurable,
           CTM.[DisableMode] = @DisableMode,
           CTM.[Deleted] = @Deleted
      FROM [dbo].[KHCtlMaster] CTM
     WHERE CTM.[id] = @id
  END
ELSE
  BEGIN
    INSERT
      INTO [dbo].[KHCtlMaster]
          ([id],
           [FormName],
           [ControlName],
           [FriendlyName],
           [IsConfigurable],
           [DisableMode],
           [Deleted])
    VALUES(@id,
           @FormName,
           @ControlName,
           @FriendlyName,
           @IsConfigurable,
           @DisableMode,
           @Deleted)
  END

SET IDENTITY_INSERT [dbo].[KHCtlMaster] OFF

DECLARE @maxVal INT
SELECT @maxVal = ISNULL(max([id]),0)+1 from [dbo].[KHCtlMaster]
DBCC CHECKIDENT('KHCtlMaster', RESEED, @maxVal) WITH NO_INFOMSGS
GO

DECLARE @id INT
DECLARE @GroupName VARCHAR(100)
DECLARE @Deleted bit

SET @id = 1000035
SET @GroupName = 'Document Reproduction Maintenance'
SET @Deleted = 0

SET @GroupName = RTRIM(ISNULL(@GroupName, ''))

IF NOT EXISTS
  (SELECT *
     FROM [dbo].[KHCtlGroups] KTG
    WHERE KTG.[id] = @id)
  BEGIN
    INSERT
      INTO [dbo].[KHCtlGroups]
          ([id],
           [GroupName],
           [KeyhouseLock],
           [Deleted])
    VALUES(@id,
           RTRIM(ISNULL(@GroupName, '')),
           1,
           @Deleted)
  END       
ELSE
  BEGIN
    UPDATE CTG
       SET CTG.[GroupName] = RTRIM(ISNULL(@GroupName, '')),
           CTG.[Deleted] = @Deleted
      FROM [dbo].[KHCtlGroups] CTG
     WHERE CTG.[id] = @id
  END


IF NOT EXISTS
  (SELECT *
     FROM [dbo].[KHCtlGroupControls] CTC
    WHERE CTC.[groupid] = 1000035
      AND CTC.[masterid] = 7752)
  BEGIN
    INSERT
      INTO [dbo].[KHCtlGroupControls]
          ([groupid],
           [masterid])
    VALUES(1000035,
           7752)
  END

GO 

DECLARE @id INT
DECLARE @GroupName VARCHAR(100)
DECLARE @Deleted bit

SET @id = 1000035
SET @GroupName = 'Document Reproduction Maintenance'
SET @Deleted = 0

SET @GroupName = RTRIM(ISNULL(@GroupName, ''))

IF NOT EXISTS
  (SELECT *
     FROM [dbo].[KHCtlGroups] KTG
    WHERE KTG.[id] = @id)
  BEGIN
    INSERT
      INTO [dbo].[KHCtlGroups]
          ([id],
           [GroupName],
           [KeyhouseLock],
           [Deleted])
    VALUES(@id,
           RTRIM(ISNULL(@GroupName, '')),
           1,
           @Deleted)
  END       
ELSE
  BEGIN
    UPDATE CTG
       SET CTG.[GroupName] = RTRIM(ISNULL(@GroupName, '')),
           CTG.[Deleted] = @Deleted
      FROM [dbo].[KHCtlGroups] CTG
     WHERE CTG.[id] = @id
  END


IF NOT EXISTS
  (SELECT *
     FROM [dbo].[KHCtlGroupControls] CTC
    WHERE CTC.[groupid] = 1000035
      AND CTC.[masterid] = 7451)
  BEGIN
    INSERT
      INTO [dbo].[KHCtlGroupControls]
          ([groupid],
           [masterid])
    VALUES(1000035,
           7451)
  END

GO 

IF OBJECT_ID('[dbo].[ky_NETRetireHandler]') IS NOT NULL
	DROP PROCEDURE [dbo].[ky_NETRetireHandler]
GO

CREATE PROCEDURE [dbo].[ky_NETRetireHandler] (
       @CODE VARCHAR(20) = '')
AS
/*******************************************************************************************************************************
 *
 * [dbo].[ky_NETRetireHandler]
 *
 * Retires the specified handler
 *
 * Modification History
 * 2017-08-17   John Ginnane       Created
 *
 *******************************************************************************************************************************/
BEGIN
	SET NOCOUNT ON
	
	UPDATE HAN
	   SET HAN.[Retired] = 'Y'
	  FROM [dbo].[Handlers]
	 WHERE HAN.[Code] = LTRIM(RTRIM(ISNULL(@CODE, '')))
	
	SET NOCOUNT OFF
END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM [dbo].[TimeEntry] [TIE]
    WHERE [TIE].[Charge] <> [TIE].[Charged])
  BEGIN
    DECLARE @COMMAND VARCHAR(MAX)

    SET @COMMAND = '
UPDATE [TIE]
   SET [TIE].[Charge] = ROUND([TIE].[Time] * [TIE].[Rate] / 60, 2)
  FROM [dbo].[TimeEntry] [TIE]
 WHERE [TIE].[Charge] <> [TIE].[Charged]
   AND (   [TIE].[Charge] = ROUND([TIE].[Time] * [TIE].[Rate] / 60, 2)
        OR [TIE].[Charged] = ROUND([TIE].[Time] * [TIE].[Rate] / 60, 2))
   AND [TIE].[Rate] <> 0'

   EXEC (@COMMAND)

   SET @COMMAND = '
UPDATE [TIE]
   SET [TIE].[Charged] = ROUND([TIE].[Time] * [TIE].[Rate] / 60, 2)
  FROM [dbo].[TimeEntry] [TIE]
 WHERE [TIE].[Charge] <> [TIE].[Charged]
   AND (   [TIE].[Charge] = ROUND([TIE].[Time] * [TIE].[Rate] / 60, 2)
        OR [TIE].[Charged] = ROUND([TIE].[Time] * [TIE].[Rate] / 60, 2))
   AND [TIE].[Rate] <> 0'

   EXEC (@COMMAND)

   SET @COMMAND = '
UPDATE [TIE]
   SET [TIE].[Charged] = 0
  FROM [dbo].[TimeEntry] [TIE]
 WHERE [TIE].[Rate] = 0
   AND [TIE].[Charge] = 0
   AND [TIE].[Charged] <> 0'

   EXEC (@COMMAND)

   SET @COMMAND = '
UPDATE [TIE]
   SET [TIE].[Charge] = [TIE].[Charged]
  FROM [dbo].[TimeEntry] [TIE]
 WHERE [TIE].[Charge] <> [TIE].[Charged]
   AND ABS(ROUND([TIE].[Time] * [TIE].[Rate] / 60, 2) - [TIE].[Charged]) < 1
   AND [TIE].[Charged] <> 0
   AND [TIE].[Rate] NOT IN (0, 1)'

   EXEC (@COMMAND)

   SET @COMMAND = '
UPDATE [TIE]
   SET [TIE].[Charged] = [TIE].[Charge]
  FROM [dbo].[TimeEntry] [TIE]
 WHERE [TIE].[Charge] < 0
   AND [TIE].[Charged] = - [TIE].[Charge]
   AND [TIE].[TimeOrCharge] = ''B''
   AND [TIE].[Task] = ''BILLED'''

   EXEC (@COMMAND)

   SET @COMMAND = '
UPDATE [TIE]
   SET [TIE].[Charged] = [TIE].[Charge]
  FROM [dbo].[TimeEntry] [TIE]
 WHERE [TIE].[Charge] <> [TIE].[Charged]
   AND [TIE].[Charge] < 0'

   EXEC (@COMMAND)

   SET @COMMAND = '
UPDATE [TIE]
   SET [TIE].[Charge] = [TIE].[Charged]
  FROM [dbo].[TimeEntry] [TIE]
 WHERE [TIE].[Charge] <> [TIE].[Charged]
   AND [TIE].[Charged] < 0'

   EXEC (@COMMAND)

   SET @COMMAND = '
UPDATE [TIE]
   SET [TIE].[Charged] = [TIE].[Charge]
  FROM [dbo].[TimeEntry] [TIE]
 WHERE [TIE].[Charge] <> [TIE].[Charged]
   AND [TIE].[Charge] <> 0
   AND [TIE].[TimeOrCharge] = ''C'''

   EXEC (@COMMAND)

   SET @COMMAND = '
UPDATE [TIE]
   SET [TIE].[Charge] = [TIE].[Charged]
  FROM [dbo].[TimeEntry] [TIE]
 WHERE [TIE].[Charge] <> [TIE].[Charged]
   AND [TIE].[Charged] <> 0
   AND [TIE].[TimeOrCharge] = ''C'''

   EXEC (@COMMAND)
  END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM [dbo].[TimeEntry] [TIE]
    WHERE [TIE].[Charge] <> [TIE].[Charged])
  BEGIN
    SELECT 'Found Inconsistent Time Entries. Please review the list below and address these manually if necessary.
    
Found Inconsistent Time Entries - START OF LIST' AS [Please Note]

  SELECT [TIE].[Time],
         [TIE].[Rate],
         [TIE].[Charge],
         [TIE].[Charged],
         [TIE].[Units],
         [TIE].[TimeOrCharge],
         [TIE].[Task],
         [TIE].[Date],
         [TIE].[Matter],
         ROUND([TIE].[Time] * [TIE].[Rate] / 60, 2)
    FROM [dbo].[TimeEntry] [TIE]
   WHERE [TIE].[Charge] <> [TIE].[Charged]
ORDER BY [TIE].[DATE] DESC

SELECT '
Found Inconsistent Time Entries - END OF LIST' AS [Please Note]
  END
GO

IF NOT EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETTRTimeEntryInsert'
      AND SO.[type] = 'TR')
  BEGIN
    DECLARE @COMMAND NVARCHAR(MAX)

    SET @COMMAND = 'CREATE TRIGGER
       [dbo].[ky_NETTRTimeEntryInsert]
    ON [dbo].[TimeEntry]
       AFTER INSERT
AS
BEGIN
  IF ((SELECT TRIGGER_NESTLEVEL()) < 2)
    BEGIN
      DECLARE @CHARGE DECIMAL(10, 2)

      UPDATE TIE
         SET TIE.[Charge] = CASE WHEN ISNULL(TIE.[Charge], 0) <> 0
                                      THEN ISNULL(TIE.[Charge], 0)
                                 ELSE ISNULL(TIE.[Charged], 0) END,
             TIE.[Charged] = CASE WHEN ISNULL(TIE.[Charge], 0) <> 0
                                      THEN ISNULL(TIE.[Charge], 0)
                                 ELSE ISNULL(TIE.[Charged], 0) END,
             TIE.[BillingDate] = CASE WHEN TIE.[BillingDate] IS NULL
                                           THEN TIE.[BilledDate]
                                      WHEN TIE.[BilledDate] IS NULL
                                           THEN TIE.[BillingDate]
                                      ELSE TIE.[BillingDate] END,
             TIE.[BilledDate] = CASE WHEN TIE.[BillingDate] IS NULL
                                           THEN TIE.[BilledDate]
                                      WHEN TIE.[BilledDate] IS NULL
                                           THEN TIE.[BillingDate]
                                      ELSE TIE.[BillingDate] END
        FROM inserted INS
             INNER JOIN [dbo].[TimeEntry] TIE
                     ON TIE.[RecordID] = INS.[RecordID]
       WHERE ISNULL(INS.[Charge], 0) <> ISNULL(INS.[Charged], 0)
          OR ISNULL(INS.[BillingDate], CONVERT(DATETIME, ''19000101'')) <> ISNULL(INS.[BilledDate], CONVERT(DATETIME, ''19000101''))

    END
END'

    EXEC (@COMMAND)
  END
GO

IF NOT EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETTRTimeEntryUpdate'
      AND SO.[type] = 'TR')
  BEGIN
    DECLARE @COMMAND NVARCHAR(MAX)

    SET @COMMAND = 'CREATE TRIGGER
       [dbo].[ky_NETTRTimeEntryUpdate]
    ON [dbo].[TimeEntry]
       AFTER Update
AS
BEGIN
  IF ((SELECT TRIGGER_NESTLEVEL()) < 2)
    BEGIN
      UPDATE TIE
         SET TIE.[Charge] = CASE WHEN ISNULL(INS.[Charged], 0) <> ISNULL(DEL.[Charged], 0) 
                                      THEN ISNULL(INS.[Charged], 0)
                                 ELSE ISNULL(INS.[Charge], 0) END,
             TIE.[Charged] = CASE WHEN ISNULL(INS.[Charge], 0) <> ISNULL(DEL.[Charge], 0) 
                                      THEN ISNULL(INS.[Charge], 0)
                                  ELSE ISNULL(INS.[Charged], 0) END,
             TIE.[BillingDate] = CASE WHEN ISNULL(INS.[BilledDate], CONVERT(DATETIME, ''19000101'')) <> ISNULL(DEL.[BilledDate], CONVERT(DATETIME, ''19000101'')) 
                                      THEN INS.[BilledDate]
                                 ELSE INS.[BillingDate] END,
             TIE.[BilledDate] = CASE WHEN ISNULL(INS.[BillingDate], CONVERT(DATETIME, ''19000101'')) <> ISNULL(DEL.[BillingDate], CONVERT(DATETIME, ''19000101'')) 
                                      THEN INS.[BillingDate]
                                ELSE INS.[BilledDate] END
        FROM inserted INS
             INNER JOIN deleted DEL
                     ON DEL.[RecordID] = INS.[RecordID]
             INNER JOIN [dbo].[TimeEntry] TIE
                     ON TIE.[RecordID] = INS.[RecordID]
       WHERE ISNULL(INS.[Charge], 0) <> ISNULL(DEL.[Charge], 0)
          OR ISNULL(INS.[Charged], 0) <> ISNULL(DEL.[Charged], 0)
          OR ISNULL(INS.[BillingDate], CONVERT(DATETIME, ''19000101'')) <> ISNULL(DEL.[BillingDate], CONVERT(DATETIME, ''19000101''))
          OR ISNULL(INS.[BilledDate], CONVERT(DATETIME, ''19000101'')) <> ISNULL(DEL.[BilledDate], CONVERT(DATETIME, ''19000101''))
    END
END'

    EXEC (@COMMAND)
  END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSetRevenueSettings'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSetRevenueSettings]
  END
GO

CREATE PROCEDURE [dbo].[ky_NETSetRevenueSettings] (  
       @KeyName    VARCHAR(50),  
       @KeyValue   VARCHAR(100))  
AS
/****************************************************************************************
 *
 * [dbo].[ky_NETSetRevenueSettings]
 *
 * Description: This SP is used to set a specific Revenue setting
 *
 * Modification History
 * 2017-09-13   John            Created
 *
 ****************************************************************************************/
BEGIN  
	SET NOCOUNT ON  
	
	IF EXISTS (SELECT TOP 1 1 FROM [dbo].[RevenueSettings] WHERE [KeyName] = @KeyName)
	BEGIN
		UPDATE [dbo].[RevenueSettings]
		   SET [KeyValue] = @KeyValue
		 WHERE [KeyName] = @KeyName
	END
	ELSE
	BEGIN
		INSERT INTO [dbo].[RevenueSettings] (
		            [KeyName],
		            [KeyValue])
		     SELECT @KeyName, 
		            @KeyValue
	END
	
	SET NOCOUNT OFF
END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETGetRevenueSettings'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETGetRevenueSettings]
  END
GO

CREATE PROCEDURE [dbo].[ky_NETGetRevenueSettings] (
       @KEYNAME VARCHAR(50))
AS
/****************************************************************************************
 *
 * Description: this sp is used to fetch a specific Revenue setting
 *
 * Modification History
 * 2017-09-13   John            Created
 *
 ****************************************************************************************/
BEGIN
	SET NOCOUNT ON;

	DECLARE @VALUE VARCHAR(100)
	   
	SELECT @VALUE = [SET].[KeyValue]
	  FROM [dbo].[RevenueSettings] [SET]
	 WHERE [SET].[KeyName] = @keyname

	SELECT ISNULL(@VALUE, '') AS [KeyValue]

	SET NOCOUNT OFF
END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM sys.objects [SO]
          LEFT OUTER JOIN sys.[columns] [SC]
                       ON [SC].[object_id] = [SO].[object_id]
                      AND [SC].[name] = 'CourtFeeCode'
    WHERE [SO].[name] = 'DebtCostingLedger'
      AND [SO].[type] = 'U'
      AND [SC].[column_id] IS NULL)
BEGIN
	ALTER TABLE [dbo].[DebtCostingLedger]
	  ADD [CourtFeeCode] VARCHAR(30) NULL 
END

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETDebtSpPostCosts2'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETDebtSpPostCosts2]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETDebtSpPostCosts2]
      (@date              DATETIME,
       @mattercode        VARCHAR(20),
       @description       VARCHAR(200),
       @costs             DECIMAL(18,2)    =    0.00,
       @entryby           VARCHAR(12),
       @feeearner         VARCHAR(10),
       @rate              DECIMAL(18,2)    =    0.00,
       @chargevalue       DECIMAL(18,2)    =    0.00,
       @timeorcharge      CHAR(1),
       @time              INT,
       @CourtFeeCode      VARCHAR(30))
AS
/***********************************************************************************************
 *
 * Description:    Insert Costs INTO Debtcosting legder
 *
 * Modification History
 * 2014-03-17  Magesh Kumar          Created
 * 2014-09-18  Pino Carafa           Fixed MAX function - will return NULL on an empty table
 * 2015-06-23  Sridharen             KEYD - 2332 - Use of LTRIM(RTRIM(
 * 2016-02-29  John                  KEYD-3377 - Setting XnID to -1
 * 2016-09-23  Pino                  Tidy Up for KEYD-4079
 * 2016-09-23  Pino                  It was linking TimeEntry through TrackRef. This should
 *                                   actually be Docket
 * 2016-09-27  Pino                  Issues writing to TimeEntry
 * 2017-09-14  John                  Created THIS file, adding new argument to be saved - KEYD-4768
 ***********************************************************************************************/
BEGIN

DECLARE @postref INT

SET @postref = (SELECT MAX([DCL].[POSTINGREF]) + 1
                  FROM [dbo].[DebtCostingLedger] [DCL])

SET @postref = ISNULL(@postref, 1)

INSERT
  INTO [dbo].[DebtCostingLedger]
      ([DATE],
       [DESCRIPTION],
       [MATTERCODE],
       [TYPE],
       [COSTS],
       [ENTRYDATE],
       [ENTEREDBY],
       [POSTINGREF],
       [INTEREST],
       [PREMIUMPAID],
       [COSTSPAID],
       [INTERESTPAID],
       [Value],
       [XnId],
       [CourtFeeCode])
VALUES(@date,
       @description,
       @mattercode,
       'Costs',
       @costs,
       GETDATE(),
       @entryby,
       @postref,
       0.00,
       0.00,
       0.00,
       0.00,
       0.00,
       -1,
       @CourtFeeCode)

UPDATE [MAT]
   SET [MAT].[RecoverableCosts] = ISNULL([MAT].[RecoverableCosts], 0) + @costs
  FROM [dbo].[matters] [MAT]
 WHERE [MAT].[Code] = RTRIM(@mattercode)

IF (@timeorcharge='C')
  ---Insert into timeentry
  BEGIN
    SET @date = CONVERT(datetime, CONVERT(VARCHAR, @date, 112))
    SET @rate = @chargevalue

    INSERT
      INTO [dbo].[TimeEntry]
          ([Matter],
           [FeeEarn],
           [Date],
           [Comment],
           [Time],
           [Rate],
           [Charge],
           [TimeOrCharge],
           [Docket],
           [Rec_Irr])
    VALUES(@mattercode,
           @feeearner,
           @date,
           @description,
           @time,
           @rate,
           @chargevalue,
           @timeorcharge,
           @postref,
           'C')
  END
END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPFileSharingGetDocumentInfo'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPFileSharingGetDocumentInfo]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETSPFileSharingGetDocumentInfo]
      (@DocFolderID       INT,
       @TrackReference    INT,
       @Handler           VARCHAR(10))
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETSPFileSharingGetDocumentInfo]                                                                *
 *                                                                                                           *
 * Retrieve File Sharing information for a Document in a Shared Folder                                       *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.5.1.2                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2017-09-15   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

SET NOCOUNT ON

DECLARE @RESULT TABLE
       ([FileSharingDocumentID]                         VARCHAR(500),
        [FileSharingServersID]                          INT,
        [ServerTypeID]                                  INT,
        [ServerType]                                    VARCHAR(20),
        [ServerName]                                    VARCHAR(50),
        [Server]                                        VARCHAR(50),
        [ServerDomain]                                  VARCHAR(100),
        [UserId]                                        VARCHAR(100),
        [Password]                                      VARCHAR(100),
        [SiteID]                                        VARCHAR(100),
        [ClientID]                                      VARCHAR(100),
        [ClientSecret]                                  VARCHAR(500),
        [ProxyType]                                     INT,
        [ProxyServer]                                   VARCHAR(500),
        [ProxyPort]                                     INT)

--Get the best match
INSERT
  INTO @RESULT
      ([FileSharingDocumentID],
       [FileSharingServersID],
       [ServerTypeID],
       [ServerType],
       [ServerName],
       [Server],
       [ServerDomain],
       [UserId],
       [Password],
       [SiteID],
       [ClientID],
       [ClientSecret],
       [ProxyType],
       [ProxyServer],
       [ProxyPort])        
SELECT [FSD].[FileSharingDocumentID]                    AS [FileSharingDocumentID],
       [FSM].[FileSharingServersID]                     AS [FileSharingServersID],
       [FSS].[TypeId]                                   AS [ServerTypeID],
       [FSST].[TypeName]                                AS [ServerType],
       [FSS].[Name]                                     AS [ServerName],
       [FSS].[Server]                                   AS [Server],
       [FSS].[Domain]                                   AS [ServerDomain],
       ISNULL([FSH].[UserId], [FSS].[AdminUserId])      AS [UserId],
       ISNULL([FSH].[Password], [FSS].[AdminPassword])  AS [Password],
       [FSS].[SiteID]                                   AS [SiteID],
       [FSS].[ClientID]                                 AS [ClientID],
       [FSS].[ClientSecret]                             AS [ClientSecret],
       [FSS].[ProxyType]                                AS [ProxyType],
       [FSS].[ProxyServer]                              AS [ProxyServer],
       [FSS].[ProxyPort]                                AS [ProxyPort]
  FROM [dbo].[FileSharingDocuments] [FSD]
       INNER JOIN [dbo].[FileSharingMapping] [FSM]
                  INNER JOIN [dbo].[FileSharingDocFolders] [FSDF]
                          ON [FSDF].[DocFolderID] = [FSM].[FileSharingDocFoldersID]
                         AND [FSDF].[Shared] = 1
                  INNER JOIN [dbo].[FileSharingServers] [FSS]
                             INNER JOIN [dbo].[FileSharingServerTypes] [FSST]
                                     ON [FSST].[id] = [FSS].[TypeId]
                             LEFT OUTER JOIN [dbo].[FileSharingHandlers] [FSH]
                                          ON [FSH].[FileSharingServersID] = [FSS].[id]
                                         AND [FSH].[FileSharingHandlerCode] = @Handler
                          ON [FSS].[id] = [FSM].[FileSharingServersID]
                         AND RTRIM(ISNULL([FSS].[CollabMatter], '')) = ''
               ON [FSM].[id] = [FSD].[FileSharingMappingID]
              AND [FSM].[FileSharingDocFoldersID] = @DocFolderID
 WHERE [FSD].[TrackReference] = @TrackReference

IF EXISTS (SELECT TOP 1 1 FROM @RESULT)
  BEGIN
    SELECT [RES].[FileSharingDocumentID],
           [RES].[FileSharingServersID],
           [RES].[ServerTypeID],
           [RES].[ServerType],
           [RES].[ServerName],
           [RES].[Server],
           [RES].[ServerDomain],
           [RES].[UserId],
           [RES].[Password],
           [RES].[SiteID],
           [RES].[ClientID],
           [RES].[ClientSecret],
           [RES].[ProxyType],
           [RES].[ProxyServer],
           [RES].[ProxyPort]
      FROM @RESULT [RES]

    SET NOCOUNT OFF
    RETURN
  END

--If it wasn't found through Best Match, try and get it from another Shared folder.
INSERT
  INTO @RESULT
      ([FileSharingDocumentID],
       [FileSharingServersID],
       [ServerTypeID],
       [ServerType],
       [ServerName],
       [Server],
       [ServerDomain],
       [UserId],
       [Password],
       [SiteID],
       [ClientID],
       [ClientSecret],
       [ProxyType],
       [ProxyServer],
       [ProxyPort])        
SELECT TOP 1
       [FSD].[FileSharingDocumentID]                    AS [FileSharingDocumentID],
       [FSM].[FileSharingServersID]                     AS [FileSharingServersID],
       [FSS].[TypeId]                                   AS [ServerTypeID],
       [FSST].[TypeName]                                AS [ServerType],
       [FSS].[Name]                                     AS [ServerName],
       [FSS].[Server]                                   AS [Server],
       [FSS].[Domain]                                   AS [ServerDomain],
       ISNULL([FSH].[UserId], [FSS].[AdminUserId])      AS [UserId],
       ISNULL([FSH].[Password], [FSS].[AdminPassword])  AS [Password],
       [FSS].[SiteID]                                   AS [SiteID],
       [FSS].[ClientID]                                 AS [ClientID],
       [FSS].[ClientSecret]                             AS [ClientSecret],
       [FSS].[ProxyType]                                AS [ProxyType],
       [FSS].[ProxyServer]                              AS [ProxyServer],
       [FSS].[ProxyPort]                                AS [ProxyPort]
  FROM [dbo].[FileSharingDocuments] [FSD]
       INNER JOIN [dbo].[FileSharingMapping] [FSM]
                  INNER JOIN [dbo].[FileSharingDocFolders] [FSDF]
                          ON [FSDF].[DocFolderID] = [FSM].[FileSharingDocFoldersID]
                         AND [FSDF].[Shared] = 1
                  INNER JOIN [dbo].[FileSharingServers] [FSS]
                             INNER JOIN [dbo].[FileSharingServerTypes] [FSST]
                                     ON [FSST].[id] = [FSS].[TypeId]
                             LEFT OUTER JOIN [dbo].[FileSharingHandlers] [FSH]
                                          ON [FSH].[FileSharingServersID] = [FSS].[id]
                                         AND [FSH].[FileSharingHandlerCode] = @Handler
                          ON [FSS].[id] = [FSM].[FileSharingServersID]
                         AND RTRIM(ISNULL([FSS].[CollabMatter], '')) = ''
               ON [FSM].[id] = [FSD].[FileSharingMappingID]
 WHERE [FSD].[TrackReference] = @TrackReference

IF EXISTS (SELECT TOP 1 1 FROM @RESULT)
  BEGIN
    SELECT [RES].[FileSharingDocumentID],
           [RES].[FileSharingServersID],
           [RES].[ServerTypeID],
           [RES].[ServerType],
           [RES].[ServerName],
           [RES].[Server],
           [RES].[ServerDomain],
           [RES].[UserId],
           [RES].[Password],
           [RES].[SiteID],
           [RES].[ClientID],
           [RES].[ClientSecret],
           [RES].[ProxyType],
           [RES].[ProxyServer],
           [RES].[ProxyPort]
      FROM @RESULT [RES]

    SET NOCOUNT OFF
    RETURN
  END

--Just get Server info. Let the software take care of creating a share.
INSERT
  INTO @RESULT
      ([FileSharingDocumentID],
       [FileSharingServersID],
       [ServerTypeID],
       [ServerType],
       [ServerName],
       [Server],
       [ServerDomain],
       [UserId],
       [Password],
       [SiteID],
       [ClientID],
       [ClientSecret],
       [ProxyType],
       [ProxyServer],
       [ProxyPort])        
SELECT TOP 1
       ''                                               AS [FileSharingDocumentID],
       [FSS].[id]                                       AS [FileSharingServersID],
       [FSS].[TypeId]                                   AS [ServerTypeID],
       [FSST].[TypeName]                                AS [ServerType],
       [FSS].[Name]                                     AS [ServerName],
       [FSS].[Server]                                   AS [Server],
       [FSS].[Domain]                                   AS [ServerDomain],
       ISNULL([FSH].[UserId], [FSS].[AdminUserId])      AS [UserId],
       ISNULL([FSH].[Password], [FSS].[AdminPassword])  AS [Password],
       [FSS].[SiteID]                                   AS [SiteID],
       [FSS].[ClientID]                                 AS [ClientID],
       [FSS].[ClientSecret]                             AS [ClientSecret],
       [FSS].[ProxyType]                                AS [ProxyType],
       [FSS].[ProxyServer]                              AS [ProxyServer],
       [FSS].[ProxyPort]                                AS [ProxyPort]
  FROM [dbo].[FileSharingServers] [FSS]
       INNER JOIN [dbo].[FileSharingServerTypes] [FSST]
               ON [FSST].[id] = [FSS].[TypeId]
       LEFT OUTER JOIN [dbo].[FileSharingHandlers] [FSH]
                    ON [FSH].[FileSharingServersID] = [FSS].[id]
                   AND [FSH].[FileSharingHandlerCode] = @Handler
 WHERE [FSS].[CollabDocFolderID] IS NULL
   AND RTRIM(ISNULL([FSS].[CollabMatter], '')) = ''

IF EXISTS (SELECT TOP 1 1 FROM @RESULT)
  BEGIN
    SELECT [RES].[FileSharingDocumentID],
           [RES].[FileSharingServersID],
           [RES].[ServerTypeID],
           [RES].[ServerType],
           [RES].[ServerName],
           [RES].[Server],
           [RES].[ServerDomain],
           [RES].[UserId],
           [RES].[Password],
           [RES].[SiteID],
           [RES].[ClientID],
           [RES].[ClientSecret],
           [RES].[ProxyType],
           [RES].[ProxyServer],
           [RES].[ProxyPort]
      FROM @RESULT [RES]

    SET NOCOUNT OFF
    RETURN
  END

SET NOCOUNT ON

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSYSCONUpdateConfigurationSettings3'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSYSCONUpdateConfigurationSettings3]
  END
GO

CREATE PROCEDURE [ky_NETSYSCONUpdateConfigurationSettings3]
(
 @PrecedentFolder				VARCHAR(300),
 @StorageMethod					VARCHAR(1),
 @WpFolder						VARCHAR(300),
 @EmailOn						CHAR(1),
 @AllowMultipleDraftBills		CHAR(1),
 @LicenseName					VARCHAR(300),
 @VatNo							VARCHAR(20),
 @Debtors						VARCHAR(20),
 @Creditor						VARCHAR(20),
 @Outlay						VARCHAR(20),
 @VAT							VARCHAR(20),
 @PL							VARCHAR(20),
 @Client						VARCHAR(20),
 @FeesWriteOff					VARCHAR(20),
 @OutlayWriteOff				VARCHAR(20),
 @BillsWriteOff					VARCHAR(20),
 @DefaultOfficeAc				VARCHAR(20),
 @DefaultClientAc				VARCHAR(20),
 @DefaultClientDEP				VARCHAR(20),
 @MiscOutlay					VARCHAR(20),
 @PettyCash						VARCHAR(20),
 @WithTax						VARCHAR(20),
 @Year							VARCHAR(20),
 @CurPeriod						VARCHAR(20),
 @OutlayBudget					VARCHAR(20),
 @LastBill						VARCHAR(20),
 @VatExclude					VARCHAR(20),
 @DefaultVAT					VARCHAR(20),
 @FeeExclude					VARCHAR(20),
 @MiscFe						VARCHAR(20),
 @EnableFNL						VARCHAR(20),
 @LaunchStartUp					VARCHAR(1),
 @Ref1							VARCHAR(20),
 @Ref2							VARCHAR(20),
 @Ref3							VARCHAR(20),
 @Solicitor						VARCHAR(30),
 @OtherParty					VARCHAR(30),
 @SupplierCategory				VARCHAR(30),
 @DefaultCategory				VARCHAR(20),
 @AutoNumber					VARCHAR(1),
 @AutoAlphaClientNumbering		VARCHAR(1),
 @ThirdPartyLimit				DECIMAL(7,2),
 @DateFutValidation				VARCHAR(20),
 @DatePastValidation			VARCHAR(20),
 @NextFile						VARCHAR(20),
 @ClosedFileNo					VARCHAR(20),
 @ChequePrinting				VARCHAR(20),
 @EnableCP						VARCHAR(1),
 @EIConvert						DECIMAL(9,6),
 @DocumentNamingMethod			VARCHAR(20),
 @DefaultVATCode				VARCHAR(3),
 @WIPWarningLimit				DECIMAL(18,2),
 @DebtorWarningDaysLimit		INT,
 @LastActionMonth				INT,
 @LockupValue					DECIMAL(18,2),
 @ShowAlarm						BIT,
 @EstimateFeeRequired			BIT,
 @ExpectedinvoiceDateRequired	BIT,
 @TimeDayBookLimit				VARCHAR(50),
 @OSInvoiceLimit				VARCHAR(50),
 @CurrentWIPLimit				VARCHAR(50),
 @DraftInvoiceLimit				VARCHAR(50),
 @EnableCD						VARCHAR(1),
 @EnableDW						VARCHAR(1),
 @EnableFM						VARCHAR(1),
 @RECINSTALLED					VARCHAR(1),
 @NOBAR							VARCHAR(1),
 @WORDPERFECT					VARCHAR(1),
 @OPSISLINK						VARCHAR(1),
 @TIMECOSTING					VARCHAR(1),
 @FORCEFE						VARCHAR(1),
 @UK							VARCHAR(1),
 @USEOUTLAYTYPE					VARCHAR(1),
 @PENDINGOUTLAY					VARCHAR(1),
 @INDICATOR18					VARCHAR(1),
 @INDICATOR19					VARCHAR(1),
 @INDICATOR20					VARCHAR(1),
 @DEMOEXPIRE					INT,
 @MAXNUMBEROFMATTER				INT,
 @LABLEPRINTER					VARCHAR(1),
 @YEARENDCOMPLETED				VARCHAR(1),
 @INDICATOR22					VARCHAR(1),
 @INDICATOR23					VARCHAR(1),
 @INDICATOR24					VARCHAR(1),
 @INDICATOR25					VARCHAR(1),
 @INDICATOR26					VARCHAR(1),
 @INDICATOR27					VARCHAR(1),
 @INDICATOR28					VARCHAR(1),
 @INDICATOR29					VARCHAR(1),
 @INDICATOR30					VARCHAR(1),
 @FEESREPSEQ					VARCHAR(20),
 @SPLITCHEQUES					VARCHAR(1),
 @INDICATOR3					VARCHAR(1),
 @INDICATOR4					VARCHAR(1),
 @INDICATOR6					VARCHAR(1),
 @INDICATOR7					VARCHAR(1),
 @INDICATOR8					VARCHAR(1),
 @INDICATOR9					VARCHAR(1),
 @INDICATOR10					VARCHAR(1),
 @ENQUIRY						VARCHAR(1),
 @DOTimeCosting					VARCHAR(1),
 @Billing						VARCHAR(1),
 @IsChequeReq					VARCHAR(1),
 @IsClientReq					VARCHAR(1),
 @CaseMan						VARCHAR(1),
 @MaxClientCode					VARCHAR(2),
 @MaxMatterCode					VARCHAR(2),
 @YourRef						VARCHAR(20),
 @DeleteAction					VARCHAR(1),
 @FileNoLabel					VARCHAR(20),
 @DeleteAttachment				VARCHAR(1),
 @SDLTpath						VARCHAR(500),
 @BranchRequired				BIT,
 @DeleteOtherAction				VARCHAR(1),
 @ReadyonlyAttachment			BIT,
 @RevenueHoldDate				VARCHAR(20),
 @ClientGroup					VARCHAR(20),
 @TeamLeader                    VARCHAR(10),
 @CompositeBilling              VARCHAR(10),
 @EnableDictation               VARCHAR(10),
 @ExportReportsToLegacyFormats  VARCHAR(10),
 @DocumentSyncVisible           VARCHAR(10),
 @EnablePaymentReceipt			BIT
)
AS
/*
ky_NETSYSCONUpdateConfigurationSettings

This stored procedrue is used to update configuration settings

 *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software
 *                                    remains fully functional
 *   ******************************************************************************************
 *   *
 *   * Supersedes:                              [dbo].[ky_NETSYSCONUpdateConfigurationSettings]
 *   * First compatible version:                5.5.1.1 (5.4.6.10)
 *   * Last compatible software version:        -
 *   * Superseded by:                           -
 *   *
 *   ******************************************************************************************

Created: 05June2014 07.58 PM (IST)

Modification History
-----------------------------------------------------
Date  | Modified By   | Description
-----------------------------------------------------
14Nov2014  Arun.v     Keyhouse Option tab added
20Nov2014  Arun.V     Third Party Limit and @EIConvert is parameter type changed from VARCHAR to decimal
21Nov2014  Arun.V     MaxClientCode, MaxMatterCode added
23Dec2014  Arun.v     Added Your Ref field
02FEB2015  Arun.v     Added ExpectedinvoiceDateRequired field
27APR2015  Magesh     Added a parameter named deleteAction to add the user delete permission to settings table.
04JUN2015  Magesh     Added a parameter named DeleteAttachment to add the user Attachment delete permission to settings table.
10JUL2015  Magesh     Added a parameter to add SDLT global path.
11AUG2015  Sridharan  RDJ-135 - New Matter - Branch
11SEP2015  Pino       Added NULL checks for numeric fields that are passed in using VARCHAR parameters
15SEP2015  Sridharan  KEYD-2595 - Option to Take and Delete tasks in one click
20NOV2015  Magesh	  KEYD-2891-Once an action is complete ALL documents for that action are READ ONLY	
04JAN2016  Magesh	  Added query to add revenue holdays and client group value to settings table
11AUG2016  Sridharan  Composite Billing Indicator
06OCT2016  Suriya     KEYD-4088: Added a new parameter for Enabling Dictation
18OCT2016  Sridharan  KEYD-4103: Export Report output to DOCX or XSLX
02JUN2017  Suriya     KEYD-4630: System option to Switch ON/OFF the "Document Sync" functionality
22SEP2017  Arun V	  KEYD-4804: Added Payment Receipt option in setting
*/
BEGIN
SET NOCOUNT ON

 UPDATE
  systemConfig
 SET
  ScprecedentFolder=@PrecedentFolder, Storagemethod=@StorageMethod, ScWpFolder=@WpFolder, EmailOn  =@EmailOn

 --Allow Multiple Draft Bills
 UPDATE
  DesktopControl
 SET
    Indicator5	 = @AllowMultipleDraftBills
  , REQUESTER	 = @DefaultVATCode
  , SPLITCHEQUES = @SPLITCHEQUES
  , INDICATOR3	 = @INDICATOR3
  , INDICATOR4	 = @INDICATOR4
  , INDICATOR6	 = @INDICATOR6
  , INDICATOR7	 = @INDICATOR7
  , INDICATOR8	 = @INDICATOR8
  , INDICATOR9	 = @INDICATOR9
  , INDICATOR10	 = @INDICATOR10
  , ENQUIRY		 = @ENQUIRY
  , TIMECOSTING  = @DOTimeCosting
  , BILLING		 = @Billing
  , CHEQUEREQ	 = @IsChequeReq
  , CLIENTREQ	 = @IsClientReq
  , CASEMAN		 = @CaseMan

 UPDATE [CONTROL] SET
    Name			= @LicenseName
  , Vatno			= @VatNo
  , Debtors			= @Debtors
  , Creditor		= @Creditor
  , Outlay			= @Outlay
  , VAT				= @VAT
  , PL				= @PL
  , Client			= @Client
  , FeesWriteOff    = @FeesWriteOff
  , OutlayWriteOff  = @OutlayWriteOff
  , BillsWriteOff   = @BillsWriteOff
  , DefaultOfficeAc = @DefaultOfficeAc
  , DefaultClientAc = @DefaultClientAc
  , DefaultClientDEP = @DefaultClientDEP
  , MiscOutlay    = @MiscOutlay
  , PettyCash     = @PettyCash
  , WithTax		  = @WithTax
  , [YEAR]        = CASE WHEN ISNULL(@Year, '') = '' THEN YEAR(GETDATE()) ELSE CONVERT(SMALLINT, CONVERT(FLOAT, @YEAR)) END
  , [CurPer]      = CASE WHEN ISNULL(@CurPeriod, '') = '' THEN MONTH(GETDATE()) ELSE CONVERT(SMALLINT, CONVERT(FLOAT, @CurPeriod)) END
  , Outlaybudget  = CASE WHEN ISNULL(@OutlayBudget, '') = '' THEN 0 ELSE CONVERT(DECIMAL(7, 2), @OutlayBudget) END
  , LastBill      = CASE WHEN ISNULL(@LastBill, '') = '' THEN 0 ELSE CONVERT(INT, CONVERT(FLOAT, @LastBill)) END
  , VatExclude    = @VatExclude
  , DefaultVAT    = @DefaultVAT
  , FeeExclude    = @FeeExclude
  , MiscFe        = @MiscFe
  , EnableFNL     = @EnableFNL
  , AlarmStartupLaunch  = @LaunchStartUp
  , UserPrompt1   = @Ref1
  , userPrompt2   = @Ref2
  , UserPrompt3   = @Ref3
  , Solicitor     = @Solicitor
  , Othparty      = @OtherParty
  , SupplierCategory  = @SupplierCategory
  , DefaultCategory   = @DefaultCategory
  , AutoNumber		  = @AutoNumber
  , AutoAlphaClientNumbering = @AutoAlphaClientNumbering
  , ThirdPartyLimit   = @ThirdPartyLimit
  , DateFutValidation   = CASE WHEN ISNULL(@DateFutValidation, '') = '' THEN 0 ELSE CONVERT(SMALLINT, CONVERT(FLOAT, @DateFutValidation)) END
  , DatePastValidation  = CASE WHEN ISNULL(@DatePastValidation, '') = '' THEN 0 ELSE CONVERT(SMALLINT, CONVERT(FLOAT, @DatePastValidation)) END
  , NextFile     = CASE WHEN ISNULL(@NextFile, '') = '' THEN 0 ELSE CONVERT(INT, CONVERT(FLOAT, @NextFile)) END
  , ClosedFileNo    = CASE WHEN ISNULL(@ClosedFileNo, '') = '' THEN 0 ELSE CONVERT(INT, CONVERT(FLOAT, @ClosedFileNo)) END
  , ChequePrinting   = @ChequePrinting
  , EnableCP     = @EnableCP
  , EIConvert     = @EIConvert
  , DocumentNamingMethod  = @DocumentNamingMethod
  , ENABLECDB     = @EnableCD
  , EnableDW      = @EnableDW
  , ENABLEFM      = @EnableFM
  , RECINSTALLED     = @RECINSTALLED
  , NOBAR      = @NOBAR
  , WORDPERFECT    = @WORDPERFECT
  , OPSISLINK     =  @OPSISLINK
  , TIMECOSTING    =  @TIMECOSTING
  , FORCEFE     =  @FORCEFE
  , UK      =  @UK
  , USEOUTLAYTYPE    = @USEOUTLAYTYPE
  , PENDINGOUTLAY    =  @PENDINGOUTLAY
  , INDICATOR18    =  @INDICATOR18
  , INDICATOR19    =  @INDICATOR19
  , INDICATOR20      = @INDICATOR20
  , DEMOEXPIRE     = @DEMOEXPIRE
  , MAXNUMBEROFMATTERS  =  @MAXNUMBEROFMATTER
  , LABELPRINTER    =  @LABLEPRINTER
  , YEARENDCOMPLETED   = @YEARENDCOMPLETED
  , INDICATOR22    =  @INDICATOR22
  , INDICATOR23    = @INDICATOR23
  , INDICATOR24    = @INDICATOR24
  , INDICATOR25    = @INDICATOR25
  , INDICATOR26    = @INDICATOR26
  , INDICATOR27    = @INDICATOR27
  , INDICATOR28    = @INDICATOR28
  , INDICATOR29    = @INDICATOR29
  , INDICATOR30    = @INDICATOR30
  , FEESREPSEQ     =  @FEESREPSEQ
  , YourRef        =   @YourRef

  UPDATE Settings SET KeyValue = @WIPWarningLimit WHERE KeyName = 'EstFeeWarningLimit'
  UPDATE Settings SET KeyValue = @DebtorWarningDaysLimit WHERE KeyName = 'DebtorWarningDaysLimit'
  UPDATE Settings SET KeyValue = @LastActionMonth WHERE KeyName = 'LastActionMonth'
  UPDATE Settings SET KeyValue = @LockupValue WHERE KeyName = 'LockupUpValue'
  UPDATE Settings SET KeyValue = @ShowAlarm WHERE KeyName = 'ShowAlarm'
  UPDATE Settings SET KeyValue = @EstimateFeeRequired WHERE KeyName = 'EstFeeRequiredatMatterSetup'
  UPDATE Settings SET KeyValue = @ExpectedinvoiceDateRequired WHERE KeyName = 'ExpInvoiceDateRequiredatMatterSetup'
  UPDATE Settings SET KeyValue = @BranchRequired WHERE KeyName = 'NewCaseWizardBranch'

    IF NOT EXISTS (SELECT TOP 1 1 FROM Settings WHERE KeyName = 'AllowToDeleteActionAttachment')
        INSERT INTO Settings (KeyName,KeyValue ) VALUES ('AllowToDeleteActionAttachment',@DeleteAttachment)
    ELSE
        UPDATE Settings SET KeyValue = @DeleteAttachment WHERE KeyName = 'AllowToDeleteActionAttachment'

    IF NOT EXISTS (SELECT TOP 1 1 FROM Settings WHERE KeyName = 'AllowToDeleteAction')
        INSERT INTO Settings (KeyName,KeyValue) VALUES ('AllowToDeleteAction',@DeleteAction)
    ELSE
        UPDATE Settings SET KeyValue = @DeleteAction WHERE KeyName = 'AllowToDeleteAction'

    IF EXISTS (SELECT TOP 1 1 FROM Settings WHERE KeyName = 'TimeDayBookLimit')
        UPDATE Settings SET KeyValue = @TimeDayBookLimit WHERE KeyName = 'TimeDayBookLimit'
    ELSE
        INSERT INTO Settings (KeyName,KeyValue ) VALUES ('TimeDayBookLimit',@TimeDayBookLimit)


    IF EXISTS (SELECT TOP 1 1 FROM Settings WHERE KeyName = 'OSInvoiceLimit')
        UPDATE Settings SET KeyValue = @OSInvoiceLimit WHERE KeyName = 'OSInvoiceLimit'
    ELSE
        INSERT INTO Settings (KeyName,KeyValue ) VALUES ('OSInvoiceLimit',@OSInvoiceLimit)


    IF EXISTS (SELECT TOP 1 1 FROM Settings WHERE KeyName = 'CurrentWIPLimit')
        UPDATE Settings SET KeyValue = @CurrentWIPLimit WHERE KeyName = 'CurrentWIPLimit'
    ELSE
        INSERT INTO Settings (KeyName,KeyValue ) VALUES ('CurrentWIPLimit',@CurrentWIPLimit)


    IF EXISTS (SELECT TOP 1 1 FROM Settings WHERE KeyName = 'DraftInvoiceLimit')
        UPDATE Settings SET KeyValue = @DraftInvoiceLimit WHERE KeyName = 'DraftInvoiceLimit'
    ELSE
        INSERT INTO Settings (KeyName,KeyValue ) VALUES ('DraftInvoiceLimit',@DraftInvoiceLimit)

    --Client Maximum Code
    IF EXISTS(SELECT TOP 1 1 FROM Settings WHERE KeyName = 'MaxClCode')
        UPDATE Settings SET KeyValue = @MaxClientCode WHERE KeyName = 'MaxClCode'
    ELSE
        INSERT INTO Settings (KeyName,KeyValue ) VALUES ('MaxClCode',@MaxClientCode)

    --Matter Minimum Code
    IF EXISTS(SELECT TOP 1 1 FROM Settings WHERE KeyName = 'MaxMatNum')
        UPDATE Settings SET KeyValue = @MaxMatterCode WHERE KeyName = 'MaxMatNum'
    ELSE
        INSERT INTO Settings (KeyName,KeyValue ) VALUES ('MaxMatNum',@MaxMatterCode)


    IF EXISTS(SELECT TOP 1 1 FROM Settings WHERE KeyName = 'FileNoLabelText')
        UPDATE Settings SET KeyValue = @FileNoLabel WHERE KeyName = 'FileNoLabelText'
    ELSE
        INSERT INTO Settings (KeyName,KeyValue ) VALUES ('FileNoLabelText',@FileNoLabel )

    IF EXISTS(SELECT TOP 1 1 FROM Settings WHERE KeyName = 'SDLTGlobalPath')
        UPDATE Settings SET KeyValue = @SDLTpath  WHERE KeyName = 'SDLTGlobalPath'
    ELSE
        INSERT INTO Settings (KeyName,KeyValue ) VALUES ('SDLTGlobalPath',@SDLTpath )

	IF EXISTS(SELECT TOP 1 1 FROM Settings WHERE KeyName = 'AllowToDeleteOtherAction')  
        UPDATE Settings SET KeyValue = @DeleteOtherAction  WHERE KeyName = 'AllowToDeleteOtherAction'  
    ELSE  
        INSERT INTO Settings (KeyName,KeyValue ) VALUES ('AllowToDeleteOtherAction',@DeleteOtherAction ) 
        
    IF EXISTS(SELECT TOP 1 1 FROM Settings WHERE KeyName = 'MakeFileReadonlyWhenComplete')  
        UPDATE Settings SET KeyValue = @ReadyonlyAttachment  WHERE KeyName = 'MakeFileReadonlyWhenComplete'  
    ELSE  
        INSERT INTO Settings (KeyName,KeyValue ) VALUES ('MakeFileReadonlyWhenComplete',@ReadyonlyAttachment )
        
    IF EXISTS(SELECT TOP 1 1 FROM Settings WHERE KeyName = 'REV_HOLDDATE')  
        UPDATE Settings SET KeyValue = @RevenueHoldDate   WHERE KeyName = 'REV_HOLDDATE'  
    ELSE  
        INSERT INTO Settings (KeyName,KeyValue ) VALUES ('REV_HOLDDATE',@RevenueHoldDate )
        
    IF EXISTS(SELECT TOP 1 1 FROM Settings WHERE KeyName = 'REV_CLIENTGROUP')  
        UPDATE Settings SET KeyValue = @ClientGroup  WHERE KeyName = 'REV_CLIENTGROUP'  
    ELSE  
        INSERT INTO Settings (KeyName,KeyValue ) VALUES ('REV_CLIENTGROUP',@ClientGroup ) 
        
   IF EXISTS(SELECT TOP 1 1 FROM Settings WHERE KeyName = 'UseTeamLeader')  
        UPDATE Settings SET KeyValue = @TeamLeader  WHERE KeyName = 'UseTeamLeader'  
    ELSE  
        INSERT INTO Settings (KeyName,KeyValue ) VALUES ('UseTeamLeader',@TeamLeader ) 
        
   IF EXISTS(SELECT TOP 1 1 FROM Settings WHERE KeyName = 'CompositeBilling')  
        UPDATE Settings SET KeyValue = @CompositeBilling  WHERE KeyName = 'CompositeBilling'  
    ELSE  
        INSERT INTO Settings (KeyName,KeyValue ) VALUES ('CompositeBilling',@CompositeBilling ) 
        
    IF EXISTS(SELECT TOP 1 1 FROM Settings WHERE KeyName = 'EnableDictation')  
        UPDATE Settings SET KeyValue = @EnableDictation  WHERE KeyName = 'EnableDictation'  
    ELSE  
        INSERT INTO Settings (KeyName,KeyValue ) VALUES ('EnableDictation',@EnableDictation )
   
   IF EXISTS(SELECT TOP 1 1 FROM Settings WHERE KeyName = 'ExportReportsToLegacyFormats')  
        UPDATE Settings SET KeyValue = @ExportReportsToLegacyFormats  WHERE KeyName = 'ExportReportsToLegacyFormats'  
    ELSE  
        INSERT INTO Settings (KeyName,KeyValue ) VALUES ('ExportReportsToLegacyFormats',@ExportReportsToLegacyFormats)

   IF EXISTS(SELECT TOP 1 1 FROM Settings WHERE KeyName = 'DocumentSyncVisible')  
        UPDATE Settings SET KeyValue = @DocumentSyncVisible WHERE KeyName = 'DocumentSyncVisible'  
    ELSE  
        INSERT INTO Settings (KeyName,KeyValue ) VALUES ('DocumentSyncVisible',@DocumentSyncVisible ) 
        
	IF EXISTS(SELECT TOP 1 1 FROM Settings WHERE KeyName = 'EnablePaymentReceipt')
		UPDATE Settings SET KeyValue = @EnablePaymentReceipt WHERE KeyName = 'EnablePaymentReceipt'
	ELSE
		INSERT INTO Settings (KeyName, KeyValue) VALUES ('EnablePaymentReceipt', @EnablePaymentReceipt)


SET NOCOUNT OFF
END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPFileSharingLinkToServer'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPFileSharingLinkToServer]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETSPFileSharingLinkToServer]
      (@TypeID              INT,
       @ServerName          VARCHAR(50),
       @Server              VARCHAR(50),
       @AdminUserID         VARCHAR(100),
       @AdminPassword       VARCHAR(100),
       @RootFolder          VARCHAR(200),
       @CollabMatterFolder  VARCHAR(200),
       @CollabFolderFolder  VARCHAR(200),
       @ClientID            VARCHAR(100),
       @ClientSecret        VARCHAR(500),
       @SiteID              VARCHAR(100),
       @CollabDocFolderID   INT,
       @CollabMatter        VARCHAR(50),
       @CollabFolder        VARCHAR(50),
       @Domain              VARCHAR(100),
       @ProxyType           INT,
       @ProxyServer         VARCHAR(500),
       @ProxyPort           INT)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETSPFileSharingLinkToServer]                                                                   *
 *                                                                                                           *
 * Link a Document Folder to a File Sharing Server using Link Information sent to you by the Other Party     *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.5.1.2                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2017-09-22   Pino Carafa       Created                                                                    *
 * 2017-12-05   Pino Carafa       Need to get the root Document Folder for the matter to insert into         *
 *                                [dbo].[FileSharingMatterMapping]                                           *
 *                                                                                                           *
 *************************************************************************************************************/
 BEGIN

DECLARE @ClientFolderURL            VARCHAR(500)
DECLARE @ClientFolderPermissionsURL VARCHAR(500)
SELECT @ClientFolderURL = REPLACE(REPLACE(REPLACE([FSST].[ClientFolderURL], '%02', @Server), '%03', @SiteID), '%04', @Domain),
       @ClientFolderPermissionsURL = REPLACE(REPLACE(REPLACE([FSST].[ClientFolderPermissionsURL], '%02', @Server), '%03', @SiteID), '%04', @Domain)
  FROM [dbo].[FileSharingServerTypes] [FSST]
 WHERE [FSST].[id] = @TypeID

 IF NOT EXISTS
   (SELECT TOP 1 1
      FROM [dbo].[FileSharingServers] [FSS]
     WHERE [FSS].[Name] = @ServerName)
AND NOT EXISTS
   (SELECT TOP 1 1
      FROM [dbo].[FileSharingServers] [FSS]
     WHERE [FSS].[Server] = @Server
       AND [FSS].[TypeId] = @TypeID
       AND [FSS].[Domain] = @Domain
       AND ISNULL([FSS].[CollabDocFolderID], 0) = ISNULL(@CollabDocFolderID, 0))
AND NOT EXISTS
   (SELECT TOP 1 1
      FROM [dbo].[FileSharingServers] [FSS]
           INNER JOIN [dbo].[FileSharingMapping] [FSM]
                   ON [FSM].[FileSharingServersID] = [FSS].[id]
                  AND [FSM].[FileSharingServerFolderID] = @CollabFolderFolder
     WHERE [FSS].[Server] = @Server
       AND [FSS].[TypeId] = @TypeID
       AND [FSS].[Domain] = @Domain)
AND EXISTS
   (SELECT TOP 1 1
      FROM [dbo].[DocFolders] [DF]
     WHERE [DF].[id] = @CollabDocFolderID)
AND NOT EXISTS
   (SELECT TOP 1 1
      FROM [dbo].[FileSharingDocFolders] [FSDF]
     WHERE [FSDF].[DocFolderID] = @CollabDocFolderID
       AND [FSDF].[Shared] = CONVERT(BIT, 1))
  BEGIN
    DECLARE @NewID INT

    INSERT
      INTO [dbo].[FileSharingServers]
          ([TypeID],
           [Name],
           [Server],
           [AdminUserId],
           [AdminPassword],
           [Retired],
           [RootFolder],
           [ClientFolderURL],
           [ClientFolderPermissionsURL],
           [ClientID],
           [ClientSecret],
           [SiteID],
           [CollabDocFolderID],
           [CollabMatter],
           [CollabFolder],
           [Domain],
           [ProxyType],
           [ProxyServer],
           [ProxyPort])
    SELECT @TypeID                     AS [TypeID],
           @ServerName                 AS [Name],
           @Server                     AS [Server],
           @AdminUserID                AS [AdminUserId],
           @AdminPassword              AS [AdminPassword],
           0                           AS [Retired],
           @RootFolder                 AS [RootFolder],
           @ClientFolderURL            AS [ClientFolderURL],
           @ClientFolderPermissionsURL AS [ClientFolderPermissionsURL],
           @ClientID                   AS [ClientID],
           @ClientSecret               AS [ClientSecret],
           @SiteID                     AS [SiteID],
           @CollabDocFolderID          AS [CollabDocFolderID],
           @CollabMatter               AS [CollabMatter],
           @CollabFolder               AS [CollabFolder],
           @Domain                     AS [Domain],
           @ProxyType                  AS [ProxyType],
           @ProxyServer                AS [ProxyServer],
           @ProxyPort                  AS [ProxyPort]

    SET @NewID = CONVERT(INT, SCOPE_IDENTITY())

    IF NOT EXISTS
      (SELECT TOP 1 1
         FROM [dbo].[FileSharingDocFolders] [FSDF]
        WHERE [FSDF].[DocFolderID] = @CollabDocFolderID)
      BEGIN
        INSERT
          INTO [dbo].[FileSharingDocFolders]
              ([DocFolderID],
               [Shared])
        VALUES(@CollabDocFolderID,
               CONVERT(BIT, 1))
      END
    ELSE
      BEGIN
        UPDATE [FSDF]
           SET [FSDF].[Shared] = CONVERT(BIT, 1)
          FROM [dbo].[FileSharingDocFolders] [FSDF]
         WHERE [FSDF].[DocFolderID] = @CollabDocFolderID
      END

    DECLARE @CollabMatterRootDocFolderId INT
    DECLARE @CollabDocFolderMatter VARCHAR(20)
    DECLARE @parent INT

    SET @CollabMatterRootDocFolderId = @CollabDocFolderID
    SET @CollabDocFolderMatter = NULL

    WHILE EXISTS
         (SELECT TOP 1 1
            FROM [dbo].[DocFolders] [DF]
           WHERE [DF].[id] = @CollabMatterRootDocFolderId)
     AND @CollabDocFolderMatter IS NULL
      BEGIN
        SELECT @CollabDocFolderMatter = [MDF].[MatterCode]
          FROM [dbo].[MatterDocFolders] [MDF]
         WHERE [MDF].[DocFolderID] = @CollabMatterRootDocFolderId

        IF @CollabDocFolderMatter IS NULL
          BEGIN
            SET @parent = NULL

            SELECT @parent = [DF].[parent]
              FROM [dbo].[DocFolders] [DF]
             WHERE [DF].[id] = @CollabMatterRootDocFolderId

            SET @CollabMatterRootDocFolderId = @parent
          END
      END

    IF NOT @CollabDocFolderMatter IS NULL
      BEGIN
        INSERT
          INTO [dbo].[FileSharingMatterMapping]
              ([FileSharingMatter],
               [FileSharingServersID],
               [FileSharingServerFolderID])
        VALUES(@CollabDocFolderMatter,
               @NewID,
               @CollabMatterFolder)
      END

    INSERT
      INTO [dbo].[FileSharingMapping]
          ([FileSharingDocFoldersID],
           [FileSharingServersID],
           [FileSharingServerFolderID])
    VALUES(@CollabDocFolderID,
           @NewID,
           @CollabFolderFolder)

    SELECT @NewID AS [NewServerID]
    RETURN @NewID
  END

END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPAddFileSharingLinkFolder'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPAddFileSharingLinkFolder]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETSPAddFileSharingLinkFolder]
      (@MatterCode VARCHAR(20),
       @FolderName VARCHAR(500))
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETSPAddFileSharingLinkFolder]                                                                  *
 *                                                                                                           *
 * Create a new Document Folder to hold documents linked to from another Keyhouse site                       *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.5.1.2                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2017-09-25   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

IF NOT EXISTS
  (SELECT TOP 1 1
     FROM [dbo].[SearchMatters] [MAT]
    WHERE [MAT].[Code] = @MatterCode)
  BEGIN
    RETURN
  END

IF NOT EXISTS
  (SELECT TOP 1 1
     FROM [dbo].[MatterDocFolders] [MDF]
    WHERE [MDF].[MatterCode] = @MatterCode)
  BEGIN
    DECLARE @DefaultRootName VARCHAR(500)

    SET @DefaultRootName = ISNULL((SELECT [KeyValue] 
                                     FROM [dbo].[Settings]
                                    WHERE [KeyName] = 'DocFolderDefaultRootName'),
                                  'Documents')
    INSERT
      INTO [dbo].[DocFolders]
          ([foldername],
           [foldericon])
    VALUES(@DefaultRootName,
           1)
           
    DECLARE @NewMatterFolder INT
    
    SET @NewMatterFolder = CONVERT(INT, SCOPE_IDENTITY())
    
    INSERT
      INTO [dbo].[MatterDocFolders]
          ([MatterCode],
           [DocFolderID])
    VALUES(@MatterCode,
           @NewMatterFolder)      
  END

IF NOT EXISTS
  (SELECT TOP 1 1
     FROM [dbo].[MatterDocFolders] [MDF]
          INNER JOIN [dbo].[DocFolders] [DF]
                  ON [DF].[parent] = [MDF].[DocFolderID]
                 AND [DF].[foldername] = @FolderName
    WHERE [MDF].[MatterCode] = @MatterCode)
  BEGIN
    INSERT
      INTO [dbo].[DocFolders]
          ([parent],
           [foldername],
           [foldericon])
    SELECT [MDF].[DocFolderID],
           @FolderName,
           1
      FROM [dbo].[MatterDocFolders] [MDF]
     WHERE [MDF].[MatterCode] = @MatterCode

    SELECT CONVERT(INT, SCOPE_IDENTITY()) AS [NewDocFolderID]
  END

END
GO

--"Unknown WorkType" with blank code must be retired
UPDATE [dbo].[WorkTypes] SET [RETIRED] = 'Y' WHERE RTRIM(ISNULL([CODE],'')) = ''
GO

-- New column for court fee code - KEYD-4768
IF EXISTS
  (SELECT TOP 1 1
     FROM sys.objects [SO]
          LEFT OUTER JOIN sys.[columns] [SC]
                       ON [SC].[object_id] = [SO].[object_id]
                      AND [SC].[name] = 'CourtFeeCode'
    WHERE [SO].[name] = 'InitialDebtCharges'
      AND [SO].[type] = 'U'
      AND [SC].[column_id] IS NULL)
BEGIN
	ALTER TABLE [dbo].[InitialDebtCharges]
	  ADD [CourtFeeCode] VARCHAR(30) NULL 
END
GO

-- Add table to allow for saving Briefs to Case
IF NOT EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] [SO]
    WHERE [SO].[name] = 'BriefAttachments'
      AND [SO].[type] = 'U')
  BEGIN
    EXEC ('
    CREATE
     TABLE [dbo].[BriefAttachments]
          ([id]                      INT                          IDENTITY(1, 1)                 NOT NULL,
           [BriefID]                 INT                                                         NOT NULL,
           [TrackReference]          INT                                                         NOT NULL,
           CONSTRAINT [PK_BriefAttachments]
                      PRIMARY KEY CLUSTERED
                     ([id]))')

    EXEC ('
    ALTER 
    TABLE [dbo].[BriefAttachments] 
     WITH CHECK 
      ADD CONSTRAINT 
          [FK_BriefAttachments_Brief]
          FOREIGN KEY([BriefId])
          REFERENCES [dbo].[Brief]
                    ([id])
          ON DELETE CASCADE')

    EXEC ('
    ALTER 
    TABLE [dbo].[BriefAttachments] 
    CHECK CONSTRAINT [FK_BriefAttachments_Brief]')

    EXEC ('
    ALTER 
    TABLE [dbo].[BriefAttachments] 
     WITH CHECK 
      ADD CONSTRAINT 
          [FK_BriefAttachments_DAT]
          FOREIGN KEY([TrackReference])
          REFERENCES [dbo].[DiaryAttachments]
                    ([TrackReference])
          ON DELETE CASCADE')

    EXEC ('
    ALTER 
    TABLE [dbo].[BriefAttachments] 
    CHECK CONSTRAINT [FK_BriefAttachments_DAT]')

  END
GO

IF NOT EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] [SO]
          INNER JOIN SYS.[indexes] [SI]
                  ON [SI].[object_id] = [SO].[object_id]
                 AND [SI].[name] = 'IDX_BriefTrack'
    WHERE [SO].[name] = 'BriefAttachments')
  BEGIN
    EXEC ('
    CREATE UNIQUE
           NONCLUSTERED
           INDEX
           [IDX_BriefTrack]
        ON [dbo].[BriefAttachments]
          ([BriefID],
           [TrackReference])
           INCLUDE ([id])
        ON [PRIMARY]')
  END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPGetBriefLocation'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPGetBriefLocation]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETSPGetBriefLocation]
      (@BriefID              INT,
       @DocumentType         INT)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETSPGetBriefLocation]                                                                          *
 *                                                                                                           *
 * Gets the File Location of the Brief. If the Brief was Saved To Case it will get this from                 *
 * [dbo].[BriefAttachments], else from [dbo].[Brief]                                                         *
 * It will get:                                                                                              *
 * @DocumentType = 1 - the first PDF Diary Attachment in [dbo].[BriefAttachments] c.q. the [Location]        *
 *                     from [dbo].[Brief]                                                                    *
 * @DocumentType = 2 - the first DOC or DOCX Diary Attachment in [dbo].[BriefAttachments]                    *
 *                     c.q. replace .pdf with .doc in the [Location] from [dbo].[Brief]                      *
 * @DocumentType = 3 - the first XML Diary Attachment in [dbo].[BriefAttachments]                            *
 *                     c.q. replace .pdf with \Brief.xml in the [Location] from [dbo].[Brief]                *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.5.1.2                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2017-08-26   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN
  DECLARE @FilePath VARCHAR(256)
  DECLARE @TrackReference INT
  DECLARE @ActionID INT

  IF @DocumentType NOT IN (1, 2, 3)
    BEGIN
      DECLARE @Files TABLE
             ([FilePath]            VARCHAR(256),
              [TrackReference]      INT,
              [ActionID]            INT)

      INSERT
        INTO @Files
            ([FilePath],
             [TrackReference],
             [ActionID])
        EXEC [dbo].[ky_NETSPGetBriefLocation] @BriefID, 1

      INSERT
        INTO @Files
            ([FilePath],
             [TrackReference],
             [ActionID])
        EXEC [dbo].[ky_NETSPGetBriefLocation] @BriefID, 2

      INSERT
        INTO @Files
            ([FilePath],
             [TrackReference],
             [ActionID])
        EXEC [dbo].[ky_NETSPGetBriefLocation] @BriefID, 3

      SELECT [FilePath], [TrackReference], [ActionID] FROM @Files
      RETURN
    END

    SELECT TOP 1
           @FilePath =
           CASE WHEN UNC.[UNC] IS NULL
                     THEN RTRIM(ISNULL([DAT].[FilePath], ''))
                ELSE   UNC.[UNC]
                     + SUBSTRING(RTRIM(ISNULL([DAT].[FilePath],
                                              '')),
                                 3,
                                 LEN(RTRIM(ISNULL([DAT].[FilePath],
                                                  ''))) - 2) END,
           @TrackReference = [DAT].[TrackReference],
           @ActionID = [DAT].[DiaryID]
      FROM [dbo].[Brief] [BRI]
           INNER JOIN [dbo].[BriefAttachments] [BRA]
                   ON [BRA].[BriefID] = [BRI].[ID]
           INNER JOIN [dbo].[DiaryAttachments] [DAT]
                   ON [DAT].[TrackReference] = [BRA].[TrackReference]
                  AND CASE WHEN @DocumentType = 1 --PDF
                            AND [DAT].[TYPE] = 'PDF'
                                THEN 1
                           WHEN @DocumentType = 2 --DOC
                            AND [DAT].[TYPE] LIKE 'DOC%'
                                THEN 1
                           WHEN @DocumentType = 3 --XML
                            AND [DAT].[TYPE] = 'XML'
                                THEN 1
                           ELSE 0 END = 1
           LEFT OUTER JOIN [dbo].[UNCAlias] UNC
                        ON SUBSTRING([DAT].[FilePath],
                                     2,
                                     1) = ':'
                       AND UNC.[Drive] = SUBSTRING([DAT].[FilePath],
                                                   1,
                                                   1)
     WHERE [BRI].[ID] = @BriefID
  ORDER BY [BRA].[TrackReference] DESC

IF @FilePath IS NULL
  BEGIN
    SET @TrackReference = 0
    SET @ActionID = 0

    SELECT TOP 1
           @FilePath =
           CASE WHEN UNC.[UNC] IS NULL
                     THEN RTRIM(ISNULL([BRI].[Location], ''))
                ELSE   UNC.[UNC]
                     + SUBSTRING(RTRIM(ISNULL([BRI].[Location],
                                              '')),
                                 3,
                                 LEN(RTRIM(ISNULL([BRI].[Location],
                                                  ''))) - 2) END
      FROM [dbo].[Brief] [BRI]
           LEFT OUTER JOIN [dbo].[UNCAlias] UNC
                        ON SUBSTRING([BRI].[Location],
                                     2,
                                     1) = ':'
                       AND UNC.[Drive] = SUBSTRING([BRI].[Location],
                                                   1,
                                                   1)
     WHERE [BRI].[ID] = @BriefID

    IF NOT @FilePath IS NULL
      BEGIN
        IF @FilePath LIKE '%.pdf'
          BEGIN
            IF @DocumentType = 2
              BEGIN
                SET @FilePath = SUBSTRING(@FilePath, 1, LEN(@FilePath) - 3) + 'DOC'
              END

            IF @DocumentType = 3
              BEGIN
                SET @FilePath = SUBSTRING(@FilePath, 1, LEN(@FilePath) - 4) + '\Brief.xml'
              END
          END
      END
  END

IF NOT @FilePath IS NULL
  BEGIN
    SELECT @FilePath AS [FilePath],
           @TrackReference AS [TrackReference],
           @ActionID AS [ActionID]
  END

END

GO

IF OBJECT_ID('ky_NETFetchPrintLedger') IS NOT NULL
	DROP PROCEDURE [dbo].[ky_NETFetchPrintLedger]
GO

CREATE PROCEDURE [dbo].[ky_NETFetchPrintLedger] (
       @XML NVARCHAR(MAX) = ''
)
AS
/*******************************************************************************************************************************
 *
 * [dbo].[ky_NETFetchPrintLedger]
 *
 * Fetches the print ledger
 *
 * Example search criteria: N'<search matter="BAR001/0002" 
 *                                    function="" 
 *                                    sortby="" 
 *                                    sortorder="" 
 *                                    search="" 
 *                                    maxresult="2000" />'
 *
 * If function is "SELECTCOUNT" then it returns the number of records rather than the data.
 * This is used to determine whether or not we want to show the "Print Ledger" button when in Client/Case.
 *
 * Modification History
 * 2017-01-04   John Ginnane       Created
 * 2017-01-06   John Ginnane       Added function parameter, will operate similar to ky_NETGetLinkedCases
 * 2017-02-02   John Ginnane       Made it use XML to fetch N records rather than all, similar to ky_NETStrongRoomFetchItem
 * 2017-03-08   John Ginnane       Summary totals now sum number of pages, not just records
 * 2017-03-09   John Ginnane       Can now fetch individual record for editing
 * 2017-04-13   John Ginnane       Now returning cost and invoiceno
 * 2017-04-25   John Ginnane       Same as above but for summary totals
 * 2017-04-26   John Ginnane       Added missing column track reference
 * 2017-04-27   John Ginnane       Changed total summary
 * 2017-04-28   John Ginnane       Added counts to summary
 * 2017-05-11   John Ginnane       Added back the page count totals
 * 2017-06-01   John Ginnane       Added handler department and cumulative charge (latter taking into account search criteria)
 * 2017-06-02   John Ginnane       Put the main query into table variable for performance reasons
 * 2017-06-15   John Ginnane       Now calculating profit and loss
 * 2017-06-23   John Ginnane       Fixed potential NULL values appearing in summary. Removed debugging results
 * 2017-08-16   John Ginnane       Changed how unbilled, billed and profit/loss figures are calculated - KEYD-4348
 * 2017-08-16   John Ginnane       Added write off total
 * 2017-09-06   John Ginnane       Fixed NULL value issues when getting summary
 * 2017-09-27   John Ginnane       KEYD-4814 - Changed "Write Off Value" to be all items whose status is NOT "Printed"
 *
 *******************************************************************************************************************************/
BEGIN
	SET NOCOUNT ON
	
	DECLARE @idoc       INT
	DECLARE @MATTERCODE VARCHAR(20)
	DECLARE @FUNCTION   VARCHAR(20)
	DECLARE @SORTBY     VARCHAR(100)
	DECLARE @SORTORDER  VARCHAR(4)
	DECLARE @SEARCH     VARCHAR(500)
	DECLARE @MAXRESULT  INT
	DECLARE @SQL        NVARCHAR(MAX)
	
	BEGIN TRY
		EXEC sp_xml_preparedocument @idoc OUTPUT, @XML
	END TRY
	BEGIN CATCH
		EXEC sp_xml_preparedocument @idoc OUTPUT, N'<search matter="" function="" sortby="" sortorder="" search="" maxresult="2000"/>'
	END CATCH
	
	SELECT @MATTERCODE = '%' + LTRIM(RTRIM(ISNULL([XML].[matter], ''))) + '%',
	       @FUNCTION   = LTRIM(RTRIM(ISNULL([XML].[function], ''))),
	       @SORTBY     = LTRIM(RTRIM(ISNULL([XML].[sortby], ''))),
	       @SORTORDER  = LTRIM(RTRIM(ISNULL([XML].[sortorder], ''))),
	       @SEARCH     = '%' + LTRIM(RTRIM(ISNULL([XML].[search], ''))) + '%',
	       @MAXRESULT  = ISNULL([XML].[maxresult], 2000)
	  FROM OPENXML(@idoc, N'search')
	 WITH ([matter]    VARCHAR(20)  '@matter',
	       [function]  VARCHAR(20)  '@function',
	       [sortby]    VARCHAR(50)  '@sortby',
	       [sortorder] VARCHAR(4)   '@sortorder',
	       [search]    VARCHAR(500) '@search',
	       [maxresult] INT          '@maxresult') AS [XML]
	
	-- Return count of records rather than data
	IF @FUNCTION = 'SELECTCOUNT'
	BEGIN
		SELECT @SQL = 'SELECT COUNT(1)
		                 FROM [dbo].[PrintJobTracker] AS PJT
		                WHERE PJT.[MatterCode] LIKE @MATTERCODE'
	END
	ELSE IF @FUNCTION = 'SUMMARY'
	BEGIN
		SELECT @SQL = 'DECLARE @BILLED DECIMAL(19, 2)
		               
		               IF [dbo].[ISSAM4]() = 1
		               BEGIN
		                   SELECT @BILLED = SUM(ISNULL(TRN.[NominalValue], 0)) * -1
		                     FROM [dbo].[TransactionsHeaders] AS HED
		                          INNER JOIN [dbo].[Transactions] AS TRN
		                                  ON HED.[BatchNo] = TRN.[BatchNo]
 		                    WHERE HED.[Posted] = ''Y''
   		                      AND HED.[XnType] = ''I''
   		                      AND TRN.[RecType] = ''D''
   		                      AND TRN.[LineType] = ''M''
   		                      AND TRN.[OutlayCode] = ''PHO''
   		                      AND TRN.[CorrectionInd] = ''L''
   		                      AND HED.[MatterCode] = REPLACE(@MATTERCODE, ''%'', '''')
   		               END
		               		
		               SELECT ISNULL(COUNT(1), 0) AS [Total],
		                      ISNULL(SUM(ISNULL(PJT.[Cost], 0)), 0) AS [TotalValue],
		                      ISNULL(SUM(CASE WHEN ISNULL(PJT.[InvoiceNo], 0) <= 0 AND ISNULL(PJT.[Status], '''') = ''Printed'' THEN ISNULL(PJT.[NoOfPages], 0) ELSE 0 END), 0) AS [UnbilledCount],
		                      ISNULL(SUM(CASE WHEN ISNULL(PJT.[InvoiceNo], 0)  > 0 AND ISNULL(PJT.[Status], '''') = ''Printed'' THEN ISNULL(PJT.[NoOfPages], 0) ELSE 0 END), 0) AS [BilledCount],
		                      ISNULL(SUM(CASE WHEN ISNULL(PJT.[InvoiceNo], 0) <= 0 AND ISNULL(PJT.[Status], '''') = ''Printed'' THEN ISNULL(PJT.[Cost], 0) ELSE 0 END), 0) AS [UnbilledValue],
		                      --ISNULL(SUM(CASE WHEN ISNULL(PJT.[InvoiceNo], 0)  > 0 AND ISNULL(PJT.[Status], '''') = ''Printed'' THEN ISNULL(PJT.[Cost], 0) ELSE 0 END), 0) AS [BilledValue],
		                      ISNULL(@BILLED, 0) AS [BilledValue],
		                      ISNULL(SUM(CASE WHEN ISNULL(PJT.[Status], '''') <> ''Printed'' THEN ISNULL(PJT.[Cost], 0) ELSE 0 END), 0) AS [WriteOffValue],
		                      SUM(ISNULL(PJT.[Cost], 0)) AS [TotalCost],
		                      ISNULL(SUM(CASE WHEN ISNULL(PJT.[IsColorPrinter], 0) <> 0 THEN ISNULL(PJT.[Cost], 0) ELSE 0 END), 0) AS [ColourTotalCost],
                              ISNULL(SUM(CASE WHEN ISNULL(PJT.[IsColorPrinter], 0)  = 0 THEN ISNULL(PJT.[Cost], 0) ELSE 0 END), 0) AS [BWTotalCost],
                              ISNULL(SUM(CASE WHEN ISNULL(PJT.[IsColorPrinter], 0) <> 0 THEN ISNULL(PJT.[NoOfPages], 0) ELSE 0 END), 0) AS [ColourTotal],
                              ISNULL(SUM(CASE WHEN ISNULL(PJT.[IsColorPrinter], 0)  = 0 THEN ISNULL(PJT.[NoOfPages], 0) ELSE 0 END), 0) AS [BWTotal],
	                          ISNULL(SUM(CASE WHEN ISNULL(PJT.[IsColorPrinter], 0) <> 0 AND ISNULL(PJT.[PageSize], '''') = ''A4'' THEN ISNULL(PJT.[Cost], 0) ELSE 0 END), 0) AS [ColourA4Cost],
	                          ISNULL(SUM(CASE WHEN ISNULL(PJT.[IsColorPrinter], 0)  = 0 AND ISNULL(PJT.[PageSize], '''') = ''A4'' THEN ISNULL(PJT.[Cost], 0) ELSE 0 END), 0) AS [BWA4Cost],
	                          ISNULL(SUM(CASE WHEN ISNULL(PJT.[IsColorPrinter], 0) <> 0 AND ISNULL(PJT.[PageSize], '''') = ''A4'' THEN ISNULL(PJT.[NoOfPages], 0) ELSE 0 END), 0) AS [ColourA4],
	                          ISNULL(SUM(CASE WHEN ISNULL(PJT.[IsColorPrinter], 0)  = 0 AND ISNULL(PJT.[PageSize], '''') = ''A4'' THEN ISNULL(PJT.[NoOfPages], 0) ELSE 0 END), 0) AS [BWA4],
	                          ISNULL(SUM(CASE WHEN ISNULL(PJT.[IsColorPrinter], 0) <> 0 AND ISNULL(PJT.[PageSize], '''') = ''A3'' THEN ISNULL(PJT.[Cost], 0) ELSE 0 END), 0) AS [ColourA3Cost],
	                          ISNULL(SUM(CASE WHEN ISNULL(PJT.[IsColorPrinter], 0)  = 0 AND ISNULL(PJT.[PageSize], '''') = ''A3'' THEN ISNULL(PJT.[Cost], 0) ELSE 0 END), 0) AS [BWA3Cost],
	                          ISNULL(SUM(CASE WHEN ISNULL(PJT.[IsColorPrinter], 0) <> 0 AND ISNULL(PJT.[PageSize], '''') = ''A3'' THEN ISNULL(PJT.[NoOfPages], 0) ELSE 0 END), 0) AS [ColourA3],
	                          ISNULL(SUM(CASE WHEN ISNULL(PJT.[IsColorPrinter], 0)  = 0 AND ISNULL(PJT.[PageSize], '''') = ''A3'' THEN ISNULL(PJT.[NoOfPages], 0) ELSE 0 END), 0) AS [BWA3],
	                          ISNULL(SUM(CASE WHEN ISNULL(PJT.[IsColorPrinter], 0) <> 0 AND ISNULL(PJT.[PageSize], '''') = ''Letter'' THEN ISNULL(PJT.[Cost], 0) ELSE 0 END), 0) AS [ColourLetterCost],
	                          ISNULL(SUM(CASE WHEN ISNULL(PJT.[IsColorPrinter], 0)  = 0 AND ISNULL(PJT.[PageSize], '''') = ''Letter'' THEN ISNULL(PJT.[Cost], 0) ELSE 0 END), 0) AS [BWLetterCost],
	                          ISNULL(SUM(CASE WHEN ISNULL(PJT.[IsColorPrinter], 0) <> 0 AND ISNULL(PJT.[PageSize], '''') = ''Letter'' THEN ISNULL(PJT.[NoOfPages], 0) ELSE 0 END), 0) AS [ColourLetter],
	                          ISNULL(SUM(CASE WHEN ISNULL(PJT.[IsColorPrinter], 0)  = 0 AND ISNULL(PJT.[PageSize], '''') = ''Letter'' THEN ISNULL(PJT.[NoOfPages], 0) ELSE 0 END), 0) AS [BWLetter],
	                          ISNULL(SUM(CASE WHEN ISNULL(PJT.[IsColorPrinter], 0) <> 0 AND ISNULL(PJT.[PageSize], '''') NOT IN (''A4'', ''A3'', ''Letter'') THEN ISNULL(PJT.[Cost], 0) ELSE 0 END), 0) AS [ColourOtherCost],
	                          ISNULL(SUM(CASE WHEN ISNULL(PJT.[IsColorPrinter], 0)  = 0 AND ISNULL(PJT.[PageSize], '''') NOT IN (''A4'', ''A3'', ''Letter'') THEN ISNULL(PJT.[Cost], 0) ELSE 0 END), 0) AS [BWOtherCost],
	                          ISNULL(SUM(CASE WHEN ISNULL(PJT.[IsColorPrinter], 0) <> 0 AND ISNULL(PJT.[PageSize], '''') NOT IN (''A4'', ''A3'', ''Letter'') THEN ISNULL(PJT.[NoOfPages], 0) ELSE 0 END), 0) AS [ColourOther],
	                          ISNULL(SUM(CASE WHEN ISNULL(PJT.[IsColorPrinter], 0)  = 0 AND ISNULL(PJT.[PageSize], '''') NOT IN (''A4'', ''A3'', ''Letter'') THEN ISNULL(PJT.[NoOfPages], 0) ELSE 0 END), 0) AS [BWOther],
	                          ISNULL(@BILLED, 0) - ISNULL(SUM(ISNULL(PJT.[Cost], 0)), 0) AS [ProfitLoss]
		                 FROM [dbo].[PrintJobTracker] AS PJT
		                WHERE PJT.[MatterCode] LIKE @MATTERCODE'
	END
	ELSE IF @FUNCTION = 'FETCH'
	BEGIN
		DECLARE @RECORDID INT
		SELECT @RECORDID = ISNULL([XML].[recordid], 0)
		  FROM OPENXML(@idoc, N'search')
		  WITH ([recordid] INT '@recordid') AS [XML]
		
		SELECT TOP 1
		       ISNULL(PJT.[Handler], '')       AS [Handler],
		       ISNULL(PJT.[FileName], '')      AS [FileName],
		       ISNULL(PJT.[DeviceName], '')    AS [DeviceName],
		       ISNULL(PJT.[NoOfPages], 0)      AS [NoOfPages],
		       ISNULL(PJT.[IsColorPrinter], 0) AS [IsColorPrinter],
		       ISNULL(PJT.[PageSize], '')      AS [PageSize],
		       ISNULL(PJT.[Status], '')        AS [Status]
		  FROM [dbo].[PrintJobTracker] AS PJT
		 WHERE PJT.[RecordID] = @RECORDID
	END
	ELSE
	BEGIN
		SELECT @SEARCH = CASE WHEN @SEARCH = '%%'
			                       THEN ''
			                  ELSE @SEARCH END,
			   @SORTORDER = CASE WHEN @SORTORDER NOT IN ('ASC', 'DESC')
			                      AND @SORTBY <> ''
			                          THEN 'ASC'
			                     ELSE @SORTORDER END
		
	    SELECT @SQL = 'DECLARE @TABLE TABLE (
	                          [RecordID]              INT,
	                          [ClientCode]            VARCHAR(20),
	                          [ClientName]            VARCHAR(256),
	                          [MatterCode]            VARCHAR(20),
	                          [MatterDescription]     VARCHAR(2000),
	                          [Handler]               VARCHAR(10),
	                          [HandlerName]           VARCHAR(30),
	                          [JobID]                 INT,
	                          [FileName]              VARCHAR(200),
	                          [DeviceName]            VARCHAR(500),
	                          [NumPages]              INT,
	                          [LoggedInUser]          VARCHAR(300),
	                          [LastUpdated]           DATETIME,
	                          [Status]                VARCHAR(20),
	                          [IP]                    VARCHAR(30),
	                          [MachineName]           VARCHAR(50),
	                          [PageSize]              VARCHAR(50),
	                          [IsColour]              VARCHAR(10),
	                          [ApplicationType]       VARCHAR(60),
	                          [Cost]                  DECIMAL(19, 2),
	                          [InvoiceNo]             INT,
	                          [TrackReference]        INT,
	                          [HandlerDepartmentCode] VARCHAR(10),
	                          [HandlerDepartment]     VARCHAR(200))
	                          
	                   INSERT INTO @TABLE (
	                          [RecordID],
	                          [ClientCode],
	                          [ClientName],
	                          [MatterCode],
	                          [MatterDescription],
	                          [Handler],
	                          [HandlerName],
	                          [JobID],
	                          [FileName],
	                          [DeviceName],
	                          [NumPages],
	                          [LoggedInUser],
	                          [LastUpdated],
	                          [Status],
	                          [IP],
	                          [MachineName],
	                          [PageSize],
	                          [IsColour],
	                          [ApplicationType],
	                          [Cost],
	                          [InvoiceNo],
	                          [TrackReference],
	                          [HandlerDepartmentCode],
	                          [HandlerDepartment])
	                   SELECT TOP ' + LTRIM(RTRIM(CONVERT(VARCHAR(100), @MAXRESULT))) + '
	                          PJT.[RecordID]        AS [RecordID],
			                  CNT.[Code]            AS [ClientCode],
			                  CNT.[Name]            AS [ClientName],
			                  MAT.[Code]            AS [MatterCode],
			                  MAT.[Description]     AS [MatterDescription],
			                  PJT.[Handler]         AS [Handler],
			                  HAN.[Name]            AS [HandlerName],
			                  PJT.[JobID]           AS [JobID],
			                  PJT.[FileName]        AS [FileName],
			                  PJT.[DeviceName]      AS [DeviceName],
			                  PJT.[NoOfPages]       AS [NumPages],
			                  PJT.[LoggedInUser]    AS [LoggedInUser],
			                  PJT.[LastUpdated]     AS [LastUpdated],
			                  PJT.[Status]          AS [Status],
			                  PJT.[SystemIP]        AS [IP],
			                  PJT.[Machinename]     AS [MachineName],
			                  PJT.[PageSize]        AS [PageSize],
			                  CASE WHEN ISNULL(PJT.[IsColorPrinter], 0) = 0
						                THEN ''No''
					               ELSE ''Yes'' END AS [IsColour],
			                  PJT.[ApplicationType] AS [ApplicationType],
			                  ISNULL(PJT.[Cost], 0) AS [Cost],
			                  ISNULL(PJT.[InvoiceNo], 0) AS [InvoiceNo],
			                  CONVERT(VARCHAR(15), NULLIF(PJT.[TrackReference], 0)) AS [TrackReference],
			                  ISNULL(HAN.[Dept], '''') AS [HandlerDepartmentCode],
			                  ISNULL(DEP.[DESC], '''') AS [HandlerDepartment]
		                 FROM [dbo].[PrintJobTracker] AS PJT
			                  LEFT OUTER JOIN [dbo].[matters] AS MAT
							               ON MAT.[Code] = PJT.[MatterCode]
			                  LEFT OUTER JOIN [dbo].[Contacts] AS CNT
							               ON CNT.[Code] = MAT.[ClientCode]
			                  LEFT OUTER JOIN [dbo].[Handlers] AS HAN
							               ON HAN.[Code] = PJT.[Handler]
							  LEFT OUTER JOIN [dbo].[Departments] AS DEP
							               ON HAN.[Dept] = DEP.[CODE]
		                WHERE PJT.[MatterCode] LIKE @MATTERCODE
		                  AND CASE WHEN @SEARCH = ''''
		                                THEN 1
		                           WHEN CNT.[CODE] LIKE @SEARCH
		                                THEN 1
		                           WHEN CNT.[Name] LIKE @SEARCH
		                                THEN 1
		                           WHEN MAT.[Description] LIKE @SEARCH
		                                THEN 1
		                           WHEN PJT.[Handler] LIKE @SEARCH
		                                THEN 1
		                           WHEN HAN.[Name] LIKE @SEARCH
		                                THEN 1
		                           WHEN PJT.[FileName] LIKE @SEARCH
		                                THEN 1
		                           WHEN PJT.[DeviceName] LIKE @SEARCH
		                                THEN 1
		                           WHEN PJT.[LoggedInUser] LIKE @SEARCH
		                                THEN 1
		                           WHEN PJT.[SystemIP] LIKE @SEARCH
		                                THEN 1
		                           WHEN PJT.[Machinename] LIKE @SEARCH
		                                THEN 1
		                           WHEN PJT.[PageSize] LIKE @SEARCH
		                                THEN 1
		                           WHEN PJT.[ApplicationType] LIKE @SEARCH
		                                THEN 1
		                           WHEN PJT.[InvoiceNo] LIKE @SEARCH
		                                THEN 1
		                           ELSE 0 END = 1
		                  
		                  SELECT [RecordID],
	                             [ClientCode],
	                             [ClientName],
	                             [MatterCode],
	                             [MatterDescription],
	                             [Handler],
	                             [HandlerName],
	                             [JobID],
	                             [FileName],
	                             [DeviceName],
	                             [NumPages],
	                             [LoggedInUser],
	                             [LastUpdated],
	                             [Status],
	                             [IP],
	                             [MachineName],
	                             [PageSize],
	                             [IsColour],
	                             [ApplicationType],
	                             [Cost],
	                             (SELECT SUM(INR.[Cost]) FROM @TABLE AS INR WHERE INR.[RecordID] <= OTR.[RecordID]) AS [CumulativeCharge],
	                             [InvoiceNo],
	                             [TrackReference],
	                             [HandlerDepartmentCode],
	                             [HandlerDepartment]
	                        FROM @TABLE AS OTR
			                     ORDER BY [' + ISNULL(NULLIF(@SORTBY, ''), 'RecordID') + '] ' + @SORTORDER
	END
	
	EXEC sp_xml_removedocument @idoc
	
	EXEC sp_executesql @SQL,
	                   N'@MATTERCODE VARCHAR(20),
	                     @SEARCH     VARCHAR(500)',
	                   @MATTERCODE=@MATTERCODE,
	                   @SEARCH=@SEARCH	                      
	
	SET NOCOUNT OFF
END
GO

IF OBJECT_ID(N'[ky_NETCCDeleteDiaryDelegations]',N'P')IS NOT NULL
	DROP PROCEDURE [ky_NETCCDeleteDiaryDelegations]
GO
CREATE PROCEDURE [ky_NETCCDeleteDiaryDelegations]  
(  
  @ASSIGNNO		INT,
  @CODE			VARCHAR(20), 
  @ACTIONID		INT
)  
/*****************************************************************************  
  
Stored Procedure Name: [ky_NETCCFetchDiaryDelegations]  
Description:  This procedure is used load the data for TaskAddAction page Delegation Grid.  
Last Updated: 
12 FEB 2015 Arun- Change NVARCHAR to VARCHAR
26 SEP 2017 Arun - KEYD-4577
26 SEP 2017 Arun - KEYD-4577 - Added an condition to check any action is delegated by the owner..
*****************************************************************************/  
AS  
BEGIN  
  SET NOCOUNT ON    
   IF EXISTS(SELECT 1 FROM DiaryDelegations WHERE AssignNo = @ASSIGNNO AND DelType = 'Released')
   BEGIN
	    UPDATE 
		DiaryDelegations 
	SET 
		  DELEGATE		= @CODE
		, FromHandler	= @CODE 
	WHERE 
		ACTIONID = @ACTIONID 
	AND 
		[OWNER] = 'Y'
   END
   
   Delete From 
          DiaryDelegations 
   WHERE
          AssignNo  = @ASSIGNNO 
          AND DelType <> 'Created'   
   
   IF (	SELECT 
			COUNT(*) 
		FROM 
			DiaryDelegations 
		WHERE 
			DELEGATE = (SELECT 
							HANDLER 
						FROM 
							DiaryDelegations 
						WHERE 
							[OWNER] = 'Y' AND 
							ACTIONID = @ACTIONID) AND 
			DelType = 'Delegated' AND 
			ACTIONID = @ACTIONID) = 0
   BEGIN
	   UPDATE   
			diary   
	   SET   
			DELEGATIONSTATUS = 0   
	   WHERE   
			ACTIONID = @ACTIONID  
   END
   
    SET NOCOUNT OFF 
END
GO

GO

IF NOT EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] [SO]
    WHERE [SO].[name] = 'DiaryAttachmentAccess'
      AND [SO].[type] = 'U')
  BEGIN
    EXEC ('
    CREATE
     TABLE [dbo].[DiaryAttachmentAccess]
          ([id]                  INT           IDENTITY(1, 1)                                     NOT NULL,
           [TrackReference]      INT                                                              NOT NULL,
           [Handler]             VARCHAR(10)                                                      NOT NULL,
           [AccessDate]          DATETIME      CONSTRAINT [DF_DAA_AccessDate] DEFAULT GETDATE()   NOT NULL,
           [Comment]             VARCHAR(2000) CONSTRAINT [DF_DAA_Comment] DEFAULT('''')          NOT NULL,
           CONSTRAINT [PK_DiaryAttachmentAccess]
                      PRIMARY KEY CLUSTERED
                     ([id])
                   ON [PRIMARY])
        ON [PRIMARY]')
  END
GO

IF NOT EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] [SO]
          INNER JOIN SYS.[indexes] [SI]
                  ON [SI].[object_id] = [SO].[object_id]
                 AND [SI].[name] = 'IDX_AccessByTime'
    WHERE [SO].[name] = 'DiaryAttachmentAccess'
      AND [SO].[type] = 'U')
  BEGIN
    EXEC ('
    CREATE INDEX
           [IDX_AccessByTime]
        ON [dbo].[DiaryAttachmentAccess]
          ([AccessDate],
           [TrackReference])
           INCLUDE ([id],
                    [Handler],
                    [Comment])
        ON [PRIMARY]')
  END
GO

IF NOT EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] [SO]
          INNER JOIN SYS.[indexes] [SI]
                  ON [SI].[object_id] = [SO].[object_id]
                 AND [SI].[name] = 'IDX_DATAccessByHandler'
    WHERE [SO].[name] = 'DiaryAttachmentAccess'
      AND [SO].[type] = 'U')
  BEGIN
    EXEC ('
    CREATE INDEX
           [IDX_DATAccessByHandler]
        ON [dbo].[DiaryAttachmentAccess]
          ([Handler],
           [TrackReference],
           [AccessDate])
           INCLUDE ([id],
                    [Comment])
        ON [PRIMARY]')
  END
GO

IF NOT EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] [SO]
          INNER JOIN SYS.[indexes] [SI]
                  ON [SI].[object_id] = [SO].[object_id]
                 AND [SI].[name] = 'IDX_TimeAccessByHandler'
    WHERE [SO].[name] = 'DiaryAttachmentAccess'
      AND [SO].[type] = 'U')
  BEGIN
    EXEC ('
    CREATE INDEX
           [IDX_TimeAccessByHandler]
        ON [dbo].[DiaryAttachmentAccess]
          ([Handler],
           [AccessDate],
           [TrackReference])
           INCLUDE ([id],
                    [Comment])
        ON [PRIMARY]')
  END
GO

IF NOT EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] [SO]
    WHERE [SO].[name] = 'DiaryAttachmentAccessArchive'
      AND [SO].[type] = 'U')
  BEGIN
    EXEC ('
    CREATE
     TABLE [dbo].[DiaryAttachmentAccessArchive]
          ([id]                  INT           IDENTITY(1, 1)     NOT NULL,
           [TrackReference]      INT                              NOT NULL,
           [Handler]             VARCHAR(10)                      NOT NULL,
           [AccessDate]          DATETIME                         NOT NULL,
           [Comment]             VARCHAR(2000)                    NOT NULL,
           CONSTRAINT [PK_DiaryAttachmentAccessArchive]
                      PRIMARY KEY CLUSTERED
                     ([id])
                   ON [PRIMARY])
        ON [PRIMARY]')
  END
GO

IF NOT EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] [SO]
          INNER JOIN SYS.[indexes] [SI]
                  ON [SI].[object_id] = [SO].[object_id]
                 AND [SI].[name] = 'IDX_AccessByTime'
    WHERE [SO].[name] = 'DiaryAttachmentAccessArchive'
      AND [SO].[type] = 'U')
  BEGIN
    EXEC ('
    CREATE INDEX
           [IDX_AccessByTime]
        ON [dbo].[DiaryAttachmentAccessArchive]
          ([AccessDate],
           [TrackReference])
           INCLUDE ([id],
                    [Handler],
                    [Comment])
        ON [PRIMARY]')
  END
GO

IF NOT EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] [SO]
          INNER JOIN SYS.[indexes] [SI]
                  ON [SI].[object_id] = [SO].[object_id]
                 AND [SI].[name] = 'IDX_DATAccessByHandler'
    WHERE [SO].[name] = 'DiaryAttachmentAccessArchive'
      AND [SO].[type] = 'U')
  BEGIN
    EXEC ('
    CREATE INDEX
           [IDX_DATAccessByHandler]
        ON [dbo].[DiaryAttachmentAccessArchive]
          ([Handler],
           [TrackReference],
           [AccessDate])
           INCLUDE ([id],
                    [Comment])
        ON [PRIMARY]')
  END
GO

IF NOT EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] [SO]
          INNER JOIN SYS.[indexes] [SI]
                  ON [SI].[object_id] = [SO].[object_id]
                 AND [SI].[name] = 'IDX_TimeAccessByHandler'
    WHERE [SO].[name] = 'DiaryAttachmentAccessArchive'
      AND [SO].[type] = 'U')
  BEGIN
    EXEC ('
    CREATE INDEX
           [IDX_TimeAccessByHandler]
        ON [dbo].[DiaryAttachmentAccessArchive]
          ([Handler],
           [AccessDate],
           [TrackReference])
           INCLUDE ([id],
                    [Comment])
        ON [PRIMARY]')
  END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETTRUpdateDisplayText'
      AND SO.[type] = 'TR')
  BEGIN
    DROP TRIGGER [dbo].[ky_NETTRUpdateDisplayText]
  END
GO

WHILE EXISTS
     (SELECT TOP 1 1
        FROM [dbo].[diary] [DIA]
       WHERE [DIA].[ATTACHMENTS] IS NULL)
BEGIN

BEGIN TRANSACTION

DECLARE @ACTIONS TABLE
       ([ACTIONID] INT PRIMARY KEY)

DELETE @ACTIONS

INSERT
  INTO @ACTIONS
      ([ACTIONID])
SELECT TOP 50
       [DIA].[ACTIONID]
  FROM [dbo].[diary] [DIA]
 WHERE [DIA].[ATTACHMENTS] IS NULL

UPDATE [DIA]
   SET [DIA].[ATTACHMENTS] = CASE WHEN [HAT].[TOTAL] = 0
                                       THEN 'N'
                                  ELSE 'Y' END
  FROM @ACTIONS [NID]
       INNER JOIN [dbo].[diary] [DIA]
               ON [DIA].[ACTIONID] = [NID].[ACTIONID]
       CROSS APPLY (SELECT COUNT(1) AS [TOTAL]
                      FROM [dbo].[DiaryAttachments] [DAT]
                     WHERE [DAT].[DiaryID] = [DIA].[ACTIONID]) [HAT]

COMMIT

END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM [dbo].[diary] [DIA]
    WHERE [DIA].[ATTACHMENTS] = '1')
  BEGIN
    UPDATE [dbo].[diary] SET [ATTACHMENTS] = 'Y' WHERE [ATTACHMENTS] = '1'
  END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM [dbo].[diary] [DIA]
    WHERE [DIA].[ATTACHMENTS] = '0')
  BEGIN
    UPDATE [dbo].[diary] SET [ATTACHMENTS] = 'N' WHERE [ATTACHMENTS] = '0'
  END
GO

WHILE EXISTS
     (SELECT TOP 1 1
        FROM [dbo].[diary] [DIA]
             CROSS APPLY (SELECT COUNT(1) AS [TOTAL]
                            FROM [dbo].[DiaryAttachments] [DAT]
                           WHERE [DAT].[DiaryID] = [DIA].[ACTIONID]) [HAT]
       WHERE [DIA].[ATTACHMENTS] <> 'Y'
         AND [HAT].[TOTAL] > 0)
BEGIN

BEGIN TRANSACTION

DECLARE @ACTIONS TABLE
       ([ACTIONID] INT PRIMARY KEY)

DELETE @ACTIONS

INSERT
  INTO @ACTIONS
      ([ACTIONID])
SELECT [DIA].[ACTIONID]
  FROM [dbo].[diary] [DIA]
       CROSS APPLY (SELECT COUNT(1) AS [TOTAL]
                      FROM [dbo].[DiaryAttachments] [DAT]
                     WHERE [DAT].[DiaryID] = [DIA].[ACTIONID]) [HAT]
 WHERE [DIA].[ATTACHMENTS] <> 'Y'
   AND [HAT].[TOTAL] > 0

UPDATE [DIA]
   SET [DIA].[ATTACHMENTS] = CASE WHEN [HAT].[TOTAL] = 0
                                       THEN 'N'
                                  ELSE 'Y' END
  FROM @ACTIONS [NID]
       INNER JOIN [dbo].[diary] [DIA]
               ON [DIA].[ACTIONID] = [NID].[ACTIONID]
       CROSS APPLY (SELECT COUNT(1) AS [TOTAL]
                      FROM [dbo].[DiaryAttachments] [DAT]
                     WHERE [DAT].[DiaryID] = [DIA].[ACTIONID]) [HAT]

COMMIT

END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM [dbo].[diary] [DIA]
          LEFT OUTER JOIN [dbo].[DiaryAttachments] [DAT]
                       ON [DAT].[DiaryID] = [DIA].[ACTIONID]
    WHERE [DIA].[ATTACHMENTS] <> 'N'
      AND [DAT].[TrackReference] IS NULL)
BEGIN

DECLARE @ACTIONS TABLE
       ([ACTIONID] INT PRIMARY KEY)

DELETE @ACTIONS

INSERT
  INTO @ACTIONS
      ([ACTIONID])
SELECT [DIA].[ACTIONID]
  FROM [dbo].[diary] [DIA]
       LEFT OUTER JOIN [dbo].[DiaryAttachments] [DAT]
                    ON [DAT].[DiaryID] = [DIA].[ACTIONID]
 WHERE [DIA].[ATTACHMENTS] <> 'N'
   AND [DAT].[TrackReference] IS NULL

UPDATE [DIA]
   SET [DIA].[ATTACHMENTS] = 'N'
  FROM @ACTIONS [NID]
       INNER JOIN [dbo].[diary] [DIA]
               ON [DIA].[ACTIONID] = [NID].[ACTIONID]

END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETTRUpdateDAT'
      AND SO.[type] = 'TR')
  BEGIN
    DROP TRIGGER [dbo].[ky_NETTRUpdateDAT]
  END
GO

CREATE TRIGGER
       [dbo].[ky_NETTRUpdateDAT]
    ON [dbo].[DiaryAttachments]
       AFTER INSERT, UPDATE
AS
IF ((SELECT TRIGGER_NESTLEVEL()) <2)
BEGIN

  BEGIN TRY
    IF EXISTS
      (SELECT TOP 1 1 FROM [deleted])
      BEGIN
        INSERT
          INTO [dbo].[DiaryAttachmentAccess]
              ([TrackReference],
               [Handler],
               [Comment])
        SELECT [DEL].[TrackReference],
               CONVERT(VARCHAR(10),
                       CASE WHEN CHARINDEX(CHAR(0),
                                           CONVERT(VARCHAR(MAX),
                                                   [P].[context_info])) > 0
                                 THEN SUBSTRING(CONVERT(VARCHAR(MAX),
                                                        [P].[context_info]),
                                                1,
                                                CHARINDEX(CHAR(0),
                                                          CONVERT(VARCHAR(MAX),
                                                          [P].[context_info])) - 1)
                            ELSE CONVERT(VARCHAR(MAX),
                                         [P].[context_info]) END),
               'Updated'
          FROM [deleted] DEL
               INNER JOIN [sys].[sysprocesses] P
                       ON P.[spid] = @@SPID
      END
    ELSE
      BEGIN
        INSERT
          INTO [dbo].[DiaryAttachmentAccess]
              ([TrackReference],
               [Handler],
               [Comment])
        SELECT [INS].[TrackReference],
               CONVERT(VARCHAR(10),
                       CASE WHEN CHARINDEX(CHAR(0),
                                           CONVERT(VARCHAR(MAX),
                                                   [P].[context_info])) > 0
                                 THEN SUBSTRING(CONVERT(VARCHAR(MAX),
                                                        [P].[context_info]),
                                                1,
                                                CHARINDEX(CHAR(0),
                                                          CONVERT(VARCHAR(MAX),
                                                          [P].[context_info])) - 1)
                            ELSE CONVERT(VARCHAR(MAX),
                                         [P].[context_info]) END),
               'Inserted'
          FROM [inserted] INS
               INNER JOIN [sys].[sysprocesses] P
                       ON P.[spid] = @@SPID
      END

    UPDATE [DIA]
       SET [DIA].[ATTACHMENTS] = CASE WHEN [HAT].[TOTAL] = 0
                                           THEN 'N'
                                      ELSE 'Y' END
      FROM deleted [DEL]
           INNER JOIN [dbo].[diary] [DIA]
                      CROSS APPLY (SELECT COUNT(1) AS [TOTAL]
                                     FROM [dbo].[DiaryAttachments] [DAT]
                                    WHERE [DAT].[DiaryID] = [DIA].[ACTIONID]) [HAT]
                   ON [DIA].[ACTIONID] = [DEL].[DiaryID]

    UPDATE [DIA]
       SET [DIA].[ATTACHMENTS] = CASE WHEN [HAT].[TOTAL] = 0
                                           THEN 'N'
                                      ELSE 'Y' END
      FROM inserted [INS]
           INNER JOIN [dbo].[diary] [DIA]
                      CROSS APPLY (SELECT COUNT(1) AS [TOTAL]
                                     FROM [dbo].[DiaryAttachments] [DAT]
                                    WHERE [DAT].[DiaryID] = [DIA].[ACTIONID]) [HAT]
                   ON [DIA].[ACTIONID] = [INS].[DiaryID]
  END TRY
  BEGIN CATCH
  END CATCH
END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPDocumentAccess'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPDocumentAccess]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETSPDocumentAccess]
      (@TrackReference          INT,
       @Handler                 VARCHAR(10),
       @Comment                 VARCHAR(2000))
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETSPDocumentAccess]                                                                            *
 *                                                                                                           *
 * Log Document Access                                                                                       *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.5.1.2                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2017-10-02   Pino Carafa       Created                                                                    *
 * 2018-08-09   Pino Carafa       Performance Improvement                                                    *
 * 2019-03-20   Pino Carafa       Azure Compatibility                                                        *
 * 2019-10-09   John Ginnane      KEYD-6197 - Replaced sysprocesses code with function                       *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

SET @TrackReference = ISNULL(@TrackReference, 0)

IF @TrackReference = 0
  BEGIN
    RETURN
  END

IF (SELECT TOP 1 1 FROM [dbo].[DiaryAttachments] [DAT] WITH (NOLOCK) WHERE [DAT].[TrackReference] = @TrackReference) IS NULL
  BEGIN
    RETURN
  END

IF EXISTS
  (SELECT TOP 1 1
     FROM [dbo].[Handlers] [HAN] WITH (NOLOCK)
    WHERE [HAN].[CODE] = RTRIM(ISNULL(@Handler, '')))
  BEGIN
    SELECT @Handler     = RTRIM(ISNULL([HAN].[Code], ''))
      FROM [dbo].[Handlers] [HAN] WITH (NOLOCK)
     WHERE [HAN].[CODE] = RTRIM(ISNULL(@Handler, ''))
  END
ELSE
  BEGIN
    SET @Handler = ''
  END

IF RTRIM(ISNULL(@Handler, '')) = ''
  BEGIN
    SET @Handler = [dbo].[ky_NETFNGetLoggedOnHandler]()
  END

SET @Comment        = CASE WHEN RTRIM(ISNULL(@Comment, '')) = ''
                                THEN 'No reason given'
                           ELSE RTRIM(ISNULL(@Comment, '')) END

  INSERT
    INTO [dbo].[DiaryAttachmentAccess]
        ([TrackReference],
         [Handler],
         [Comment])
  VALUES(@TrackReference,
         @Handler,
         @Comment)

END

GO

-- ENTEREDBY should contain a HANDLER CODE which can be up to 10 characters wide.
IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] [SO]
          INNER JOIN SYS.[columns] [SC]
                  ON [SC].[object_id] = [SO].[object_id]
                 AND [SC].[name] = 'ENTEREDBY'
                 AND [SC].[max_length] < 10
    WHERE [SO].[name] = 'DiaryAttachments'
      AND [SO].[type] = 'U')
  BEGIN
    EXEC ('ALTER TABLE [dbo].[DiaryAttachments] ALTER COLUMN [ENTEREDBY] VARCHAR(10) NOT NULL')
  END
GO

-- Just in case somebody is trialing Contact Intelligence; this trigger would cause the following
-- statements to grind to a halt.
IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] [SO]
    WHERE [SO].[name] = 'ky_NETCIAddNoteForTrackReference'
      AND [SO].[type] = 'TR')
  BEGIN
    EXEC ('
    DISABLE TRIGGER [dbo].[ky_NETCIAddNoteForTrackReference]
                 ON [dbo].[DiaryAttachments]')
  END
GO

-- Update those Diary Attachments where the EnteredBy column contains a Login instead of a
-- handler code (older entries)
-- This is causing a huge bottleneck. Not sure this is even necessary anymore. Review after 5.6.3.1
--UPDATE [DAT]
--   SET [DAT].[ENTEREDBY] = [HAN2].[CODE]
--  FROM [dbo].[DiaryAttachments] [DAT]
--       LEFT OUTER JOIN [dbo].[Handlers] [HAN]
--                    ON [HAN].[CODE] = [DAT].[ENTEREDBY]
--       LEFT OUTER JOIN [dbo].[Handlers] [HAN2]
--                    ON [HAN2].[LOGON] = [DAT].[ENTEREDBY]
-- WHERE RTRIM(ISNULL([DAT].[ENTEREDBY], '')) <> ''
--   AND [HAN].[CODE] IS NULL
--   AND [HAN2].[CODE] IS NOT NULL
--GO

-- Update those Diary Attachments where the ENTEREDBY column contains a Login instead of a
-- handler code (older entries) - this time those where the LOGON is actually longer than
-- 10 characters so it was truncated when it was originally put into ENTEREDBY
-- This is causing a huge bottleneck. Not sure this is even necessary anymore. Review after 5.6.3.1
--UPDATE [DAT]
--   SET [DAT].[ENTEREDBY] = [HAN2].[CODE]
--  FROM [dbo].[DiaryAttachments] [DAT]
--       LEFT OUTER JOIN [dbo].[Handlers] [HAN]
--                    ON [HAN].[CODE] = [DAT].[ENTEREDBY]
--       LEFT OUTER JOIN [dbo].[Handlers] [HAN2]
--                    ON SUBSTRING([HAN2].[LOGON], 1, LEN([DAT].[ENTEREDBY])) = [DAT].[ENTEREDBY]
-- WHERE RTRIM(ISNULL([DAT].[ENTEREDBY], '')) <> ''
--   AND [HAN].[CODE] IS NULL
--   AND [HAN2].[CODE] IS NOT NULL
--GO

-- The ones that contain a blank ENTEREDBY can be deemed to have been entered by the
-- Admin account.
UPDATE [DAT]
   SET [DAT].[ENTEREDBY] = 'ADM'
  FROM [dbo].[DiaryAttachments] [DAT]
       LEFT OUTER JOIN [dbo].[Handlers] [HAN]
                    ON [HAN].[CODE] = [DAT].[ENTEREDBY]
 WHERE RTRIM(ISNULL([DAT].[ENTEREDBY], '')) = ''
   AND [HAN].[CODE] IS NULL
GO

-- Likewise, those who contain "KEYHOUSE" or a variation thereof in ENTEREDBY
UPDATE [DAT]
   SET [DAT].[ENTEREDBY] = 'ADM'
  FROM [dbo].[DiaryAttachments] [DAT]
       LEFT OUTER JOIN [dbo].[Handlers] [HAN]
                    ON [HAN].[CODE] = [DAT].[ENTEREDBY]
 WHERE RTRIM(ISNULL([DAT].[ENTEREDBY], '')) LIKE 'KEY%'
   AND [HAN].[CODE] IS NULL
GO

-- After this there may be an odd entry where ENTEREDBY contains part of the Handler Name
-- This is causing a huge bottleneck. Not sure this is even necessary anymore. Review after 5.6.3.1
--UPDATE [DAT]
--   SET [DAT].[ENTEREDBY] = [HAN2].[CODE]
--  FROM [dbo].[DiaryAttachments] [DAT]
--       LEFT OUTER JOIN [dbo].[Handlers] [HAN]
--                    ON [HAN].[CODE] = [DAT].[ENTEREDBY]
--       LEFT OUTER JOIN [dbo].[Handlers] [HAN2]
--                    ON SUBSTRING([HAN2].[NAME], 1, LEN([DAT].[ENTEREDBY])) = [DAT].[ENTEREDBY]
-- WHERE RTRIM(ISNULL([DAT].[ENTEREDBY], '')) <> ''
--   AND [HAN].[CODE] IS NULL
--   AND [HAN2].[CODE] IS NOT NULL
--GO

-- Finally, change anything that's left and that can't be matched no matter how hard we try
-- and remove their meaningless ENTEREDBY values, replacing them with ADM for the Admin account.
UPDATE [DAT]
   SET [DAT].[ENTEREDBY] = 'ADM'
  FROM [dbo].[DiaryAttachments] [DAT]
       LEFT OUTER JOIN [dbo].[Handlers] [HAN]
                    ON [HAN].[CODE] = [DAT].[ENTEREDBY]
 WHERE RTRIM(ISNULL([DAT].[ENTEREDBY], '')) <> ''
   AND [HAN].[CODE] IS NULL
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETGetDocumentsVersion'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETGetDocumentsVersion]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETGetDocumentsVersion]
       (@TrackReference      INT)
     
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETGetDocumentsVersion]                                                                         *
 *                                                                                                           *
 * This sp is used to get next Version number of document based on trackreference                            *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.4.4.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2017-03-03   Suriya Mohan      Created                                                                    *
 * 2017-05-08   Suriya Mohan      KEYD-4315-Export/Import Document Folders .. sync                           *
 * 2017-06-06   Suriya Mohan      KEYD-4315-Export/Import Document Folders .. sync                           *
 * 2017-10-02   Pino Carafa       Log access and Coding Standards                                            *
 * 2019-03-21   Pino Carafa       Azure compatibility                                                        *
 * 2019-10-09   John Ginnane      KEYD-6197 - Replaced sysprocesses code with function                       *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

DECLARE @Handler VARCHAR(10)

SET @Handler = [dbo].[ky_NETFNGetLoggedOnHandler]()

INSERT
  INTO [dbo].[DiaryAttachmentAccess]
      ([TrackReference],
       [Handler],
       [Comment])
SELECT @TrackReference,
       @Handler,
       'Retrieved Version For Export/Import Document Folders .. sync'
  FROM [dbo].[DiaryAttachments] [DAT]
 WHERE [DAT].[TrackReference] = @TrackReference

;WITH [LatestVersion]
AS
(  SELECT ISNULL(MAX([DAV].[Version]), 0) + 1 AS [Version],
          [DAT].[LASTACCESSBY]
     FROM [dbo].[DiaryAttachments] [DAT]
          INNER JOIN [dbo].[DiaryAttachmentVersioning] [DAV]
                  ON [DAV].[TrackReference] = [DAT].[TrackReference]
    WHERE [DAT].[Trackreference] = @TrackReference
 GROUP BY [DAT].[LASTACCESSBY])
  SELECT TOP 1 
         [LAV].[LASTACCESSBY] AS [Handler],
         [HAN].[NAME], 
         [LAV].[Version] 
    FROM [LatestVersion] [LAV]
         LEFT OUTER JOIN [dbo].[Handlers] [HAN] 
                      ON [HAN].[CODE] = [LAV].[LASTACCESSBY]
ORDER BY [LAV].[Version] DESC

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETInsertDocumentsVersion'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETInsertDocumentsVersion]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETInsertDocumentsVersion]
       (@TrackReference      INT,
        @Version             INT,
        @FilePath            VARCHAR(250),
        @HandlerIns          VARCHAR(10))

AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETInsertDocumentsVersion]                                                                      *
 *                                                                                                           *
 * This sp is used to insert the document version                                                            *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.4.4.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2017-03-03   Suriya Mohan      Created                                                                    *
 * 2017-03-07   Suriya Mohan      Added the comments column                                                  *
 * 2017-05-15   Suriya Mohan      KEYD-4315-Export/Import Document Folders .. sync                           *
 * 2017-06-08   Suriya Mohan      KEYD-4315-Export/Import Document Folders .. sync                           *
 * 2017-10-02   Pino Carafa       Coding Standards and Activity Logging                                      *
 *                                AND HANDLER SHOULD BE VARCHAR(10) FOR CRYING OUT LOUD                      *
 * 2019-03-20   Pino Carafa       Azure compatibility                                                        *
 * 2019-10-09   John Ginnane      KEYD-6197 - Replaced sysprocesses code with function                       *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

DECLARE @Handler1 VARCHAR(10)
DECLARE @TimeStamp DATETIME

SELECT @Handler1  = [DAT].[LASTACCESSBY],
       @TimeStamp = [DAT].[DATEENTERED]
  FROM [dbo].[DiaryAttachments] [DAT]
 WHERE [DAT].[TrackReference] = @TrackReference

INSERT
  INTO [dbo].[DiaryAttachmentVersioning]
      ([TrackReference],
       [Version],
       [Filepath],
       [Handler],
       [Timestamp],
       [Comments])
Values(@TrackReference,
       @Version,
       @FilePath,
       @Handler1,
       @TimeStamp,
       '')

UPDATE [DAT]
   SET [DAT].[ENTEREDBY]    = @HandlerIns,
       [DAT].[LASTACCESSBY] = @HandlerIns,
       [DAT].[DATEENTERED]  = GETDATE()
  FROM [dbo].[DiaryAttachments] [DAT]
 WHERE [DAT].[trackreference] = @TrackReference

DECLARE @Handler VARCHAR(10)

SET @Handler = [dbo].[ky_NETFNGetLoggedOnHandler]()

INSERT
  INTO [dbo].[DiaryAttachmentAccess]
      ([TrackReference],
       [Handler],
       [Comment])
SELECT @TrackReference,
       @Handler,
       'Inserted New Version For Export/Import Document Folders .. sync'
  FROM [dbo].[DiaryAttachments] [DAT]
 WHERE [DAT].[TrackReference] = @TrackReference

END

GO


IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPGetNextDocumentVersion'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPGetNextDocumentVersion]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETSPGetNextDocumentVersion]
      (@TrackReference INT,
       @Comment        VARCHAR(MAX))
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETSPGetNextDocumentVersion]                                                                    *
 *                                                                                                           *
 * Inserts a row for the next version of a document and return the new row including the id. After calling   *
 * this procedure the calling software will copy files to their new locations, and if that fails it will     *
 * delete the newly created line as a "rollback" using ky_NETSPDeleteDocumentVersion                         *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.5.1.2                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2017-06-30   Pino Carafa       Created                                                                    *
 * 2017-10-02   Pino Carafa       Access Logging                                                             *
 * 2019-03-20   Pino Carafa       Azure compatibility                                                        *
 * 2019-10-09   John Ginnane      KEYD-6197 - Replaced sysprocesses code with function                       *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

SET NOCOUNT ON

DECLARE @MaxVersion INT
DECLARE @FilePath VARCHAR(255)

SELECT @MaxVersion = ISNULL(MAX([DATV].[Version]), 0)
  FROM [dbo].[DiaryAttachments] [DAT]
       LEFT OUTER JOIN [dbo].[DiaryAttachmentVersioning] [DATV]
                    ON [DATV].[TrackReference] = [DAT].[TrackReference]
 WHERE [DAT].[TrackReference] = @TrackReference

SELECT @FilePath
     = CASE WHEN UNC.[UNC] IS NULL
                 THEN RTRIM(ISNULL(ISNULL([DXR].[FilePath],
                                          [DAT].[FilePath]), ''))
            ELSE   UNC.[UNC]
                 + SUBSTRING(RTRIM(ISNULL(ISNULL([DXR].[FilePath],
                                                 [DAT].[FilePath]),
                                          '')),
                             3,
                             LEN(RTRIM(ISNULL(ISNULL([DXR].[FilePath],
                                                     [DAT].[FilePath]),
                                              ''))) - 2) END
  FROM [dbo].[DiaryAttachments] [DAT]
       LEFT OUTER JOIN [dbo].[DAIMXRef] [DXR]
                    ON [DXR].[TrackReference] = [DAT].[TrackReference]
       LEFT OUTER JOIN [dbo].[UNCAlias] UNC
                    ON SUBSTRING(ISNULL([DXR].[FilePath],
                                        [DAT].[FilePath]),
                                 2,
                                 1) = ':'
                   AND UNC.[Drive] = SUBSTRING(ISNULL([DXR].[FilePath],
                                                      [DAT].[FilePath]),
                                               1,
                                               1)
 WHERE [DAT].[TrackReference] = @TrackReference

SELECT @FilePath = REVERSE(@FilePath)
DECLARE @DOTPOS INT

SET @DOTPOS = CHARINDEX('.', @FilePath)
IF @DOTPOS = 0
  BEGIN
    SET @FilePath = REVERSE(@FilePath) + '_V' + CONVERT(VARCHAR, @MaxVersion)
  END
ELSe
  BEGIN
    SET @FilePath = REVERSE(SUBSTRING(@FilePath, @DOTPOS + 1, LEN(@FilePath) - @DotPos)) + '_V' + CONVERT(VARCHAR, @MaxVersion + 1) + REVERSE(SUBSTRING(@FilePath, 1, @DOTPOS))
  END

DECLARE @Handler VARCHAR(10)

SET @Handler = [dbo].[ky_NETFNGetLoggedOnHandler]()

INSERT
  INTO [dbo].[DiaryAttachmentAccess]
      ([TrackReference],
       [Handler],
       [Comment])
SELECT @TrackReference,
       @Handler,
       'Inserted New Version ' + Convert(VARCHAR(10), @MaxVersion + 1)
  FROM [dbo].[DiaryAttachments] [DAT]
 WHERE [DAT].[TrackReference] = @TrackReference

INSERT
  INTO [dbo].[DiaryAttachmentVersioning]
      ([TrackReference],
       [IMDocID],
       [Version],
       [FilePath],
       [Comments],
       [TimeStamp],
       [Handler])
SELECT [DAT].[TrackReference]                                                AS [TrackReference],
       ISNULL([DXR].[IMDocID], '')                                           AS [IMDocID],
       @MaxVersion + 1                                                       AS [Version],
       @FilePath                                                             AS [FILEPATH],
       @Comment                                                              AS [Comments],
       [DAT].[LASTACCESSDATE]                                                AS [TimeStamp],
       [DAT].[ENTEREDBY]                                                     AS [Handler]
  FROM [dbo].[DiaryAttachments] [DAT]
       LEFT OUTER JOIN [dbo].[DAIMXRef] [DXR]
                    ON [DXR].[TrackReference] = [DAT].[TrackReference]
 WHERE [DAT].[TrackReference] = @TrackReference

  SELECT TOP 1
         [DATV].[id],
         [DATV].[TrackReference],
         [DATV].[IMDocID],
         [DATV].[Version],
         [DATV].[Comments],
         [DATV].[FilePath],
         [DATV].[TimeStamp],
         [DATV].[Handler]
    FROM [dbo].[DiaryAttachmentVersioning] [DATV]
   WHERE [DATV].[TrackReference] = @TrackReference
ORDER BY [DATV].[Version] DESC

SET NOCOUNT OFF

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPAddBriefAttachment'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPAddBriefAttachment]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETSPAddBriefAttachment]
      (@BriefID INT,
       @TrackReference INT)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETSPAddBriefAttachment]                                                                        *
 *                                                                                                           *
 * Stores the Diary Location for a Brief File                                                                *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.5.1.2                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2017-08-27   Pino Carafa       Created                                                                    *
 * 2017-10-03   Pino Carafa       Added logging                                                              *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

DECLARE @Action VARCHAR(2000)

SET @Action = 'Document included in Brief (' + CONVERT(VARCHAR(10), @BriefID) + ')'

EXEC [dbo].[ky_NETSPDocumentAccess] @TrackReference, '', @Action

INSERT
  INTO [dbo].[BriefAttachments]
      ([BriefID],
       [TrackReference])
VALUES(@BriefID,
       @TrackReference)
END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETCASEDIARYFetchAttachments'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETCASEDIARYFetchAttachments]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETCASEDIARYFetchAttachments]
      (@DiaryId  INT)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETCASEDIARYFetchAttachments]                                                                   *
 *                                                                                                           *
 * Fetch Attachment list by diary number                                                                     *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.-.-.-                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2014-??-??   ???????????       Created                                                                    *
 * 2014-07-15   Pino Carafa       Add Version Information                                                    *
 * 2017-10-03   Pino Carafa       Reformatted for Framework / Coding Standards                               *
 *                                Added Logging                                                              *
 * 2019-03-21   Pino Carafa       Azure compatibility                                                        *
 * 2019-10-09   John Ginnane      KEYD-6197 - Replaced sysprocesses code with function                       *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

SET NOCOUNT ON

DECLARE @Handler VARCHAR(10)

SET @Handler = [dbo].[ky_NETFNGetLoggedOnHandler]()

INSERT
  INTO [dbo].[DiaryAttachmentAccess]
      ([TrackReference],
       [Handler],
       [Comment])
SELECT [DAT].[TrackReference],
       @Handler,
       'Listed for Diary'
  FROM [dbo].[DiaryAttachments] [DAT]
 WHERE [DAT].[DiaryID] = @DiaryId

  --Attachments list
  SELECT [DAT].[DiaryID]                                                                   AS [DIARYID],
         [DAT].[TrackReference]                                                            AS [TRACKREFERENCE],
         RTRIM(ISNULL([DAT].[CASECODE], ''))                                               AS [CASECODE],
         RTRIM(ISNULL([DAT].[NAME], ''))                                                   AS [NAME],
         RTRIM(ISNULL([DAT].[DOCUMENT], ''))                                               AS [DOCUMENT],
         CASE WHEN [UNC].[UNC] IS NULL
                   THEN RTRIM(ISNULL([DAT].[FILEPATH], ''))
              ELSE   [UNC].[UNC]
                   + SUBSTRING(RTRIM(ISNULL([DAT].[FILEPATH],
                                            '')),
                               3,
                                 LEN(RTRIM(ISNULL([DAT].[FILEPATH],
                                                  '')))
                               - 2) END                                                    AS [FILEPATH],
         RTRIM(ISNULL([DAT].[TYPE], ''))                                                   AS [TYPE],
         RTRIM(ISNULL(DCL.[CLASSDESCRIPTION], ''))                                         AS [CLASSDESCRIPTION],
         [DAT].[DATEENTERED]                                                               AS [DATEENTERED],
         RTRIM(ISNULL([DAT].[SYSTEM], 'N'))                                                AS [SYSTEM],
         RTRIM(ISNULL([DAT].[DOCCLASS], ''))                                               AS [DOCCLASS],
         [DV].[CurrentVersion]                                                             AS [CurrentVersion],
         RTRIM(ISNULL(DX.[IMDocID], ''))                                                   AS [IMDocID],
         RTRIM(ISNULL([DAT].[ENTEREDBY], ''))                                              AS [ENTEREDBY],
         RTRIM(ISNULL([DAT].[LASTACCESSBY], ''))                                           AS [LASTACCESSBY],
         RTRIM(ISNULL([DAT].[DICTATIONFILE], ''))                                          AS [DICTATIONFILE],
         RTRIM(ISNULL([DAT].[SOURCE], ''))                                                 AS [Source]
    FROM [dbo].[DiaryAttachments] [DAT]
         CROSS APPLY (SELECT   ISNULL(MAX([DAV].[Version]),
                                      0)
                             + 1 AS [CurrentVersion]
                        FROM [dbo].[DiaryAttachmentVersioning] [DAV]
                       WHERE [DAV].[TrackReference] = [DAT].[TRACKREFERENCE]) [DV]
         LEFT OUTER JOIN [dbo].[DocumentClasses] [DCL]
                      ON [DCL].[CLASSCODE] = [DAT].[DOCCLASS]
         LEFT OUTER JOIN [dbo].[UNCAlias] [UNC]
                      ON SUBSTRING([DAT].[FilePath], 2, 1) = ':'
                     AND [UNC].[Drive] = SUBSTRING([DAT].[FilePath], 1, 1)
         LEFT OUTER JOIN [dbo].[DAIMXRef] [DX]
                      ON [DX].[TrackReference] = [DAT].[TrackReference]
   WHERE [DAT].[DiaryID] = @DiaryId
ORDER BY [DAT].[TrackReference] DESC

SET NOCOUNT OFF

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETRevenueInsertintoDiary'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETRevenueInsertintoDiary]
  END
GO
-- See below for Create!

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETRevenueInsertintoDiary2'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETRevenueInsertintoDiary2]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETRevenueInsertintoDiary2]
      (@CASECODE          VARCHAR(20),
       @value             VARCHAR(100),
       @principal         VARCHAR(100),
       @interest          VARCHAR(100),
       @From              VARCHAR(100)='',
       @DebtDescription   VARCHAR(MAX),
       @TimeEntryId       INT,
       @LoginCode         VARCHAR(30))
 AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[KY_NETRevenueInsertintoDiary2]                                                                     *
 *                                                                                                           *
 * This Procedure is used to insert a Dairy Action whenever changes are made in debt details                 *
 * or while allocating payment in Debt Ledger.                                                               *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              [dbo].[ky_NETRevenueInsertintoDiary]                     *   *
 *   * First compatible version:                5.5.1.2                                                  *   *
 *   * Last compatible software version:        5.5.1.1                                                  *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2016-??-??   Magesh?????       Created                                                                    *
 * 2016-02-02   Magesh            Added 'NA' to the action status column in diary                            *
 * 2016-02-10   Magesh            Added Debtledger post payment description to Text2 column in diary table   *
 * 2016-02-16   Magesh            Added values for Priority, Publish, ActionType,                            *
 *                                ProcessType, DueDate, OrginalActionID                                      *
 * 2016-02-17   Magesh            Added Text1 column value based on payment type                             *
 * 2016-06-07   Arun V            Aligment issue fixed                                                       *
 * 2017-09-15   John              Added scenario for costs, so they are also report to case diary            *
 * 2017-10-02   John              KEYD-4768 Created v2 which accepts logged in user                          *
 *                                          and assigns action to them.                                      *
 * 2017-10-03   Pino Carafa       CODING STANDARDS ......                                                    *
 * 2017-10-03   Pino Carafa       "Handlers" table mis-spelled as Handles                                    *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN

DECLARE @ActionId    AS BIGINT
DECLARE @Subject    AS VARCHAR(500)
DECLARE @ActionCode AS VARCHAR(15)
DECLARE @FnCode     AS VARCHAR(10)
DECLARE @TeamCode   AS VARCHAR(10)

EXEC @ActionId = [dbo].[ky_NETGetNextActionID]

DECLARE @TimeConvert INT
SET @TimeConvert = dbo.ky_ConvertTimeToClarion(GETDATE())
SET @Principal =(SELECT CAST(@Principal AS NUMERIC(18,2)))
SET @interest =(SELECT CAST(@interest AS NUMERIC(18,2)))

SELECT @FnCode   = ISNULL([HAN].[Code], ''),
       @TeamCode = ISNULL([HAN].[Team], '')
  FROM [dbo].[Handlers] AS [HAN]
 WHERE [HAN].[Code] = @LoginCode

DECLARE @Text1 AS VARCHAR(5000)

IF @From ='AddDebtDetails'
  BEGIN
    SET @Text1 =   'Manual Amendment.'
                 + CHAR(13)
                 + 'Sequence No: '
                 + @Value
                 + CHAR(13)
                 + 'New Tax Amount: '
                 + CONVERT(VARCHAR(100), @Principal)
                 + CHAR(13)
                 + 'New Interest Amount: '
                 + CONVERT(VARCHAR(100), @interest )

    SET @ActionCode ='R004'

    SET @Subject ='Manual Debt Breakdown Amendment'
  END
ELSE IF @From = 'Cost'
  BEGIN
    SELECT @Text1 =   'Cost added: '
                    + LTRIM(RTRIM(@DebtDescription))
                    + CHAR(13)
                    + 'Amount: '
                    + LTRIM(RTRIM(@Value)),
           @ActionCode = 'C01',
           @Subject =   'Cost added: '
                      + LTRIM(RTRIM(@DebtDescription))
  END
ELSE
  BEGIN
    SET @Text1 = CASE WHEN @From='Direct Payment'
                           THEN 'Direct Payment' + CHAR(13)
                      WHEN @From ='Credit'
                           THEN 'Credit' + CHAR(13)
                      WHEN @From ='Set off'
                           THEN 'Set off' + CHAR(13)
                      ELSE 'Payment Received From Tax Payer' + CHAR(13) END

    SET @Text1 =   @Text1
                 + 'Amount:'
                 + @Value
                 + CHAR(13)
                 + 'Tax:'
                 + CONVERT(VARCHAR(100), @Principal)
                 + CHAR(13)
                 + 'Interest:'
                 + CONVERT(VARCHAR(100), @interest )

    SET @ActionCode ='O01'

    SET @Subject ='Payment received'
  END

DECLARE @DisplaytextValue VARCHAR(220)

SET @DisplaytextValue = [dbo].[ky_RemoveSpuriousWhitespace2](ISNULL(CONVERT(VARCHAR(MAX), @Text1), ''), 200)

INSERT
  INTO [dbo].[Diary]
      ([CASECODE],
       [DATE],
       [STATUS],
       [ACTIONCODE],
       [ACTIONSTATUS],
       [ACTIONTYPE],
       [PROCESSTYPE],
       [FNCODE],
       [TEAMCODE],
       [TEXT1],
       [TEXT2],
       [DELEGATEDFNR],
       [DELEGATEDDATE],
       [DELEGATEDBACKDATE],
       [DEFERRED],
       [DUEDATE],
       [PUBLISH],
       [DYSTARTTIME],
       [DYENDTIME],
       [DURATION],
       [ACTIONID],
       [ORGINALACTIONID],
       [PRIORITY],
       [HIGHLIGHTED],
       [MILESTEONETYPE],
       [ATTACHMENTS],
       [PROCESSSTATUS],
       [WORKPROCESS],
       [BILLABLE],
       [BILLDESCRIPTION],
       [EMAILADDRESS],
       [ADDRESSTO],
       [CCTo],
       [BCCTo],
       [EMAIL],
       [SUBJECT],
       [DELEGATIONSTATUS],
       [DRAFTBILLNO],
       [CHEQUEREQNO],
       [TxmSent],
       [Location],
       [HearingType],
       [ForCopy],
       [TxmDate],
       [TxmSeqNo],
       [DisplayText],
       [Flag])
VALUES(@CASECODE,
       GETDATE(),
       1,
       @ACTIONCODE,
       'NA',
       'A',
       'I',
       @FNCODE,
       @TEAMCODE,
       @TEXT1 ,
       @DebtDescription,
       null,
       null,
       null,
       null,
       GETDATE(),
       'N',
       @TIMECONVERT,
       @TIMECONVERT,
       0,
       @ACTIONID,
       @ACTIONID,
       'N',
       null,
       null,
       null,
       0,
       0,
       0,
       null,
       null,
       null,
       null,
       null,
       null,
       @Subject,
       0,
       0,
       0,
       0,
       null,
       'Not Applicable',
       null,
       null,
       0,
       @DisplaytextValue,
       0)

INSERT
  INTO [dbo].[DiaryDelegations]
      ([ACTIONID],
       [HANDLER],
       [TEAM],
       [DATE],
       [TIME],
       [DATER],
       [TIMER],
       [DUEDATE],
       [DUETIME],
       [REVIEW],
       [STATUS],
       [OWNER],
       [DELEGATE],
       [DELEGATESTATUS],
       [ActionType],
       [FromHandler],
       [ReturnedBy],
       [DelType])
VALUES(@ACTIONID,
       @FNCODE,
       @TEAMCODE,
       GETDATE(),
       @TIMECONVERT,
       null,
       @TIMECONVERT,
       GETDATE(),
       @TIMECONVERT,
       null,
       1,
       'Y',
       @FNCODE,
       0,
       null,
       @FNCODE,
       null,
       'Created')

UPDATE [TIE]
   SET [TIE].[ActionID] = @ActionID
  FROM [dbo].[TimeEntry] [TIE]
 WHERE [TIE].[RecordID] = @TimeEntryId

END

GO

CREATE PROCEDURE
       [dbo].[ky_NETRevenueInsertintoDiary]
      (@CASECODE          VARCHAR(20),
       @value             VARCHAR(100),
       @principal         VARCHAR(100),
       @interest          VARCHAR(100),
       @From              VARCHAR(100)='',
       @DebtDescription   VARCHAR(MAX),
       @TimeEntryId       INT)
 AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[KY_NETRevenueInsertintoDiary]                                                                      *
 *                                                                                                           *
 * This Procedure is used to insert a Dairy Action whenever changes are made in debt details                 *
 * or while allocating payment in Debt Ledger.                                                               *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.-.-.-                                                  *   *
 *   * Last compatible software version:        5.5.1.1                                                  *   *
 *   * Superseded by:                           [dbo].[ky_NETRevenueInsertintoDiary2]                    *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2016-??-??   Magesh?????       Created                                                                    *
 * 2016-02-02   Magesh            Added 'NA' to the action status column in diary                            *
 * 2016-02-10   Magesh            Added Debtledger post payment description to Text2 column in diary table   *
 * 2016-02-16   Magesh            Added values for Priority, Publish, ActionType,                            *
 *                                ProcessType, DueDate, OrginalActionID                                      *
 * 2016-02-17   Magesh            Added Text1 column value based on payment type                             *
 * 2016-06-07   Arun V            Aligment issue fixed                                                       *
 * 2017-09-15   John              Added scenario for costs, so they are also report to case diary            *
 * 2017-10-03   Pino Carafa       Call V2, establish Handler code from Process                               *
 * 2019-03-20   Pino Carafa       Azure compatibility                                                        *
 * 2019-10-09   John Ginnane      KEYD-6197 - Replaced sysprocesses code with function                       *
 *************************************************************************************************************/
BEGIN

DECLARE @Handler VARCHAR(10)

SET @Handler = [dbo].[ky_NETFNGetLoggedOnHandler]()

EXEC [dbo].[ky_NETRevenueInsertintoDiary2] @CASECODE,
                                           @value,
                                           @principal,
                                           @interest,
                                           @From,
                                           @DebtDescription,
                                           @TimeEntryId,
                                           @Handler
END
GO

-- Oops typo
IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPArciveDiaryAttachmentAccess'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPArciveDiaryAttachmentAccess]
  END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSPArchiveDiaryAttachmentAccess'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSPArchiveDiaryAttachmentAccess]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETSPArchiveDiaryAttachmentAccess]
      (@Days           INT)
AS
/*************************************************************************************************************
 *                                                                                                           *
 * [dbo].[ky_NETSPArchiveDiaryAttachmentAccess]                                                              *
 *                                                                                                           *
 * Archives Diary Attachment Access data so that it won't interfere with normal daily use of the system.     *
 *                                                                                                           *
 * Compatibility information - PLEASE update older versions if necessary to ensure the compatible software   *
 *                                    remains fully functional                                               *
 *   *****************************************************************************************************   *
 *   *                                                                                                   *   *
 *   * Supersedes:                              -                                                        *   *
 *   * First compatible version:                5.5.2.1                                                  *   *
 *   * Last compatible software version:        -                                                        *   *
 *   * Superseded by:                           -                                                        *   *
 *   *                                                                                                   *   *
 *   *****************************************************************************************************   *
 *                                                                                                           *
 * Modification History                                                                                      *
 * 2017-10-03   Pino Carafa       Created                                                                    *
 *                                                                                                           *
 *************************************************************************************************************/
BEGIN
SET NOCOUNT ON

DECLARE @CutoffDate DATETIME

IF @Days < 1
  BEGIN
    RETURN
  END

SET @Days = @Days - 1

SET @CutoffDate = DATEADD(DAY, -@Days, CONVERT(DATETIME, CONVERT(VARCHAR, GETDATE(), 112)))

INSERT
  INTO [dbo].[DiaryAttachmentAccessArchive]
      ([AccessDate],
       [Comment],
       [Handler],
       [TrackReference])
SELECT [AccessDate],
       [Comment],
       [Handler],
       [TrackReference]
  FROM [dbo].[DiaryAttachmentAccess] [DAA]
 WHERE [DAA].[AccessDate] <= @CutoffDate

DELETE [DAA]
  FROM [dbo].[DiaryAttachmentAccess] [DAA]
 WHERE [DAA].[AccessDate] <= @CutoffDate

SET NOCOUNT OFF

END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSaveRevenueClientCode2'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSaveRevenueClientCode2]
  END
GO

CREATE PROCEDURE [dbo].[ky_NETSaveRevenueClientCode2]
    @CLIENTNO1           VARCHAR(10)='',
    @CLIENTNO2           VARCHAR(10)='',
    @CLIENTNO3           VARCHAR(10)='',
    @ClientGroup         VARCHAR(10)='',
    @HoldDays            VARCHAR(10)='',
    @Team                VARCHAR(10)='',
    @FE                  VARCHAR(10)='',
    @Dept                VARCHAR(10)='',
    @WorkType            VARCHAR(10)='',
    @LastOutlay          VARCHAR(50)='',
    @CasePlan            VARCHAR(10)='',
    @NewReferral         VARCHAR(50)='',
    @AmdReferral         VARCHAR(50)='',
    @RejReferral         VARCHAR(50)='',
    @MandateNo           VARCHAR(50)='',
    @SolicitorNo         VARCHAR(50)='',
    @BankAcc             VARCHAR(50)='',
    @EnableClaimSystem   VARCHAR(10) = '',
    
    @EFTSheriffBankName         VARCHAR(100) = '',
    @EFTSheriffSortCode         VARCHAR(100) = '',
    @EFTSheriffAccountNumber    VARCHAR(100) = '',
    @EFTSheriffIBAN             VARCHAR(100) = '',
    @EFTSheriffBIC              VARCHAR(100) = '',
    @EFTSheriffOriginalIDNumber VARCHAR(100) = '',
    
    @EFTPayeeBankName         VARCHAR(100) = '',
    @EFTPayeeSortCode         VARCHAR(100) = '',
    @EFTPayeeAccountNumber    VARCHAR(100) = '',
    @EFTPayeeIBAN             VARCHAR(100) = '',
    @EFTPayeeBIC              VARCHAR(100) = ''
AS
/***************************************************************************************************************
 *  [dbo].[ky_NETSaveRevneueClientCode2]
 *
 *  This procedure is used to save the revenue client code. This is the updated version of [dbo].[ky_NETSaveRevneueClientCode]
 *
 *  2016-01-28   Magesh    Changed the @count to get top first count from the setting table for REV_CLIENTNO
 *  2016-02-08   Magesh    Introduced renvenue table and added the setting onto the table with unique [KeyName]
 *  2017-09-12   John      Created v2 copy of the procedure. Added three new fields
 *  2017-09-26   John      Added new fields that are saved. Still in same sprint as above so no new version - KEYD-4769
 ***************************************************************************************************************/
BEGIN
    IF EXISTS(SELECT TOP 1 1 FROM [dbo].[RevenueSettings] WHERE [KeyName] = 'ClientNo1')
        UPDATE [dbo].[RevenueSettings] SET [KeyValue] = @CLIENTNO1  WHERE [KeyName] = 'ClientNo1'
    ELSE
        INSERT INTO [dbo].[RevenueSettings] ([KeyName], [KeyValue]) VALUES ('ClientNo1', @CLIENTNO1)

    IF EXISTS(SELECT TOP 1 1 FROM [dbo].[RevenueSettings] WHERE [KeyName] = 'ClientNo2')
        UPDATE [dbo].[RevenueSettings] SET [KeyValue] = @CLIENTNO2  WHERE [KeyName] = 'ClientNo2'
    ELSE
        INSERT INTO [dbo].[RevenueSettings] ([KeyName], [KeyValue]) VALUES ('ClientNo2', @CLIENTNO2)

    IF EXISTS(SELECT TOP 1 1 FROM [dbo].[RevenueSettings] WHERE [KeyName] = 'ClientNo3')
        UPDATE [dbo].[RevenueSettings] SET [KeyValue] = @CLIENTNO3  WHERE [KeyName] = 'ClientNo3'
    ELSE
        INSERT INTO [dbo].[RevenueSettings] ([KeyName], [KeyValue]) VALUES ('ClientNo3', @CLIENTNO3)

    IF EXISTS(SELECT TOP 1 1 FROM [dbo].[RevenueSettings] WHERE [KeyName] = 'ClientGroup')
        UPDATE [dbo].[RevenueSettings] SET [KeyValue] = @ClientGroup  WHERE [KeyName] = 'ClientGroup'
    ELSE
        INSERT INTO [dbo].[RevenueSettings] ([KeyName], [KeyValue]) VALUES ('ClientGroup', @ClientGroup)

    IF EXISTS(SELECT TOP 1 1 FROM [dbo].[RevenueSettings] WHERE [KeyName] = 'HoldDays')
        UPDATE [dbo].[RevenueSettings] SET [KeyValue] = @HoldDays  WHERE [KeyName] = 'HoldDays'
    ELSE
        INSERT INTO [dbo].[RevenueSettings] ([KeyName], [KeyValue]) VALUES ('HoldDays', @HoldDays)

    IF EXISTS(SELECT TOP 1 1 FROM [dbo].[RevenueSettings] WHERE [KeyName] = 'Team')
        UPDATE [dbo].[RevenueSettings] SET [KeyValue] = @Team  WHERE [KeyName] = 'Team'
    ELSE
        INSERT INTO [dbo].[RevenueSettings] ([KeyName], [KeyValue]) VALUES ('Team', @Team)

    IF EXISTS(SELECT TOP 1 1 FROM [dbo].[RevenueSettings] WHERE [KeyName] = 'FE')
        UPDATE [dbo].[RevenueSettings] SET [KeyValue] = @FE  WHERE [KeyName] = 'FE'
    ELSE
        INSERT INTO [dbo].[RevenueSettings] ([KeyName], [KeyValue]) VALUES ('FE', @FE)

    IF EXISTS(SELECT TOP 1 1 FROM [dbo].[RevenueSettings] WHERE [KeyName] = 'Dept')
        UPDATE [dbo].[RevenueSettings] SET [KeyValue] = @Dept  WHERE [KeyName] = 'Dept'
    ELSE
        INSERT INTO [dbo].[RevenueSettings] ([KeyName], [KeyValue]) VALUES ('Dept', @Dept)

    IF EXISTS(SELECT TOP 1 1 FROM [dbo].[RevenueSettings] WHERE [KeyName] = 'WorkType')
        UPDATE [dbo].[RevenueSettings] SET [KeyValue] = @WorkType  WHERE [KeyName] = 'WorkType'
    ELSE
        INSERT INTO [dbo].[RevenueSettings] ([KeyName], [KeyValue]) VALUES ('WorkType', @WorkType)

    IF EXISTS(SELECT TOP 1 1 FROM [dbo].[RevenueSettings] WHERE [KeyName] = 'LastOutlay')
        UPDATE [dbo].[RevenueSettings] SET [KeyValue] = @LastOutlay  WHERE [KeyName] = 'LastOutlay'
    ELSE
        INSERT INTO [dbo].[RevenueSettings] ([KeyName], [KeyValue]) VALUES ('LastOutlay', @LastOutlay)

    IF EXISTS(SELECT TOP 1 1 FROM [dbo].[RevenueSettings] WHERE [KeyName] = 'CasePlan')
        UPDATE [dbo].[RevenueSettings] SET [KeyValue] = @CasePlan  WHERE [KeyName] = 'CasePlan'
    ELSE
        INSERT INTO [dbo].[RevenueSettings] ([KeyName], [KeyValue]) VALUES ('CasePlan', @CasePlan)

    IF EXISTS(SELECT TOP 1 1 FROM [dbo].[RevenueSettings] WHERE [KeyName] = 'NewReferral')
        UPDATE [dbo].[RevenueSettings] SET [KeyValue] = @NewReferral  WHERE [KeyName] = 'NewReferral'
    ELSE
        INSERT INTO [dbo].[RevenueSettings] ([KeyName], [KeyValue]) VALUES ('NewReferral', @NewReferral)

    IF EXISTS(SELECT TOP 1 1 FROM [dbo].[RevenueSettings] WHERE [KeyName] = 'AmdReferral')
        UPDATE [dbo].[RevenueSettings] SET [KeyValue] = @AmdReferral  WHERE [KeyName] = 'AmdReferral'
    ELSE
        INSERT INTO [dbo].[RevenueSettings] ([KeyName], [KeyValue]) VALUES ('AmdReferral', @AmdReferral)

    IF EXISTS(SELECT TOP 1 1 FROM [dbo].[RevenueSettings] WHERE [KeyName] = 'RejReferral')
        UPDATE [dbo].[RevenueSettings] SET [KeyValue] = @RejReferral  WHERE [KeyName] = 'RejReferral'
    ELSE
        INSERT INTO [dbo].[RevenueSettings] ([KeyName], [KeyValue]) VALUES ('RejReferral', @RejReferral)

    IF EXISTS(SELECT TOP 1 1 FROM [dbo].[RevenueSettings] WHERE [KeyName] = 'MandateNo')
        UPDATE [dbo].[RevenueSettings] SET [KeyValue] = @MandateNo  WHERE [KeyName] = 'MandateNo'
    ELSE
        INSERT INTO [dbo].[RevenueSettings] ([KeyName], [KeyValue]) VALUES ('MandateNo', @MandateNo)

    IF EXISTS(SELECT TOP 1 1 FROM [dbo].[RevenueSettings] WHERE [KeyName] = 'SolicitorNo')
        UPDATE [dbo].[RevenueSettings] SET [KeyValue] = @SolicitorNo  WHERE [KeyName] = 'SolicitorNo'
    ELSE
        INSERT INTO [dbo].[RevenueSettings] ([KeyName], [KeyValue]) VALUES ('SolicitorNo', @SolicitorNo)

    IF EXISTS(SELECT TOP 1 1 FROM [dbo].[RevenueSettings] WHERE [KeyName] = 'BankAcc')
        UPDATE [dbo].[RevenueSettings] SET [KeyValue] = @BankAcc  WHERE [KeyName] = 'BankAcc'
    ELSE
        INSERT INTO [dbo].[RevenueSettings] ([KeyName], [KeyValue]) VALUES ('BankAcc', @BankAcc)
    
    IF EXISTS(SELECT TOP 1 1 FROM [dbo].[RevenueSettings] WHERE [KeyName] = 'ClaimSystemEnabled')
        UPDATE [dbo].[RevenueSettings] SET [KeyValue] = @EnableClaimSystem WHERE [KeyName] = 'ClaimSystemEnabled'
    ELSE
		INSERT INTO [dbo].[RevenueSettings] ([KeyName], [KeyValue]) VALUES ('ClaimSystemEnabled', @EnableClaimSystem)
	
    IF EXISTS(SELECT TOP 1 1 FROM [dbo].[RevenueSettings] WHERE [KeyName] = 'EFTSheriffBankName')
        UPDATE [dbo].[RevenueSettings] SET [KeyValue] = @EFTSheriffBankName WHERE [KeyName] = 'EFTSheriffBankName'
    ELSE
		INSERT INTO [dbo].[RevenueSettings] ([KeyName], [KeyValue]) VALUES ('EFTSheriffBankName', @EFTSheriffBankName)
	
    IF EXISTS(SELECT TOP 1 1 FROM [dbo].[RevenueSettings] WHERE [KeyName] = 'EFTSheriffSortCode')
        UPDATE [dbo].[RevenueSettings] SET [KeyValue] = @EFTSheriffSortCode WHERE [KeyName] = 'EFTSheriffSortCode'
    ELSE
		INSERT INTO [dbo].[RevenueSettings] ([KeyName], [KeyValue]) VALUES ('EFTSheriffSortCode', @EFTSheriffSortCode)
	
    IF EXISTS(SELECT TOP 1 1 FROM [dbo].[RevenueSettings] WHERE [KeyName] = 'EFTSheriffAccountNumber')
        UPDATE [dbo].[RevenueSettings] SET [KeyValue] = @EFTSheriffAccountNumber WHERE [KeyName] = 'EFTSheriffAccountNumber'
    ELSE
		INSERT INTO [dbo].[RevenueSettings] ([KeyName], [KeyValue]) VALUES ('EFTSheriffAccountNumber', @EFTSheriffAccountNumber)
	
    IF EXISTS(SELECT TOP 1 1 FROM [dbo].[RevenueSettings] WHERE [KeyName] = 'EFTSheriffIBAN')
        UPDATE [dbo].[RevenueSettings] SET [KeyValue] = @EFTSheriffIBAN WHERE [KeyName] = 'EFTSheriffIBAN'
    ELSE
		INSERT INTO [dbo].[RevenueSettings] ([KeyName], [KeyValue]) VALUES ('EFTSheriffIBAN', @EFTSheriffIBAN)
	
    IF EXISTS(SELECT TOP 1 1 FROM [dbo].[RevenueSettings] WHERE [KeyName] = 'EFTSheriffBIC')
        UPDATE [dbo].[RevenueSettings] SET [KeyValue] = @EFTSheriffBIC WHERE [KeyName] = 'EFTSheriffBIC'
    ELSE
		INSERT INTO [dbo].[RevenueSettings] ([KeyName], [KeyValue]) VALUES ('EFTSheriffBIC', @EFTSheriffBIC)
	
    IF EXISTS(SELECT TOP 1 1 FROM [dbo].[RevenueSettings] WHERE [KeyName] = 'EFTSheriffOriginalIDNumber')
        UPDATE [dbo].[RevenueSettings] SET [KeyValue] = @EFTSheriffOriginalIDNumber WHERE [KeyName] = 'EFTSheriffOriginalIDNumber'
    ELSE
		INSERT INTO [dbo].[RevenueSettings] ([KeyName], [KeyValue]) VALUES ('EFTSheriffOriginalIDNumber', @EFTSheriffOriginalIDNumber)
	
    IF EXISTS(SELECT TOP 1 1 FROM [dbo].[RevenueSettings] WHERE [KeyName] = 'EFTPayeeBankName')
        UPDATE [dbo].[RevenueSettings] SET [KeyValue] = @EFTPayeeBankName WHERE [KeyName] = 'EFTPayeeBankName'
    ELSE
		INSERT INTO [dbo].[RevenueSettings] ([KeyName], [KeyValue]) VALUES ('EFTPayeeBankName', @EFTPayeeBankName)
	
    IF EXISTS(SELECT TOP 1 1 FROM [dbo].[RevenueSettings] WHERE [KeyName] = 'EFTPayeeSortCode')
        UPDATE [dbo].[RevenueSettings] SET [KeyValue] = @EFTPayeeSortCode WHERE [KeyName] = 'EFTPayeeSortCode'
    ELSE
		INSERT INTO [dbo].[RevenueSettings] ([KeyName], [KeyValue]) VALUES ('EFTPayeeSortCode', @EFTPayeeSortCode)
	
    IF EXISTS(SELECT TOP 1 1 FROM [dbo].[RevenueSettings] WHERE [KeyName] = 'EFTPayeeAccountNumber')
        UPDATE [dbo].[RevenueSettings] SET [KeyValue] = @EFTPayeeAccountNumber WHERE [KeyName] = 'EFTPayeeAccountNumber'
    ELSE
		INSERT INTO [dbo].[RevenueSettings] ([KeyName], [KeyValue]) VALUES ('EFTPayeeAccountNumber', @EFTPayeeAccountNumber)
	
    IF EXISTS(SELECT TOP 1 1 FROM [dbo].[RevenueSettings] WHERE [KeyName] = 'EFTPayeeIBAN')
        UPDATE [dbo].[RevenueSettings] SET [KeyValue] = @EFTPayeeIBAN WHERE [KeyName] = 'EFTPayeeIBAN'
    ELSE
		INSERT INTO [dbo].[RevenueSettings] ([KeyName], [KeyValue]) VALUES ('EFTPayeeIBAN', @EFTPayeeIBAN)
	
    IF EXISTS(SELECT TOP 1 1 FROM [dbo].[RevenueSettings] WHERE [KeyName] = 'EFTPayeeBIC')
        UPDATE [dbo].[RevenueSettings] SET [KeyValue] = @EFTPayeeBIC WHERE [KeyName] = 'EFTPayeeBIC'
    ELSE
		INSERT INTO [dbo].[RevenueSettings] ([KeyName], [KeyValue]) VALUES ('EFTPayeeBIC', @EFTPayeeBIC)
END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETSYSCONFetchConfigurationSettings2'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETSYSCONFetchConfigurationSettings2]
  END
GO

CREATE PROCEDURE [ky_NETSYSCONFetchConfigurationSettings2]
AS
/*
ky_NETSYSCONFetchConfigurationSettings2

This Stored procedure is used to fetch Configuration settings.

Created: 05June2014 07.58 PM (IST)
Modification History
-----------------------------------------------------------------------------------------
Date      | Modified By   | Description
-----------------------------------------------------------------------------------------
14Nov2014   Arun.v          Keyhouse Option tab added
21Nov2014   Arun.v          MaxClCode and MaxMatterLen added
23Dec2014   Arun.v          Added Your Ref field
02FEB2015   Arun.v          Added ExpInvoiceDateRequiredAtMatterStartup field
24FEB2015   Pino            Add "Allowassigntoclosed" and "RoundupUnits"
24FEB2015   Pino            Get Number of Minutes per Unit
30MAR2015   Pino            Number of Minutes per Unit also available in SAM3. Default 6.
27APR2015   Magesh          Added a parameter named deleteAction to get the user delete permission to settings table.
12May2015   Arun            Added AllowAnimation parameter
04JUN2015   Magesh          Added a parameter named DeleteAttachment to get the user Attachment delete permission to settings table.
10JUL2015   Magesh          Added a parameter to get SDLT global path.
11AUG2015   Sridharan       RDJ-135 - New Matter - Branch
15SEP2015   Sridharan       KEYD-2595 - Option to Take and Delete tasks in one click
20NOV2015   Magesh          KEYD-2891-Once an action is complete ALL documents for that action are READ ONLY
04JAN2016   magesh          Fetching revenue client code, holddays and client grp value from settings table
07FEB2016   John            Added extra setting to stop users who are not in the charge out rate list from posting time to those
                            clients/matters - http://jira1.dsrc.in:8080/browse/KEYD-3239
08FEB2016   Magesh          Introduced revenuesettings table and getting values from it.
08FEB2016   Pino            Had to fix this procedure because we have been ignoring KEYD-1166 AGAIN.
11MAR2016   Sridharan       KEYD-3426-URGENT Cannot eneter a long e-mail address in the admin em ID.
23MAR2016   Pino            KEYD-3468-Emailusername needs to be 100 characters
03May2016   Arun            http://jira1.dsrc.in:8080/browse/KEYD-3532 - Not allow Multiple draft bill
19Jul2016   Sridharan       Make Team Leader field configurable (System Configuration)
11AUG2016   Sridharan       Composite Billing Indicator
06OCT2016   Suriya          KEYD-4088: Added a new parameter for Enabling Dictation
18OCT2016   Sridharan       KEYD-4103: Export Report output to DOCX or XSLX
28NOV2016   Arun V          KEYD-4286: After upgrading to 5.4.6.1, the system not allows to put up multiple draft bills, 
                            however the option to "NOT allow multiple draft bills" is not ticked
02JUN2017   Suriya          KEYD-4630: System option to Switch ON/OFF the "Document Sync" functionality
13SEP2017   John            KEYD-4765: Created v2 file, as there are three new fields to do with Sheriff functionality.
22SEP2017   Arun V			KEYD-4804: Added Payment Receipt option in setting
27SEP2017   John            KEYD-4769: Added EFT fields for Sheriff system
*/
BEGIN
SET NOCOUNT ON
  DECLARE @PrecedentFolder VARCHAR(200)
  declare @StorageMethod VARCHAR(10)
  DECLARE @WpFolder VARCHAR(200)
  DECLARE @EmailOn VARCHAR(2)
  DECLARE @AllowMultipleDraftBills VARCHAR(2)
  DECLARE @SolicitorCategoryType VARCHAR(10)
  DECLARE @OtherPartyCategoryType VARCHAR(10)
  DECLARE @SupplierCategoryType VARCHAR(10)
  DECLARE @DefaultVATCode  VARCHAR(3)
  DECLARE @EstFeeRequiredAtMatterStartup bit
  DECLARE @ExpInvoiceDateRequiredAtMatterStartup bit
  DECLARE @BranchRequired   BIT
  DECLARE @WIPWarningLimit DECIMAL(18,2)
  DECLARE @DebtorWarningDaysLimit INT
  DECLARE @LastActionMonth INT
  DECLARE @LockupValue DECIMAL(18,2)
  DECLARE @ShowAlarm BIT

  DECLARE @IsSplitCheque VARCHAR(1)
  DECLARE @IsHideCheque VARCHAR(1)
  DECLARE @IsHideClient VARCHAR(1)
  DECLARE @IsNewDocSearch VARCHAR(1)
  DECLARE @IsAdestLink VARCHAR(1)
  DECLARE @IsPostReq VARCHAR(1)
  DECLARE @IsCaseBooks VARCHAR(1)
  DECLARE @IsAxxiaLinks VARCHAR(1)
  DECLARE @IsEnquiry VARCHAR(1)
  DECLARE @IsTimeCosting VARCHAR(1)
  DECLARE @IsBilling VARCHAR(1)
  DECLARE @IsChqueReq VARCHAR(1)
  DECLARE @IsClientReq VARCHAR(1)
  DECLARE @IsCaseMan VARCHAR(1)
  DECLARE @MaxClientCode VARCHAR(2)
  DECLARE @MaxMatterCode VARCHAR(2)
  DECLARE @LKVALUE VARCHAR(30)
  DECLARE @IsSAM4 BIT
  DECLARE @TUTNUMVALUE DECIMAL(17, 2)
  DECLARE @DELETEACTION BIT
  DECLARE @DELETEOTHERACTION BIT
  DECLARE @SDLTPath VARCHAR(500)
  DECLARE @UseTeamLeader VARCHAR(10)
  DECLARE @DocumentSyncVisible VARCHAR(10)
  DECLARE @CompositeBilling VARCHAR(10)
  DEClARE @EnableDictation VARCHAR(10)
  DECLARE @ExportReportsToLegacyFormats VARCHAR(10)

  SET @IsSAM4 = [dbo].[ISSAM4]()

  DECLARE @TimeDaybookLimit     VARCHAR(100)
  DECLARE @OSInvoiceLimit       VARCHAR(100)
  DECLARE @CurrentWIPLimit      VARCHAR(100)
  DECLARE @DraftInvoiceLimit    VARCHAR(100)
  DECLARE @RoundupUnits         BIT
  DECLARE @AllowAssignToClosed  BIT
  DECLARE @AllowAnimation BIT


  DECLARE @EmailUserName                VARCHAR(100)
  DECLARE @EmailPassword                VARCHAR(25)
  DECLARE @EmailSMTP                    VARCHAR(25)
  DECLARE @EmailPort                    VARCHAR(7)
  DECLARE @EmailEnableSSL               BIT
  DECLARE @EmailUseDefaultCredentials   BIT
  DECLARE @StrongRoomAdminEmail         VARCHAR(100)
  DECLARE @StrongRoomAdmin              VARCHAR(20)
  DECLARE @FileNoLabelText              VARCHAR(20)
  DECLARE @DeleteAttachment             BIT
  DECLARE @ReadyonlyAttachment          BIT
  DECLARE @ShowAddAssociates            BIT
  DECLARE @AddAssociatesDefaultValue    VARCHAR(50)
  DECLARE @ChargeOutRateCheck           BIT
  DECLARE @NotAllowMultipleDraftBill	BIT
  DECLARE @EnablePaymentReceipt			BIT
  
  SET @EmailEnableSSL = 0
  SET @EmailUseDefaultCredentials = 0
  -- Default Values
  SET @WIPWarningLimit                          = 0
  SET @DebtorWarningDaysLimit                   = 0
  SET @LastActionMonth                          = 0
  SET @DebtorWarningDaysLimit                   = '0'
  SET @TimeDaybookLimit                         = '0'
  SET @OSInvoiceLimit                           = '0'
  SET @CurrentWIPLimit                          = '0'
  SET @DraftInvoiceLimit                        = '0'
  SET @RoundupUnits                             = 0
  SET @AllowAssignToClosed                      = 0
  IF @IsSAM4 = 0
    BEGIN
      SET @MaxClientCode                        = '6'
      SET @MaxMatterCode                        = '4'
    END
  ELSE
    BEGIN
      SET @MaxClientCode                        = '8'
      SET @MaxMatterCode                        = '8'
    END
  SET @LockupValue                              = 0
  SET @ShowAlarm                                = 0
  SET @EstFeeRequiredAtMatterStartup            = 0
  SET @ExpInvoiceDateRequiredAtMatterStartup    = 0
  SET @BranchRequired                           = 0
  SET @DELETEACTION                             = 0
  SET @DELETEOTHERACTION                        = 0
  SET @DeleteAttachment                         = 1
  SET @AllowAnimation                           = 'False'
  SET @ReadyonlyAttachment                        = 0
  SET @ShowAddAssociates                        = 0
  SET @AddAssociatesDefaultValue                = ''
  SET @ChargeOutRateCheck                       = 0
  SET @NotAllowMultipleDraftBill                = 1
  SET @UseTeamLeader                            = 'False'
  SET @DocumentSyncVisible                      = 'False'
  SET @CompositeBilling                         = 'N'
  SET @EnableDictation                          = 'False'
  SET @ExportReportsToLegacyFormats             = 'False'
  SET @EnablePaymentReceipt						= 'False'

  --And try get them from the database. If not present, default values will remain intact.
  SELECT @WIPWarningLimit                       = CASE WHEN [SET].[KeyName] = 'EstFeeWarningLimit'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @WIPWarningLimit END,
         @DebtorWarningDaysLimit                = CASE WHEN [SET].[KeyName] = 'DebtorWarningDaysLimit'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @DebtorWarningDaysLimit END,
         @MaxClientCode                         = CASE WHEN [SET].[KeyName] = 'MaxClCode'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @MaxClientCode END,
         @MaxMatterCode                         = CASE WHEN [SET].[KeyName] = 'MaxMatNum'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @MaxMatterCode END,
         @LockupValue                           = CASE WHEN [SET].[KeyName] = 'LockupUpValue'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @LockupValue END,
         @ShowAlarm                             = CASE WHEN [SET].[KeyName] = 'ShowAlarm'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @ShowAlarm END,
         @EstFeeRequiredAtMatterStartup         = CASE WHEN [SET].[KeyName] = 'EstFeeRequiredatMatterSetup'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @EstFeeRequiredAtMatterStartup END,
         @ExpInvoiceDateRequiredAtMatterStartup = CASE WHEN [SET].[KeyName] = 'ExpInvoiceDateRequiredatMatterSetup'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @ExpInvoiceDateRequiredAtMatterStartup END,
         @LastActionMonth                       = CASE WHEN [SET].[KeyName] = 'LastActionMonth'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @LastActionMonth END,
         @TimeDaybookLimit                      = CASE WHEN [SET].[KeyName] = 'TimeDayBookLimit'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @TimeDaybookLimit END,
         @OSInvoiceLimit                        = CASE WHEN [SET].[KeyName] = 'OSInvoiceLimit'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @OSInvoiceLimit END,
         @CurrentWIPLimit                       = CASE WHEN [SET].[KeyName] = 'CurrentWIPLimit'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @CurrentWIPLimit END,
         @DraftInvoiceLimit                     = CASE WHEN [SET].[KeyName] = 'DraftInvoiceLimit'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @DraftInvoiceLimit END,
         @RoundupUnits                          = CASE WHEN [SET].[KeyName] = 'RoundupUnits'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @RoundupUnits END,
         @AllowAssignToClosed                   = CASE WHEN [SET].[KeyName] = 'AllowAssignToClosed'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @AllowAssignToClosed END,
         @DELETEACTION                            = CASE WHEN [SET].[KeyName] = 'AllowToDeleteAction'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @DELETEACTION END,
         @DELETEOTHERACTION                        = CASE WHEN [SET].[KeyName] = 'AllowToDeleteOtherAction'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @DELETEOTHERACTION END,
         @AllowAnimation                        = CASE WHEN [SET].[KeyName] = 'AllowAnimation'
                                                       THEN [SET].[KeyValue]
                                                       ELSE @AllowAnimation END,
        @EmailUserName                             = CASE WHEN [SET].[KeyName] = 'EmailUserName'
                                                       THEN [SET].[KeyValue]
                                                       ELSE @EmailUserName END,
        @EmailPassword                             = CASE WHEN [SET].[KeyName] = 'EmailPassword'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @EmailPassword END,
        @EmailSMTP                                 = CASE WHEN [SET].[KeyName] = 'EMailSMTP'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @EmailSMTP END,
        @EmailPort                                 = CASE WHEN [SET].[KeyName] = 'EmailPort'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @EmailPort END,
        @EmailEnableSSL                         = CASE WHEN [SET].[KeyName] = 'EmailEnableSSL'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @EmailEnableSSL END,
        @EmailUseDefaultCredentials             = CASE WHEN [SET].[KeyName] = 'EmailUseDefaultCredentials'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @EmailUseDefaultCredentials END,
        @StrongRoomAdminEmail                    = CASE WHEN [SET].[KeyName] = 'StrongRoomAdminEmail'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @StrongRoomAdminEmail END,
        @StrongRoomAdmin                        = CASE WHEN [SET].[KeyName] = 'StrongRoomAdmin'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @StrongRoomAdmin END,
        @FileNoLabelText                        = CASE WHEN [SET].[KeyName] = 'FileNoLabelText'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @FileNoLabelText END,
        @DeleteAttachment                        =CASE WHEN [SET].[KeyName] = 'AllowToDeleteActionAttachment'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @DeleteAttachment END,
        @SDLTPath                                =CASE WHEN [SET].[KeyName] = 'SDLTGlobalPath'
                                                            THEN [SET].[KeyValue]
                                                       ELSE '' END,
        @BranchRequired                            =CASE WHEN [set].[KeyName]='NewCaseWizardBranch'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @BranchRequired END,
        @ReadyonlyAttachment                     =CASE WHEN [set].[KeyName]='MakeFileReadonlyWhenComplete'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @ReadyonlyAttachment END,
        @ShowAddAssociates                        =CASE WHEN [set].[KeyName] = 'ShowAddAssociates'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @ShowAddAssociates END,
        @AddAssociatesDefaultValue                =CASE WHEN [set].[KeyName] = 'AddAssociatesDefaultValue'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @AddAssociatesDefaultValue END,
        @ChargeOutRateCheck                        = CASE WHEN [SET].[KeyName] = 'ChargeOutRateCheck'
                                                       THEN [SET].[KeyValue]
                                                       ELSE @ChargeOutRateCheck END,
        @NotAllowMultipleDraftBill                = CASE WHEN [SET].[KeyName] = 'NotAllowMulipleDraftBill'
                                                       THEN [SET].[KeyValue]
                                                       ELSE @NotAllowMultipleDraftBill
                                                  END,
        @UseTeamLeader                               = CASE WHEN [SET].[KeyName] = 'UseTeamLeader'
                                                       THEN [SET].[KeyValue]
                                                       ELSE @UseTeamLeader
                                                  END,
        @DocumentSyncVisible                         = CASE WHEN [SET].[KeyName] = 'DocumentSyncVisible'
                                                       THEN [SET].[KeyValue]
                                                       ELSE @DocumentSyncVisible
                                                  END,
        @CompositeBilling                            = CASE WHEN [SET].[KeyName] = 'CompositeBilling'
                                                       THEN [SET].[KeyValue]
                                                       ELSE @CompositeBilling
                                                  END,
        @EnableDictation                            = CASE WHEN [SET].[KeyName] = 'EnableDictation'
                                                       THEN [SET].[KeyValue]
                                                       ELSE @EnableDictation
                                                  END,
		@ExportReportsToLegacyFormats               = CASE WHEN [SET].[KeyName] = 'ExportReportsToLegacyFormats'
                                                       THEN [SET].[KeyValue]
                                                       ELSE @ExportReportsToLegacyFormats
                                                  END,
         @EnablePaymentReceipt				        = CASE WHEN [SET].[KeyName] = 'EnablePaymentReceipt'
													   THEN [SET].[KeyValue]
													   ELSE @EnablePaymentReceipt
												  END

    FROM [dbo].[Settings] [SET]



    --IF @IsSAM4 = 1
    --  BEGIN
        SELECT @TUTNUMVALUE = CTN.[NumValue]
          FROM [dbo].[CtrlNum] CTN
         WHERE CTN.[Code] = 'TimeUnit'
    --  END

    SET @TUTNUMVALUE = CASE WHEN ISNULL(@TUTNUMVALUE, 0) = 0 THEN 6 ELSE @TUTNUMVALUE END

  --PINO 2015-02-24 replaced 9 select statements with single one above.
  --SET @WIPWarningLimit  = (SELECT KeyValue  FROM Settings WHERE KeyName='EstFeeWarningLimit')
  --SET @DebtorWarningDaysLimit = (SELECT KeyValue  FROM Settings WHERE KeyName='DebtorWarningDaysLimit')
  --SET @LastActionMonth = (SELECT KeyValue  FROM Settings WHERE KeyName='LastActionMonth')

  --SET @MaxClientCode  = (SELECT KeyValue  FROM Settings WHERE KeyName='MaxClCode')
  --SET @MaxMatterCode  = (SELECT KeyValue  FROM Settings WHERE KeyName='MaxMatNum')

  --SET @LKVALUE =  (SELECT KeyValue FROM Settings WHERE KeyName='LockupUpValue')
  --SET @LockupValue = CONVERT(DECIMAL(18,2),@LKVALUE)

  --SET @ShowAlarm = (SELECT KeyValue  FROM Settings WHERE KeyName='ShowAlarm')
  --SET @ShowAlarm = ISNULL(@ShowAlarm, 0)
  --SET @EstFeeRequiredAtMatterStartup = (SELECT ISNULL(KeyValue, 'False')  FROM Settings WHERE KeyName='EstFeeRequiredatMatterSetup')
  --SET @EstFeeRequiredAtMatterStartup = ISNULL(@EstFeeRequiredAtMatterStartup, 0)

  --SET @ExpInvoiceDateRequiredAtMatterStartup = (SELECT ISNULL(KeyValue, 'False')  FROM Settings WHERE KeyName='ExpInvoiceDateRequiredatMatterSetup')
  --SET @ExpInvoiceDateRequiredAtMatterStartup = ISNULL(@ExpInvoiceDateRequiredAtMatterStartup, 0)


  SET @SolicitorCategoryType = (SELECT TOP 1 CatType  FROM Category where Code = (select TOP 1 Solicitor from [control] ))
  SET @OtherPartyCategoryType = (SELECT TOP 1 CatType  FROM Category where Code = (select TOP 1 Othparty from [control] ))
  SET @SupplierCategoryType = (SELECT TOP 1 CatType  FROM Category where Code = (select TOP 1 SupplierCategory from [control] ))

  SELECT
   @PrecedentFolder= ScprecedentFolder, @StorageMethod= Storagemethod, @WpFolder=ScWpFolder, @EmailOn=EmailOn
  FROM
   systemConfig

  --Allow Multiple Draft Bills
  SELECT
   @AllowMultipleDraftBills = Indicator5, @DefaultVATCode = REQUESTER
  , @IsSplitCheque =SPLITCHEQUES
  , @IsHideCheque  = INDICATOR3
  , @IsHideClient = INDICATOR4
  , @IsNewDocSearch = INDICATOR6
  , @IsAdestLink  = INDICATOR7       , @IsPostReq  = INDICATOR8
  , @IsCaseBooks = INDICATOR9
  , @IsAxxiaLinks  = INDICATOR10
  , @IsEnquiry  = ENQUIRY
  , @IsTimeCosting = TIMECOSTING
  , @IsBilling = BILLING
  , @IsChqueReq = CHEQUEREQ
  , @IsClientReq = CLIENTREQ
  , @IsCaseMan  = CASEMAN
  FROM
   DEsktopControl

  SELECT
    Name,
    Vatno,
    Debtors,
    Creditor,
    Outlay,
    VAT,
    PL,
    Client,
    FeesWriteOff,
    OutlayWriteOff,
    BillsWriteOff,
    DefaultOfficeAc,
    DefaultClientAc,
    DefaultClientDEP,
    MiscOutlay,
    PettyCash,
    WithTax,
    [YEAR],
    [CurPer],
    Outlaybudget,
    LastBill,
    VatExclude,
    DefaultVAT,
    FeeExclude,
    MiscFe,
    EnableFNL,
    'VAT' as VAT,
    AlarmStartupLaunch AS LaunchAlarmSystemAtStartup,
    'ReserveLedger' as ReserveLedger,
    UserPrompt1,
    userPrompt2,
    UserPrompt3,
    Solicitor,
    Othparty,
    SupplierCategory,
    DefaultCategory,
    AutoNumber,
    AutoAlphaClientNumbering,
    ThirdPartyLimit,
    DateFutValidation,
    DatePastValidation,
    NextFile,
    ClosedFileNo,
    ChequePrinting,
    EnableCP,
    EIConvert,
    @PrecedentFolder as PrecedentFolder,
    @StorageMethod as StorageMethod,
    @WpFolder as WPFolder,
    DocumentNamingMethod as DocumentNamingMethod,
    @EmailOn as EmailOn,
    @AllowMultipleDraftBills as AllowMultipleDraftBills,
    @SolicitorCategoryType as SolicitorCategory,
    @OtherPartyCategoryType as OtherPartyCategory,
    @SupplierCategoryType as SupplierCategoryType,
    @DefaultVATCode as DefaultVATCode,
    @WIPWarningLimit AS WIPWarningLimit,
    @DebtorWarningDaysLimit as DebtorWarningDaysLimit,
    @LastActionMonth  as LastActionMonth,
    CONVERT(DECIMAL(18,2),@LockupValue)  as LockupValue,
    @ShowAlarm as ShowAlarm,
    @EstFeeRequiredAtMatterStartup as EstFeeRequiredAtMatterStartup,
    @ExpInvoiceDateRequiredAtMatterStartup as ExpInvoiceDateRequiredAtMatterStartup,
    ENABLECDB,
    ENABLEDW,
    ENABLEFM,
    RECINSTALLED,
    NOBAR,
    WORDPERFECT,
    OPSISLINK,
    TIMECOSTING,
    FORCEFE,
    UK,
    USEOUTLAYTYPE,
    PENDINGOUTLAY,
    INDICATOR18,
    INDICATOR19,
    INDICATOR20,
    DEMOEXPIRE,
    MAXNUMBEROFMATTERS,
    LABELPRINTER,
    YEARENDCOMPLETED,
    INDICATOR22,
    INDICATOR23,
    INDICATOR24,
    INDICATOR25,
    INDICATOR26,
    INDICATOR27,
    INDICATOR28,
    INDICATOR29,
    INDICATOR30,
    FEESREPSEQ,
    @IsSplitCheque as SplitCheque,
    @IsHideCheque as HideCheque,
    @IsHideClient as HideClient,
    @IsNewDocSearch as NewDocSearch,
    @IsAdestLink as AdestLink,
    @IsPostReq as IsPostReq,
    @IsCaseBooks as IsCaseBook,
    @IsAxxiaLinks as AxxiaLinks,
    @IsEnquiry as Enquiry,
    @IsTimeCosting as DOTimeCosting,
    @IsBilling as Billing,
    @IsChqueReq as IsChequeReq,
    @IsClientReq as IsClientReq,
    @IsCaseMan as CaseMan,
    AUTONUMBER as AutoNumber,
    ISNULL(@MaxClientCode,6) AS MaxClientCode,
    ISNULL(@MaxMatterCode,4) as MaxMatterCode,
    YourRef,
    @DELETEACTION as DeleteAction,
    @DELETEOTHERACTION as DeleteOtherAction,
    @AllowAnimation AS AllowAnimation,
    @EmailUserName AS [EmailUserName],
    @EmailPassword AS [EmailPassword],
    @EmailSMTP AS [EmailSMTP],
    @EmailPort AS [EmailPort],
    @EmailEnableSSL AS [EmailEnableSSL],
    @EmailUseDefaultCredentials AS [UseDefaultCredentials],
    @StrongRoomAdminEmail AS [StrongRoomAdminEmail],
    @StrongRoomAdmin AS [StrongRoomAdmin],
    ISNULL(@FileNoLabelText,'File No') AS [FileNoLabelText],
    @DeleteAttachment AS DeleteAttachement,
    ISNULL(@SDLTPath,'') AS SDLTPath,
    @BranchRequired AS BranchRequired,
    @ReadyonlyAttachment AS [ReadyonlyAttachment],
    @ShowAddAssociates   AS ShowAddAssociates,
    @AddAssociatesDefaultValue AS AddAssociatesDefaultValue,
    @ChargeOutRateCheck AS ChargeOutRateCheck,
    @NotAllowMultipleDraftBill AS NotAllowMutlipleDraftBill,
    @UseTeamLeader as UseTeamLeader,
    @DocumentSyncVisible as DocumentSyncVisible,
    @CompositeBilling as CompositeBilling,
    @EnableDictation as EnableDictation,
	@ExportReportsToLegacyFormats as ExportReportsToLegacyFormats,
	@EnablePaymentReceipt as EnablePaymentReceipt
   FROM
   [CONTROL]

SELECT @TimeDaybookLimit    AS [TimeDayBookLimit],
       @OSInvoiceLimit      AS [OSInvoiceLimit],
       @CurrentWIPLimit     AS [CurrentWIPLimit],
       @DraftInvoiceLimit   AS [DraftInvoiceLimit],
       @RoundupUnits        AS [RoundupUnits],
       @AllowAssignToClosed AS [AllowAssignToClosed],
       @TUTNUMVALUE         AS [MinutesPerUnit]

  --fetch revenue details
     --fetch revenue details

    --KEYD-1166.... AGAIN.... 
    DECLARE @CLIENTNO1              VARCHAR(20)
    DECLARE @CLIENTNO2              VARCHAR(20)
    DECLARE @CLIENTNO3              VARCHAR(20)
    DECLARE @ClientGroup            VARCHAR(10)
    DECLARE @HoldDays               VARCHAR(10)
    DECLARE @Team                   VARCHAR(10)
    DECLARE @FE                     VARCHAR(10)
    DECLARE @Dept                   VARCHAR(10)
    DECLARE @WorkType               VARCHAR(10)
    DECLARE @LastOutlay             VARCHAR(50)
    DECLARE @CasePlan               VARCHAR(10)
    DECLARE @NewReferral            VARCHAR(50)
    DECLARE @AmdReferral            VARCHAR(50)
    DECLARE @RejReferral            VARCHAR(50)
    DECLARE @MandateNo              VARCHAR(50)
    DECLARE @SolicitorNo            VARCHAR(50)
    DECLARE @BankAcc                VARCHAR(50)
    DECLARE @ClaimSystemEnabled     VARCHAR(10)
    
    DECLARE @EFTSheriffBankName         VARCHAR(100)
    DECLARE @EFTSheriffSortCode         VARCHAR(100)
    DECLARE @EFTSheriffAccountNumber    VARCHAR(100)
    DECLARE @EFTSheriffIBAN             VARCHAR(100)
    DECLARE @EFTSheriffBIC              VARCHAR(100)
    DECLARE @EFTSheriffOriginalIDNumber VARCHAR(100)
    
    DECLARE @EFTPayeeBankName           VARCHAR(100)
    DECLARE @EFTPayeeSortCode           VARCHAR(100)
    DECLARE @EFTPayeeAccountNumber      VARCHAR(100)
    DECLARE @EFTPayeeIBAN               VARCHAR(100)
    DECLARE @EFTPayeeBIC                VARCHAR(100)

    --KEYD-1166.... AGAIN.... 
    SELECT @CLIENTNO1           = '',
           @CLIENTNO2           = '',
           @CLIENTNO3           = '',
           @ClientGroup         = '',
           @HoldDays            = '',
           @Team                = '',
           @FE                  = '',
           @Dept                = '',
           @WorkType            = '',
           @LastOutlay          = '',
           @CasePlan            = '',
           @NewReferral         = '',
           @AmdReferral         = '',
           @RejReferral         = '',
           @MandateNo           = '',
           @SolicitorNo         = '',
           @BankAcc             = '',
           @ClaimSystemEnabled  = 'False',
           
           @EFTSheriffBankName         = '',
           @EFTSheriffSortCode         = '',
           @EFTSheriffAccountNumber    = '',
           @EFTSheriffIBAN             = '',
           @EFTSheriffBIC              = '',
           @EFTSheriffOriginalIDNumber = '',
           
           @EFTPayeeBankName         = '',
           @EFTPayeeSortCode         = '',
           @EFTPayeeAccountNumber    = '',
           @EFTPayeeIBAN             = '',
           @EFTPayeeBIC              = ''
           
 
    SELECT
        @CLIENTNO1                                = CASE WHEN [SET].[KeyName] = 'ClientNo1'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @CLIENTNO1 END,
        @CLIENTNO2                                = CASE WHEN [SET].[KeyName] = 'ClientNo2'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @CLIENTNO2 END,
        @CLIENTNO3                                = CASE WHEN [SET].[KeyName] = 'ClientNo3'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @CLIENTNO3 END,
        @ClientGroup                            = CASE WHEN [SET].[KeyName] = 'ClientGroup'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @ClientGroup  END,
        @HoldDays                                = CASE WHEN [SET].[KeyName] = 'HoldDays'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @HoldDays END,
        @Team                                    = CASE WHEN [SET].[KeyName] = 'Team'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @Team END,
        @FE                                        = CASE WHEN [SET].[KeyName] = 'FE'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @FE END,
        @Dept                                    = CASE WHEN [SET].[KeyName] = 'Dept'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @Dept END,
        @WorkType                                = CASE WHEN [SET].[KeyName] = 'WorkType'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @WorkType END,
        @LastOutlay                                = CASE WHEN [SET].[KeyName] = 'LastOutlay'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @LastOutlay END,
        @CasePlan                                = CASE WHEN [SET].[KeyName] = 'CasePlan'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @CasePlan END,
        @NewReferral                            = CASE WHEN [SET].[KeyName] = 'NewReferral'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @NewReferral END,
        @AmdReferral                            = CASE WHEN [SET].[KeyName] = 'AmdReferral'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @AmdReferral END,
        @MandateNo                              = CASE WHEN [SET].[KeyName] = 'MandateNo'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @MandateNo END,
        @SolicitorNo                            = CASE WHEN [SET].[KeyName] = 'SolicitorNo'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @SolicitorNo END,
        @BankAcc                                = CASE WHEN [SET].[KeyName] = 'BankAcc'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @BankAcc END,
        @RejReferral                            = CASE WHEN [SET].[KeyName] = 'RejReferral'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @RejReferral END,
        @ClaimSystemEnabled                     = CASE WHEN [SET].[KeyName] = 'ClaimSystemEnabled'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @ClaimSystemEnabled END,
                                                       
        @EFTSheriffBankName                     = CASE WHEN [SET].[KeyName] = 'EFTSheriffBankName'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @EFTSheriffBankName END,
        @EFTSheriffSortCode                     = CASE WHEN [SET].[KeyName] = 'EFTSheriffSortCode'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @EFTSheriffSortCode END,
        @EFTSheriffAccountNumber                = CASE WHEN [SET].[KeyName] = 'EFTSheriffAccountNumber'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @EFTSheriffAccountNumber END,
        @EFTSheriffIBAN                         = CASE WHEN [SET].[KeyName] = 'EFTSheriffIBAN'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @EFTSheriffIBAN END,
        @EFTSheriffBIC                          = CASE WHEN [SET].[KeyName] = 'EFTSheriffBIC'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @EFTSheriffBIC END,
        @EFTSheriffOriginalIDNumber             = CASE WHEN [SET].[KeyName] = 'EFTSheriffOriginalIDNumber'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @EFTSheriffOriginalIDNumber END,
                                                       
        @EFTPayeeBankName                       = CASE WHEN [SET].[KeyName] = 'EFTPayeeBankName'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @EFTPayeeBankName END,
        @EFTPayeeSortCode                       = CASE WHEN [SET].[KeyName] = 'EFTPayeeSortCode'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @EFTPayeeSortCode END,
        @EFTPayeeAccountNumber                  = CASE WHEN [SET].[KeyName] = 'EFTPayeeAccountNumber'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @EFTPayeeAccountNumber END,
        @EFTPayeeIBAN                           = CASE WHEN [SET].[KeyName] = 'EFTPayeeIBAN'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @EFTPayeeIBAN END,
        @EFTPayeeBIC                            = CASE WHEN [SET].[KeyName] = 'EFTPayeeBIC'
                                                            THEN [SET].[KeyValue]
                                                       ELSE @EFTPayeeBIC END
    FROM
        REVENUESETTINGS [SET]



    SELECT
        @CLIENTNO1           AS [CLIENTNO1],
        @CLIENTNO2           AS [CLIENTNO2],
        @CLIENTNO3           AS [CLIENTNO3],
        @ClientGroup         AS [CLIENTGROUP],
        @HoldDays            AS [HOLDDAYS],
        @Team                AS [TEAM],
        @FE                  AS [FE],
        @Dept                AS [DEPT],
        @WorkType            AS [WORKTYPE],
        @LastOutlay          AS [LASTOUTLAY],
        @CasePlan            AS [CASEPLAN],
        @NewReferral         AS [NEWREFERRAL],
        @AmdReferral         AS [AMDREFERRAL],
        @RejReferral         AS [REJREFERRAL],
        @MandateNo           AS [MANDATENO],
        @SolicitorNo         AS [SOLICITORNO],
        @BankAcc             AS [BANKACC],
        @ClaimSystemEnabled  AS [ClaimSystemEnabled],
        
        @EFTSheriffBankName         AS [EFTSheriffBankName],
        @EFTSheriffSortCode         AS [EFTSheriffSortCode],
        @EFTSheriffAccountNumber    AS [EFTSheriffAccountNumber],
        @EFTSheriffIBAN             AS [EFTSheriffIBAN],
        @EFTSheriffBIC              AS [EFTSheriffBIC],
        @EFTSheriffOriginalIDNumber AS [EFTSheriffOriginalIDNumber],
        
        @EFTPayeeBankName         AS [EFTPayeeBankName],
        @EFTPayeeSortCode         AS [EFTPayeeSortCode],
        @EFTPayeeAccountNumber    AS [EFTPayeeAccountNumber],
        @EFTPayeeIBAN             AS [EFTPayeeIBAN],
        @EFTPayeeBIC              AS [EFTPayeeBIC]


SET NOCOUNT OFF
END

GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETUpdateEFTDetails2'
      AND SO.[type] = 'P')
  BEGIN
    DROP PROCEDURE [dbo].[ky_NETUpdateEFTDetails2]
  END
GO

CREATE PROCEDURE
       [dbo].[ky_NETUpdateEFTDetails2]
      (@EFTYN                 VARCHAR(1),
       @EFTLimit              DECIMAL(18,2),
       @NoTransaction         INT,
       @BankFileName          VARCHAR(500),
       @FileLocation          VARCHAR(500),
       @FileNumber            INT,
       @EFTGeneratorLocation  VARCHAR(500))
      
AS
/*********************************************************************************************
 *
 * [dbo].ky_NETUpdateEFTDetails
 * This stored procedure is used to update EFT Details
 *
 * Modification History
 *
 * 2017-10-03   John           Created v2, added EFT Generator Location
 *
 *********************************************************************************************/
BEGIN
SET NOCOUNT ON

IF EXISTS
  (SELECT TOP 1 1
     FROM [dbo].[EFTDetails] )
  BEGIN
    UPDATE
      dbo.[EFTDetails]
    SET
      EFTYN=@EFTYN,
      EFTLimit=@EFTLimit,
      EFTTransactionLimit=@NoTransaction,
      BankFileName=@BankFileName,
      Location=@FileLocation,
      FileNumber=@FileNumber	
 END
ELSE
 BEGIN
    INSERT INTO
       dbo.[EFTDetails]
       (EFTYN,
       EFTLimit,
       EFTTransactionLimit,
       BankFileName,
       Location,
       FileNumber
       )
    VALUES
      (@EFTYN,
      @EFTLimit,
      @NoTransaction,
      @BankFileName,
      @FileLocation,
      0
      )
 END
 
	IF EXISTS (SELECT TOP 1 1 FROM [dbo].[Settings] WHERE [KeyName] = 'EFTGeneratorLocation')
	BEGIN
		UPDATE [dbo].[Settings]
		   SET [KeyValue] = @EFTGeneratorLocation
		 WHERE [KeyName] = 'EFTGeneratorLocation'
	END
	ELSE
	BEGIN
		INSERT INTO [dbo].[Settings] (
		       [KeyName],
		       [KeyValue])
	    SELECT 'EFTGeneratorLocation',
	           @EFTGeneratorLocation
	END
SET NOCOUNT OFF
END
GO

IF OBJECT_ID(N'[ky_NETFetchEFTSent]',N'P')IS NOT NULL
	DROP PROCEDURE [ky_NETFetchEFTSent]
GO

CREATE PROCEDURE [dbo].[ky_NETFetchEFTSent] (
       @SEQNO AS INT)
AS
/*********************************************************************************************
 *
 * Stored Procedure Name: [dbo].[ky_NETFetchEFTSent]
 *
 * Description:  This procedure is used to FETCH a given [dbo].[EFTSent] record
 * Modification History: 
 *
 * 2017-10-03   John        Created
 *
*********************************************************************************************/
BEGIN   
SET NOCOUNT ON

     SELECT EFTS.[SequenceNo],
            EFTS.[EFTXML]
       FROM [dbo].[EFTSent] AS EFTS
      WHERE EFTS.[SequenceNo] = @SEQNO 
          
SET NOCOUNT OFF
END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] [SO]
          LEFT OUTER JOIN SYS.[columns] [SC]
                       ON [SC].[object_id] = [SO].[object_id]
                      AND [SC].[name] = 'HandlerCode'
    WHERE [SO].[name] = 'BatchH'
      AND [SO].[type] = 'U'
      AND [SC].[column_id] IS NULL)
  BEGIN
    EXEC ('
    ALTER TABLE
          [dbo].[BatchH]
      ADD [HandlerCode] VARCHAR(10) NULL')
  END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] [SO]
    WHERE [SO].[name] = 'BatchH'
      AND [SO].[type] = 'U')
  BEGIN
    DECLARE @COMMAND VARCHAR(MAX)

    SET @COMMAND = '
DECLARE @BATCHES TABLE
       ([BATCHNO]                 INT                                  PRIMARY KEY,
        [USERCODE]                VARCHAR(8)                           NOT NULL,
        [HANDLERCODE]             VARCHAR(10)                          NOT NULL,
        [PROCESSED]               BIT                                  NOT NULL,
        UNIQUE NONCLUSTERED ([PROCESSED], [BATCHNO]))

INSERT
  INTO @BATCHES
      ([BATCHNO],
       [USERCODE],
       [HANDLERCODE],
       [PROCESSED])
SELECT [BAH].[BATCHNO],
       RTRIM(ISNULL([BAH].[USERCODE], '''')),
       '''',
       0
  FROM [dbo].[BatchH] [BAH]
 WHERE [BAH].[HandlerCode] IS NULL

UPDATE [BH]
   SET [BH].[HANDLERCODE] = [HAN].[CODE],
       [BH].[PROCESSED] = 1
  FROM @BATCHES [BH]
       INNER JOIN [dbo].[Handlers] [HAN]
               ON [HAN].[LOGON] = [BH].[USERCODE]
              AND RTRIM(ISNULL([HAN].[CODE], '''')) <> ''''
              AND RTRIM(ISNULL([HAN].[LOGON], '''')) <> ''''
 WHERE [BH].[PROCESSED]  = 0

UPDATE [BH]
   SET [BH].[HANDLERCODE] = [HAN].[CODE],
       [BH].[PROCESSED] = 1
  FROM @BATCHES [BH]
       INNER JOIN [dbo].[Handlers] [HAN]
               ON SUBSTRING([HAN].[LOGON], 1, 8) = [BH].[USERCODE]
              AND RTRIM(ISNULL([HAN].[CODE], '''')) <> ''''
              AND RTRIM(ISNULL([HAN].[LOGON], '''')) <> ''''
 WHERE [BH].[PROCESSED]  = 0

UPDATE [BH]
   SET [BH].[HANDLERCODE] = ''ADM'',
       [BH].[PROCESSED] = 1
  FROM @BATCHES [BH]
 WHERE [BH].[PROCESSED]  = 0
   AND (   [BH].[USERCODE] = ''keyhouse''
        OR RTRIM(ISNULL([BH].[USERCODE], '''')) = '''')

UPDATE [BH]
   SET [BH].[HANDLERCODE] = [HAN].[CODE],
       [BH].[PROCESSED] = 1
  FROM @BATCHES [BH]
       INNER JOIN [dbo].[Handlers] [HAN]
               ON [HAN].[CODE] = [BH].[USERCODE]
              AND RTRIM(ISNULL([HAN].[CODE], '''')) <> ''''
 WHERE [BH].[PROCESSED]  = 0

UPDATE [BH]
   SET [BH].[HANDLERCODE] = [HAN].[CODE],
       [BH].[PROCESSED] = 1
  FROM @BATCHES [BH]
       INNER JOIN [dbo].[Handlers] [HAN]
               ON SUBSTRING([HAN].[CODE], 1, 8) = [BH].[USERCODE]
              AND RTRIM(ISNULL([HAN].[CODE], '''')) <> ''''
 WHERE [BH].[PROCESSED]  = 0

UPDATE [BH]
   SET [BH].[HANDLERCODE] = [HAN].[CODE],
       [BH].[PROCESSED] = 1
  FROM @BATCHES [BH]
       INNER JOIN [dbo].[Handlers] [HAN]
               ON SUBSTRING([HAN].[NAME], 1, 8) = REPLACE([BH].[USERCODE], ''.'', '' '')
              AND RTRIM(ISNULL([HAN].[CODE], '''')) <> ''''
 WHERE [BH].[PROCESSED]  = 0

UPDATE [BH]
   SET [BH].[HANDLERCODE] = ''ADM'',
       [BH].[PROCESSED] = 1
  FROM @BATCHES [BH]
 WHERE [PROCESSED]  = 0

UPDATE [BAH]
   SET [BAH].[HANDLERCODE] = [BH].[HANDLERCODE]
  FROM @BATCHES [BH]
       INNER JOIN [dbo].[BatchH] [BAH]
               ON [BAH].[BATCHNO] = [BH].[BATCHNO]
 WHERE [BH].[PROCESSED] = 1'

    EXEC (@COMMAND)
  END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] SO
    WHERE SO.[name] = N'ky_NETTRAddOrUpdateBatchH'
      AND SO.[type] = 'TR')
  BEGIN
    DROP TRIGGER [dbo].[ky_NETTRAddOrUpdateBatchH]
  END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] [SO]
    WHERE [SO].[name] = 'BatchH'
      AND [SO].[type] = 'U')
  BEGIN
    DECLARE @COMMAND VARCHAR(MAX)

    SET @COMMAND = '
CREATE TRIGGER
       [dbo].[ky_NETTRAddOrUpdateBatchH]
    ON [dbo].[BatchH]
   FOR INSERT, UPDATE
AS
/*********************************************************************************************
 *
 * [dbo].[ky_NETTRAddOrUpdateBatchH]
 * Makes sure that USERCODE (old Clarion software, LOGON) and HandlerCode
 *                           framework and new software, CODE) are consistent
 * Modification History
 * 2017-10-05 Pino Carafa               Created
 * 2019-03-19 Pino Carafa               Azure compatibility
 * 2019-10-09 John Ginnane              KEYD-6197 - Replaced sysprocesses code with function
 * 2019-12-05 Johh Ginnane              KEYD-6319 - Reverted code because function always returned ADM
 *                                      which is not ideal
 *
 *********************************************************************************************/
IF ((SELECT TRIGGER_NESTLEVEL()) < 2)
BEGIN
  DECLARE @Handler VARCHAR(10)
  DECLARE @Logon VARCHAR(10)

  SELECT @Handler = CONVERT(VARCHAR(10),
                            CASE WHEN CHARINDEX(CHAR(0),
                                                CONVERT(VARCHAR(MAX),
                                                        [P].[context_info])) > 0
                                      THEN SUBSTRING(CONVERT(VARCHAR(MAX),
                                                             [P].[context_info]),
                                                     1,
                                                     CHARINDEX(CHAR(0),
                                                               CONVERT(VARCHAR(MAX),
                                                               [P].[context_info])) - 1)
                                 ELSE CONVERT(VARCHAR(MAX),
                                              [P].[context_info]) END)
    FROM [sysprocesses] P
   WHERE P.[spid] = @@SPID

  IF RTRIM(ISNULL(@Handler, '''')) <> ''''
    BEGIN
      SELECT @Logon = RTRIM(ISNULL([HAN].[Logon], ''''))
        FROM [dbo].[Handlers] [HAN]
       WHERE [HAN].[CODE] = @Handler

      SET @Logon = RTRIM(SUBSTRING(ISNULL(@Logon, ''''), 1, 8))

      IF EXISTS
        (SELECT TOP 1 1
           FROM inserted [INS]
          WHERE [INS].[HandlerCode] IS NULL)
        BEGIN
          UPDATE [BAH]
             SET [BAH].[USERCODE] = @Logon,
                 [BAH].[HandlerCode] = @Handler
            FROM inserted [INS]
                 INNER JOIN [dbo].[BatchH] [BAH]
                         ON [BAH].[BATCHNO] = [INS].[BATCHNO]
           WHERE [INS].[HandlerCode] IS NULL
        END
    END

  IF EXISTS
    (SELECT TOP 1 1
       FROM inserted [INS]
      WHERE [INS].[HandlerCode] IS NULL
        AND RTRIM(ISNULL(@Handler, '''')) = '''')
    BEGIN
      DECLARE @BATCHES TABLE
             ([BATCHNO]                 INT                                  PRIMARY KEY,
              [USERCODE]                VARCHAR(8)                           NOT NULL,
              [HANDLERCODE]             VARCHAR(10)                          NOT NULL,
              [PROCESSED]               BIT                                  NOT NULL,
              UNIQUE NONCLUSTERED ([PROCESSED], [BATCHNO]))

      INSERT
        INTO @BATCHES
            ([BATCHNO],
             [USERCODE],
             [HANDLERCODE],
             [PROCESSED])
      SELECT [BAH].[BATCHNO],
             RTRIM(ISNULL([BAH].[USERCODE], '''')),
             '''',
             0
        FROM inserted [BAH]
       WHERE [BAH].[HandlerCode] IS NULL

      UPDATE [BH]
         SET [BH].[HANDLERCODE] = [HAN].[CODE],
             [BH].[PROCESSED] = 1
        FROM @BATCHES [BH]
             INNER JOIN [dbo].[Handlers] [HAN]
                     ON [HAN].[LOGON] = [BH].[USERCODE]
                    AND RTRIM(ISNULL([HAN].[CODE], '''')) <> ''''
                    AND RTRIM(ISNULL([HAN].[LOGON], '''')) <> ''''
       WHERE [BH].[PROCESSED]  = 0

      UPDATE [BH]
         SET [BH].[HANDLERCODE] = [HAN].[CODE],
             [BH].[PROCESSED] = 1
        FROM @BATCHES [BH]
             INNER JOIN [dbo].[Handlers] [HAN]
                     ON SUBSTRING([HAN].[LOGON], 1, 8) = [BH].[USERCODE]
                    AND RTRIM(ISNULL([HAN].[CODE], '''')) <> ''''
                    AND RTRIM(ISNULL([HAN].[LOGON], '''')) <> ''''
       WHERE [BH].[PROCESSED]  = 0

      UPDATE [BH]
         SET [BH].[HANDLERCODE] = ''ADM'',
             [BH].[PROCESSED] = 1
        FROM @BATCHES [BH]
       WHERE [BH].[PROCESSED]  = 0
         AND (   [BH].[USERCODE] = ''keyhouse''
              OR RTRIM(ISNULL([BH].[USERCODE], '''')) = '''')

      UPDATE [BH]
         SET [BH].[HANDLERCODE] = [HAN].[CODE],
             [BH].[PROCESSED] = 1
        FROM @BATCHES [BH]
             INNER JOIN [dbo].[Handlers] [HAN]
                     ON [HAN].[CODE] = [BH].[USERCODE]
                    AND RTRIM(ISNULL([HAN].[CODE], '''')) <> ''''
       WHERE [BH].[PROCESSED]  = 0

      UPDATE [BH]
         SET [BH].[HANDLERCODE] = [HAN].[CODE],
             [BH].[PROCESSED] = 1
        FROM @BATCHES [BH]
             INNER JOIN [dbo].[Handlers] [HAN]
                     ON SUBSTRING([HAN].[CODE], 1, 8) = [BH].[USERCODE]
                    AND RTRIM(ISNULL([HAN].[CODE], '''')) <> ''''
       WHERE [BH].[PROCESSED]  = 0

      UPDATE [BH]
         SET [BH].[HANDLERCODE] = [HAN].[CODE],
             [BH].[PROCESSED] = 1
        FROM @BATCHES [BH]
             INNER JOIN [dbo].[Handlers] [HAN]
                     ON SUBSTRING([HAN].[NAME], 1, 8) = REPLACE([BH].[USERCODE], ''.'', '' '')
                    AND RTRIM(ISNULL([HAN].[CODE], '''')) <> ''''
       WHERE [BH].[PROCESSED]  = 0

      UPDATE [BH]
         SET [BH].[HANDLERCODE] = ''ADM'',
             [BH].[PROCESSED] = 1
        FROM @BATCHES [BH]
       WHERE [PROCESSED]  = 0

      UPDATE [BAH]
         SET [BAH].[HANDLERCODE] = [BH].[HANDLERCODE]
        FROM @BATCHES [BH]
             INNER JOIN [dbo].[BatchH] [BAH]
                     ON [BAH].[BATCHNO] = [BH].[BATCHNO]
       WHERE [BH].[PROCESSED] = 1
    END

UPDATE [BAH]
   SET [BAH].[USERCODE] = RTRIM(SUBSTRING(ISNULL([HAN].[LOGON], ''''), 1, 8))
  FROM inserted [INS]
       INNER JOIN [dbo].[BatchH] [BAH]
               ON [BAH].[BATCHNO] = [INS].[BATCHNO]
       INNER JOIN [dbo].[Handlers] [HAN]
               ON [HAN].[CODE] = [INS].[HandlerCode]
              AND RTRIM(SUBSTRING(ISNULL([HAN].[LOGON], ''''), 1, 8)) <> RTRIM(ISNULL([INS].[USERCODE], ''''))
END'
    EXEC (@COMMAND)
END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] [SO]
          LEFT OUTER JOIN SYS.[columns] [SC]
                       ON [SC].[object_id] = [SO].[object_id]
                      AND [SC].[name] = 'EnteredHandlerCode'
    WHERE [SO].[name] = 'Transactions'
      AND [SO].[type] = 'U'
      AND [SC].[column_id] IS NULL)
  BEGIN
    EXEC ('
    ALTER TABLE
          [dbo].[Transactions]
      ADD [EnteredHandlerCode] VARCHAR(10) NULL')
  END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] [SO]
          LEFT OUTER JOIN SYS.[columns] [SC]
                       ON [SC].[object_id] = [SO].[object_id]
                      AND [SC].[name] = 'UpdatedHandlerCode'
    WHERE [SO].[name] = 'Transactions'
      AND [SO].[type] = 'U'
      AND [SC].[column_id] IS NULL)
  BEGIN
    EXEC ('
    ALTER TABLE
          [dbo].[Transactions]
      ADD [UpdatedHandlerCode] VARCHAR(10) NULL')
  END
GO

IF EXISTS
  (SELECT TOP 1 1
     FROM SYS.[objects] [SO]
    WHERE [SO].[name] = 'Transactions'
      AND [SO].[type] = 'U')
  BEGIN
    DECLARE @COMMAND VARCHAR(MAX)

    SET @COMMAND = '
DECLARE @BATCHES TABLE
       ([XnID]                    INT                                  PRIMARY KEY,
        [EnteredBy]               VARCHAR(8)                           NOT NULL,
        [EnteredHandlerCode]      VARCHAR(10)                          NOT NULL,
        [PROCESSED]               BIT                                  NOT NULL,
        UNIQUE NONCLUSTERED ([PROCESSED], [XnID]))

INSERT
  INTO @BATCHES
      ([XnID],
       [EnteredBy],
       [EnteredHandlerCode],
       [PROCESSED])
SELECT [BAH].[XnID],
       RTRIM(ISNULL([BAH].[EnteredBy], '''')),
       '''',
       0
  FROM [dbo].[Transactions] [BAH]
 WHERE [BAH].[EnteredHandlerCode] IS NULL

UPDATE [BH]
   SET [BH].[EnteredHandlerCode] = [HAN].[CODE],
       [BH].[PROCESSED] = 1
  FROM @BATCHES [BH]
       INNER JOIN [dbo].[Handlers] [HAN]
               ON [HAN].[LOGON] = [BH].[EnteredBy]
              AND RTRIM(ISNULL([HAN].[CODE], '''')) <> ''''
              AND RTRIM(ISNULL([HAN].[LOGON], '''')) <> ''''
 WHERE [BH].[PROCESSED]  = 0

UPDATE [BH]
   SET [BH].[EnteredHandlerCode] = [HAN].[CODE],
       [BH].[PROCESSED] = 1
  FROM @BATCHES [BH]
       INNER JOIN [dbo].[Handlers] [HAN]
               ON SUBSTRING([HAN].[LOGON], 1, 8) = [BH].[EnteredBy]
              AND RTRIM(ISNULL([HAN].[CODE], '''')) <> ''''
              AND RTRIM(ISNULL([HAN].[LOGON], '''')) <> ''''
 WHERE [BH].[PROCESSED]  = 0

UPDATE [BH]
   SET [BH].[EnteredHandlerCode] = ''ADM'',
       [BH].[PROCESSED] = 1
  FROM @BATCHES [BH]
 WHERE [BH].[PROCESSED]  = 0
   AND (   [BH].[EnteredBy] = ''keyhouse''
        OR RTRIM(ISNULL([BH].[EnteredBy], '''')) = '''')

UPDATE [BH]
   SET [BH].[EnteredHandlerCode] = [HAN].[CODE],
       [BH].[PROCESSED] = 1
  FROM @BATCHES [BH]
       INNER JOIN [dbo].[Handlers] [HAN]
               ON [HAN].[CODE] = [BH].[EnteredBy]
              AND RTRIM(ISNULL([HAN].[CODE], '''')) <> ''''
 WHERE [BH].[PROCESSED]  = 0

UPDATE [BH]
   SET [BH].[EnteredHandlerCode] = [HAN].[CODE],
       [BH].[PROCESSED] = 1
  FROM @BATCHES [BH]
       INNER JOIN [dbo].[Handlers] [HAN]
               ON SUBSTRING([HAN].[CODE], 1, 8) = [BH].[EnteredBy]
              AND RTRIM(ISNULL([HAN].[CODE], '''')) <> ''''
 WHERE [BH].[PROCESSED]  = 0

UPDATE [BH]
   SET [BH].[EnteredHandlerCode] = [HAN].[CODE],
       [BH].[PROCESSED] = 1
  FROM @BATCHES [BH]
       INNER JOIN [dbo].[Handlers] [HAN]
               ON SUBSTRING([HAN].[NAME